/*************************************************************************
* Copyright (C) 2021 Intel Corporation
*
* Licensed under the Apache License,  Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* 	http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law  or agreed  to  in  writing,  software
* distributed under  the License  is  distributed  on  an  "AS IS"  BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the  specific  language  governing  permissions  and
* limitations under the License.
*************************************************************************/

#ifndef IFMA_ECPRECOMP7_P256_H
#define IFMA_ECPRECOMP7_P256_H

#include "owndefs.h"

#if (_IPP32E >= _IPP32E_K1)

#include "gfpec/pcpgfpecstuff.h"

#include "gfpec/ecnist/ifma_defs.h"
#include "gfpec/ecnist/ifma_ecpoint_p256.h"


#define BASE_POINT_WIN_SIZE (7)
#define BASE_POINT_N_ENTRY  (1 << ((BASE_POINT_WIN_SIZE)-1))

#define OPERAND_BITSIZE (256)
#define LEN52_P256      (NUMBER_OF_DIGITS(OPERAND_BITSIZE, DIGIT_SIZE))

/* P256 affine point */
typedef struct {
    BNU_CHUNK_T X[LEN52_P256];
    BNU_CHUNK_T Y[LEN52_P256];
} P256_POINT_AFFINE_IFMA_MEM;

extern const __ALIGN64 P256_POINT_AFFINE_IFMA_MEM
    ifma_ec_nistp256r1_bp_precomp[37][BASE_POINT_N_ENTRY];

#if !defined(_DISABLE_ECP_256R1_HARDCODED_BP_TBL_)

/* clang-format off */
/* Montgomery coefficient R = 2^(6*52) mod p */
const __ALIGN64 P256_POINT_AFFINE_IFMA_MEM ifma_ec_nistp256r1_bp_precomp[][BASE_POINT_N_ENTRY] = {
    {
        /* digit=0 (1,2,..,64)*(2^{0})*G */
        {{ 0x0008905f76bdc7b5, 0x0007bd418a9143c1, 0x000c475d568abc1f, 0x0009a701075ba95f, 0x00003d1f32c8b4b9 }, { 0x000571ff18aafa5c, 0x000f757ce95560a8, 0x000434a7e54432f7, 0x0002797258b4ab8e, 0x00009df870e37032 }},
        {{ 0x0006bb32e5348a6d, 0x000d9d410ddd64df, 0x0000ad784f985075, 0x0008a23d9aa6ae3c, 0x00001fb0f13f1e58 }, { 0x0008c57751832101, 0x00047d361bee1a57, 0x000b725df8a6ac15, 0x000e32cbe152cd7c, 0x00008c240484bd0e }},
        {{ 0x0006936a3fdd92de, 0x00061904eebc1272, 0x0009e1ea17bc2219, 0x00038de98b027f84, 0x0000be1daceb9daa }, { 0x0005f06a2abb7836, 0x000261fc983a7ebd, 0x000c327193eff4d4, 0x000b6b38e583e47a, 0x0000315e09d4065e }},
        {{ 0x00077362f599237a, 0x0003b0d46918dcc5, 0x000d6eb05e7ddb8d, 0x000ec90f24650a6e, 0x0000604e8ac3b74e }, { 0x0004b14c39d10c04, 0x000ee4ce4cbaba68, 0x00017625a80d5c8a, 0x000d1ce2e1762847, 0x00009cb7c6eea514 }},
        {{ 0x0009079606520cb9, 0x000d1aec45c61f5c, 0x0009cbb1bd776c0e, 0x0006a090c90ec649, 0x0000ce1d21d8a47e }, { 0x0003a076bba1725f, 0x0003c4ae7ba4f107, 0x000f724b117049be, 0x00007c06873c568e, 0x0000d1631291cbdd }},
        {{ 0x0003074b59672881, 0x000c6373e77664a1, 0x000f2165a2e4d910, 0x000ef22ad55ae744, 0x0000cd292bcbc0f3 }, { 0x0004c9916deed188, 0x000da20d377e44ba, 0x000309358347a501, 0x000921711299c2b5, 0x0000cd3e2ccadf00 }},
        {{ 0x0003ba5119d6cc8f, 0x000a64ea0173b4f1, 0x0003be81afdd3079, 0x000572c082bd2021, 0x00001db750e89b35 }, { 0x000aedd9165911e1, 0x000ef303f5b9d4de, 0x000172b8a2c6cf35, 0x000b760956742f2f, 0x00007d5db743c61e }},
        {{ 0x000763891f69c8ac, 0x000a0d19499a78fb, 0x0004b837ab35be28, 0x00064506b462ab5c, 0x00002c41f6130b86 }, { 0x0006ec12f2cdf2f2, 0x000b1a9532d49775, 0x000d78b2a72327aa, 0x0006dc9f021e3327, 0x00006889435a91f0 }},
        {{ 0x00005b3081d12011, 0x0005d8f264e20e8e, 0x000c794c77bfa4a9, 0x000a8da00abe6bfe, 0x0000ec1d857c8273 }, { 0x00086659ce05e9b9, 0x000be7aa45f33140, 0x000dc5f6ec1518cd, 0x000c7761a56af7be, 0x00006928f160cc82 }},
        {{ 0x000c61c947f1150b, 0x000d0d19dc21ec8c, 0x0009436db7f13b03, 0x00010004998f9868, 0x0000d698306e8e57 }, { 0x0008719a5563ace5, 0x0005b7880dd9e767, 0x0000bfce23b2a87e, 0x000ef00d2c43a899, 0x00000a4ab8781013 }},
        {{ 0x000e7090f1a30d21, 0x000050a6245e4043, 0x0005300f4eef4770, 0x0009ef9b59de4079, 0x00006484903423ea }, { 0x000893002449b8b8, 0x000cd612b944e886, 0x000e7cec61a3d0bd, 0x000f4ac3d250f939, 0x0000b1ed336264d4 }},
        {{ 0x000ee1df781a65c0, 0x0005aa7d26977684, 0x00057b527abaea51, 0x000d1785eabdedef, 0x0000cdef8db325d3 }, { 0x000ff57c95d31907, 0x0000bd91cbb5b437, 0x00064a93d9a13993, 0x000c56666170ed2f, 0x00005f7b461764b7 }},
        {{ 0x00038477acc71a2f, 0x000f6634b2ed7097, 0x000911eb5b6105a9, 0x0008469110e35676, 0x00000610528b928e }, { 0x000bc0876ac5a83f, 0x000ec90c00ee17ff, 0x000f786e4b786fcc, 0x000771cc16874838, 0x00000146b81bb7f7 }},
        {{ 0x000aae7333f7c13a, 0x000695bb0cf664af, 0x000684e894afaa81, 0x000c1ec60126e48f, 0x000061142a8620ce }, { 0x00052706dc43d122, 0x0000bb7995d586b1, 0x000e6dfb30e5e3e4, 0x00094c5dbbe29569, 0x000098178e3461e6 }},
        {{ 0x0002aa0e43ed7de8, 0x000f0b7bc60055be, 0x000d74387007853e, 0x00003ab7003cc23e, 0x00004cf9074de0f7 }, { 0x000042170a90bbc4, 0x0008e4f6383c45d2, 0x00042262041aaffc, 0x000ce8397d766355, 0x0000fed56953d3e6 }},
        {{ 0x0004d63c80e5d9f3, 0x00018650bc6fb805, 0x0004eb27f1ea9ab1, 0x000aa02495882e07, 0x0000a466f2e5fb46 }, { 0x0004edf7b266ddd9, 0x00042d652a23f9b6, 0x0008e61d6dd58c13, 0x0007a359e3670c31, 0x0000d257b443894a }},
        {{ 0x0001a35c0b2e1db3, 0x0006acbd53c5c9dc, 0x0005ea66bb7068eb, 0x000e07afff17624b, 0x0000591d6b9069fe }, { 0x000bd512271e49d5, 0x0005c4693e79987e, 0x000ffcc7a5e70433, 0x000ceaebb0575bf2, 0x0000f661c2e757ee }},
        {{ 0x00075abd1bd6dba8, 0x000a9c856ada97a6, 0x000d283083228862, 0x0005281794afc964, 0x0000f61673328f6a }, { 0x000c6fc7d9717201, 0x000bc725e485a1d6, 0x000f22df4362f530, 0x000127aba4b2a5ca, 0x0000262b28cba5aa }},
        {{ 0x000e545c2926ec20, 0x000f2baa5659ae8f, 0x0009278d4545d7bc, 0x0004021ac68363ab, 0x000024e11be64e4e }, { 0x000720ee26643347, 0x000a2e7dc4c696bf, 0x000a0488899ec903, 0x00061c1c15806244, 0x0000a0f278b7757f }},
        {{ 0x0002d1eeb7e12dea, 0x000a729d2337a31e, 0x000e2cb43d24a969, 0x000a499ba5916868, 0x00000502dc95217c }, { 0x000ee104e224dfe5, 0x00037417884005d5, 0x0007a11ec6cc5e0f, 0x000c97966a2d4ec1, 0x0000b25760a55d65 }},
        {{ 0x000cead9f5ae523c, 0x00010ecc135b208e, 0x00045a991a87a7dc, 0x000ecff3074e1b26, 0x0000f12e91b276ab }, { 0x000086d5f289ce55, 0x0007fb2e286e5b9c, 0x000fc334666f68cf, 0x000ddd4a41b0fadb, 0x00009543bc814bf5 }},
        {{ 0x000dc777e9e6fad0, 0x00041352f6f3076f, 0x0002e5c9b2b028cb, 0x000fdea3dc99ffa2, 0x00008d533e5e9beb }, { 0x00093e8d07d69b5d, 0x0002abb35415f047, 0x0009d3e5719cbe0c, 0x0009ed8b5640c2d6, 0x00007bfb45e7d23e }},
        {{ 0x000f2cea7cc654a9, 0x000cd532de45068a, 0x0007ef0134f22ea3, 0x00003976937c7a7e, 0x0000bee7aa1e25dd }, { 0x0003d0687c1447ce, 0x000d6fb9e4785a98, 0x000ba35936cdef18, 0x000646171e5fda49, 0x0000ead87df3975d }},
        {{ 0x00097b2330287754, 0x0002fb215d02819f, 0x000f4762770544c1, 0x000b3f29f6d0d754, 0x00004490d3fcc8f4 }, { 0x00095dc55f23d140, 0x0009750b24bcbc75, 0x00092f441df78180, 0x000fa350111ea494, 0x0000e940dd6ad26f }},
        {{ 0x00066742ebfbaa6d, 0x000acacd837879f9, 0x000e570424c4a8f8, 0x0004786af6fc1b49, 0x00001a48d8c34737 }, { 0x000b948dc3d28c27, 0x000a2feb49662287, 0x0001c85fab8c7901, 0x0002b0aee96cc631, 0x00002b68fe2195f7 }},
        {{ 0x00094baaa8ef3ea5, 0x0000832ed1f80089, 0x000a3f65ca8f72a6, 0x0004151fbe6c9267, 0x0000b870bac04296 }, { 0x000d178031be45da, 0x0003ae11d248018e, 0x000a757344d6c689, 0x000636af55a39898, 0x0000f7e62c4349a1 }},
        {{ 0x000f8aa54bce826c, 0x000bef497e2feb49, 0x00070f34cdc735a9, 0x000cd91d7d36a42d, 0x0000b578ca909f39 }, { 0x000d431069058eee, 0x000c9ba09895e702, 0x000fb074eaef8c71, 0x0005c5ebb3bd0c66, 0x00009a05e3ae9f73 }},
        {{ 0x00097f6b6bdc2a47, 0x000f7b7172ccd1fe, 0x000747273d266f27, 0x000b70ceb7323cb2, 0x00000289b30a54f3 }, { 0x0009b4cc5fea0d1e, 0x0001b3ed1a83da1a, 0x0006d6965e083330, 0x000ff991ea6a7f9c, 0x0000d78ac0dce842 }},
        {{ 0x000266f95a2ac736, 0x0007e6acb66e132e, 0x00092b2b5f263396, 0x0005083e2f1be963, 0x0000a1dcaaacccd8 }, { 0x000798142a61d3f6, 0x00070735c2088993, 0x00076b6304793707, 0x000a86b9f9047244, 0x000045b4d24b0f15 }},
        {{ 0x0000288cb622df88, 0x00094b13298b343d, 0x0008e426d8eb9144, 0x000e7f2cda3a14e4, 0x0000f218abd21717 }, { 0x00060d45ce976d65, 0x000272f1c040abc4, 0x000dc0b2557c3e5f, 0x000cd3f41b675511, 0x00004127c7f96903 }},
        {{ 0x00048dc4f9468a4c, 0x000f0696755ff899, 0x0006b2a252ed9c6c, 0x0006d140ddd3cf7e, 0x0000ad40f1efa3f6 }, { 0x00038ae89241062f, 0x000cf814ea53299f, 0x0001a560779698e3, 0x00091d4fc2d921ca, 0x0000d74868346acb }},
        {{ 0x0009227077a277a3, 0x00059cfc5e3a3d83, 0x000c07576f48364e, 0x000355e97befd904, 0x00001c25c29ce15c }, { 0x000231d792f7c39a, 0x000a46ddbef42f56, 0x0003e5d8c175121e, 0x000d694407e449b6, 0x00003711285e87b5 }},
        {{ 0x0003ec6869243119, 0x000fb560f6645343, 0x000c177c889a62e3, 0x0002671b985ceae7, 0x0000759c4e754541 }, { 0x00016c4304749ba6, 0x000dd2b3a8ec1f1b, 0x0007a6db92eaf628, 0x000a36f745509d12, 0x0000c8bc38c34b01 }},
        {{ 0x000982ddd5a6b31c, 0x00004147ca23cd3c, 0x0005da0ea6c80ff8, 0x000b47194caa7a5c, 0x0000393836b1cdce }, { 0x000159100cf86685, 0x0007b0b5db6f971a, 0x000ece2e5f860514, 0x000f00f26bba2cb1, 0x0000051e80a79643 }},
        {{ 0x000a640b870d07f0, 0x000947fdec67ef5c, 0x000358da6b30f00e, 0x000bac4c8742ee46, 0x0000a74e3225ade2 }, { 0x000ba90162b75d7c, 0x00085908ceb6aa92, 0x000c28cf5172addc, 0x000c5fef7313c300, 0x00005ab09925afb3 }},
        {{ 0x000569656d0ab3a3, 0x000b5d6a8219bb73, 0x000ce59d601232aa, 0x000a3b46ec691d0b, 0x00002fe02f08a3c7 }, { 0x00080b9e12257943, 0x0006bbd94cda03a4, 0x0002ce206001c072, 0x000d85dc6934e82d, 0x000038227865a8bc }},
        {{ 0x000763a388070738, 0x00043c58cce08b52, 0x000c57b3e9ccf93f, 0x00070a006b636458, 0x000073a51820269f }, { 0x0008b36143bab3e9, 0x0005b8a7d3cebb65, 0x000e166d770ce677, 0x000a45d2ce7adda3, 0x0000ec583f44a15a }},
        {{ 0x000df0c7dd2a5105, 0x000beca4d63c0eea, 0x000661f880f15fdb, 0x000ddee3b5123311, 0x000048f0c43aa7f7 }, { 0x0009a4c281762585, 0x000a83bfe7765e5d, 0x0008e5934187989d, 0x00033ac9cccaab35, 0x0000ce5a2b475518 }},
        {{ 0x000853e4c48e856b, 0x0002988a059c1425, 0x0004a6b7ae821b53, 0x00075cb646f5ae71, 0x0000703cc0a9b888 }, { 0x000c0bc0e57524e4, 0x000992363c52f980, 0x00067576c7916db3, 0x00029971c2ec4a76, 0x0000edb0d188156c }},
        {{ 0x00062d7b00bc24ad, 0x000bb1db82e9f3e4, 0x000b98f8663c28b2, 0x000cdc58083200f0, 0x000027ff00ca4baa }, { 0x0003466612d4f0d6, 0x000cb88f26517980, 0x000c3364b622041a, 0x000046d5974c6e18, 0x0000427ccfde908a }},
        {{ 0x0008100a5073116b, 0x000f5492ed22e919, 0x0004d6f71dd1ef3b, 0x000bcb7716fdfe0b, 0x00009fc9cb71c4c6 }, { 0x00070bfeecfa9449, 0x00091a2a35c628ff, 0x000dd1729b5abc1c, 0x000c390ceb6f94d2, 0x000067c9ff7c6c22 }},
        {{ 0x00049c89cec09788, 0x000d4cfed2bad016, 0x0002b90a1bf89f1b, 0x00011a78fa155cbe, 0x0000121edd577d70 }, { 0x0003a8cfe47d654d, 0x000197de9ff257af, 0x0000ff8380ca53f3, 0x000fac7cf9b35961, 0x000020d4bd2f2712 }},
        {{ 0x00079c85db554c9f, 0x000a0fd248a7d061, 0x00046febee1efaa5, 0x00027ba8180c5bfb, 0x00006ff589c6b4b0 }, { 0x0007795f4fdd2c3c, 0x000746261a6966c4, 0x0002a3638658dd15, 0x000ba67184d82d05, 0x0000f03444a9ebc8 }},
        {{ 0x000f3425f708c1f8, 0x000841fd55a897c5, 0x000b535ceb04e43b, 0x000bfdbb0009194a, 0x000088089bf24fda }, { 0x000b629f9123de43, 0x0002387730d50dae, 0x000b5fa59bee9bc2, 0x000bef1170423446, 0x0000ba71f5c7ecb2 }},
        {{ 0x00094f7d3556d1af, 0x00077ccd5cd79bfe, 0x000aab1cf0e2c812, 0x00075d0580032940, 0x0000ac88a0bcab00 }, { 0x00073acbff0414b2, 0x0001996782ba21ad, 0x00091ff067a6c333, 0x000e35a81c525446, 0x0000b3a408dcdfc3 }},
        {{ 0x000ec684c472efb6, 0x000ffb149ee90d9c, 0x0009d1a819f5cd85, 0x000086fe0785c339, 0x0000a0452c3e9739 }, { 0x00033da338c4f59b, 0x000c7c9c1586e630, 0x000657769bf27db5, 0x00001e85d45431d6, 0x00009e1980aa8791 }},
        {{ 0x000b969975373e25, 0x0005013caa760974, 0x0008eb319f90f0fe, 0x0001ccdc70a624fa, 0x00007256c83b773a }, { 0x000bbff86f099df5, 0x00060ccd921d60e9, 0x000fdfeed1a41869, 0x0004f9ed49b944e0, 0x000095725f99e254 }},
        {{ 0x00084039030f437c, 0x000c53077adc612c, 0x0008f397ca9d5bf0, 0x0004489bda749652, 0x0000af611eeda45e }, { 0x0005efc7e9c72224, 0x0007b66175adff18, 0x000b31e8fdcd72d3, 0x0007b51e30b26d7f, 0x0000abe9a851e48b }},
        {{ 0x000bdbe608fb93c1, 0x0004a421ff6acd3a, 0x0003a80083002601, 0x000e2a027ffe7048, 0x0000344f04aba92c }, { 0x000394077fc5dc01, 0x00091285e1109e8d, 0x000720d053dabda0, 0x000a58dd826147d2, 0x0000b53fc0ef1331 }},
        {{ 0x000dceb8db394d64, 0x0005ed0488c171a0, 0x000a0599999cd66c, 0x00064966aa78bfba, 0x000083197a8d8d9c }, { 0x0007d341959b35cd, 0x000b3899ae2b710d, 0x0005cc715b012468, 0x000de6d46efde7ae, 0x0000a80719fcd630 }},
        {{ 0x00020ebe0e2e4e62, 0x0008a1c3b3f64c9d, 0x00094c502131223e, 0x000f4b6d4e20bc0b, 0x000051365d8f7700 }, { 0x000731e3847a272f, 0x0001fa773241ea3d, 0x000b86e87be0bce4, 0x00040ed6261f1511, 0x0000b13c9b2db908 }},
        {{ 0x0008f77f3559dd64, 0x0008f6e834459048, 0x0009264e7a8accb5, 0x00097c4b72e0ec46, 0x000077614016cc6c }, { 0x000ef2f2271b0201, 0x0008c295864687c7, 0x000da3ec433bdb78, 0x000bcc29e23b92ea, 0x0000b3f836102480 }},
        {{ 0x000c3a318e1ca066, 0x0003038546c4d8de, 0x0003a6b814eb7a95, 0x0002846d95f9c3fc, 0x00003dc194c05f8c }, { 0x000ba1da5a3ee82c, 0x0005341c31cb264f, 0x000e567f3998412e, 0x000c7b5973db82f6, 0x0000dcf49c0e789d }},
        {{ 0x000b8a0feba9b0d8, 0x000838eb212cf53b, 0x000e3ccb6c73f691, 0x00013fd6c4f2e512, 0x00003caf4f380d7c }, { 0x000a556e4004eb04, 0x0006cd2e7d6107e1, 0x0004cd0cf76abda4, 0x0002e32b17ee0c44, 0x00002b28880d6418 }},
        {{ 0x000cee844a04a5b5, 0x00006f9e73056835, 0x000eab00d79c20e6, 0x000225fc131eea5b, 0x000026b5a23f0042 }, { 0x0005e801401734f5, 0x000a0c5a0182bde2, 0x0007783fa574d37f, 0x000b023caee759f8, 0x000034b41c5e2869 }},
        {{ 0x000a3f0c4b2e90ba, 0x00087407e8d7a14b, 0x0009db17e71f6133, 0x000cf4644bb1ff3c, 0x0000added5f139e5 }, { 0x000392768185fea2, 0x000205b1f9af32a8, 0x00062e5faaca59cc, 0x000b6f4ac01e0db8, 0x000089510687411e }},
        {{ 0x00013298312c0224, 0x0000ee77acaca28c, 0x00070605c2f812c6, 0x0001c8b381bfeea5, 0x0000fc71519eb2d1 }, { 0x000a44c6c6da463c, 0x000b6db2aac9e59d, 0x000cdc1cb163dec3, 0x000553e015748060, 0x0000ecc5ac9b6bed }},
        {{ 0x00005f725f94ec75, 0x0008be8a7ea82f03, 0x000db3ca28505d7f, 0x000eb16e199cac80, 0x00002d3afce656e9 }, { 0x00063749945c9e90, 0x0008bc4623fb21b0, 0x000e742bd018d74e, 0x000c16b3efa70533, 0x0000d56e129c9961 }},
        {{ 0x000c2bab1bd708fe, 0x00050d65c46aa8e4, 0x00035ca4b8b592cb, 0x000c87e362100d5d, 0x0000d356747acecb }, { 0x0000624998e75dc4, 0x000250c15426704a, 0x000307d5a251b927, 0x00086b3f8778afcd, 0x000023ad4d716895 }},
        {{ 0x00040c4cce907be5, 0x0008328ed8a6e19d, 0x00056d7d8cf94ba2, 0x000a054bc33fc233, 0x0000e2bedf2813d6 }, { 0x0004820109c066e9, 0x0004cbbf80e75ca0, 0x00001e2bfe22ef3a, 0x000eac49612c651a, 0x00003c7feeecb38e }},
        {{ 0x00094a44a734cd7a, 0x000711a7f4c04cc9, 0x00010ff60add019a, 0x000013b0a5955662, 0x0000a10d512bac4a }, { 0x000d1bc7809447a3, 0x0005519dbeb9b690, 0x0000e1b86d8d43b3, 0x000fd18a2473c568, 0x0000743d915fe645 }},
        {{ 0x000dbcb03af2f664, 0x0008d75a1962f910, 0x0001df9c456cf23f, 0x000933e336ed7e06, 0x00004dfcb458852d }, { 0x000b60cfffa10db0, 0x000924ebb32e38e6, 0x000efca5a01f42ce, 0x0003e39ee6472e5e, 0x00000912f4df98e0 }},
        {{ 0x000763a4348a5ffa, 0x000df319ff91fe50, 0x0000e8edeab92e73, 0x00022d126039c480, 0x0000c218bfcbfe07 }, { 0x0000876a01d11d05, 0x000ffd5383e76ba9, 0x000ff1c717959f5e, 0x0000608e0ac98b92, 0x000071d355178dc0 }},
        {{ 0x0005fb8394ec96bc, 0x000daf305968b809, 0x000272a3f024eed6, 0x0002f6201380a091, 0x00005bd44243eed6 }, { 0x000967204437eb69, 0x000fc1280edfe2f2, 0x000966478f3e6250, 0x00056e2f88f726bb, 0x0000b48e94ed5c40 }},
    },
    {
        /* digit=1 (1,2,..,64)*(2^{7})*G */
        {{ 0x000d311f2cf406c5, 0x00006817a2ad62aa, 0x0008ff31e0276fbf, 0x000a6d8b57ef2b6b, 0x0000437e1f6177bf }, { 0x000a38d321e63954, 0x0002c87c2ff3b6df, 0x00023c3f61ddd141, 0x0009d5d38b46feaf, 0x0000a9e4aaa74b8f }},
        {{ 0x000c3da1963d1b37, 0x00058fa696946fc7, 0x00083d8e03d70b52, 0x000d8231550fbc6d, 0x0000c23f1ad7a853 }, { 0x00099d49ebcaeaa8, 0x00055b03cfd5d8b3, 0x00024f00b2ba1b89, 0x000385daa0704b7c, 0x00003f881bb6342a }},
        {{ 0x000e4b32b147920e, 0x0003b6d6ec293cde, 0x0006a0251f43f078, 0x000b7060a733dbda, 0x0000a65cc513dc70 }, { 0x000af675f284aa5c, 0x000a1a9cae368d14, 0x0000b2ba3d012d1a, 0x00090686e5001a7b, 0x000048a7a97981a9 }},
        {{ 0x00042370d35642b3, 0x000ce10399492969, 0x000aa1fa40e1eea0, 0x0002bfcf55b63827, 0x00006300726619ca }, { 0x000dc61f92142820, 0x0008f2ee0d985a17, 0x000af47364dbdea2, 0x0003af2f8192cc64, 0x00009a126d8fb59a }},
        {{ 0x000e8aa3ab0b8178, 0x000835bc2da7de94, 0x000ec16e3013acaf, 0x000091e41d0e643d, 0x0000aceee54c0f0d }, { 0x000d296a148a8106, 0x0005f329a49110f0, 0x000620ad480c6f7a, 0x00042e45ad18317f, 0x00005efb94892cc8 }},
        {{ 0x0005d5bbfc230970, 0x000353eed4c37174, 0x000ef73916a712cd, 0x0004acaa59f8240c, 0x0000b069410109c2 }, { 0x0004d5e0fcf149d6, 0x000f673137fd28e3, 0x000f92655f0108ba, 0x000ab48f2862ac6e, 0x00009efcb3eef840 }},
        {{ 0x000662dcbd70a604, 0x000642008913f4f8, 0x0001646660d881ac, 0x00000ae6cea20ed6, 0x00006d3c305ff6fc }, { 0x00037f7913789dfb, 0x0002e5c725d2aaaa, 0x000eead363c6f511, 0x00079dfccbad2752, 0x000044f72f690d92 }},
        {{ 0x000da26e1aec73e6, 0x0009be144cc3adda, 0x0005da1c14b462fc, 0x000697e72d5ffa1d, 0x00007dc85a835122 }, { 0x0001c794b1a357c1, 0x000661563483caaf, 0x00083d98e34b8417, 0x000eead3ce6924cd, 0x0000cfb7382ae666 }},
        {{ 0x000adbb3f3f634c4, 0x000728882a8042c0, 0x000fc24bcea4eeec, 0x00062e340d931cfa, 0x00006ff1f31aecff }, { 0x000577c11e875a33, 0x000621e5eaa8eb87, 0x00049c3ccd5a10e4, 0x000a9bfc5e663a8b, 0x00000028143def02 }},
        {{ 0x000a786d62a93d0d, 0x0002b5c08369a907, 0x000a9e9647958a75, 0x000519f802990c59, 0x000076b05f00e31d }, { 0x000b2ce9e430d6f9, 0x000b4be20ac3a8dd, 0x00007f8d434c0323, 0x000cdfc6531a2178, 0x00009e7647365eaa }},
        {{ 0x000c56feb051eaca, 0x000694b168984df5, 0x0006d73097883b04, 0x0004c56f0e81dce1, 0x00009cec7a5febd4 }, { 0x000233e35ef6807b, 0x000db01fde58c84a, 0x00068c98cfb5957e, 0x0001a0f7874e2411, 0x0000c7113cb30256 }},
        {{ 0x000f3a8f9d8cc3f9, 0x00072f38bd7aff18, 0x000a8e4721a92651, 0x00064b0151b5ee4c, 0x00003b792b15a449 }, { 0x000255f53b1b9da9, 0x00010c9a0748be7d, 0x0004d6d86ce4578c, 0x00072ff371d391c9, 0x0000702e661ee631 }},
        {{ 0x000f627439d898dc, 0x000c204f1788ee3f, 0x000ea906c4682ecd, 0x000a125c0c14f27a, 0x000041647e0afb10 }, { 0x000c1c33bbabfba6, 0x000dfd8ca2a59655, 0x00027acd85f7ce0d, 0x000fdcce83d9a554, 0x0000b9121e2a4fa0 }},
        {{ 0x0008c889de2e32a8, 0x000b0d474a861082, 0x000fc583d3b1de17, 0x0003aa480f8048a8, 0x00003ee931e746ba }, { 0x0001c240230a2d47, 0x0001f2eb214a040f, 0x0007461ec3244db5, 0x0007f2d8ae8c48e9, 0x0000149fdf2f3f76 }},
        {{ 0x000359ed623dd564, 0x00070c35f74040ab, 0x00014990dbf56e58, 0x000ef8823409aeb7, 0x00004dda364e2017 }, { 0x0002f1d4df601f90, 0x000b926ed6f4a607, 0x000ed89f5903b254, 0x00050ca59e21e8d7, 0x0000edcf85157a9c }},
        {{ 0x000c5fe1306bd6ce, 0x00056b9eb7926b83, 0x0003915e73d8d154, 0x00006960bbff88cb, 0x000076006a867f79 }, { 0x000daf6b21ea15f4, 0x0006ef43764fb3df, 0x000733a24b0b851f, 0x00086f1f37b5af31, 0x000030c22394a5ed }},
        {{ 0x0008cc13d69c73d0, 0x00097ec6063540cf, 0x0001ed2db7b8d6e8, 0x00025f6ab50b259c, 0x00005b5c1b0bc3a3 }, { 0x000f8efeb1d7be88, 0x0002b05c52176597, 0x0003cf322ba2a903, 0x00054bdfefe5364d, 0x00003115b68dac90 }},
        {{ 0x000e583c57029879, 0x00078e5d59233595, 0x000cf5436359e633, 0x000ed2298db0cea9, 0x00001cb09b717846 }, { 0x000b4d4d5956cfae, 0x0006c2dd48185b77, 0x000df304ca2dbb2c, 0x00040ad8bc93fbc7, 0x000021236a00feae }},
        {{ 0x0003768f8316e330, 0x00000b046eb842ab, 0x0009c86455eacc56, 0x00014cdfc8e844a9, 0x0000f8985bdb1afd }, { 0x00051b6eefc870ce, 0x00026e04e09e61c1, 0x00018bac5eec5fb1, 0x0002391cd414dc3a, 0x0000e1888cf3478c }},
        {{ 0x000e19084bc4ca0c, 0x000ca136339c0831, 0x0005c19d3eb5ee51, 0x000cbf8ab39ff815, 0x00008e22e48c871d }, { 0x000222840c0fb45d, 0x000f7c0eccd47ef9, 0x0004b83b7724081a, 0x000269cf68969338, 0x000064e761fcef38 }},
        {{ 0x000c5d75f6acf986, 0x00082f5803bac625, 0x000cdb3e5e689697, 0x0006a3f810b7dd91, 0x00004626edf8311d }, { 0x000036b9b2545d4c, 0x0007b381f2dc308f, 0x000ee94d9b0c2902, 0x000ba643120faa7b, 0x00005df95dde3c50 }},
        {{ 0x00004884ebe91706, 0x0005b0939842194e, 0x000d6987cda9dbf9, 0x0009d28e0b7e2353, 0x000008055a9e7106 }, { 0x00013520c2fe68f0, 0x0006d817419f40e7, 0x00023860a17ef38e, 0x0005c66465b0ac16, 0x0000599f3cf0f8c6 }},
        {{ 0x00024a8434f084ae, 0x000ac5ce98c51002, 0x000d0b928b25c421, 0x0001e398b6cec871, 0x000093f359354aab }, { 0x0006936830d34d21, 0x0005e6125f5c46fd, 0x000b71c0f68f5335, 0x0006e3bdc6a22bed, 0x0000bf2268186fde }},
        {{ 0x0007c2eec1224f3e, 0x00041e891e5d7d3f, 0x00067bdd87b70fb3, 0x000909f61f14b7d1, 0x0000092310e6f29e }, { 0x0000472beb05816d, 0x00006f43756e621a, 0x000f77de8d5da4d4, 0x000061872c013a23, 0x00006e813df45a5e }},
        {{ 0x000e87acdcb83dff, 0x0002f270ac69a804, 0x0005a696a81a3be4, 0x000c1a69134096d2, 0x00005b6a5c9aeed7 }, { 0x0004a56cd8946d50, 0x0009037e285ccb47, 0x0003dc3862797108, 0x000a2a719188089e, 0x000041ca5eaccff2 }},
        {{ 0x000421cc52cbf744, 0x0000ae21726931ae, 0x0007523f20aa75d9, 0x0008a4a180bbbb2c, 0x0000d98ec4e59258 }, { 0x000359b3dc444ac7, 0x0000b08bea87be65, 0x00035e7422a6acdd, 0x000509cf716a475d, 0x0000928ea2c9a7ff }},
        {{ 0x000fc3a5983894c4, 0x0002dde8bf06e31d, 0x0003fd749369a738, 0x0008323d21fdf8f4, 0x00008669487b1517 }, { 0x0001b4f11e41bf9b, 0x0002da0b1ecff106, 0x000bc68b142b2d0a, 0x0008ff273ee6ed6c, 0x0000d795b8258fd9 }},
        {{ 0x000f3d47897647f6, 0x000e09bd4eff2d71, 0x0001234b6e8604ea, 0x00059f12388ae677, 0x0000156c9f102ac1 }, { 0x000794b13b286f95, 0x00083c07f9871da6, 0x00088ea1313d5e21, 0x000e488ac35fbda7, 0x00009a346a1a06b8 }},
        {{ 0x00023a9f8173b2f9, 0x0006bac09cc0917d, 0x0008f44aa17b929d, 0x0006f7d7127c5372, 0x000084aea5a60d7b }, { 0x00040f1c67e5aa9c, 0x000f92852650359a, 0x0000d0929045b31a, 0x000ab235cfa09ad0, 0x0000eb49c8d9c125 }},
        {{ 0x0002383e48552e85, 0x000f02edcc18770a, 0x000ee5ed855ce52d, 0x00082ceed4baf1f2, 0x00006b9ecc862bcc }, { 0x000b2dfd7c4b99b3, 0x000af0a2646d19a8, 0x000b789a42819c25, 0x000d236f156cb44e, 0x0000656fde1d4d03 }},
        {{ 0x0007bdb99d7bc622, 0x00050f2a672c5c73, 0x0007d5960c453caa, 0x0007d3732eacb11c, 0x0000d966e32b0083 }, { 0x00026693bd8b6c09, 0x00026b62899c20f7, 0x0001d2183a676b7a, 0x000eb73e33535d51, 0x0000b014e1e0cb73 }},
        {{ 0x000a20ed2a8c1c6f, 0x00041e5ab4b35a24, 0x0001abcdbda4db56, 0x0003f26801d8b600, 0x000004b48afeb9ed }, { 0x000a24142725a709, 0x00053d44fec01e63, 0x000c6ddb3c8679c1, 0x000868efd3ed7ddc, 0x0000203192c73ce6 }},
        {{ 0x0007fd26ae8cf314, 0x000cb2c6a8ea8208, 0x000b6a2f9ba58ade, 0x000360026b2b50d3, 0x0000e7087b5a0c2a }, { 0x00066f7ffa7c7f1a, 0x0007ccc02bee4ba6, 0x0003a0128fdc2037, 0x00093cf855312180, 0x000003bbef5c6e85 }},
        {{ 0x0006c3200fa35ff3, 0x000da33bccd9617a, 0x000413b4109421a3, 0x0007b1486365dede, 0x0000cb76293645ba }, { 0x000a57354a206551, 0x000aa2ce5e35efd0, 0x000b4afe6c76e5aa, 0x000a5e65199a4e25, 0x00004027e76f2bd6 }},
        {{ 0x0006a6f79278950c, 0x000a24bf8e4b0651, 0x000417ac0a99741e, 0x000fde19607a834c, 0x00007e7ec115210d }, { 0x000ea49b3d3fd55e, 0x00005c66a8649edc, 0x000db7faa2d74f3b, 0x000d255ecfbcdfcf, 0x00002d9e2c5a83c0 }},
        {{ 0x0000ab7b49321f95, 0x0006f4fe9d964888, 0x000e00d72578140d, 0x0007254e94b53d52, 0x000052362af789d1 }, { 0x000995d53e000a78, 0x0009dd427d3317fc, 0x000a50ce91e0ec03, 0x00091e3928dfc1dd, 0x00006eca410021d4 }},
        {{ 0x000c32fbd2c44064, 0x000ddec090090ae0, 0x000ce17d6c5d6bed, 0x0008932902278a0c, 0x0000acd50fc16037 }, { 0x000cf307bda4feac, 0x0009adaf341a6c11, 0x0008fefc60e33292, 0x000789f3aae0ba13, 0x0000954ec449a5e8 }},
        {{ 0x0003b88a9ddcf1a4, 0x00019c719803511c, 0x000ac4d03920e342, 0x000d52c5ea9f97b3, 0x00008856a1c81ea8 }, { 0x000bd715882bc1be, 0x0004948c48b0ee35, 0x000babd03139016e, 0x000ab99f37506bc7, 0x0000a2cc32ebf442 }},
        {{ 0x0005c8485a9d19bd, 0x0004708c1bc52257, 0x000ab150da1307d3, 0x000737043d0b946d, 0x0000f537f04207d8 }, { 0x000df3874cc30296, 0x000c9499ce4177a4, 0x000e2bcd84f0edb4, 0x0006566d74fa61a1, 0x0000f53571e1b3de }},
        {{ 0x0008d158672d3eea, 0x00042f983dfedc91, 0x000474b60ab2fc6f, 0x000eaded2f843713, 0x0000d9c3c4a8aa46 }, { 0x000a5956bae7a26b, 0x000d3cf69769bb7a, 0x00061a2054c79b62, 0x0000e89b2c55eb85, 0x0000df9c408d613a }},
        {{ 0x00045c51f0556ac4, 0x000d2c29fe20ebae, 0x000b00ce08e2eb0f, 0x000b2d989f27168b, 0x000069b4c47cc369 }, { 0x000a9db957a6f395, 0x000bd76099b42210, 0x000c1ab9e80238d6, 0x0000fc3958dc6407, 0x000089ab035eddea }},
        {{ 0x0006a52433a64b6b, 0x000cbd3df61bc767, 0x000fc8d268a83dbb, 0x00023378f13a619b, 0x0000e79d7f68711f }, { 0x000ac045e020fb6c, 0x000ac91ac5bdc5d3, 0x000280079621025d, 0x000e41853ff4a1a7, 0x00003f7f41088141 }},
        {{ 0x000fb4f9a3e59e5a, 0x000da1bad53dba78, 0x000bf3c668b6a4a0, 0x00044b7898ec269c, 0x0000c42d0387de55 }, { 0x0001da22fc32c200, 0x000f823ff9875cf9, 0x0006cb6a079f4d88, 0x00004267923224f5, 0x0000acbd3518a2ba }},
        {{ 0x000364ebbf27875b, 0x000d74d6c04a661c, 0x00059ad281587be8, 0x000bd88579710d3b, 0x0000db51af083108 }, { 0x000aa7641d8ee984, 0x000d9517733d61c0, 0x0009c400c2a5a595, 0x000b8d148d55644b, 0x0000c50509008fc6 }},
        {{ 0x000b6066dd443936, 0x000b681318106601, 0x0007c2ead965cfff, 0x00004cc739dfa650, 0x00008f398de4fb88 }, { 0x000bf773f8103f84, 0x000c1919c9efebdb, 0x000ee624abfcc370, 0x0003683d3425d4bd, 0x00007bb42dbf21c5 }},
        {{ 0x000ed26d818f0304, 0x000b96994b03ed1e, 0x0002747a57c4f532, 0x000e11ccad9ad1de, 0x00005dcd24ba7a71 }, { 0x000ff2db2716bd02, 0x00009e7a4db09d2e, 0x000ca61347d0c139, 0x000d9c529f47f569, 0x00003c0c5c86b05b }},
        {{ 0x000187d6657e670b, 0x000c514b913e759b, 0x0005de6d082e2895, 0x000b95de153da478, 0x000092e140a21444 }, { 0x000e8d345fd30ff2, 0x000c4501dc90b596, 0x00030f1ce6c7249f, 0x000fc18848170e94, 0x00005a2d667b789c }},
        {{ 0x0004b7839d2bce29, 0x00096a1c5c5ea50e, 0x000b82de82e45901, 0x00006e042d3baf14, 0x0000ff6d502965e2 }, { 0x000d93355322d13b, 0x000bc94e7de6dc08, 0x0003d497727bb320, 0x000dbc3ec5cd0f4d, 0x000059e7d3f99e36 }},
        {{ 0x000be924c158e0e0, 0x000aec5776c57224, 0x0001df76044b6efa, 0x0001345a9dc467e6, 0x000033e12139aef7 }, { 0x0000c9ca4e9b5641, 0x00084aea18921310, 0x0003a63bff437207, 0x000da19c58bf8a8e, 0x0000f70e2d2715de }},
        {{ 0x000bc2ffb0c0a600, 0x000e80f83774bdde, 0x000199b044d49d58, 0x000a469a96313160, 0x000087769824fdd0 }, { 0x00009015dea24c28, 0x00015b77cfa46a5c, 0x000e60d52571b07d, 0x00070300a71e6161, 0x00003117310e7902 }},
        {{ 0x000f2b9ffa5ce94f, 0x000f8b2d174d7aa6, 0x00088475bd09ee91, 0x00010b8ed4072d8e, 0x0000abdc7583857a }, { 0x0001914e9bad73f4, 0x00022153760fe8aa, 0x000f4edb1d776b51, 0x0002edc717a96e6b, 0x000069595ca97461 }},
        {{ 0x0005b20d1b136d4c, 0x0009757d8a3c5cf1, 0x000315091df843b9, 0x000413cdd1197ecc, 0x00001969c8be641e }, { 0x00099ea238db2755, 0x0005526705840398, 0x00091e3df50ef1df, 0x0002e83152277c96, 0x0000630e2f484e9d }},
        {{ 0x000875384b505bc8, 0x0005a8b760c1c9d4, 0x000152e6e08ee77d, 0x0009b13c5c7efbb1, 0x0000a686f3ab7cc9 }, { 0x000912a263785680, 0x0008baa3ba8cd86a, 0x000b197dbb7d4f6e, 0x000cc9952b0ce40f, 0x0000893fc7e41001 }},
        {{ 0x0000049c55f94a22, 0x000bbcceb5b76c16, 0x0004c6f76e5ea2ee, 0x000e2049f746f528, 0x000076ebebbffadc }, { 0x000d3fe2ee578482, 0x0000d823c2f7c9eb, 0x0002d43d7a059070, 0x0007d72696bd9904, 0x0000ce8557b6ba2f }},
        {{ 0x000ff4f96876ad1b, 0x0005eca12ef3d36a, 0x000b8e89ac75cb71, 0x000521bf79ee3c36, 0x000090ab39ba4d57 }, { 0x000a347a48af4239, 0x00064d0c47adb7e5, 0x0002a8fa03b1669f, 0x0006bc96a45699b2, 0x0000959575cb4270 }},
        {{ 0x000e6e24892d0ac0, 0x000933b3cef0d7d8, 0x00079d4d1ef8b947, 0x000763500d94f7b1, 0x0000842223ee34e1 }, { 0x000d016ec18d3932, 0x000e49716caca6a2, 0x00080b51ac183c3b, 0x000187be1866c49d, 0x0000c9294fa577e4 }},
        {{ 0x000e969937eb4c0c, 0x000c0c900eac3f90, 0x0000d456ae10b8bb, 0x0007d820d835f628, 0x000023a90a360f31 }, { 0x00089014af462bca, 0x000fee88f30c9cbf, 0x00043eac31b1d4ea, 0x0002462aba606ae7, 0x0000fd95461debef }},
        {{ 0x000b9912384ae29c, 0x000ce9c31c71f7be, 0x00023dd6ec23d54c, 0x000ec27a301f95aa, 0x00007cbf09e9dc08 }, { 0x000c8b46801a5c63, 0x000305e5d91ba877, 0x00008ecb0399943e, 0x00077eb69c6ac628, 0x0000c52861049c21 }},
        {{ 0x000e5d49ffd84deb, 0x0003a1fea5b8e69b, 0x000d075a15bd1f24, 0x0003205e5aeecb3b, 0x00003034bad431b0 }, { 0x000f13f5a83d106c, 0x000a7aae5c8646f1, 0x000801cda3d81b1b, 0x0009b5194dbaa12e, 0x00000bad3d6972ba }},
        {{ 0x000608b69829f0bc, 0x00035eeaceb4fc02, 0x0002386658aa3391, 0x00022b4c83625600, 0x0000214f4fdc3f4b }, { 0x000209f113e9fbcd, 0x00005fc05268301b, 0x00088e26a0fee5c5, 0x000d514c2a6943d3, 0x000089511c6cae35 }},
        {{ 0x0003ffdde3be7cc0, 0x0009e0697bf1298a, 0x000f7d9d85bb314f, 0x000a1cc6fcbdb672, 0x0000c8ea86a332f9 }, { 0x000a3d7005cb8287, 0x00033aaa914d3ba5, 0x000f2ea215e8f9bd, 0x0006b8051836d500, 0x0000034ad2de159b }},
        {{ 0x000da3ce9ebe72c9, 0x0009a4d39eedbbab, 0x000746b1d5ce6360, 0x00065220c35cd2e6, 0x0000dcc999f8c73b }, { 0x00032fd30a1b75d3, 0x000a7602c9cf7a47, 0x0004089bb469c24b, 0x000ff75df01572ee, 0x0000ed51712af770 }},
        {{ 0x000a4544aa046b21, 0x000e8c716543a402, 0x0006d498f7c2ddb5, 0x000b9c5149acafd3, 0x00004ad54644a863 }, { 0x0003a83affc7b669, 0x000e96291d60b063, 0x0008f4ccfa455de1, 0x0003ad227281b7be, 0x0000b48074a91ac9 }},
        {{ 0x00058ae74fac8317, 0x0008162d5d721d5b, 0x0002a648ba32d37d, 0x0001fb6a40c3357a, 0x0000d41eebd7ebce }, { 0x0007312ec0f21311, 0x00009812e6dad6be, 0x00010d0c3d9a387e, 0x00083b3d8de28605, 0x00000e56beae8107 }},
    },
    {
        /* digit=2 (1,2,..,64)*(2^{14})*G */
        {{ 0x000161957d933aea, 0x00028e76e6485b37, 0x000b05babd7b6596, 0x0000f456d4d3e24d, 0x0000ce6abd42c3d5 }, { 0x000aefa7aee9d873, 0x0009f83cdb12a6c3, 0x0001f36139b52f8a, 0x0008e5973bf3fa88, 0x0000e7a50e16b447 }},
        {{ 0x0003f63771beff8f, 0x0006f623b36f9fd2, 0x0003d9eb684ebab1, 0x00084f3ef26543b2, 0x000096ae77bc2ed6 }, { 0x000fb89d795ff65e, 0x0007796b6a1142e5, 0x0005d3a20e91b4d9, 0x00011bfd55ea6aac, 0x0000a386dea94493 }},
        {{ 0x00086584c9b213cb, 0x000f83af72e34d46, 0x000fa8a9676c4401, 0x000cad84e0bd9ea3, 0x00006265c102920f }, { 0x00057594c72c920d, 0x00088d38c97b942f, 0x000958e6e09ae71b, 0x000e172894609561, 0x00007cd5b11ce7c5 }},
        {{ 0x000564bade8d111c, 0x00049fce3779ee34, 0x000d77e4a2bf81f0, 0x00017327ae00e7f9, 0x0000f7675150c393 }, { 0x0007f7ccb1668c80, 0x000b12ac4708e8e4, 0x000b5f26ce2e2a59, 0x000dbcb9f4f5725f, 0x0000c707f408fcd6 }},
        {{ 0x0003a7c17140472d, 0x000581b8d9465810, 0x0003ba4cdea132db, 0x0003dade0f0d1313, 0x0000dedff07af825 }, { 0x00058b1494211ca0, 0x000df38e475381a1, 0x00026ff71ea03acc, 0x000c4dddaca1ba42, 0x0000a403ce5132ea }},
        {{ 0x0007c64e36e21202, 0x00075e5522e6b693, 0x000384071de1f828, 0x00012fb5669c908c, 0x00002dc9675211f7 }, { 0x000fb08f0866ab60, 0x0007bdfee478d7d1, 0x000be1e36897f3a4, 0x00021b6dfba75c8b, 0x000059cbb6ef7bf0 }},
        {{ 0x0007560abcab01ab, 0x00000f9842e9f302, 0x000f375578f2df7b, 0x0007d7f6aa331d4b, 0x0000a0561e6ffd56 }, { 0x0003b77509e9eaaa, 0x00089870e3a6b750, 0x000267e7dbebaacd, 0x000134cd025c6aea, 0x00001e66709e13cc }},
        {{ 0x000dbde4ac826d0d, 0x0007c0361a341c13, 0x000fcf3f7c95b3aa, 0x000c6246c3604dc5, 0x0000ce46d1be3f02 }, { 0x000f0d1f30affc92, 0x0006a5d1c7eef696, 0x000562423070d98e, 0x0003f8fca5889680, 0x00003b45ef942f70 }},
        {{ 0x0006f04d3172e6a8, 0x0009cd4196224f85, 0x0004e05cba6c1ba5, 0x00062492375a4ab9, 0x00000994feb99b0a }, { 0x000000efa0311470, 0x0004f1cbda602b24, 0x0002b9603cc9b10c, 0x0003b606b73ee174, 0x00007167141027d7 }},
        {{ 0x00099b055d1c6101, 0x0001a4b12cfe2978, 0x000b9f946f882bbf, 0x0008773e83eee129, 0x00009e74c40cf1ad }, { 0x0005132b2b61a7c7, 0x000b3643a39c8cf8, 0x000df383119dec9f, 0x0008c316382f09ef, 0x00008e4c68996471 }},
        {{ 0x000b4afef949e9b3, 0x000f0ab516eb17bb, 0x000f177c3832341c, 0x0009fbaa673bec06, 0x0000087396035c70 }, { 0x000f54befe9fd403, 0x000bbc8e61d66b0f, 0x0003efd4710c8da1, 0x00097a75a02bda4b, 0x00003e2c39ecdf62 }},
        {{ 0x000c9a9d2168c124, 0x000671dbe7a2af37, 0x0008d31cad5154da, 0x000c67297ec51caa, 0x0000f6a5ba0129fa }, { 0x00003937564cf036, 0x000ae4f6ef5f0145, 0x000298d331a9ca83, 0x00051e01e8abcf41, 0x0000803532cecc34 }},
        {{ 0x0001e274de55c8d4, 0x0002956ed6def63f, 0x00018b709b8e424b, 0x00056a6c153cf648, 0x00008cfd0094deed }, { 0x0009fdddc13851d6, 0x000243c44cc78808, 0x000af2278bc0bdee, 0x000524ae0411a426, 0x0000bf64ecb23c5b }},
        {{ 0x00088c68b8e8aff1, 0x000162232398baaa, 0x0001b47037788dbe, 0x00052abf2205fee4, 0x0000c5e4c2db8891 }, { 0x000e9e19413381a3, 0x000ad417337aaa82, 0x000ce0156bb388a5, 0x0002a062f9ed364a, 0x0000c9e564f943f6 }},
        {{ 0x0009d84aaee0a3f0, 0x000044a3dafd2d50, 0x00058e154e29774b, 0x0006409b135d1fae, 0x000075272e315a42 }, { 0x000286418cf25deb, 0x00030c788aaa6917, 0x00093d60897674a9, 0x0001ac1012dea57a, 0x0000de5fb8c5274b }},
        {{ 0x000688350470c6e0, 0x000970977f7195ad, 0x0006d7634e8b89f8, 0x00088212d8ec8616, 0x000043845130a44a }, { 0x000f5c2ee645cfc9, 0x0005bf8309004c28, 0x000584a3ba78a88d, 0x0005275a4a96e4b7, 0x000038c353982c77 }},
        {{ 0x000270f0d3fad15e, 0x0004ca5dd968198b, 0x0001bfd0499861e3, 0x000657b03fa131c5, 0x0000619c34d7bc9a }, { 0x000c59900ea25f24, 0x0005164a053a3bc5, 0x0006be2d3ae0af2d, 0x00056cbd0088254b, 0x00006bf9c020a568 }},
        {{ 0x00058215b141a06f, 0x00072dda1887395c, 0x0005cda8502dab0c, 0x00008e62940960f3, 0x0000e7123058aaf9 }, { 0x000dd036eeb10f06, 0x0009558f0e1a5240, 0x000be7c0a8ca237d, 0x000ca5062b406856, 0x0000716a9f2d15eb }},
        {{ 0x00060caadd4449ef, 0x00082acc33156b3e, 0x000964736c5160b7, 0x000929feff09d24e, 0x0000caac5cb6e932 }, { 0x00018427c2846cf1, 0x00007a6979b1d8f6, 0x000c35f465637c02, 0x00002fdc060908a1, 0x00001090f1c298a2 }},
        {{ 0x0006877b288e2940, 0x000f33959145a65f, 0x000803b8e80eaf37, 0x0006112a0cd9bc36, 0x0000b81a72e0ba9f }, { 0x00040fa5a0fd54f6, 0x0008b9bf5cbdb258, 0x000859f95ca88625, 0x00044185f2a4fd1d, 0x0000bbdb6fce4677 }},
        {{ 0x000251fe80573ec6, 0x000aebe6558842d7, 0x0003a5a1d225d97a, 0x000041e6d70df8c6, 0x0000a3616cfca01e }, { 0x000d97a9062076a5, 0x0008fcb5e9aac9a8, 0x0004626cb46bc51a, 0x00054784e11bb093, 0x000096e6c4d5afbb }},
        {{ 0x00077708c667d059, 0x0003a277946d3f97, 0x00038fd019ad9658, 0x000e65586e132bd2, 0x00002ddf44a145b6 }, { 0x000ac06c3764df11, 0x000b7cd1297029e8, 0x000c3138c426b403, 0x000a6c5d92c61095, 0x000067be85672106 }},
        {{ 0x000e2347ffc2edcb, 0x0001139a13046683, 0x0004c9d54e06e078, 0x000d9578de4a9c88, 0x000002ea6f0c8fe9 }, { 0x000ff875671915b2, 0x0004977ea7d23a38, 0x000cd731fc668ce1, 0x0005492d32fac257, 0x0000e0f225fb9165 }},
        {{ 0x0000d8b2dae7ca94, 0x000fcb36b0cf82eb, 0x0005753d030cfdc3, 0x000d1805a6438d24, 0x0000a53ea3da217e }, { 0x000afa8c8dcb9f12, 0x000c8dbd9184c4d2, 0x0005ce430ce88936, 0x000756b473dbb18d, 0x000096aae294c905 }},
        {{ 0x000ad8848abb5fc6, 0x0008de8bc36742c8, 0x000e63c869056bf3, 0x000e7b500898f427, 0x00006d577bd5f382 }, { 0x00074a2d9a7f8c04, 0x0003838d74a2a464, 0x000954574e07ece4, 0x0002da5e45e3e05a, 0x000086fa3c45cbcb }},
        {{ 0x000df865364f0739, 0x00000ad653ae3263, 0x000a6954560d380b, 0x000cd878314bcf72, 0x0000e5984638d7d1 }, { 0x000ccfe3cd6ce14b, 0x00024e7b53ca5557, 0x000d2f05f16b65f7, 0x000f23968688cb00, 0x000083bf9324d5ad }},
        {{ 0x00074a1d2d22f9b4, 0x00093e06c4c1fe61, 0x000a12419e4cac55, 0x000daa614557e1f1, 0x0000f204eb2e220d }, { 0x00075797ca396209, 0x0003230c51fbf539, 0x000157111a5b409e, 0x000ad3873d894099, 0x00001cfd1518ece2 }},
        {{ 0x000dfe0d903a457a, 0x0001207e3533d77b, 0x000970360f08ec9b, 0x000e84aedd7222e6, 0x00008819a858ad52 }, { 0x000eb874ec49378f, 0x0000bb61d056605f, 0x000587efe032d6ae, 0x000c3fe8ca9ea9df, 0x0000fbcf08ae31fe }},
        {{ 0x0005f49faf4f53a7, 0x00043867ee0d98fd, 0x000b060178b5af0c, 0x0006c8ad4201ad34, 0x0000d0bf380597e8 }, { 0x000a24911e00876c, 0x0004691c7f4a40c8, 0x0000e8e94c74da88, 0x0004ffd872e69d9c, 0x0000a4ebb0eb3649 }},
        {{ 0x000d6de7bd491998, 0x000801060a716760, 0x0008e8fca31418e2, 0x000fd8b1ee852d1a, 0x0000dc52d76da81a }, { 0x0007f0f335115f76, 0x000f988c8e347f89, 0x000703e8a98e5e7e, 0x000a7f3501c58754, 0x0000eef253d19b22 }},
        {{ 0x000b3fe0f9da5724, 0x000bdce4924867ad, 0x000f8578681e0a84, 0x000597679bd5f51a, 0x0000d1366a434662 }, { 0x000ca8b8c079486a, 0x0002ad8556fa9db6, 0x0008b69876ff7430, 0x000332891fa26216, 0x0000f462db921b41 }},
        {{ 0x00022a049341c337, 0x00033a9bd7638026, 0x00000306390eab7d, 0x00022f7c4c21486a, 0x0000d141475b8b86 }, { 0x000540cb1b1be032, 0x000da9dbb7a8ee0d, 0x000b96c0e3c063f7, 0x000de42ecdc2ed3b, 0x0000397c62c0f6ec }},
        {{ 0x000736f9291ef406, 0x000e74ecf84f6c7a, 0x000b85d2161ab278, 0x0002f97b48561fb9, 0x0000bcf6264a5050 }, { 0x000c70cb296a5d7b, 0x0005e292c1950ee1, 0x00033a6b46899d57, 0x0000d0c8eda17754, 0x0000b3f3cc6b8110 }},
        {{ 0x00070c1c8efd9357, 0x000358b6f0f7c500, 0x00087e41afdefe0f, 0x000242f42a8eba2b, 0x00006e5680230b39 }, { 0x00079d74c341868c, 0x00037716484d2e49, 0x000e75a3cc667a4d, 0x00052af0bfb78318, 0x0000b9d00abad929 }},
        {{ 0x000667ac0d2f8437, 0x0001f4761fb04286, 0x000a587e78c7cc95, 0x0002a4ffd4d78954, 0x000020826521d389 }, { 0x000ac4bacee14c89, 0x0007c1243bcf832f, 0x0006a53c4668bdcb, 0x0006e01a7fbadf8b, 0x000053ad230ebad2 }},
        {{ 0x00058566c410a298, 0x000708d977eab403, 0x0005f1db4ccc80c1, 0x0006b3d5351f4c5b, 0x00008dad5554cb23 }, { 0x0009e5f0b7cfc603, 0x000d39e5068f5ff5, 0x000c81197a4e6a94, 0x00018bcf0f31435f, 0x0000e92434aca407 }},
        {{ 0x000943b3af624454, 0x000dbb898559acf1, 0x000e3473e7a0fb04, 0x0005b12aa96018c2, 0x0000fe9ef544dffe }, { 0x000ec75726be2472, 0x00024df15a40d39b, 0x00038d59a09f0400, 0x00073ce4cb2a4468, 0x00009d760075d37d }},
        {{ 0x000213754eefea23, 0x0007e63bea064441, 0x0002e51141a3c690, 0x0004ea5ab27e95bb, 0x00006619f7919ae2 }, { 0x000cc970d32e84eb, 0x0002520716e0f742, 0x000ed2f724ef5dbe, 0x000273e326679c82, 0x000049306f2821a2 }},
        {{ 0x000f8396342a360d, 0x000122f740f0e66f, 0x000a2be78d984691, 0x000d47ed7545c616, 0x0000556fdb469789 }, { 0x000fcb8983c3880e, 0x00019d9025ac99a0, 0x00001b9859b74109, 0x000503719d314d4a, 0x00003e7c7f956365 }},
        {{ 0x000fd58ff1b72f01, 0x000216138634818c, 0x0003e4769e197596, 0x000b0daa3501814f, 0x00007ddf3af2b89d }, { 0x000db6eef931c2b7, 0x0002414d3b507a84, 0x000e7754ebdfc74a, 0x0002342aa57bd478, 0x00004e0dcd5fe91f }},
        {{ 0x000de92572a83fdd, 0x000151ad1fc0654d, 0x0002a33d78008489, 0x0003f6cf9c71cc96, 0x000005b75269915f }, { 0x0005f44d1e992bfc, 0x000daa3f97ad8f78, 0x000f0c59969f0192, 0x00014ac00e6c19d3, 0x000025a10641d608 }},
        {{ 0x000332b6dc4e101e, 0x000a28b09d64c52b, 0x000f42ace06d0b43, 0x00063417da1b5e49, 0x000046ce47f6dc04 }, { 0x000f0bc5a141ae2f, 0x00059a0b4511aa47, 0x00025cca06b413dd, 0x0001f23bd0978875, 0x00000bd956a4fb97 }},
        {{ 0x0009ff29ad65488e, 0x00084d4c11f61ef0, 0x00043084bdf0af25, 0x000773c159ce1075, 0x0000ec6435d82c07 }, { 0x0008b5fef46ca97d, 0x000f5bde7da946e1, 0x000bd5caa5fe7671, 0x000b00fe4ebf9eb3, 0x000075259e3b13aa }},
        {{ 0x00038c534e92ad2a, 0x000a3d35958cd79a, 0x000479aaddc64328, 0x000e62d263580a1b, 0x0000ae928f234367 }, { 0x0003cc3e51cd352e, 0x000b60a2ff5845be, 0x000cd9abd92da353, 0x0004a6b20e5bb40b, 0x0000ae386f9756b8 }},
        {{ 0x0008ec52c22da1f9, 0x000befee5682d265, 0x000f55dba5b9efc6, 0x0007758ab5206f76, 0x0000dc15c6ab6dd6 }, { 0x000ae060d5586b0b, 0x00084990f75cf9ae, 0x000ecfa270009add, 0x0001d8aabf411951, 0x0000e895791fe909 }},
        {{ 0x000bd450efe0661a, 0x000c5737fb54dc25, 0x000ac9c1e559ba7b, 0x0009f36811f3e391, 0x0000aca8c08c3b1d }, { 0x000890e30907b5a8, 0x0005467981eb389e, 0x000751746dcc2ec9, 0x000873f5ced7e192, 0x00002b55533c62a3 }},
        {{ 0x000d7f7a52f077e7, 0x000cf82feccef760, 0x00057e3af33fc8e9, 0x000ab8ec48b8e11f, 0x00004d722538f93a }, { 0x000e2b5b0ebc17e7, 0x000ec0a9efe5633d, 0x000a76269e26417c, 0x0003683002d8d246, 0x00005e002152d468 }},
        {{ 0x0007d2501ccbde9a, 0x000a25abf776f5b8, 0x0000d8632ae86106, 0x000e0a57a6f72407, 0x0000bbf66580faa5 }, { 0x000a4228cbe2da60, 0x000f2e187b053405, 0x0007c7202366612a, 0x000206ddbb7bf7cc, 0x0000bac9edde0240 }},
        {{ 0x000632d18502246d, 0x00069d6fd4fd6717, 0x000e4ca7cf89ed63, 0x0009c7e96a233687, 0x00006b0153bdbf01 }, { 0x000381366bc36648, 0x000b0e940735ff4f, 0x0002cc87fb85426a, 0x00097f7183a3e6e1, 0x00007201b1efff17 }},
        {{ 0x00034d89dd22ce0a, 0x00018a960167d923, 0x000841dc67af01f4, 0x00085a71462f9d6f, 0x00002030a417e2e0 }, { 0x0007d6912f7833d4, 0x0008fba2a4383ce6, 0x0006f5229a7161b8, 0x0008aa0c0cec8e93, 0x0000c10be8855950 }},
        {{ 0x00084ec70d77159c, 0x000d0461b9131492, 0x000f11ff3bcd0483, 0x00018499436aae2e, 0x000053055174f620 }, { 0x0001596fa6d718a3, 0x0000fd061391d54e, 0x00061cb7deec1cba, 0x0002f8aff69c5d5d, 0x00006bb3ce340edd }},
        {{ 0x0008ec05f3a623c5, 0x000965696a71cba9, 0x000264ded8c36a35, 0x000a9bf71944938b, 0x0000d4afba995956 }, { 0x00012ecd8b87d02c, 0x000a21b0698304a7, 0x0003b08f2a1059a1, 0x00051b2c12d69144, 0x0000fce76e99e893 }},
        {{ 0x000176fbaba509a9, 0x00057a700fd56e19, 0x000267db806ce842, 0x0003499e102ec969, 0x0000b53556f68ba5 }, { 0x000e116c316d7216, 0x000dbba44b7b6187, 0x000973ccdceb19b4, 0x000436cc0b4d7aae, 0x00006be87ea1879a }},
        {{ 0x0008553aba1f2c4a, 0x000c581466265d74, 0x000205a2733679fa, 0x00047457fb15b6fe, 0x00004fd50ac219da }, { 0x0001bd339486e911, 0x00019c8c21e94db8, 0x000d3c038859a5de, 0x00057368f795ac38, 0x0000e90f47c1b4a7 }},
        {{ 0x000cf9b00c68264b, 0x00090e956b3c4f2a, 0x000ee347c7bffba2, 0x000dac4c1017a99c, 0x00002d4fa858cee4 }, { 0x0003e8b689fdab85, 0x000378d47173611a, 0x00080894bee52bc5, 0x00029bca21a4cbe0, 0x0000cd609cc3206a }},
        {{ 0x00010dfae15b10c8, 0x0009eda4d88f1dda, 0x0004d9c563094fbf, 0x000680267c604f16, 0x0000e738ded43ee9 }, { 0x0004f544eac66991, 0x0008b76af36f4e47, 0x00023a4a4a5e2681, 0x0005f158c994b229, 0x00008811132390b1 }},
        {{ 0x0002b76b184bc485, 0x000d6b98113a7572, 0x000467e11acc370b, 0x000e3015f4ef2d2b, 0x00006a474045ccce }, { 0x000def0c94233a80, 0x000a310443516830, 0x000696031bac0d4f, 0x000e459479d86189, 0x0000536ea5513960 }},
        {{ 0x000f5796a5a19f3a, 0x0001a174c7c15e09, 0x00019acfe58d4e71, 0x0003d955f539bfb0, 0x000062e8e0c9e958 }, { 0x0004a8ce6d3e344e, 0x0004fe84f157269f, 0x000485874afe4d4f, 0x000560c91c825c1e, 0x000052b87c2a4f9e }},
        {{ 0x000a6c9d078887c3, 0x00072a770cbac78d, 0x000b1c459c509446, 0x000e4ed1183488e9, 0x00005eb15cc6694e }, { 0x000d2b810054b0ec, 0x0000f30dd648c523, 0x000c15d01ed16251, 0x000a54486994ca02, 0x0000ab1e189c171a }},
        {{ 0x0006c4fcdd32e0c3, 0x00085c85963a46e3, 0x000572d964eab5be, 0x000a454b4658ab87, 0x0000336a4535aa18 }, { 0x000adbe47f352065, 0x000c2a1a3906def5, 0x00070765a4c6a931, 0x000b907de9fe95f5, 0x00005cfb0d9f02c8 }},
        {{ 0x0003d6b0785941c9, 0x0006e0f396de2b60, 0x000bed16174546c7, 0x000db3d7298583d4, 0x0000fd49ba83df8b }, { 0x000c93db80ebbbf3, 0x000ea93b0b64912e, 0x000328d49f883c1f, 0x000302da99b3a343, 0x0000fc0ce27be066 }},
        {{ 0x000a22cd41ae626e, 0x0001e2466e48bdd8, 0x000887aa0a4c529d, 0x000c298032514f2f, 0x0000980fe0c00e32 }, { 0x00016284b5cf64ac, 0x000e1f4b283e4c6c, 0x0008c9d6f389d580, 0x000c0ec0978c2985, 0x0000926037d62cd5 }},
        {{ 0x0001d36874bc33f5, 0x00084161338830dc, 0x000b103ed755e9c1, 0x0009151f86d4b8a6, 0x0000aa1dcf33163d }, { 0x000f1ebe4e82f431, 0x000b3958d436d13c, 0x0000c091c2614a82, 0x000b90eb5d4d9c22, 0x00002e1a3189da65 }},
        {{ 0x000e6e2135aa3e34, 0x000406a584c5e205, 0x00015b6e4efffd6a, 0x00016589b267e4ea, 0x000028a50a289a0f }, { 0x000fef4e6229a8b1, 0x0001b6a8c8f936e6, 0x000445d2f37ab589, 0x000b64a868af84d0, 0x00002d1d526d0d0a }},
    },
    {
        /* digit=3 (1,2,..,64)*(2^{21})*G */
        {{ 0x0005c78880bdb670, 0x0001f09257226088, 0x000a8d556d463880, 0x000fe87ac131260d, 0x0000a978da1cbb3c }, { 0x00081d1c0fbab2fc, 0x00085ffc3bdf24eb, 0x000b3da6a9989162, 0x0009b73405bff75c, 0x0000d9387ea82fe6 }},
        {{ 0x0002b79aff1b7640, 0x0003cc8f43a730f8, 0x000bb0f3be0fe168, 0x0008ce40be89b6f3, 0x0000da60e29a2a91 }, { 0x000b662498ee8663, 0x000c556962fe5de3, 0x000b255f6d5a6808, 0x000a245580c590ad, 0x0000e4d2e7887b5b }},
        {{ 0x000414582cf35827, 0x000a80d6c39765a2, 0x000d882871de9e8b, 0x0003d1450a2db3ac, 0x000046f7d2fddbd8 }, { 0x000672303b75f511, 0x0007abc60eee9a8a, 0x0001cc28700717f7, 0x0003de4d554c79f0, 0x00009aa79fb6b0ef }},
        {{ 0x0004ff052f5917b5, 0x000faa8719c05631, 0x000ca760c26fde52, 0x0005d57d3cd8ad2d, 0x0000bd74588b7072 }, { 0x0006ad71d02a024a, 0x0003acf3e2d65208, 0x00078da16339cd4e, 0x000f011987ebdf0f, 0x0000079790c922a9 }},
        {{ 0x0005c0e37aec2320, 0x000de9226f435728, 0x000ad7c163ddcc04, 0x0006d74c1dfd3ab5, 0x0000e297c00ebf67 }, { 0x00035b9485776668, 0x00011727be19ae06, 0x00067bb1a22fdff6, 0x0004b977e616a339, 0x0000d8b5f68742af }},
        {{ 0x0005de4f4a416a75, 0x000c7dada430c0b9, 0x000da4b88b7c5154, 0x00041bd6a4702850, 0x00002e74a9165ad4 }, { 0x00017cb319502a4a, 0x0006309e265c17ce, 0x000326fe85685ff1, 0x000d07bca24e4546, 0x0000bf42ebef2391 }},
        {{ 0x000819dd9659911d, 0x000754374dcec468, 0x000ccca8d9a91332, 0x000dbbc9c33cfc02, 0x00001f5408b7ae98 }, { 0x000eb357eb3611e6, 0x00044a60871f4274, 0x0005a83cc448d345, 0x000e00faf8e0cf36, 0x00003b6703b79d89 }},
        {{ 0x000f5d55f460757a, 0x0004ec4868af75df, 0x000e466b71f6795b, 0x0000c72b0d7325cf, 0x0000e10c5f3da26c }, { 0x00094ad8a7522d79, 0x000cc411977a0ee8, 0x0004e8d5601f6c03, 0x0001e3b955085c4c, 0x000028f635d618a7 }},
        {{ 0x000f311e0eabc31c, 0x0000741ada354383, 0x0008c56a6e3fb07b, 0x00054b334f4a9fc8, 0x00001f2b7d334bf3 }, { 0x00027ce89b6036fc, 0x000043892ab98919, 0x0000ac10010e629b, 0x0008e3fee257bdcc, 0x0000d4e8e7aaee28 }},
        {{ 0x000cdb6e7895d161, 0x000cbca523db2808, 0x0009c3f8c90b1ad1, 0x000c6706f5c889f8, 0x0000605933f73ab2 }, { 0x000076d41f277058, 0x000c4f0179c8e744, 0x000207aa49414348, 0x00014ba00d8874e5, 0x000029e04258b6c3 }},
        {{ 0x0008a790b23045f3, 0x000333c47b517ea8, 0x000ca7c0cdac91c8, 0x000c03ea079a448f, 0x0000f88cfdd1c6c8 }, { 0x00044cc5afb2f094, 0x000787d80969fe8f, 0x00034cfc5609af25, 0x000a4e12654f92fd, 0x00002a65082a73a9 }},
        {{ 0x0006c0d50cd87d6c, 0x00017278b1e68b79, 0x0009997e54c71ee5, 0x00052d2cbc731ad7, 0x0000b44017339c7f }, { 0x000bf64f7d3a4aac, 0x000f3e79b4f7f248, 0x0007b1a318728a1b, 0x000e1748dfdd781d, 0x00003026e7244622 }},
        {{ 0x00007a3c1ceeb879, 0x0000d2c467be912a, 0x000926c2f4afd167, 0x000ffc5c7f4c85ee, 0x0000156de531a17b }, { 0x0002c133aac4e622, 0x000cd450c00beb33, 0x0002cd4c1893c312, 0x000229d5b30e2c2b, 0x0000fcacc4f9ea39 }},
        {{ 0x000894a646301fb9, 0x0005dbb32e5685df, 0x000860524efe08bf, 0x000a632c668a9e06, 0x0000a480e5c2b19d }, { 0x0007ec2caf1e7fea, 0x0004b0a09e27ecf4, 0x000e110aeaeed496, 0x000f877baeced201, 0x0000dd057fe48dca }},
        {{ 0x0007eca65a996482, 0x00015c4f68367ced, 0x000ae66db0fb60e8, 0x000342779e4f47d0, 0x0000d6d9565d0319 }, { 0x000b6873efddb317, 0x000a47149ded0a33, 0x0004bfcc5fd5434a, 0x000d273334cb2aad, 0x0000fecf3dc03882 }},
        {{ 0x00066010689e1641, 0x000f3b22c11bb373, 0x000db6ec643ab5eb, 0x000afb15d2d0366d, 0x0000ab615f997250 }, { 0x000b5f7e8a49c22d, 0x000402c68d7b6d44, 0x0008c124aab7be0b, 0x000a684eea8f74f0, 0x00006ee0d7c27a2d }},
        {{ 0x0008fc242b74d79d, 0x000d47789070d267, 0x000e6420679c51cf, 0x000c68217a1f28e4, 0x00000be63ac7dad1 }, { 0x00042c485ec72b60, 0x00065928b0588f1b, 0x000c610949c0029c, 0x000ba4e35b6b7a0f, 0x0000c2df5af332f6 }},
        {{ 0x000752e537b3aeec, 0x0005213dbab98aa3, 0x000b6a282aeced62, 0x00097413ade9d469, 0x0000c4a05092000c }, { 0x000617ba15d6404b, 0x0001fdd15544eb9e, 0x00069ae3e8640d68, 0x000dfb6b5f013aea, 0x0000c07b8448571b }},
        {{ 0x000d7158f2903d4a, 0x00038a6936219d36, 0x000440e072274354, 0x000ca5d4c5ce1f19, 0x0000aeea5b76a62e }, { 0x0008beb35eb7c48c, 0x000df21e300ce18d, 0x000f545273fe83bf, 0x0005f0f9b35fba62, 0x00009e3ee834d7e8 }},
        {{ 0x0002699749223725, 0x00006b01f7e320d9, 0x0003db3989a12b79, 0x000dc64a30641c36, 0x000085fd6388145f }, { 0x000b92552feb649d, 0x0005217ceb6c1434, 0x00018da120fb460b, 0x000267af867fcb84, 0x0000515b55c60526 }},
        {{ 0x00099bdc6fc777ca, 0x000fd0e46aca7930, 0x0004dbdd54abe6f3, 0x0008b2d161cf0b0e, 0x0000bd4ba21f4e90 }, { 0x000ce18473bb041f, 0x0000cb20caf8b051, 0x000a55f4b97f1131, 0x000c0d508f74d505, 0x0000a157aa07d7d6 }},
        {{ 0x0004d831d9980ef8, 0x0007d9783abdb4a0, 0x0006da10193eb141, 0x0008f6fdc850fbcb, 0x0000b1f65be994c8 }, { 0x000ca16d905a0431, 0x0007d3bfa12d45da, 0x000d9aa574b633e6, 0x000f1c21ee425f8c, 0x0000472d6784e4ca }},
        {{ 0x0006be873329a887, 0x000d36a9152f9347, 0x000d764f8b8de418, 0x0009ef6acf1235e5, 0x000039762606b9b0 }, { 0x0003ee10d17b708a, 0x000ee310c0cd3130, 0x0003a226153c53ce, 0x00041a3153c52455, 0x0000f1a929d0274f }},
        {{ 0x00079739f7318d32, 0x00027b21994ef202, 0x00093bf3a921a053, 0x000dc5e102a653b6, 0x0000417acbe360fd }, { 0x000b40cff123155f, 0x000073c8b799336a, 0x000c74fb67fa3a8f, 0x000c69c2dfdfdf14, 0x00008baaad38eb7e }},
        {{ 0x00090bdcfccc29da, 0x000d983d20b42d5b, 0x000ee66ca762f3ee, 0x00090c3f0f9a810e, 0x0000f553cd3456f1 }, { 0x000e248f78d9528b, 0x0007a91f01ab782d, 0x00008ccc9b0f3f1f, 0x0006eb901c823f21, 0x00005ba0497c7a34 }},
        {{ 0x0005b53add7650bf, 0x00085e50cafe751d, 0x000bff2797447909, 0x00034812473997c9, 0x000097d3990af468 }, { 0x000e25f2ce2b08a5, 0x000d511dcedd8d12, 0x0008468d39b5d584, 0x000d543b1406bd1c, 0x0000fd5a272530da }},
        {{ 0x000a70e6bf33183a, 0x000a35e29dfc254e, 0x00054127cba88ba8, 0x00015b8ad4455fcf, 0x00005553f496e694 }, { 0x00036bb6459a57f5, 0x0006c695820f3bf6, 0x000d3acc40499055, 0x0008db783f410d2d, 0x000031a7e7bc74e1 }},
        {{ 0x000fa724e36fba72, 0x000dca39754e21c0, 0x00067dbc9e8fb755, 0x0004f699abc87a3e, 0x00008aa8469ef3d1 }, { 0x0006c5db934e2c84, 0x000c829b0720b169, 0x00026fe7131a51c7, 0x0003ca5a62dd0cf1, 0x0000be10a91df994 }},
        {{ 0x00074764ddb736f7, 0x000ddc05f18395e0, 0x000418396b551790, 0x0004f105bd3f2274, 0x0000b92a58459d14 }, { 0x000f0b4e7fc1d0d6, 0x00013507bc7f3893, 0x00007a6a15e11d01, 0x000d3ab57c580c85, 0x00003331b44a24bb }},
        {{ 0x000f11236279ebc4, 0x000a9e8a9f19c539, 0x0002ca63bfeba930, 0x00064cba95717bd7, 0x000086b4bc72b477 }, { 0x00076fcfeec45e44, 0x0001a31823b73482, 0x000584f3707abbb6, 0x000397e584dd8f0d, 0x00002080c6a3d85e }},
        {{ 0x0001374518550834, 0x000e75e2bc0aea9e, 0x0004fcfb4f1939ad, 0x000dc847e4f668fd, 0x00002277741998e3 }, { 0x00010649ba9269ce, 0x00029e6853ac983a, 0x000dc7e1409c87dd, 0x0003f9fd1c3bdf42, 0x00001a9e558bca9c }},
        {{ 0x000984d8c0756da9, 0x000f60b879fbbed4, 0x000691183fa97cc8, 0x00008c6450ff0ae8, 0x0000bc7e3aa798e1 }, { 0x0002c0087c8490da, 0x000bba6d21008f03, 0x0000075a13ccfa3b, 0x000db76f329bd54a, 0x0000cdfd68fa494d }},
        {{ 0x000eda0108b2b2b2, 0x0008ae60c1cdb260, 0x000746a2a0fce8f9, 0x000ff4bcdcd94d94, 0x000051bef4868fea }, { 0x000105f6c2d3fcbc, 0x000aa1056152fcc1, 0x0006d6133470c109, 0x0003149a4c2f037e, 0x0000f1a1f67957f5 }},
        {{ 0x00013c1b9dfbee2e, 0x00007008733913cd, 0x00026dfec0a26cf9, 0x000b44ae9cce8616, 0x0000301eb9bba43c }, { 0x00058c5a3c8e1fd6, 0x000a8c71c74cf5c3, 0x0008bc18677805cd, 0x0001a3bf18887dc4, 0x0000f290c47af3bd }},
        {{ 0x000c23f6bd8f7001, 0x000c304890228299, 0x00069f5598bb730d, 0x0004e6dad40e9397, 0x00001a83f93d79a9 }, { 0x0004233cffc4cc06, 0x0001435fbffd8bb2, 0x0008f52696de0ca5, 0x00059d33a60fcfb4, 0x000067aa71b84bb3 }},
        {{ 0x0006fe382d94c44a, 0x0009a6ce6d982af4, 0x000ea056154ed2e9, 0x000c78687ebb6bf3, 0x0000ae247fcc48a8 }, { 0x000a247e575d5c4d, 0x0005ae8b5bdbd756, 0x00014ecdaaa75fdd, 0x000bff600ab38133, 0x000002fa41762570 }},
        {{ 0x000df2688e5c0bfa, 0x000f81b636288bec, 0x0003fab55a773328, 0x00085a4049dfdca7, 0x000085fc068e8d8e }, { 0x000976ef578a35f1, 0x000ee44d0f8a67ee, 0x000a21e1e0b95bd1, 0x000ef07f4d985df4, 0x000004f8b7366d81 }},
        {{ 0x000b539e244d918a, 0x0000baca1c323732, 0x000a47f86bef05d2, 0x000da4bbc351027a, 0x000011f52fa1b210 }, { 0x000bea3fa5e2c7e5, 0x00067ec0eaa1d710, 0x000d5a338fbff1d1, 0x0008ce2402a83381, 0x00009435f8928998 }},
        {{ 0x000aad621cd624b4, 0x000028436cc61347, 0x000345f6e1c0c72b, 0x00016c92a8eb9bbf, 0x00007457c7601162 }, { 0x0003e95ca5e2ca28, 0x00025740a8f7cc07, 0x000d936cd0bf865e, 0x000cd30eee820c2c, 0x0000d32e75fd1366 }},
        {{ 0x00075553bbd18331, 0x0005fabfbaf50a55, 0x000b10f8b3952494, 0x000f37f05df55e76, 0x000083667f4aec1f }, { 0x0001b91149c64f10, 0x0006ca3553afa736, 0x000beb9cb1077179, 0x000ef4d35315f3ff, 0x0000724cdc42e0e8 }},
        {{ 0x0006710053e87e6e, 0x00027d3fa326dc3b, 0x000e628b2c963b4e, 0x000250f591df4da1, 0x00007156cd0888e8 }, { 0x000ea109e476ac64, 0x000027e618344d14, 0x000251c33bcb204d, 0x000e1583acc7ce04, 0x00003fc838df305f }},
        {{ 0x00061e97bf3ccbad, 0x000faacb40961bf5, 0x00071486d43b1a75, 0x000ebb8d14ada593, 0x0000416698a685c5 }, { 0x00083dfe218457e7, 0x000beb6da3982f5d, 0x000cdd08f647376c, 0x0000946200c2659e, 0x0000016501cf88d5 }},
        {{ 0x000970713804550a, 0x000ba5fb835398c3, 0x000f7a19d08a80db, 0x000a0d2db6106a8b, 0x000024eccd926e31 }, { 0x000c5926d6e5fabb, 0x0007d34d8249f00d, 0x000ececc10c98ac7, 0x0006705c29f14b58, 0x00007dd9e5578e79 }},
        {{ 0x00059b045ae4cdd6, 0x0002d398e77738ae, 0x000f16095fcbf91e, 0x000b19886d07a63e, 0x00009b2d8bb94209 }, { 0x000115fba320b394, 0x000f0e520fa03389, 0x000859e95f2ba538, 0x000efef1397939ac, 0x00002792ae3d4093 }},
        {{ 0x00097bb53067a096, 0x0007ac25fadf8c35, 0x0002749e0bb22636, 0x0005dba2266be38a, 0x0000e8e24f086be5 }, { 0x0008e7ffb7148254, 0x00032b7c153ca7f9, 0x0002d7956406285c, 0x000de7e1ea8d30fb, 0x000062282ab3dba8 }},
        {{ 0x000b613b880d4a8f, 0x000eef8e9726a306, 0x000f446cf9abc9c9, 0x000a75d40b5e216f, 0x000085573a0b1122 }, { 0x00016530e4f15d5b, 0x00032ba792d56100, 0x000aa02bcafc965b, 0x0009826da2ee1270, 0x000079f2ed0cf73d }},
        {{ 0x000b9754a0266bcf, 0x0006efd8fd6ee891, 0x00062aed3b5fa54d, 0x00038abdf16d3d98, 0x0000116ae37983f7 }, { 0x000a47cbfca3dc3a, 0x0002cdf70c33b988, 0x000e54ec4c71ebb6, 0x000f60732aadc87a, 0x0000b626c156e4a6 }},
        {{ 0x0009f8db144f77e9, 0x000026662b5f3af3, 0x0000d0c25c1ff887, 0x0007b73b1cefab56, 0x0000cd5a1cf3f9ca }, { 0x000d5a818694755a, 0x000725b3c61bfd60, 0x000d33c6e57d6b39, 0x000d8c2c2a5a4d41, 0x0000da06aa84668d }},
        {{ 0x000f71b5f5f2e3b8, 0x000bcf7d22da9a9b, 0x00080d18e517fe23, 0x000df74ff45b8a67, 0x00009b0d34e92cca }, { 0x0001751253d7af6a, 0x000152f69fc4db96, 0x000c14e7ef8dbe16, 0x000e0dc35b6dad34, 0x0000dff44628ebf3 }},
        {{ 0x0005c4cb0846e24e, 0x000db9781af2c2d1, 0x00046bf50cd29179, 0x00070f08dc0f7d9c, 0x000019b4a7b7a5c1 }, { 0x000c9beacfa5ec60, 0x000d2858e121e14e, 0x000d98641f8ff20b, 0x000390274eea7dc1, 0x00008ec5b0cee6b7 }},
        {{ 0x000bad30cf8c7404, 0x000a6d71df978283, 0x00056ce647a78ac9, 0x000f06b3cbf851cb, 0x000022014a98c474 }, { 0x0009c81e9e8866f3, 0x000bf02496bbcea3, 0x000eded9ca1c764b, 0x0009f6d293cf9fd4, 0x000097210cbc6093 }},
        {{ 0x00067291d1c7e163, 0x000a9b97b390d355, 0x000d3fd8195610dc, 0x0000e01f801df2bb, 0x0000aeaffaeb93fc }, { 0x000e7ba92c88e286, 0x000166c5f5406d3b, 0x00089d2d39130c67, 0x00026e8ce71fdda3, 0x0000429a520ee24c }},
        {{ 0x00059814cd121bf2, 0x000e16bc6fb5a7df, 0x00097795bc1333ec, 0x0007bb16ae3eea15, 0x0000ea9234811e7d }, { 0x000b866e7b5afbdb, 0x000eab01c33c23f3, 0x00043423d2bafbcd, 0x000358a51a96c1da, 0x00000a66348cf983 }},
        {{ 0x000243ce8547c141, 0x000e6d81495ed6bc, 0x0002f43f0e622ab5, 0x000c4cff3b239424, 0x0000291035b7c92d }, { 0x000997f8cbbf2476, 0x0009771dbbf370fb, 0x0006bf48fefebc0d, 0x00056dc3fff9afa3, 0x00000d06a64f152b }},
        {{ 0x00080c94a72ec503, 0x000ca2a37bbb1840, 0x00099c508afef996, 0x000d131cb51dcec4, 0x0000989196a9bdb3 }, { 0x0008f4ead02f1a6d, 0x000d0b12283fbe37, 0x0007e19e2b1d8b3d, 0x0007bdeec902bddc, 0x0000d0bde74ee357 }},
        {{ 0x000633d510319630, 0x0000a4956f908239, 0x000fdde9a26b1a49, 0x000f5c760ba0f507, 0x000011a47fccb3c1 }, { 0x00074a4f6a780782, 0x0003543b8c9de617, 0x0003e69cf7b58be5, 0x000c69ab2b475125, 0x000059a05b225efc }},
        {{ 0x000530e06a0e952c, 0x00059f773848f22f, 0x0005fec775249e0e, 0x0002f45f953dad71, 0x000030c4df804685 }, { 0x000aa392d634bc5b, 0x00056ffdd59d31ae, 0x000e1e3714bf08e3, 0x000102dd74382e0d, 0x0000d560753775a1 }},
        {{ 0x00032c96e155ca24, 0x0001867648024ee6, 0x0002290314ec3e56, 0x000e4a435188763f, 0x0000359aad14cb65 }, { 0x000aec2c62bf9c7a, 0x0002c0e4c00a95ea, 0x00003525bd976a14, 0x0001af8232ef17f4, 0x0000950b712f6e56 }},
        {{ 0x000cb27078304cb0, 0x000d8d83d84e58c4, 0x000c48d30163a309, 0x000b9d2a62af5094, 0x0000f43399bda145 }, { 0x00025423b825ac58, 0x0006d474ff0e321a, 0x000578e0a546e9ad, 0x0000fc3a7169f34a, 0x0000fbaf5744de2d }},
        {{ 0x0001d6998c9bae7d, 0x00058d50ac8b7823, 0x000bfbdd2f05c1a7, 0x000dd3942ff6622e, 0x000068b2d4cab4cf }, { 0x00020e9934c184c5, 0x000bb04cadcaed01, 0x0002c60e6c081a2a, 0x000de49c6801b014, 0x0000c34427b5c797 }},
        {{ 0x000321a50d8134d2, 0x000c8a4fdc6a4043, 0x000e6e6ef43ad26d, 0x000b3c48119d8e01, 0x0000448b0fd8e456 }, { 0x000066350f1fe586, 0x000e263e9a861112, 0x0000e63d8756b063, 0x000ba08ef8c0cde6, 0x000050af05bd3adc }},
        {{ 0x00058deddc932110, 0x0003689e56ddfbd6, 0x000798a4aa5dbc69, 0x0001ba8cbea1b0c0, 0x00007b58c49685f4 }, { 0x0002efa53a6c1ccc, 0x000e34d0f64ef589, 0x000ce01d226df9d2, 0x000e7b170fe43287, 0x00000387e93337d0 }},
        {{ 0x000bbb69838b4dc3, 0x000c15794b565268, 0x000e7da596a97b99, 0x0008eee0f415cb80, 0x00008189781ee1eb }, { 0x0008da59e33a17c1, 0x000b85b9abc26689, 0x0003858d35ce15d0, 0x00065e1b9b5d4f53, 0x0000e8073f5adb41 }},
        {{ 0x00041b32e5ee0cd0, 0x000bea601799a527, 0x00014451f66c2b00, 0x000f25a7ca20cec4, 0x00001caa549ccc91 }, { 0x000641caaadc47ea, 0x00097814eb57d471, 0x0004006d25c11bc5, 0x000a4cb887a2d0ed, 0x000005cf797f0540 }},
    },
    {
        /* digit=4 (1,2,..,64)*(2^{28})*G */
        {{ 0x000cf9a2f2b6ddb7, 0x000cb4f20151427a, 0x0005e5495f13c8bb, 0x000336c57206828b, 0x0000deb474abe52a }, { 0x000c630fc54290e0, 0x000e9da279153564, 0x0006b3bbef1e9949, 0x00021a8e27e0734b, 0x0000614b2885f690 }},
        {{ 0x0004ab68d7e959ee, 0x000227a1ac2703b2, 0x0008471ceedd9bdd, 0x00080ded1b49258d, 0x0000011c8d9a06d3 }, { 0x0000b360f8d06cf6, 0x000bfd4734e59d08, 0x000843e26b87ae6b, 0x000b0c3b35f8740c, 0x0000b6d468726158 }},
        {{ 0x000ac1e3e5d6df58, 0x000075d5fa067d1d, 0x000ebba6b226d7b2, 0x000af82ac4134b57, 0x00001bb953caf0cd }, { 0x000cde538c8bc9c2, 0x0003669d81fbb26c, 0x000be58de047db86, 0x0007841e8449ed3e, 0x0000d643fef5794d }},
        {{ 0x0003331d3c69f2bc, 0x0000a8da61a76fa3, 0x000d96e2379febe3, 0x000486b9c598b441, 0x0000390a58c27d0d }, { 0x000cdb9daf4c5417, 0x0006d8e298de399d, 0x0006c4c6e1a6e0f8, 0x000649470cfdb8e7, 0x0000690440cf3b0b }},
        {{ 0x000cd3d0d4027b25, 0x000d3c2019f2a596, 0x00035e7beba07964, 0x000164eddb3ce5bb, 0x00000d8037d39488 }, { 0x0000c29968554c2a, 0x0009ad6efed021c1, 0x000059b49a323da0, 0x0003c9c4a09a9f9b, 0x0000c0cc53fbad1c }},
        {{ 0x000c73d2287162dd, 0x000c529ed6e82ef1, 0x00034977dd25e6f6, 0x0004a8aae16f338e, 0x0000dc61c21e5eed }, { 0x00096fe447e4eae7, 0x0006a3174a5c8c7d, 0x000fa5f9c92267ca, 0x0007c117e01cd296, 0x000068abc8039604 }},
        {{ 0x00041d6367599350, 0x000706e88fbd3813, 0x00042818412bda97, 0x0005e6bcd02cd4a8, 0x0000b40e707676fb }, { 0x0006fe0a3a3a6e7f, 0x000d8fbd238202b9, 0x000a1286e09fb174, 0x000d4efa0600b4d1, 0x00005566cd140a17 }},
        {{ 0x000246780b6116f2, 0x00015367f636a38c, 0x00064c305dbfeb38, 0x0009f998b9f943fb, 0x00002a4f058aaf47 }, { 0x000300fadd2ed37d, 0x00097156d549fc2b, 0x0000ad7686ed0d31, 0x00005d713c2953f3, 0x0000c01130ae8caa }},
        {{ 0x0000e57c5b8cc8c5, 0x000b65087e9189fa, 0x000c53485dcbf535, 0x000419404db17375, 0x0000410edb213b42 }, { 0x000cf14a112a5ea6, 0x0006bd3f40e3c80d, 0x0000b0c5bbb47bb1, 0x000a038c3b118932, 0x00002d9c468c8d26 }},
        {{ 0x0006c8b3e970a581, 0x000a55b556913429, 0x0003fa3be9dd94c6, 0x00015610ae8e0850, 0x00009ceceff70c59 }, { 0x000953c3d70998e4, 0x0001438fb4ee6b40, 0x000b3895091982c7, 0x000476cc6fc5a9ae, 0x000057745cd78df1 }},
        {{ 0x00071e305a112a00, 0x000268238dfe7e82, 0x000504996cf1100d, 0x000bdcf2dea417e1, 0x000026436feec265 }, { 0x000396770a6a23bd, 0x000c0fd7253ecbd7, 0x0004dc33ea7bf828, 0x00062b19c2f552e2, 0x00001ac28594c2e3 }},
        {{ 0x000b1dac7015ef7f, 0x0009263c27901b4b, 0x000fce7ddf838b5b, 0x000d75b0afd9236d, 0x0000069d82647cb5 }, { 0x000f374bb4fe7c59, 0x000c453a91cf76e1, 0x0002b3e7d3328269, 0x0007376774110a41, 0x00002b0cccd10515 }},
        {{ 0x0001616338327868, 0x0007af20d188dab3, 0x0000c35722b72c00, 0x0009bef0fe3968ed, 0x00005947d75b5640 }, { 0x0006a44d86a5f4a0, 0x00047e24f0c8afe3, 0x000fa8462b5372ed, 0x00068ea514caa75f, 0x0000f6f22bba6bd6 }},
        {{ 0x000ff17a8fcc8db3, 0x000a8da0333974fb, 0x000d67b586b70a9e, 0x0008146215db516a, 0x000074410c8af154 }, { 0x000c274e88147e8f, 0x000f24e12476da14, 0x0002e79a1af86475, 0x000cebccac1e1661, 0x00006666b278e72d }},
        {{ 0x0001da368c16a10b, 0x000ad8d6c2f5226c, 0x00039333461ca008, 0x000230c0914d1b94, 0x0000a4acd35a7009 }, { 0x000b480cfa6c0f39, 0x0004cced5113cf71, 0x00033eec6394bef8, 0x000bd06880670f63, 0x00008997a9636ca9 }},
        {{ 0x000953c50eb5331d, 0x00069024147519ad, 0x000b4d5fade16af5, 0x00052cec8d0981ea, 0x0000f785b3ffbbf2 }, { 0x00063ebb7f24f144, 0x0001dccfd590f8f8, 0x000abea80d987872, 0x000fa20cf72e9626, 0x0000721e8be990fd }},
        {{ 0x00083e50e54aa3f3, 0x000c4616aca8ee7c, 0x00090f75b27323c6, 0x000bbf80d97d10b3, 0x00005e98ed7591d3 }, { 0x0005f9680c64a8ca, 0x000dffec954b3132, 0x00070b13b7552acc, 0x000af06e0cc2e8f4, 0x0000143469e3b900 }},
        {{ 0x000561bf1c36fc6e, 0x000f8094354080b3, 0x000546913f00c25f, 0x000812afa5225bfa, 0x00007cc75ab3c625 }, { 0x000332d9e420107e, 0x0002117ba1514741, 0x00049d1c94e63e5f, 0x0008a4d1edec2f93, 0x00007771c10cbd06 }},
        {{ 0x000cf057fdaa6cc2, 0x0000d69df23de05b, 0x000143a2e1a2b66a, 0x000fc772366d17da, 0x00006a8d4733875f }, { 0x000c23b3ee9a963f, 0x00089c5f0b5662c7, 0x000860969b36292f, 0x000952aff25318dd, 0x0000ef7f5b56866d }},
        {{ 0x00039af66ebf381e, 0x000ec0a097e4403b, 0x000565ea4ff656a8, 0x000452a26cb3d0a8, 0x0000a7947609a317 }, { 0x000290d33bd0a0a9, 0x0009c88bd475ca8d, 0x0005ede6e4cfd558, 0x00087196ce06b796, 0x00001bfceaea4a09 }},
        {{ 0x0003e4fe080af01f, 0x000ad47f34fb0fa1, 0x00080bccad01991b, 0x00083a0e3b4760cc, 0x00004486d52a8b97 }, { 0x00069059367f304a, 0x0006f92fdb05163c, 0x0000459ac07fbc9d, 0x000c299e28c484b1, 0x0000e9d36b72292b }},
        {{ 0x000ab74e6a2a524b, 0x000249d586e74e18, 0x000cdd00b61a885b, 0x00049f2db488883a, 0x00009b15259f5d0c }, { 0x000de0781ad6fc39, 0x000f4285e21beb19, 0x000993abcade39ff, 0x000d307c33484518, 0x0000f3708fe8c17c }},
        {{ 0x000604437bc3417c, 0x0006c58693807e1b, 0x0006987d894bcc6d, 0x00041df84a386532, 0x00004f10004876ac }, { 0x000a01df0ef92b9e, 0x000ec985ca1823d5, 0x0008a9bc2415bad0, 0x0003cac33b82f7ec, 0x00008c8d609652d6 }},
        {{ 0x0005e94efbb5789d, 0x00065f31306583d2, 0x00001d931332b2e9, 0x000f1dec805b10da, 0x00003f4f29f81351 }, { 0x000ede6596524043, 0x00086b7fa3dc26dc, 0x0002c8209e2f37f5, 0x0008351cee0e5fb9, 0x0000f62ae015ad99 }},
        {{ 0x00025ef904c10f7c, 0x000a6f6a19935459, 0x0005927778b6116c, 0x0005a468eab01cc3, 0x00007a92aaab72e0 }, { 0x00060897892570ee, 0x000b5cf53aa0b32d, 0x000a9bbc05f115be, 0x00094f47fb27beb5, 0x00009605766eabab }},
        {{ 0x000cf6c431cbebca, 0x00082e0d6fbc2acd, 0x0000a469311f571d, 0x00003479c738037a, 0x0000ebc968b789a0 }, { 0x0002ad6eced662ff, 0x000ff7be3ef104a2, 0x000be758e7592a81, 0x0002f053eebdd9a2, 0x0000d738f011f4b1 }},
        {{ 0x000c11a2e84862b7, 0x0005cc039cd297dc, 0x000d39ac02f15755, 0x000ad94968ec7e16, 0x0000f7a4858d93c4 }, { 0x0007c422e7b70ed4, 0x000865396ee63809, 0x000536df645adfbb, 0x0003f267d054c865, 0x0000eed4df196b1e }},
        {{ 0x000ba62aa6a9ec8b, 0x000ea090c22b5403, 0x000a7c1c3d2e15a3, 0x000c44b932cde42a, 0x000080285f5a46b0 }, { 0x00066c5b9ee54102, 0x000f77bab9ea96a6, 0x00027d49b293cd0a, 0x0000c99d55d03964, 0x000042cd3eeb6797 }},
        {{ 0x000a56a7143dc2a2, 0x0003ad98edbd7cc8, 0x00073c76303fbc7d, 0x00024144f4660bb8, 0x0000534e55030be4 }, { 0x000b2524cccf0e8b, 0x00055965726d47f7, 0x00018ff575b2dcee, 0x000a8bf379eed01b, 0x00003984d331077f }},
        {{ 0x000539a4fc4c4c2e, 0x0003a3b8f4b002f6, 0x0009dcbb57241ddd, 0x000e75cccc2200d7, 0x000071b64d7d7ded }, { 0x000b7a26a5d05150, 0x000cc14c33c7fcf7, 0x000b4c440b85fe71, 0x000f5a6237eb6799, 0x00004066761e4395 }},
        {{ 0x0003ed2c34dbfa56, 0x0009a8d403f46f2f, 0x000226219492ef3d, 0x00037be2f94b8fc4, 0x0000aea3923eaee7 }, { 0x00047ff958e890e9, 0x00097afb639126c8, 0x00005ed54185a48f, 0x000ed4209f7b6be6, 0x0000839d619a4633 }},
        {{ 0x0009fb18a0fee18b, 0x000f1f5678a31b0f, 0x000d3f9d9d7f8899, 0x0002e2896d50301a, 0x0000467828ca24c1 }, { 0x000a2f2a98bf1696, 0x00042ebe8758851b, 0x0005a36a8fac4075, 0x000e6fdb71e64e4c, 0x00002450761a46be }},
        {{ 0x000106b953c0b0b7, 0x00055bae8053433d, 0x000f6c25dc31bcc8, 0x000283eedc8e7259, 0x000070486fc2230d }, { 0x0009b714490d84cc, 0x000ab5b0eff13a96, 0x000a76a0a4e72ed7, 0x0003240bd242442b, 0x0000e38dc2108bbc }},
        {{ 0x000ad0436ab80ab8, 0x0008ef60e2ed7422, 0x00051536312889e3, 0x0004a0b04573f104, 0x00006d3f1bc994e5 }, { 0x0002b773beca9fd2, 0x000d82d431a8121d, 0x000ab5b79e2f23aa, 0x000aa3cb7cd38b3a, 0x0000164fbcfce738 }},
        {{ 0x000d927bed2c4843, 0x00057c49b5c1f14c, 0x000970c2633e1e0d, 0x0005bd591c444be1, 0x00007472609e17c1 }, { 0x0007c6a79d00e920, 0x0006516e48c8a348, 0x0001c38dbeccea53, 0x00064d8ec1b7ac81, 0x000001913870f958 }},
        {{ 0x0000b9e865afa38c, 0x00016f09b56ddbdb, 0x000bd975348dfe5f, 0x000fa730d1339b5a, 0x0000a2f19f0d61d3 }, { 0x000bafec279d834b, 0x0007f4177bb064e9, 0x00026174b33bb275, 0x000ec1d9a145a281, 0x0000853c8b1d855a }},
        {{ 0x0004b8f6ae320ba5, 0x000a7654265aa91f, 0x00030ea60793f5f9, 0x0002e39c7a4b1830, 0x000052a5da5f3864 }, { 0x0008eb65e67a4037, 0x0005ccc971a8a130, 0x000d43a7e7218152, 0x0006c85293f159e5, 0x00000f9da5baca8f }},
        {{ 0x0007b514916e3af9, 0x000fc8930b75c3ed, 0x0001bc940801e1b4, 0x0007469bb0bb82fe, 0x0000306f321be91b }, { 0x000a91fcbaa89e73, 0x000277b04497f19d, 0x00093fe8d485032c, 0x00029fbd61b35ed0, 0x0000ab396c946696 }},
        {{ 0x00043b4fbd7ee0ab, 0x000a57bd7ab27e14, 0x00051f590bbe1f47, 0x0001bed277c19a55, 0x00004ea5e94be42c }, { 0x00079cdff78fa878, 0x000b6656962c88f7, 0x00045670d0d30682, 0x000e4f44f139da8d, 0x00001bd14a9de58b }},
        {{ 0x00003907bb055ec3, 0x0007cddae57c7b7a, 0x00017b5a1d67b643, 0x0000e87a3345342f, 0x00001c62c23c4350 }, { 0x0006a27dca73d570, 0x000a0e1ff2189336, 0x000443af93b0c4da, 0x0001d76eac066b65, 0x00007c9b470b5621 }},
        {{ 0x0007ec6e15fb0c4c, 0x000ad93048b37172, 0x000a898c40066ebb, 0x00042d16bbf5a989, 0x0000641d1a6e425b }, { 0x00009c3566d8138b, 0x00078737276e7fc9, 0x0008f93a899b74ba, 0x000ba4e7964086d7, 0x000097a1ea45df74 }},
        {{ 0x000ca56513af6eb9, 0x000036b75dd7125f, 0x00068afadd796c99, 0x0007cb29db4c6bc9, 0x0000450348223f4e }, { 0x000cf70def0b74d0, 0x000a181d0e8bd06b, 0x000709c438fb0a64, 0x000dde624384dd31, 0x0000e9a2bfac312d }},
        {{ 0x000ec09a48204c79, 0x000886e6628e8c30, 0x0007a749dada16de, 0x000e7d5e207a0d66, 0x000034e09153d2e9 }, { 0x000c7f44552abda4, 0x000994ed83dce467, 0x000d1d53b819438d, 0x00011604c43a6316, 0x00000c57dbe9342c }},
        {{ 0x0009b1ddeddfb155, 0x0005c21fbabbe926, 0x000b77812288d335, 0x000d7ab15281850f, 0x0000e27f34f8feb7 }, { 0x000c9f1b87c58c64, 0x0004d8fab1051053, 0x00016bfdb467b764, 0x00004dc09953ced3, 0x000034ba3f40d826 }},
        {{ 0x00067e10d16361cc, 0x000d5a14a7eadcb4, 0x00010bb060fe6649, 0x000cc8383928e750, 0x0000a507d1a255d6 }, { 0x000d744bd0c7e26b, 0x000bfc3b955dcf05, 0x000d766199572b61, 0x000958974ba96307, 0x0000c161d1c03e35 }},
        {{ 0x000b7e2a947ab5c8, 0x000602aa777899e0, 0x000fc4c1124a57d1, 0x000fab531e20eda8, 0x000078d9b61326d5 }, { 0x00039bad6e2b1553, 0x00011ad19f593cf4, 0x000edadb41af06b6, 0x000180d9c7b1a9e4, 0x0000bec906422f8b }},
        {{ 0x0006fb4c2d81211e, 0x0007266e032f1445, 0x000891b7a03ca07c, 0x0008011781db632b, 0x0000932932e1a77c }, { 0x00027d6babb8c904, 0x0002ae1255a03f19, 0x000256ed85caa685, 0x0009fb89761cd6af, 0x00003a067d13fd17 }},
        {{ 0x000b89a4e82f20ac, 0x0005be0e90fb21e5, 0x0000601e57b9726e, 0x000acbbb500fd2b8, 0x0000def3cf863f66 }, { 0x000f548f31add03f, 0x0006e3f894407ce5, 0x00040ccdd200e1b3, 0x00036823ba0025b9, 0x000057161113964b }},
        {{ 0x00088658197f5e5d, 0x0000e8feac6d1135, 0x000bb5fae45cabee, 0x0008397b9a67e3f9, 0x000027605a7662c0 }, { 0x0005660e1a5eff95, 0x000cf0cf7f06f200, 0x00062500dda83af3, 0x0007a8908dc61021, 0x000035ac8c485b17 }},
        {{ 0x0000a886286039e7, 0x00030626d3c46934, 0x0006f744f5c20ad5, 0x000213b6df0e7d62, 0x000063e4a8b99f57 }, { 0x000bf9a358ea18f6, 0x000c4a7bbb38ce0f, 0x0004c1e30a289e84, 0x000b0711fe0404b6, 0x0000cdecdc1fe1a1 }},
        {{ 0x000970195f38876c, 0x000fdb028f33398d, 0x000c9978f6062539, 0x00023b2b1b07ec11, 0x0000e90d968debc2 }, { 0x0003e03b335e7e4e, 0x0000ebe0181e6402, 0x00020262f609f432, 0x00074d182f5debd6, 0x0000f8b0b29420ff }},
        {{ 0x000447ad7afa3513, 0x00076494fde0c1fc, 0x000e216f3ea60540, 0x000acc86cbf8a1ab, 0x000009a60fde808d }, { 0x0006736b8b4a3189, 0x000213803d0ccf24, 0x00083b271077fea3, 0x000e645cc80acb33, 0x00000d9f85e61323 }},
        {{ 0x0007a56d943a749e, 0x000dd60c0432f963, 0x0003a70fd69b9be8, 0x0008acba2ddebe7a, 0x0000b67c71837343 }, { 0x0008b4408a4b9b5b, 0x0002fedf2350d0a8, 0x00057ac6cd104010, 0x0002b01ae13c3f27, 0x000012894baff1f9 }},
        {{ 0x000b605ef4a68b33, 0x000b2a35cb026649, 0x0000eb26683a34fd, 0x000f7a23e5d4c0bf, 0x000025f5686f6dff }, { 0x000715b3a144285b, 0x0006702c69645d0f, 0x00097305b3462d92, 0x00066f2c4a115f03, 0x00002d4b3efd98d1 }},
        {{ 0x000d8c3079ecf96a, 0x000a048d413213a4, 0x000d72bfbc61a285, 0x0008c1df52035806, 0x0000eb8c40841f70 }, { 0x00042be3c0b0a8c3, 0x00067678d95a8f6a, 0x000d4de5294c6a81, 0x000e6f73fc58c9d7, 0x000028eb6b61151e }},
        {{ 0x0001f8fbd7da4886, 0x0007142d43b5eaaa, 0x00063dac9fad46de, 0x000237745054a076, 0x000066d55ce85e38 }, { 0x0000aad9183a3f7e, 0x00002f5d563fccc1, 0x0005e1d5d00ff737, 0x00075facf48ca505, 0x00002d09a02699ed }},
        {{ 0x0005475f79a136dd, 0x0009e62b7bdc9534, 0x00021679276438a9, 0x0002811779f0f6ff, 0x00006a65c14a85c4 }, { 0x000d71419bada82b, 0x000fdfa86f702ccb, 0x00045190dd6b9201, 0x000e77dc692e6090, 0x0000c7da1d12ccd1 }},
        {{ 0x000831b6ca7da654, 0x0008dcc6028da9ad, 0x0009ea203ea89978, 0x0000343f45352fe1, 0x0000299e73c0a3c2 }, { 0x000d50ba88136cf1, 0x0003d1de350e2c27, 0x0008a0bc297b691a, 0x0000804cbc56240d, 0x0000b08ffae6ef56 }},
        {{ 0x000b049ff6e562d0, 0x000c4265a3a2518a, 0x00091b2b9d4cf195, 0x0003b8696cf81779, 0x0000718b68347a7c }, { 0x0000d0bf5e9df0d2, 0x00027b0feb13ec70, 0x00079cbf39ae46dd, 0x00009c4afd5e9711, 0x00000194165fa7fd }},
        {{ 0x000d542a16dde633, 0x000fd1f358bcdc04, 0x0008bd491de20858, 0x0005da13a07689e9, 0x00001aaffaa71871 }, { 0x0009749558e9ff03, 0x00090d59ca0a7072, 0x00056151eb0cc4a3, 0x00090fd8cd061fe4, 0x00000ed0be2cc216 }},
        {{ 0x000229acb440cbd2, 0x0002a59c1d5d0340, 0x000665b8692c33ea, 0x000362ee6f712e9f, 0x00003d3bf4320945 }, { 0x000d5dbb02d12e1f, 0x000c24cfff0c82e2, 0x00022c68e370cba3, 0x0009e9a8987684b6, 0x0000967e564a19c5 }},
        {{ 0x0001f390883520b9, 0x0001a3ae86b173c3, 0x000575ac605bbd0a, 0x000ec5310e8e41ea, 0x00008104ffff9da7 }, { 0x0005cd6ee65978c1, 0x000b0e17829839ea, 0x00039f353f7221c6, 0x00083a0150cf691b, 0x0000faa310dfd765 }},
        {{ 0x000fa20deeef2680, 0x0000d7b5f1d23474, 0x0009fd20174b3146, 0x000121e08794645b, 0x000077749c4b1069 }, { 0x000a1c9150ecf76e, 0x00004d8d797770a6, 0x000d55260cd5fea9, 0x000b814797360c91, 0x000033f4db806293 }},
        {{ 0x000379ae15e41fc7, 0x0000ca7c35d87949, 0x00076fdd0ccf2ac7, 0x000135bfd042e655, 0x00007cefff827780 }, { 0x000b09f648a49efa, 0x000b462fcc9981ea, 0x0000c21f3e7dc01a, 0x000af6bebaed3d63, 0x0000701de9b6b9b9 }},
    },
    {
        /* digit=5 (1,2,..,64)*(2^{35})*G */
        {{ 0x000a823a4160c545, 0x000f51222248acc7, 0x000eb7e2a9255142, 0x00038a9281f6ec0e, 0x0000f63c9cc1c2fa }, { 0x000efe9ed3f2c904, 0x0002604e1900a791, 0x000aaaefc6a23190, 0x00095a14acde09d4, 0x00007b2680ab6b58 }},
        {{ 0x000c47940a63bdc1, 0x000acbe305406dd9, 0x000f4115854d7ebc, 0x000bf8c8f8eb7dc7, 0x0000eebcc0956bd4 }, { 0x00040deabe2974cf, 0x0008c5d99b307781, 0x00065a6b4bc494d8, 0x0005271e325380c5, 0x00006c75941c35b1 }},
        {{ 0x000049d7316af76b, 0x0006567afd32acfd, 0x000a11ad459dd98a, 0x0009f87dad8f0fcc, 0x00002a6cb670db4a }, { 0x00027074910360a8, 0x0007e3d42ab580e7, 0x0000b1869323c43e, 0x000fd66da09411bb, 0x0000931dee156e7f }},
        {{ 0x0003c3a8ea6b2c0c, 0x0005526c49a861ec, 0x00051d8b641c7fc2, 0x0006ff017024f3b6, 0x00006af90c54fe5b }, { 0x0006848b57a6af77, 0x000fe2cb103d4c8d, 0x0002bb9428138d50, 0x000a1d94a02bc461, 0x00005196aa6193cf }},
        {{ 0x000a9423bc0124a5, 0x0005de6c30fa92bf, 0x000c9fc103c8bc73, 0x0005e719b5a70d96, 0x0000e39716462e3e }, { 0x000d96bc718ee9dd, 0x000e112fe97a5b9f, 0x00087422d4a6a114, 0x0002dab4685da604, 0x0000ff2e0f500dd4 }},
        {{ 0x000f0e96b0521716, 0x0006bdb1782269b6, 0x0008b6894ea0436c, 0x000213b53ae34bf7, 0x0000f89e64dfb40d }, { 0x000020f0e5f7aa5e, 0x000a41577884456e, 0x000b89101dc1c7bf, 0x0005074deb3b5688, 0x0000322f52afaf0d }},
        {{ 0x0002cfa977b4a1d5, 0x000cbab0976505f7, 0x0002968e854dbe69, 0x0008fd671567f681, 0x00001688d84b98ae }, { 0x000f34f1cd8b8aab, 0x000b3ad8a2373da8, 0x0003ec6f689aa85a, 0x00077f215a8d0d5f, 0x0000c0b67348ce77 }},
        {{ 0x00039b66645c58c8, 0x0007a8dbbe5a1a9a, 0x000e8d8b5703704b, 0x000b790268187fd5, 0x000003b8f3f8d5eb }, { 0x000b2f3a4a4db66a, 0x00074ef2d65a8087, 0x000f193430cf1f8c, 0x000a6ec4c4969044, 0x0000c1c8991dfd8f }},
        {{ 0x0002864e5048aa8c, 0x000f145300eb294a, 0x000d7415c6444b6a, 0x000f9eea92b2a656, 0x0000f466251b268a }, { 0x0006be862b1eb6e3, 0x0008947db89842e4, 0x00071a89198c63ba, 0x000fc13dc12dfd7d, 0x0000921392892bc1 }},
        {{ 0x000b8888bb33c9a0, 0x00038dd0f82b2148, 0x0009657b430acfe7, 0x0000dfe19460c34f, 0x0000b39b6f7347e5 }, { 0x000bf0221449dce2, 0x00055dcc07226779, 0x0008802bed13455f, 0x0001abec610d21be, 0x0000f48269445497 }},
        {{ 0x0007c2ae5722977f, 0x000fe1242b48b998, 0x000cc07f923ee912, 0x00063ddfb1bcb665, 0x00006426594d7fc9 }, { 0x0001f5e5985dc788, 0x00029220de367261, 0x00082de9d7899404, 0x000b24c60e7043c6, 0x00006397a224d1f0 }},
        {{ 0x000dd04b97f56424, 0x0001d60b7fe7b6e0, 0x000a3239f41aa1f1, 0x000c4a61e7d16189, 0x000064bed27452ed }, { 0x00079b5499dfb43f, 0x000fd9b506db8cc0, 0x000ec79d2cb2ca13, 0x0009706e65cd47aa, 0x00001dff152d73d6 }},
        {{ 0x000ad76ee910eedb, 0x0003602c1c24a3b9, 0x0007813ead31dac1, 0x000012f4aca24e56, 0x0000638656568026 }, { 0x0007b4a7bd68d6f7, 0x00028c2538e0ff72, 0x00093e500bf32e6e, 0x00072d8d794f8980, 0x000026a43392b0c9 }},
        {{ 0x000aa7cb935a507d, 0x000551ae0ac29416, 0x0004582b4da3a965, 0x0007915d0279b025, 0x00008622b071bf70 }, { 0x00035a70c90666d4, 0x00051800d37c7a50, 0x000730d2e35953b1, 0x0006c1b9213380c3, 0x00008f95b8909dcf }},
        {{ 0x000c43c6cd7fe37b, 0x000aadd4ce4f152f, 0x0000fd9bb539417c, 0x0000140141f419a7, 0x00009af0fad64712 }, { 0x0001a9c2a0337ed0, 0x0009591925d6b2de, 0x0002806ea2f81671, 0x0009c46294b37d9d, 0x0000a4de29b6d19c }},
        {{ 0x0003fba6baaee763, 0x000548571ba18615, 0x000c7ccd5282a222, 0x0004c589348b22d5, 0x0000c343e640ecd7 }, { 0x000d50cd542a5f8c, 0x000f1f3e5eea7d82, 0x00063b79a4045592, 0x0000f8e05521c879, 0x000038b6e3d1e7a4 }},
        {{ 0x000a89dbdc1abdd9, 0x0003cd6c5a3ef169, 0x000d1f7db645ad63, 0x0009fcf1c993eff9, 0x000061650e298230 }, { 0x000c6ab8862e32eb, 0x000b3bf2b892891c, 0x000090a9ec4f324e, 0x000cab92153902b2, 0x000009cc8b82cba9 }},
        {{ 0x00084b1fe4ac70cd, 0x000d433f74cff170, 0x0003e46cf40d687a, 0x000f58eee240aaa0, 0x00005873e1125bc2 }, { 0x000ada9f07b6f730, 0x0002d16d081f6a60, 0x0008e41d8d8849a6, 0x000a77a2235304fe, 0x0000eb9e1fd7cdf3 }},
        {{ 0x0001bd2ee144dc4b, 0x000cb78968a61446, 0x000ba5f5eb3d6205, 0x000a5b0e154c1f77, 0x00002bb9da949d12 }, { 0x0001db4ebe3697d8, 0x000c9e9781105fc0, 0x000f79543fc2bbc2, 0x00067b99b9cf2971, 0x00002c5b8ddd5497 }},
        {{ 0x00065daedf9277bf, 0x0007f23b40df1cf8, 0x00030e8bc2d4eb23, 0x000253a215933737, 0x000043cb71cd34d4 }, { 0x0009699f551d880e, 0x000fdd3832791254, 0x000eeb833b56212e, 0x000ed4eafbc3d5b9, 0x00005766025c2941 }},
        {{ 0x00074ce7196c4a18, 0x00058e1d4a7a15b6, 0x000a835fa0cdac0d, 0x000e71bea08f3587, 0x0000153a1d49c1da }, { 0x0001ea48a86150df, 0x000b935088b3e0e6, 0x0008fb22fef91a3c, 0x00023c4c47a0356a, 0x0000149d050cb103 }},
        {{ 0x000a6b2421ccb0e5, 0x00092580f3e48342, 0x0001ed3e8ba96390, 0x000e982b3fdbb21c, 0x0000ca17b5625d62 }, { 0x0004efa84d6f03e8, 0x0003f944af02291c, 0x000565685cb0f5ba, 0x000eca7d7ade465c, 0x0000883813fcec37 }},
        {{ 0x00007ae2719b9b5b, 0x000c02a442b0f5c9, 0x00073f1e94c5061d, 0x0004db316b748e31, 0x000031983bdb4e33 }, { 0x0001a9c0d712eeb4, 0x0009f4b0c9bcddd6, 0x0003d8040daccc42, 0x00037d06a043fb05, 0x00006a4a48036cc5 }},
        {{ 0x000ac71e23fc88f2, 0x0008e46477184276, 0x000b03bbf167de2b, 0x000112147df17ff9, 0x0000d1d24f52c4e8 }, { 0x0008aa8296dcbae1, 0x000d8fd1defd73c7, 0x000743117c2eec10, 0x000c417ea1984065, 0x0000c69deb7b1ec1 }},
        {{ 0x000974f73d3cbc56, 0x00061f0a3738c290, 0x000ab2bd5966671d, 0x000a1f6af0a2f235, 0x0000b7892c65b907 }, { 0x0006060d7d7242b0, 0x0000ba35ba2f5a80, 0x000364edf2e9613b, 0x000948e2c542c6e6, 0x0000bb93163dd653 }},
        {{ 0x00001b3ad0550c5c, 0x00095c2065fef4a2, 0x000ddd40581378b2, 0x00064275077e60f7, 0x0000900cb66b1fd1 }, { 0x000fc0738fd99566, 0x0005ca10334fc145, 0x0008d484df668635, 0x000cfca828a9a9e0, 0x000018f7ad9443bd }},
        {{ 0x000efb14b1a6205e, 0x000c64d17d2d8c38, 0x0004e187f1d2d25c, 0x000d6dce5effc634, 0x0000567500e02c3e }, { 0x000a688b252296a0, 0x0009eee2245ae7aa, 0x000e3fa966194c20, 0x000197f5dbca4061, 0x0000512da556f80e }},
        {{ 0x000e48482682519c, 0x000f271bd3502bad, 0x0003d6962d79e50e, 0x0000f89e54ef41f2, 0x000065549732387b }, { 0x0001a43c79f96f00, 0x0008d9865cc22950, 0x00016c6aa32825aa, 0x000845fc7428a377, 0x00004147615b28b4 }},
        {{ 0x0002125ed3349bd9, 0x00097590fd3659a3, 0x000c896ccec77bc1, 0x0005dabd38927f30, 0x00001842ad4bd958 }, { 0x000b61fc67c66d39, 0x000f4ad3d4142055, 0x00028898864234e9, 0x0009d1c8173000ef, 0x00000cb713a5822a }},
        {{ 0x000568faf8627d03, 0x0008a5b7cae440c3, 0x00010ac27a395ea8, 0x00016202121c08b4, 0x0000a8232e915d7c }, { 0x00081b20b280f35d, 0x00001999ecf965b6, 0x000880fc06bcf0be, 0x000199ce2f14ed24, 0x000030dfcd06ef01 }},
        {{ 0x000461905346ffbc, 0x000b78dc6c011203, 0x00037a0f508825de, 0x000b0258c71db081, 0x00003aa89a31f12d }, { 0x00074bf462da8bad, 0x000afeadb2a3b15d, 0x000e04007c10a5d5, 0x0004726670a49a19, 0x00000c115b3d65b5 }},
        {{ 0x0005910a01f9dd2f, 0x0003e65533ef2177, 0x00044e3924858ec2, 0x0004e12677158c7e, 0x0000817fb332f9e2 }, { 0x0009f8be4c5579ed, 0x000d207cf88577d2, 0x0000ba656c829dbc, 0x0002850852224525, 0x0000a45a5a5d4127 }},
        {{ 0x0000f984312881bf, 0x000fba7076fe12bc, 0x000aed1bb3c8b51c, 0x0003c3c463f2400c, 0x00007182316ff6ef }, { 0x00040b796b1fe386, 0x00067e10170911db, 0x000a80c9381435ad, 0x0004332d27562f5b, 0x0000c44dd22a86f8 }},
        {{ 0x0008f3a69b2e986f, 0x00082973467ba92a, 0x000b4e15ae15a7c3, 0x000e0825d8c9b46d, 0x0000fc1d6d6957df }, { 0x000e56dec238f6a0, 0x000c60f61e4ce9b9, 0x0002d04f9a4a094b, 0x000a1c416bf06aab, 0x0000e7611030ccb3 }},
        {{ 0x000bb61ceb16d2d4, 0x00072f49a9898d9a, 0x000aaa7db1043ba9, 0x000cb6e97d799e77, 0x00001e9007832b66 }, { 0x0003918e6dab135c, 0x0008ca7f778443cd, 0x0000564ca17913cf, 0x000f05f6f3bb42bf, 0x000063a36f543384 }},
        {{ 0x000b7faa84dc37b4, 0x0007e719a5103f11, 0x0006d98f01816fb6, 0x000a50d195243efc, 0x00008d79992c468e }, { 0x000fc83cb87ebe8b, 0x00046bd6dfcd08ca, 0x000d5c1beb25b405, 0x00067b183129dec4, 0x0000b767a1b5e39e }},
        {{ 0x00044f601bff0a52, 0x0000e882fa6be76c, 0x000a4ed96be286e1, 0x000573bda892585f, 0x000089fc0fc5d1d0 }, { 0x00076f26dd1cd1c7, 0x000b60b2be1fdfdc, 0x000c59a459115f23, 0x000e3caa8a833bd2, 0x000032e8b64f5910 }},
        {{ 0x000c72c01889f6f7, 0x000ed7d3fdf5a517, 0x0002d9570a979390, 0x000e9616e7afb5f9, 0x0000219f402c76ff }, { 0x00053985e9b35d2b, 0x0007ffb99b5c3dc4, 0x000cf30b28c6f648, 0x0000ab982a48a0f1, 0x000083df9c5cf403 }},
        {{ 0x0008892c68df5eaf, 0x000196001b25dd8c, 0x0008330050216af6, 0x0004a5dd90dd37f4, 0x000075766028a72f }, { 0x000015de22cdfad4, 0x0001bc0ef514ca5f, 0x0008f716c0c41d51, 0x000c9395f478eb67, 0x00006c79af16b80d }},
        {{ 0x0009f0694bcc6e29, 0x00024e0eaf4b8a5d, 0x000971d5a4e6ddfc, 0x00063357068cfd9c, 0x0000e859080ab5b3 }, { 0x0003e274e3cfa9ed, 0x000ddd5ba708bcd2, 0x000937d6e7472719, 0x00016b88f5cd2ba6, 0x00005455f039ef64 }},
        {{ 0x000072e8b9f6e62c, 0x0003d2ddfd20a4a9, 0x0002f1337d994ddd, 0x000ba497d091c5fd, 0x0000a3d27ef9d712 }, { 0x000fccd127ddda11, 0x000fdb080848beca, 0x0009ca3065a453f9, 0x0001332f595bd480, 0x0000e62fb4bc3988 }},
        {{ 0x000f46369ed3f8aa, 0x0001e2d1cc1884bc, 0x00008e011d1618d7, 0x000955f9fc85ac20, 0x00002ecc593e0d91 }, { 0x0008bfe3c7e56fef, 0x000e149506467ea2, 0x00020f0e0904fd77, 0x00010253a481b63a, 0x00004ffcd9b71108 }},
        {{ 0x0002f7b360a494a2, 0x0006e090643d84a3, 0x000878d95ce09757, 0x000a592d6885f366, 0x00001f4c9f31cc53 }, { 0x000d2bd553f44765, 0x0001337dad1fe7ac, 0x00088715cb5cfc25, 0x0008d105060a157a, 0x00007397650095ee }},
        {{ 0x00076feec510b8c5, 0x000332f33e24a0b2, 0x000fcae661b4ee9f, 0x000a693259cbee23, 0x000026ff7e0a62da }, { 0x0006368fffd036a1, 0x0008ac6da10032b9, 0x000d5ab09a54017f, 0x000ea46ba718351d, 0x00008ccd45317653 }},
        {{ 0x000c8455fd5923e1, 0x0004c26150a49e46, 0x0005d2801a6da7aa, 0x0008db5d00c28b63, 0x000079a775e548ff }, { 0x00039639e7b06e8c, 0x000fd5f2d71867e2, 0x000bca63b9a5389d, 0x0008860045c0be31, 0x000038b70ff37728 }},
        {{ 0x0008d12af80f55bd, 0x00096159c7c4c6b6, 0x00006e6f6fffc941, 0x0001b75fb603aa3c, 0x0000305f59cf13f1 }, { 0x00010c71d2070171, 0x00096e7bf5d92796, 0x000549ce02fcb704, 0x0007d1bcb9aa5005, 0x0000308e0578f9b3 }},
        {{ 0x000938f757b46a0c, 0x000d5a75aa07093f, 0x000aae24ad24e9c4, 0x0007d4801ca84004, 0x0000e8c27e4df989 }, { 0x00028446eb3f9ed5, 0x000a11800a43421b, 0x0001553831dd567a, 0x000783d7639a2536, 0x0000f2b4a9ff1eca }},
        {{ 0x00086d90b771416f, 0x00012bfe3b6a6611, 0x000dce64e93fa706, 0x0005d47627751cb1, 0x00005f51d3741f54 }, { 0x0004fb8c61a5d90b, 0x000abc1dfb4f754b, 0x000a762c757a6872, 0x0007b4b4d5c5cf56, 0x000074b63f62f079 }},
        {{ 0x0004d9a14957069f, 0x000f73033ff98c71, 0x000aba3dd553a17d, 0x000f7390a9619fff, 0x0000ad3a3609acc4 }, { 0x00013dbda876d927, 0x0009c606f6df9ea2, 0x000d00515a423c7b, 0x000120d80dbcf770, 0x0000ce7619212003 }},
        {{ 0x00051b92e197fca7, 0x000790b29fc81b30, 0x000f1201f4e0144f, 0x00085039eebc3e09, 0x0000dc245aa94e7c }, { 0x00064b685717ec74, 0x0001ee9ecb5537f9, 0x000b92c10c2cc7a3, 0x000f23c1644b1b2c, 0x00008328b2510312 }},
        {{ 0x0007a54f50c83a40, 0x0005d99cec036235, 0x0005978d229986db, 0x000b408490553222, 0x00005422c6af52fd }, { 0x000d092808697dfe, 0x000441d9598b3525, 0x000c0a3802abd4af, 0x00045ab12e865a49, 0x000065df0e601cb3 }},
        {{ 0x0009d129392f3324, 0x00035084088567a3, 0x000de550212844c1, 0x000bd9d3deb6b280, 0x0000b981fcd6fb9b }, { 0x0001e32d5d00ce68, 0x000d25ebc6edfb49, 0x00016c69f3b1b20d, 0x000b846727485b1b, 0x0000e475193ea3e0 }},
        {{ 0x0006e21e7dd420a9, 0x000cbd74be59224c, 0x000baa8b377599d4, 0x000b64433ebc88cc, 0x00008d34017152f6 }, { 0x000ec23629ccb29a, 0x000ea0e5bcc36bb3, 0x000e3ee5193f6012, 0x0009ef9a89292445, 0x00003d19d936f781 }},
        {{ 0x000e29493df08894, 0x0001a85b134defa9, 0x000f7d898b37d761, 0x000db2c0d6073b1c, 0x0000020c0cac2037 }, { 0x00014b0d0b49b08d, 0x00090ea4e9acf4f7, 0x000833c2ab82797a, 0x0002a046c3e1f01c, 0x00004988c2f0debb }},
        {{ 0x000ee06e882c17f3, 0x000a0fc9920cb5e5, 0x000733e735241fbc, 0x000d49f355b44b1f, 0x0000db6fdef70b90 }, { 0x00050566058761ad, 0x000d473da621607b, 0x000be4f06c8b479d, 0x0009587ac2bb03e4, 0x000036fba157651f }},
        {{ 0x0008a673a256cc00, 0x000920ce079afee3, 0x00002ee30f7d0279, 0x0007e5ae5686e17a, 0x0000e8778c4b4660 }, { 0x00015ec10c65e1bb, 0x00006f283141c954, 0x000ecd38237a573b, 0x0000441783b2ecf4, 0x000086f1bda28a0f }},
        {{ 0x0007bf22ad35869a, 0x000aa19805033800, 0x000acb5590e4df9c, 0x0004452ae513d917, 0x0000b8de7ca6a12e }, { 0x0001fa0bdb4ffa7c, 0x0006ca1e4306839f, 0x00016bddf2cf9767, 0x000ccfcf11b3be96, 0x0000d0ade7990324 }},
        {{ 0x000b1c1f39afd405, 0x00055316ec3f510d, 0x0003f77be53bd54c, 0x0005ce6fd2c7e4a1, 0x00007db2f3ff86bf }, { 0x000930967a574854, 0x000c6755dce364a2, 0x000bdc5ec569a6e0, 0x000b64f8cb7b22a3, 0x0000915aea8087e0 }},
        {{ 0x0001bd8b5ca9eac1, 0x00048de0c061c658, 0x000c6c0961e7de05, 0x000c4f0ebcb014aa, 0x0000e471ea525791 }, { 0x0004753e714b1b3f, 0x000797ef01a17aff, 0x0003f29ac7c9d1bf, 0x000e1429321f37ac, 0x0000ccb1125704f7 }},
        {{ 0x0004fb0c0cc4be26, 0x000535ea15b0a44b, 0x0006d45d9d4b3d51, 0x000f84fb58f37c8d, 0x0000f6737b028edf }, { 0x000bc5118d042dee, 0x000f839a7ff1f2e5, 0x000305d1036ee967, 0x0004363ab9880f5a, 0x000068871fdb7590 }},
        {{ 0x0004ddb6540d8771, 0x000af554c20cea5a, 0x000d6fe72046cb52, 0x000a5e89bcac2776, 0x0000a43574dd6d54 }, { 0x0005542e3d637e0e, 0x000e3511b4941b44, 0x000429a7774ac8d1, 0x0000537e0e1ff19b, 0x000001532d232661 }},
        {{ 0x0004dcf4218a8c26, 0x000a591349f7aed5, 0x00003aed8fff670b, 0x0001ed0d83b2b5a0, 0x0000d7b1c690c699 }, { 0x0007a54456147aa0, 0x000a30727d6561e7, 0x000459d06d8bf726, 0x00005ddb5879d5ee, 0x0000236499921d85 }},
        {{ 0x0006a53ca0e071a9, 0x00056130105c0720, 0x000f873541040651, 0x000bb8cc118f7a99, 0x00002325fc847f2d }, { 0x0006bd173faf618c, 0x000401ef1aa1d9e1, 0x000433b89aa8fac0, 0x0005d13dc3241433, 0x000099c31187e85f }},
        {{ 0x0006aebf577abd9d, 0x00015452133ffd9d, 0x0000ac605ac980fc, 0x000e4582a39a8b2f, 0x00004eec5aa7265a }, { 0x000045d5cf88aa51, 0x00029cb76ccdac60, 0x000c15412957a97b, 0x000342cd1af4d36e, 0x0000e6170f039470 }},
    },
    {
        /* digit=6 (1,2,..,64)*(2^{42})*G */
        {{ 0x0008c8bf5c4dddf3, 0x000e274c005979da, 0x0001c17823f45915, 0x00004f9f9c2072b4, 0x0000aaa1baf4fa40 }, { 0x0001cb9e28458fbd, 0x0002855114df14a7, 0x000d8bfa4e43521f, 0x000189718d4374a4, 0x0000a04910e166e7 }},
        {{ 0x00032caa306d36d2, 0x0007e7903605c397, 0x000e2e161f3c1bd9, 0x0005b6484f0843d9, 0x0000c3ed7e7103a9 }, { 0x0002e9423a811470, 0x000527061160170e, 0x000b6096786931d4, 0x000d933acc32788c, 0x000040d7f8af5425 }},
        {{ 0x000fa20877d18f60, 0x000739b0260e52a4, 0x000c40c32aad97d8, 0x0003b1954d42585c, 0x00002a8f892d8d50 }, { 0x0002278164a77a97, 0x00058f8e34a0bbeb, 0x000bd2cf6dffb947, 0x00034ed8356b0040, 0x0000781e1140c00a }},
        {{ 0x0001682c10b4b42c, 0x00049455922ac1c3, 0x000c73c31352d93e, 0x0002a2bc4a7b3ef5, 0x00009c11c3b203bd }, { 0x000c092172b4577e, 0x000a6f04bd55d319, 0x00057bacfaff1310, 0x0004d8a9db6d1c08, 0x0000b14a2965910e }},
        {{ 0x0004577870abf0ac, 0x000415bc00cc638c, 0x000a9cd01752318c, 0x000474cd61a2015e, 0x00001d16278617be }, { 0x00098cd105a45700, 0x0004cce2c51211a6, 0x0001bb3533930fa7, 0x00036b5e5ad9b10b, 0x0000c8cba2a94a2e }},
        {{ 0x00023c2d425da318, 0x0001507e91b536de, 0x0001bc7ddc93f16e, 0x000d9216fceb982f, 0x000065b77c65e9bb }, { 0x000b019211c4463a, 0x0001dd93f330d370, 0x00016e0efbd983c6, 0x000d362e5b3a4c8a, 0x0000e6296830cd06 }},
        {{ 0x00025fbe30039a0d, 0x0009992840bb3366, 0x000c440a3c4b2158, 0x0002fb6007c353bd, 0x0000477b5b25ccdb }, { 0x00082dd27d9f45bb, 0x000fb1adbc173281, 0x000f912383a4036b, 0x000db4446008965b, 0x000099d080f1bc26 }},
        {{ 0x000910b95bc8d4b3, 0x0009d35ea6c39976, 0x000a5950529391f9, 0x000ac2a3954259aa, 0x0000954edcb4e373 }, { 0x000982812f18224f, 0x000a61043e09aee6, 0x000f3de4d536e4c0, 0x000508c61ced56c9, 0x000097bf82337dc0 }},
        {{ 0x000ba2f548cb64d1, 0x00040c6ce8009982, 0x00028d606b905e59, 0x000e6049072bb175, 0x000008cbdb0d4e8e }, { 0x000f7d4aa9ad99b8, 0x00036bbe490ebe0c, 0x000cb030c716d24b, 0x0006a023712a0a4c, 0x0000814fb43cd83b }},
        {{ 0x00007b3c44361615, 0x00056f43fd5684c4, 0x000f3fbace5d5c99, 0x0003e55822b26eec, 0x000042b330e7ebca }, { 0x00033f9344702a99, 0x0004cfe249c82ba0, 0x00072123e10816a0, 0x00066c9729e0faec, 0x000059f4e057a424 }},
        {{ 0x000fee4cf4b36856, 0x0005ad18be82e84d, 0x000d9a69e2fa6af6, 0x00080cbfa89967f0, 0x0000ae907090e0f3 }, { 0x000caf68098cfd0f, 0x0005a65b493c465c, 0x000ba986582707f3, 0x000f34f9f638b9a1, 0x00002e36b41f5ad9 }},
        {{ 0x000fc9cdf0a602a7, 0x0000f90768fccfce, 0x000cc86be572242c, 0x000fb55cef402d37, 0x00007fc53399e03c }, { 0x00038fa2f84181fd, 0x0002e72d1669a8bc, 0x000b93e529c4e96f, 0x000b28a0e33c536b, 0x0000c1aea47311b9 }},
        {{ 0x000d9775dc6e01cc, 0x000242bbf81f3f5e, 0x000dc1e6cfa506c4, 0x0003a88ae0525a5a, 0x0000a1093d00ddd5 }, { 0x00031ec1f450f0af, 0x000c48ac0edb37d9, 0x000dd3ad5d5d822f, 0x0006f5e7625fdb6e, 0x00006a6a380a22bf }},
        {{ 0x00008d930a7b99e6, 0x000376219ee43f1f, 0x000d6ff50a8cea27, 0x000b200e85cd57c3, 0x0000c75d78e4991c }, { 0x000037762700e77d, 0x000215bc016e4eae, 0x0008319cc70493d5, 0x000889455d391683, 0x00003ba91c61b072 }},
        {{ 0x00025cc3dbf43b85, 0x00044ecaa80ba596, 0x000a05f6ca758ac6, 0x00045a3af3083411, 0x00002728e23215f4 }, { 0x000579c8e5f5af02, 0x000646c01ac3107f, 0x0006df8bb1de094a, 0x0000f5bf7b2a8d57, 0x000009b1279bb39d }},
        {{ 0x000e413c023b543e, 0x000ff931341ed7a4, 0x000c597461477da5, 0x0006486a44223272, 0x0000e62ade1b4548 }, { 0x00073540e59ce025, 0x0006c27e17e44ceb, 0x00047c7e6d15d0b9, 0x000ca71caee86bf7, 0x000062dc3b140088 }},
        {{ 0x000308c289f47083, 0x0009a7b1b4a158cc, 0x000d64d9e0df3e0f, 0x000696e97e571c99, 0x000037723337361c }, { 0x000efbce6ccabf4d, 0x00050d3eb7958f24, 0x000be10cab44dc30, 0x0001fb936a7226df, 0x000059e89431f5b4 }},
        {{ 0x00050d86fc79732f, 0x000fb058d6990216, 0x00091217dfe6fc85, 0x000a7a33a586d4c7, 0x0000a84bf24df4fc }, { 0x0004cefe9970cdc2, 0x0001257842ba251a, 0x00059d544a9a0981, 0x000dd14576b65e3e, 0x0000027b14071dc0 }},
        {{ 0x000c30477ad22c94, 0x00046909f47befbc, 0x000a7500e598e35b, 0x000c53afe6562e9f, 0x000070ac8e36df1d }, { 0x000c834e7cd5fa4f, 0x000d5610b0ffd3dd, 0x000a7eb20ad9cb68, 0x0002a9d24a1f16f3, 0x0000a6548b8dd66c }},
        {{ 0x000bd5ca4b43dca6, 0x000b875b2c69dbcc, 0x000837ee1021381f, 0x000b45c713aa77a2, 0x0000a05614cbf186 }, { 0x0006c5c5b213f9f3, 0x00065076db476cb9, 0x0001c871dd6ccdd0, 0x000210116fde15d6, 0x00006771cf226ee0 }},
        {{ 0x000f2bc1c174c5bf, 0x00084b7eb4ceb9be, 0x000f5f4b7558a096, 0x0009784af2e99300, 0x000032a2c38518f2 }, { 0x0000f8a74351233e, 0x0000d7b9342f6b2e, 0x00019b95caad4566, 0x000fc00fc201eadf, 0x0000476372345bf8 }},
        {{ 0x00015f9c975edbc2, 0x00082b6b3eba40cc, 0x0009006244398c21, 0x000c18e5ef9739f2, 0x0000d49e0d6c8302 }, { 0x000ec5b26dc91097, 0x000db8583afbe332, 0x000ed62c2e2492d3, 0x0000a0611a3b6d6a, 0x00004081b00b4bac }},
        {{ 0x00088d1c341332d0, 0x00024d902f6fb8d2, 0x00016e040530d834, 0x000bc60004063c56, 0x000024d616048d6d }, { 0x0008547775093d38, 0x00030d607806fd8f, 0x000453677d9adf3f, 0x000502e27b2f7f9d, 0x0000af367fefc5a3 }},
        {{ 0x0001dde5dfbb6e3a, 0x000b56c872b4a606, 0x0003e4356c3c10e2, 0x000cfdbd1ab2a34a, 0x0000f1935615b0dc }, { 0x000bfd9fd4818329, 0x000cd219f2275f33, 0x0006091fe9776294, 0x0002d94aa0750c8b, 0x000091bb35d3e4f7 }},
        {{ 0x0001dc0dbedb662a, 0x0001430ee9120b67, 0x0002b1c86680159b, 0x0006b239e2b51af9, 0x0000f154211bef6d }, { 0x000650c83c8fe6f4, 0x0006ed20180cca4c, 0x0006fd05c1270162, 0x00094d0441ae6067, 0x0000b656885374b3 }},
        {{ 0x000c2c8d4a51bcd4, 0x000f3ddfe7ccfc4a, 0x000c6f660ffdd205, 0x000bb6833f925c89, 0x0000a3e9f13e00b4 }, { 0x0007ece5fa9b1c9a, 0x0001a07b807bba06, 0x000d0212a9406024, 0x000d2caa49d1e362, 0x0000acd66e4a8730 }},
        {{ 0x0005a83685cf0044, 0x000659b739f10e3a, 0x000959aa229f9a00, 0x000e28c41c3341ca, 0x0000631770b3a0da }, { 0x000031413543d5f2, 0x0001a819d7de3cd3, 0x0005f0318e67af71, 0x0003afbb0edf0258, 0x0000e407d9d3c57c }},
        {{ 0x0009529ca84f2105, 0x000db2adb22b94b8, 0x0009189807b37871, 0x000fd1763993d8ae, 0x000032507bd52b7d }, { 0x0006d1b1faa44f66, 0x0007daae572a2530, 0x0007fbd06d00887f, 0x000f39d58e02b643, 0x0000ff4486cf7424 }},
        {{ 0x00026e4489446194, 0x000d932c47447335, 0x0000e53159ab1422, 0x000ad62ca40517c7, 0x0000c82158b4d5ce }, { 0x000eb5da0a8b9559, 0x00064177ee4a9765, 0x0003335d89c6f8d2, 0x000d487411d9dc91, 0x00005c05b1cb2924 }},
        {{ 0x000bea4094530df8, 0x00044b1197b994f7, 0x0004ac59aa54c901, 0x000fb86a53c95a6c, 0x0000fb4e8ab73552 }, { 0x00098a15acd05a20, 0x000ff734db378f9b, 0x000029f161ffe433, 0x000ca8d4670025b0, 0x000031ab0dd33ade }},
        {{ 0x0002618905cfd82c, 0x0000cba2743b0043, 0x000700e28ce86b04, 0x0008aa45ab1c7f4f, 0x0000bc87ba82894e }, { 0x00094df65f20b8d8, 0x00033b5a23a0f46b, 0x000ab0b39b2d0f04, 0x000bb67945bc971d, 0x000089df6cbaeb93 }},
        {{ 0x000977faf2fd74af, 0x00010abaf95894c5, 0x0006cdf3274ead88, 0x00068f58b7b9bdc0, 0x0000d28ef6376f5e }, { 0x000d37fa9f638299, 0x000a9ef052c4b5b7, 0x0000893d97515b4d, 0x0006fb5c79fe87f7, 0x0000baaee7122abf }},
        {{ 0x0002ca17164dce09, 0x0007b1a15562627c, 0x0002ff79991dd942, 0x000f87d092c01a61, 0x00000ba8174e3fdd }, { 0x000ff6218fe602a4, 0x000c82d3eb81d82a, 0x0009df77cfaee393, 0x00070103bc02abf4, 0x00003fadbfce47a9 }},
        {{ 0x0006755712b47088, 0x000ea6c8d55b9d21, 0x0009837468e85634, 0x000495f2adc0adbe, 0x0000242e92858a3b }, { 0x00077f3fc7d01a76, 0x0008384f26847f99, 0x00074833ccf00d71, 0x00057799cbcb8e38, 0x000006fa9868df81 }},
        {{ 0x000ac30885fb7702, 0x000ad2bf4babda01, 0x000ddc4e340815f7, 0x00067f326feab68b, 0x000088b297e9debc }, { 0x00080e110807e7ff, 0x0006f7b40142d9d3, 0x0002f8840b031472, 0x000b917a3839b560, 0x00007518cceff22e }},
        {{ 0x000356d404f08cea, 0x00000d858854be16, 0x000e687e009888b9, 0x0009cd8ef4111c12, 0x0000fa7a19c7e749 }, { 0x000abe5cc63d28f8, 0x000728f594d320e9, 0x000d5b93700c61e4, 0x00066e5e40989316, 0x0000820e9cacb3d2 }},
        {{ 0x00086e2220dbc255, 0x0000bfb9723f6711, 0x000d66cf371bf433, 0x000f7b19621b2a12, 0x0000a5a81f4bb948 }, { 0x000435c651e5363c, 0x00028e088312c29f, 0x000823a6b1008cf9, 0x0004fb28994282a9, 0x0000f15346f16d75 }},
        {{ 0x0000179a7dfec12f, 0x000e63943b3a50bb, 0x000b6a3c553ecb3d, 0x000ef96b77132130, 0x000049806974e5f8 }, { 0x0002dd6225d0701e, 0x00046b3c85f455b2, 0x00086c909e3c4260, 0x00005b8d94725932, 0x0000eda3e114e5e8 }},
        {{ 0x0008515b8d05849c, 0x000678eb9c3bd6dd, 0x00062368cc3bb0d3, 0x000de285d00cdb0d, 0x000009399b53aa0a }, { 0x0000a8dafe0d5d78, 0x000f19e8cbb5ecf2, 0x0008a55ffcdabe31, 0x000e088db8c4347f, 0x00001718c6220cad }},
        {{ 0x00040e94e3de442a, 0x000ea8045ee2f40e, 0x000e5d5ccbebdf33, 0x000057ec475e354a, 0x000019d840edbdaf }, { 0x000a9e01bfa34f38, 0x0006d9f62accd765, 0x000a31d826781a00, 0x0003cef7b05cf466, 0x0000fffe5d10d9a1 }},
        {{ 0x000800bf0a44ff12, 0x0006598326aad8d2, 0x00049b9638dd5846, 0x00054d092bdef495, 0x0000f18cb0c7987d }, { 0x000e5e7f141dfe5e, 0x0000f3144438fd49, 0x0002bfc720860444, 0x000f80464ef23392, 0x0000bec1282f27e5 }},
        {{ 0x0002b639eaff99d6, 0x000e16c5fcc1aba9, 0x0007c6ff860d56c4, 0x0001cfdfaea63254, 0x000005c26b5c9cab }, { 0x0003b90f531f7c6d, 0x0007a82953b135c8, 0x000dbe99599be78a, 0x0009207697308912, 0x0000c7f561f073e9 }},
        {{ 0x000f18cdffd1566d, 0x00046d0244975b3f, 0x0007127d8f7275fa, 0x000375a51f3a4435, 0x0000eee9da0674cb }, { 0x00063c9f3d509320, 0x00018332028e2472, 0x0002cd831d3c736e, 0x000f4ea8496e280e, 0x00008f205a98807f }},
        {{ 0x0005a0beee1cf41a, 0x000fbb32d2ce8114, 0x0002b2885e3319c9, 0x000abb5db846dc0c, 0x000088344d69dd5b }, { 0x000cf49ef219e334, 0x0006b0796d69af13, 0x00081ca617ead2ab, 0x000ffac9f31a7c5b, 0x00003db05bdafddf }},
        {{ 0x0001e5f7c08d641d, 0x0002ff3086cff9bc, 0x000077bd3e4ede7f, 0x0005ba01d88dbd49, 0x000011ea14de0b80 }, { 0x000c2b9818dff733, 0x0007f1428819d98c, 0x000baebf05b22550, 0x0002635c8c770dd9, 0x0000c0c2bc6fae24 }},
        {{ 0x00049dd198ff7e19, 0x000a376fe86c6072, 0x00072b8cc8fc8049, 0x000cf92636b7b933, 0x000056f429238ed1 }, { 0x0003d2fe1f767829, 0x0003c84a981a2029, 0x0007a6b6356871bd, 0x000e449093531dd1, 0x000078fe864a7151 }},
        {{ 0x000649cddbdcf87f, 0x00040e2ec6d6b975, 0x000466c0d802a52d, 0x0007f205f9cf6d6f, 0x0000cf17c30aef77 }, { 0x000497381a72d1fc, 0x00039793f26ce5a8, 0x0001d0780b03bf41, 0x000e6506fc289a10, 0x0000f143d893c000 }},
        {{ 0x00064e359de41d5c, 0x000f9e6f2606a828, 0x00010aa4c5b9d100, 0x00087877ce25f706, 0x0000a40abe928ccd }, { 0x000056538d0f8dac, 0x000bf187ce573162, 0x00001a405eb03810, 0x00063c1f72cab250, 0x000042dc7ff0d57f }},
        {{ 0x000620742019e130, 0x000565ef15d31016, 0x0009fb94dfb78dfe, 0x000f0129480c5533, 0x0000f5ea2ed57734 }, { 0x0005d06cf408388d, 0x0004cc4edc454181, 0x000eabfbdda91999, 0x0008bd5f0005b859, 0x0000bbd16a163c99 }},
        {{ 0x000378dfaf3e1b18, 0x0000cee3484be472, 0x00062693b0806164, 0x000f0ac532ff33e9, 0x00008a1b602ca81b }, { 0x000e287f8b8af0ea, 0x00036ec856720f22, 0x000b6850f3059c33, 0x000992b582ad9d95, 0x0000342dec973aeb }},
        {{ 0x000a1efa102f0077, 0x000fcd0990cecaa7, 0x0000ed3e46704a67, 0x000f05684ade55f8, 0x000011db276b92f2 }, { 0x0005cbba4dc5488d, 0x000f720037cc0618, 0x0000500883c5313c, 0x000d8fa10682e05f, 0x00004999403c4661 }},
        {{ 0x000f5f5dc0118781, 0x00016297a3cd22af, 0x000b4b0325d760e8, 0x000a9e4ed9f8cd5d, 0x00009329e15df9b0 }, { 0x0003608bc8757f53, 0x00091b5654a446d2, 0x000f4411cbf24c65, 0x000f71756d1a9496, 0x000012b5a681cc68 }},
        {{ 0x00073e00f8daf702, 0x00020d76688c4d4f, 0x00010ff939c43bae, 0x00037f8a39d6b428, 0x00007889e808b3d9 }, { 0x00072755fb51d00c, 0x00053ec49421c3e1, 0x000b26a7d07ce447, 0x00050e49fc4e44f9, 0x0000cab187f97020 }},
        {{ 0x0003ecaa12dc8429, 0x0007457446139cc3, 0x00022bf3767cde34, 0x00061aa990a0a6e0, 0x00001e7652e2cf35 }, { 0x000dcf01768fd984, 0x0000721f7c426641, 0x00028d96cf13c980, 0x00076a126b3e91b3, 0x00001f7562a57864 }},
        {{ 0x0000c3aa50085a1b, 0x0003e8f3d589e023, 0x000f811292dc489e, 0x00017b207df0478f, 0x00001d6316d96d0d }, { 0x00082736a4a1651d, 0x000e04b941ade7f2, 0x000c0a51c96c8483, 0x000da82f03d1debd, 0x000040def4c31898 }},
        {{ 0x000c4a826abdd1fd, 0x000a3c3d41d3bd3d, 0x00082b57e9743197, 0x0005ce29fb533b8c, 0x00005ac407982d1d }, { 0x00046833e45024d6, 0x00082513d590dbeb, 0x000b39e65f731bf7, 0x0008e5d7484632f6, 0x0000ca6f29d8d329 }},
        {{ 0x0001416d4f55d1cf, 0x0006a13f7fc0586d, 0x0006957dd9c9e1b7, 0x0004686dc385315f, 0x00004cf755c4445d }, { 0x00028a41409d5b93, 0x000187851047213e, 0x00010db688562158, 0x0002f8a0c447f6e6, 0x00006e8b3e82ccc6 }},
        {{ 0x000df39b884c94aa, 0x000cccb82003f867, 0x00068dda211b1bc5, 0x000101a55610d07f, 0x0000743e976e9950 }, { 0x000fa15eb2e071d1, 0x0008e96e22db2188, 0x000a7901ed91d8b3, 0x0002dc1768cc6d9e, 0x0000e81e206dd3a5 }},
        {{ 0x000ffef809be123e, 0x000fa45cf2c24b5a, 0x000ea2facbb12142, 0x0009590b389f66a9, 0x00008f9d35fbb026 }, { 0x000dbfa947e096f3, 0x000eec68eb8fac22, 0x000afaef20465dbc, 0x0000d48cb93c1e4e, 0x0000996990687d52 }},
        {{ 0x00088969f06f59c6, 0x0000b90ca84bad76, 0x000f44cf7c6c710c, 0x00071a831fb13919, 0x00002f4311cd9b82 }, { 0x00029f789c4e2b27, 0x00025e721264d452, 0x000950c7da8f86ce, 0x0001a7c9dfdfb65e, 0x0000d786a79714bb }},
        {{ 0x00006451aee0821e, 0x000013e6f49f15d1, 0x0008801ab5e3f872, 0x00024ca67678ce82, 0x00003896b89ad41c }, { 0x0009a3bf5194d5f6, 0x00001f019fc32d29, 0x000c1668938086df, 0x000c8ae4564633df, 0x0000568a458b4918 }},
        {{ 0x000bac9c2aea0873, 0x000b2a2534d4dbc4, 0x000176f66278228d, 0x00028891a45b8f1d, 0x00005f1547f876bb }, { 0x0006452e97fd7988, 0x0004f6a7b4e35876, 0x000dab3dab4b5c1d, 0x0008e693d0045cde, 0x0000d8b2227e2114 }},
        {{ 0x000e4caa4ece9077, 0x00039549f950cd0d, 0x0000f9b126264855, 0x000941f639cc72fb, 0x0000195f212e9897 }, { 0x000d4a60338b6b56, 0x0009f7a55c7ac17f, 0x00031d3b82198646, 0x000f52be7779cbd4, 0x0000760624055358 }},
        {{ 0x000e4daa4cf03ebf, 0x000a5880a750c0f3, 0x00031006e436dc3b, 0x000d2500539bacfe, 0x0000d4ef32a1291b }, { 0x000a444f665140ba, 0x0002db55ffc69ff6, 0x000c27c0e1c0d5f3, 0x000e2259446f147b, 0x0000e26f578ece85 }},
    },
    {
        /* digit=7 (1,2,..,64)*(2^{49})*G */
        {{ 0x000392f2ec04cd39, 0x0000d5ac3f16ea83, 0x000d7f5ede30754d, 0x0009b6872115a461, 0x00008418a223a967 }, { 0x000fc5ff44240c84, 0x00018cde526cdc7f, 0x000b22632c3da39d, 0x000c70d988e537fd, 0x0000709b1f542582 }},
        {{ 0x0008d2c9ae428f00, 0x0004f7ea90567e6d, 0x0006dddb93095522, 0x00081903d513257b, 0x0000d1ef01808e5c }, { 0x000ffa707b56bdd1, 0x000c2246b29cb44f, 0x000c77ce5b30e21c, 0x000d3bd42b540a21, 0x0000c8f28344b9aa }},
        {{ 0x000c41d1ca8f6911, 0x000d2f1d4e353b7b, 0x00043304ded57d7b, 0x000e262fd062d8a1, 0x0000c7373014e0c7 }, { 0x000d825d0c68baec, 0x000f5e76be77800a, 0x000717e2f324cc7d, 0x000e0471a71fe8b3, 0x00007ed811a51502 }},
        {{ 0x00067b085c91370d, 0x000d35b1cb76219d, 0x00009adb7621c58b, 0x0008dbc100ec0bf9, 0x000035a1c37429d0 }, { 0x000539991832fa6f, 0x00095595e93a96a9, 0x000a66a28b826cbe, 0x000e29cb77526c1e, 0x0000acab05a94fb9 }},
        {{ 0x000fa297b54139e5, 0x00029e99e240d181, 0x000c9516acb11ffc, 0x000ab4d5c057cdca, 0x0000f108fdca1bee }, { 0x0001468d72cb1fb0, 0x000d1e7db083d762, 0x0006d0c02fa32f20, 0x000a1ef8c31993be, 0x0000ca3322492849 }},
        {{ 0x000f2a6cb5724fa2, 0x0002383a4c506ece, 0x0007e28291e1ba7c, 0x000a20ca2957ed18, 0x0000e20ed40d7da0 }, { 0x00052999a29a324e, 0x00090428e195c43b, 0x00095936b72378db, 0x000361c5dc842025, 0x00004a12b227046e }},
        {{ 0x000cae6fbba2a2e6, 0x000aab910a5a8bda, 0x00085cb9f51f374d, 0x000c4f5c1ca91d24, 0x000007845f0a029c }, { 0x000257ba0e5ed119, 0x000ade0d21411a01, 0x000fc7b290c9b692, 0x000bb673028fa787, 0x000016219da80e31 }},
        {{ 0x00096e6d076e5fb1, 0x000e067ceca9754a, 0x0005a20acc991594, 0x000e01fbf426d2cf, 0x0000e8ea47274e03 }, { 0x0003439dd70f73cd, 0x000dbdd87880e616, 0x000581c546fe37f4, 0x0004291151554381, 0x000076b608169ba2 }},
        {{ 0x000f4d069bfe9043, 0x0003d2fc14e35c23, 0x000c03f42ca0baaf, 0x00044b06271735b7, 0x000029c2e58d6818 }, { 0x000d0809671ae302, 0x000ac01f78f5a7c7, 0x000dff46e39ff2a0, 0x000fe0d8350a8ffb, 0x000048652a819f5b }},
        {{ 0x000dbc9d0aa61099, 0x0005fe1698e04cc2, 0x0002e8d368be0b0d, 0x0002bc879877be20, 0x00000faac89a81a9 }, { 0x000fe29735ce620f, 0x00040939454d9370, 0x000da49d76e54a1b, 0x000fc59e8e682ce9, 0x000091679212a247 }},
        {{ 0x0005ef72b2ec9bc7, 0x0002d0c5b9e02b75, 0x0001dffe92356668, 0x0003a9c2ba37c969, 0x00000df5f3206d48 }, { 0x000de1f2900b0c39, 0x000231f7b91b0620, 0x0007ad51f591a0e0, 0x0009308170a79e1c, 0x0000eab82efa982f }},
        {{ 0x000043d055253a89, 0x00013cf3a4434b4b, 0x00006a3584870bc7, 0x0000ee9a7a345811, 0x0000321ed180b8be }, { 0x00093e67f479fe7d, 0x000b49e11137b1a0, 0x000ee41e584a881d, 0x000ebddd400a754c, 0x0000aa7b62dca051 }},
        {{ 0x0005185e8a26f6cd, 0x00031e9c97e3f6b8, 0x000431c215d74bd7, 0x000a139b9c10bcab, 0x0000d0dce141728e }, { 0x000380e52200f33e, 0x000551f0eb6c4d46, 0x000d717975e238f3, 0x000cb40b84d4440b, 0x0000ddf18ad14c33 }},
        {{ 0x00046e79f34d63c5, 0x0003e51d598d7107, 0x000656cbd3b0d348, 0x0000e191f272c416, 0x000002b47849277d }, { 0x00013ba3f17082d1, 0x000f6364b573e9b5, 0x0006ac0fac4efdfc, 0x00016077b75d03f4, 0x0000927b63f07b9b }},
        {{ 0x000fbb3bb17bdaa1, 0x0000e26abb177dde, 0x000788d699b2fdf6, 0x000488073f82ab56, 0x0000c9b786cdf7f6 }, { 0x00048e86e3fd3f51, 0x000c0dba6d183586, 0x00074c02e656b55c, 0x000499190519e279, 0x0000636d8ce30c94 }},
        {{ 0x000be5a560218c0f, 0x0002245983c38b5e, 0x000b7795172b6411, 0x000736ce6b14f176, 0x0000b95d3a653da9 }, { 0x00090806472dd13c, 0x000e1cd9f87dc596, 0x000392d3ef194f1a, 0x00019f6577c595cd, 0x000044201d70daf9 }},
        {{ 0x000d06a5428770ef, 0x000550381e9ede3c, 0x0003fdf039c7f7bd, 0x000964c11243c389, 0x0000a436caa56e94 }, { 0x0004e52114b12630, 0x000594949d4e2e5b, 0x00069611476b77f0, 0x00082203533649d0, 0x00005fbc01d77138 }},
        {{ 0x000b45d0b53b1b49, 0x0006e0678a6513b9, 0x000a0395488ea188, 0x00052e4d892ea4a2, 0x000048a2399a61a0 }, { 0x0005c5b4d8712eaa, 0x000c6dbc69d11ae9, 0x0008b691a486a01f, 0x0004918721477dd1, 0x000074978afddc7b }},
        {{ 0x000353dc8bc4376c, 0x000906342e532b72, 0x0002778f7604f48d, 0x000ca80bc386ba42, 0x0000629c4169b70d }, { 0x0003fe03ce10af95, 0x000f0ea68f7e978d, 0x000156d8cd51895b, 0x00046f36d96ec568, 0x00001a9ad92e9ed5 }},
        {{ 0x0003a1c621b2b2f9, 0x00026a47ea67c77a, 0x000f4048baeb4ba1, 0x0004eda74952bd2a, 0x00008f88312234da }, { 0x0005059bc76aee10, 0x000c5a9c7ec6d792, 0x000f0d20277aefe2, 0x0007b501bff7038a, 0x0000444624dac251 }},
        {{ 0x000600da329b3360, 0x00005b2f2086fbfe, 0x0005a09086465f16, 0x000d4bd6cf6840ea, 0x0000778cdefbedd4 }, { 0x000017637cd7cb9e, 0x000d163529b8a80f, 0x00073d660bc129c5, 0x000992c7ce80e485, 0x000088119f4a6bfe }},
        {{ 0x0009ef732984387b, 0x0004d0676cb25661, 0x000198591fb0163d, 0x000d083458f76acb, 0x000001e530c1a29a }, { 0x000baf76b4f735a9, 0x00002715f228bf03, 0x000945c0bc22367f, 0x000763ac0f3cdea3, 0x000010c6a57a79e5 }},
        {{ 0x000ef5d295913cb7, 0x000aee010089465d, 0x0008af81f3014353, 0x0000106953bab5d2, 0x00002e287da81088 }, { 0x000de01f9c58ac00, 0x00058b9286b24756, 0x00095fa0668f5980, 0x000ed2388a0b9393, 0x0000e81acea89e97 }},
        {{ 0x000a4bf3535ca72f, 0x000398eadd70482e, 0x000990fe5b370e05, 0x000e33fef708de92, 0x000095e812192018 }, { 0x0004f0f83a164ec7, 0x000582c87912868c, 0x00052a18313ff9b5, 0x000a4bf0ab1b1be9, 0x0000846b0bf28b93 }},
        {{ 0x000cad0a1ecf03bf, 0x0000da397f6306a7, 0x000debf69a916d7b, 0x0009e19a2744c44a, 0x000096bc23c881e4 }, { 0x000db7091f94675d, 0x000e2cd33791d9ed, 0x000c96fbd74055fd, 0x00013a2a29cc2a05, 0x0000612a63e5bd69 }},
        {{ 0x000cbdfda897260d, 0x0001abf8538a5c69, 0x000cecee016f2ff8, 0x0004bd6da195c63d, 0x000048f79961dc84 }, { 0x000c5bd833e2b6f3, 0x00083ca849d7eab8, 0x000aafcb5a3dbb01, 0x00051e9eaebe2764, 0x0000a3bc8e72b00f }},
        {{ 0x0003a594c485ad1b, 0x00099f59837fc0d8, 0x000823f75b4b4716, 0x000321e909b641ba, 0x000075762faa795a }, { 0x0000344e3d60af43, 0x000021a526c45028, 0x000deaa6f1565663, 0x0004fc9cd9d39438, 0x000000de2d56a689 }},
        {{ 0x00082cc2c2ba07f1, 0x0001c7c2fa4f1264, 0x0005146cdf6fbb88, 0x0001defc308a3b86, 0x0000a19653e4fd21 }, { 0x000d3322183959bf, 0x000bf645f321bb80, 0x000b8d4ce39bfe85, 0x000325a588a93821, 0x0000226bc98df493 }},
        {{ 0x000b779c98067abe, 0x000c319e246b0e64, 0x000487d2607f1a30, 0x0009fe300b39781e, 0x0000008d964486ac }, { 0x000b59cb15ee57a6, 0x00073e1c84705003, 0x0008d0f2bb875ce7, 0x0004f15cc45b7efe, 0x0000d78e82ce74b3 }},
        {{ 0x00020a4bb98bb7ca, 0x0004adf3fd165e81, 0x0008f4153c582e9d, 0x00037d500499fd6a, 0x000043ec7a9e2734 }, { 0x0002fa487b4aec69, 0x00067955f4a5ac5d, 0x0000c2ad5a75a813, 0x0002b51f1cfd174f, 0x0000044c15223f52 }},
        {{ 0x000d3785755ef9c9, 0x000d5feb1dc77e1f, 0x00082492110d3db6, 0x000669ebbe956693, 0x0000caee16ac8c0d }, { 0x000b93cfea66af2d, 0x0003fbe81796b334, 0x00060d54afa0c4bb, 0x0006bd4b462d550f, 0x0000fc36919b3218 }},
        {{ 0x0003043896a03a07, 0x0004b381d531696e, 0x000fa5c9a5d318fa, 0x0004ca8c757201bf, 0x00003e9b1ef0bafa }, { 0x0000bf7978610e72, 0x0004fe18555970de, 0x0008447619e614fa, 0x00020318b8267dfa, 0x00002d551502ded1 }},
        {{ 0x000b228da47438a6, 0x0000528c06fece4c, 0x000dc9c526f8a5d5, 0x000350528d405991, 0x00009f6e58cf2dfc }, { 0x00076a72c60289e4, 0x0005d64107d1cea6, 0x0003698b662b1fc4, 0x000e45dcdeb9fdab, 0x00000c5b4b14c00a }},
        {{ 0x000b95b2310a2529, 0x000b4984d6cb00b1, 0x000ac298db4bf5b1, 0x000f75036cef9c5c, 0x00005f904b6ca659 }, { 0x000bb3c9580effd9, 0x000d02ee8ac25b1c, 0x000b6e64ee6ee865, 0x000ccb066ceb04cf, 0x00005b6ae05c96fb }},
        {{ 0x0000ebfbfb92eddc, 0x00095a1a43ff93c7, 0x000730f75cae09ab, 0x000ebe0825bc2db1, 0x00004d67b6663315 }, { 0x000d6c3f4e74d6e2, 0x00082c792871591c, 0x0005e6c0c2cb3b08, 0x000020e3cea5bcae, 0x00008a85be0f06b6 }},
        {{ 0x000d7b2902aa47f5, 0x00001230bca6de32, 0x0009846bddfb146e, 0x0004864d4dd20dd9, 0x0000510d80cb9b02 }, { 0x000dbc84fa3046b6, 0x0005459ce98840f7, 0x00000653e31059c2, 0x000ecc3a5a5d947e, 0x00004ffc38657061 }},
        {{ 0x000fc1dd78654cb5, 0x000241654a1aff29, 0x00099ab71e2397f4, 0x000dd445eabf318b, 0x00007215a0065fca }, { 0x000e995e35c91e4f, 0x0002a1513595c172, 0x000860c955739687, 0x0001206b0b950466, 0x00005b2b004bef36 }},
        {{ 0x000c3c9001d90f0f, 0x000c0697e9bbcfb2, 0x0006bc3106e1aecb, 0x00086985c95d0c1d, 0x0000231244ed3a54 }, { 0x000d62b88afe3771, 0x00044616c973f57b, 0x000be7bf5db81fb3, 0x000b808d4108b7e2, 0x000050a0c350cec6 }},
        {{ 0x000320c24f7e2d8c, 0x0009dc4197c75d61, 0x0002a3e6ad9c36af, 0x000c55311266a6df, 0x00003c001f9a9acd }, { 0x00013439c1f03bd2, 0x0000ee101a595966, 0x000b8ec570ea7b76, 0x0001ed4c98317c5b, 0x000050f4c2d89c48 }},
        {{ 0x00020ab732d47663, 0x000a19f16a66e918, 0x0000781056db02b9, 0x000d5c5eae97f282, 0x000094939d8bd05b }, { 0x000b2c87b6265c14, 0x0008af9287144234, 0x000a628484fc8c50, 0x0002a90cc54ecfff, 0x0000ff62b0d3b8d2 }},
        {{ 0x0007448f03eef746, 0x00087200a44295d8, 0x0000d1822140f960, 0x0009f2f7c707d7d2, 0x0000181637d8f606 }, { 0x000cf227783ae69d, 0x0001f455ebcd58d0, 0x00098430ac48d8c3, 0x00020c94279bcced, 0x0000f56a58b7cf63 }},
        {{ 0x0008df2c35144158, 0x00056e471d444b66, 0x000f5c291de84a20, 0x0009292f6211236b, 0x00002a2977e8653b }, { 0x00023b925d834202, 0x0004d0f59bf4f363, 0x000ba5b7bb161b25, 0x00007791df883669, 0x0000991591315aae }},
        {{ 0x000aa252908f15e7, 0x000884274082008c, 0x0009fda5d93a92ff, 0x000014d79effc521, 0x00002020dbc62e83 }, { 0x000f621ec39ad5ed, 0x000232a8f1b3c4d1, 0x00010e39cd49e9fc, 0x000cf59efcf64dfc, 0x0000b4beb3b9e9dd }},
        {{ 0x000023a25232bd58, 0x0007e27373386156, 0x000aa1b54ba4382f, 0x0002b1366162082a, 0x00002d55d436c6e7 }, { 0x000df2593f2a2fa6, 0x000663abbf04b89f, 0x0008b0824105cb8b, 0x000e4383b6735eb6, 0x00000bfd1da5861c }},
        {{ 0x000ec9322dec7ec3, 0x0007e4efdd236675, 0x000a9b56e3c02f10, 0x0001762ffdd52aac, 0x0000b9cbe745e9d1 }, { 0x00021cdfd65b39cd, 0x000df85a96f5294d, 0x00028f8386946139, 0x000cedf4282aa5bf, 0x0000d1d292453a63 }},
        {{ 0x0000fc5e10e7712b, 0x0008a7b67f8be10c, 0x000b96034915aa1b, 0x000c1b393365d1a7, 0x0000124aaa91f37d }, { 0x000148c2fe2e9e96, 0x00066f89cbbb1423, 0x000a7e7c43943fb3, 0x0001e18221d436e5, 0x0000b673c1d30111 }},
        {{ 0x000e85a1614b76ce, 0x0002fc9233222fa2, 0x0002b5945a0a0751, 0x000ae7b7789ff109, 0x00000859c80837d9 }, { 0x000d07e6b6010c68, 0x000ec7cdc810bc9e, 0x000eb01d956cacc5, 0x000ca499bea3f458, 0x00005b7830a10826 }},
        {{ 0x0000f30fee8c3f0a, 0x000134286d2e0f86, 0x0001ad978fe6de9b, 0x0004d1149e592012, 0x0000ed2e1530c0e4 }, { 0x0008c9d2519e343d, 0x0007006858e6a61a, 0x00069709a27ec803, 0x00054a1fbce4c776, 0x0000b72f3fb5dbd3 }},
        {{ 0x000e5b62c68d8a29, 0x000e65eb24fe1dcd, 0x000d8d188f85cb71, 0x000dade059eaf9de, 0x0000ef583c1ec4a5 }, { 0x000a60c4ef22069a, 0x00035aed3c2b36ea, 0x000458e370eb3a6c, 0x000784745868c19d, 0x0000a0000634a4bf }},
        {{ 0x00029b382fff57d0, 0x000efd23b60c472f, 0x000bf881f819bd9d, 0x000e0c5c9af4ef13, 0x00005143fbdccce1 }, { 0x0008035f45578ded, 0x000cb5373efaef69, 0x000c10c60ffac46e, 0x0008b3124cf56ac9, 0x00000797387ec422 }},
        {{ 0x000b0f588a0f9e16, 0x0005bb91e750c846, 0x000bf2b76bdfe46c, 0x0001e03c1c57f7b0, 0x000041ee2fa51260 }, { 0x0004a5ad25356120, 0x000b084a07c4e9ff, 0x0000b68230ef27de, 0x0007f6dccc383011, 0x0000c74f049cd8fe }},
        {{ 0x000840fa43472eaa, 0x00018bf490f97ca7, 0x0005ef60d3d59758, 0x000159d8d288f09c, 0x00008305396842d1 }, { 0x0005e16ea2c38bce, 0x0002fd25a631b378, 0x000e1e9b1cda4ede, 0x00090946694761f1, 0x00004a39b68d7484 }},
        {{ 0x000a2a463cbfea70, 0x000f033924910480, 0x0003eaa9bc9b0ced, 0x00080165c0c2f696, 0x0000c418f721a31f }, { 0x0001669c5553f128, 0x00073e7bf2f12ee2, 0x000245543d335cf1, 0x0003299d7f0a2200, 0x0000ed5a8309ae3d }},
        {{ 0x0003e34ebe7538d4, 0x000c40392f145296, 0x000bffb1ff34a332, 0x0002172f6d3d056c, 0x00001640598299a2 }, { 0x000c8da4b79d1b0b, 0x000e2ee5b80cfa5b, 0x00001774371a4424, 0x0008ecb67e872a12, 0x00004a32f2dc5f4b }},
        {{ 0x000ac787ee441dff, 0x000babb1b703e75a, 0x0006296bfb74f2f1, 0x000d17cff6773b18, 0x0000c7b2314a8aa8 }, { 0x000884ecd8191468, 0x000828b0437164b0, 0x00006b798c862eed, 0x000d18373f430ad2, 0x0000e6c1f3907575 }},
        {{ 0x000c5a4b66e99b1a, 0x000a6965f5206989, 0x000da89d6d7224d3, 0x000bf4eb9640631f, 0x0000b4e03e0395b9 }, { 0x000436d53d463c5d, 0x000f7bf80b6783d1, 0x000e25fb34d34e57, 0x000cddcb280e701f, 0x000018bf1fde0cd3 }},
        {{ 0x000ba1cfa6be5052, 0x0001c6d6556c3629, 0x000046a38dd836b9, 0x00021def9bc23a3c, 0x0000a35d2970adc2 }, { 0x000fc77d5dfd93f1, 0x000147a99c5d7062, 0x000b545b986b85a6, 0x000133e128be85f4, 0x00002b9e4f8b3c69 }},
        {{ 0x0001f6c57acb1b9f, 0x000eb805616ee306, 0x000f85b44b08579c, 0x000f6d1cdfb09548, 0x000025f3fb849285 }, { 0x000bf9f676805df1, 0x000ce7290aecb08b, 0x000851fd8e2e2099, 0x000512c03849f87c, 0x000043ec2b969f4f }},
        {{ 0x0003e4e910ccd797, 0x0009565765ba5436, 0x000bdf0577ccc478, 0x0006c55b318ce3ca, 0x0000fc4ec1949ba1 }, { 0x000fc62b2afeca4c, 0x00003a3c85932eeb, 0x000a8cca99222759, 0x000c004f696883e8, 0x00000968be080da2 }},
        {{ 0x000739fcc7a7b873, 0x0000a906d3e1fa9c, 0x000836cbbbdc2a72, 0x0007329011711561, 0x0000690a8b0f5f13 }, { 0x00013b9b26e41c59, 0x0003857165e439a0, 0x00026954c8b16ff2, 0x000cccea306353a6, 0x0000a667ba136d7d }},
        {{ 0x000c2af206d24a25, 0x000c73b973497f10, 0x000dfd13ca267da0, 0x00064e08315c0f94, 0x0000473115252061 }, { 0x000aa99338a37060, 0x0005df14457397cb, 0x0007101327a82eed, 0x00010f3062e8ed01, 0x0000a0cbc7d6001f }},
        {{ 0x0008fe977ed7b8cc, 0x00065823b0ec7de9, 0x000abfe06e603098, 0x0000aea831e1b822, 0x00004c9aceb8b0ce }, { 0x0004ee56e7d95e11, 0x00005c20d5e88732, 0x000a981eeb9623c5, 0x000e1b861e48d839, 0x000045fdcadc3088 }},
        {{ 0x0002d762b3177523, 0x000825b91e4de589, 0x00054d073fd57a0f, 0x00067a643a7488ca, 0x0000af2bc1467b91 }, { 0x000e0fcd863d7b0f, 0x0003b12091ff4a95, 0x0009ae9dd4edfec9, 0x0005ddbe2581113f, 0x00007099494bbef3 }},
        {{ 0x0008e7dd3da6ac7a, 0x0007068c094dbb56, 0x000e7080f6d2bb87, 0x00006862db77d062, 0x0000957c36387eb2 }, { 0x000c48a3b41f412c, 0x0002b40013a5e585, 0x0006e05598676e41, 0x000588aab2131174, 0x0000a5cfa92af35d }},
    },
    {
        /* digit=8 (1,2,..,64)*(2^{56})*G */
        {{ 0x000b63b80c17b3a5, 0x0009991991724f36, 0x000bce681e7aafdd, 0x00021571e5eda799, 0x0000608a0582bb4c }, { 0x0001f37958fc7a0f, 0x000ed697e5fb5166, 0x000a6efe500b7226, 0x0002f1da5737708b, 0x0000d330af0a0615 }},
        {{ 0x0001d5f271b2ffcd, 0x000ce2caa6650728, 0x00046dfd327de7f3, 0x00027f03178322fa, 0x0000216e3f0e2310 }, { 0x00003bbfc59abb34, 0x000d842f48027f5b, 0x000bd4fb27522c72, 0x0007b690faa40cdb, 0x0000abded9d9b492 }},
        {{ 0x0002c6f30c43f226, 0x0009e180d738ded3, 0x0000ae17641d02b6, 0x000b5f0756f1a5bb, 0x0000e1022d63ad47 }, { 0x000709b3807f334a, 0x000c89dd8fffb620, 0x0007ad500b84f625, 0x000081d766a281b4, 0x0000baefeb53ba7b }},
        {{ 0x000e8143ae4b352e, 0x0008432a3326505c, 0x00021f9bfe1140f1, 0x000adfcfa38d6927, 0x000034810d2f90b4 }, { 0x0000bf0d00ef8992, 0x0002ffa7e6ab6665, 0x000fd1695563a3a5, 0x000ad66ed6cd2d22, 0x0000bbe464cbfd77 }},
        {{ 0x0000824eeec7168a, 0x000ce52877a21ec5, 0x0006bed12ba5446b, 0x000931ca2300414a, 0x0000eeb5b0f5515a }, { 0x000322d64e381de0, 0x0001cecf83cba5d5, 0x000840921d7268be, 0x000d46dd7f953814, 0x00002b466f4b1410 }},
        {{ 0x000bd6a7d881d987, 0x000ec843d9325f3b, 0x0000379b23abd56a, 0x00044afc5a9bef2d, 0x00004a3dd4f7b324 }, { 0x00032b0b1e2614cb, 0x000de028f61bc0b8, 0x000ceba5ab839425, 0x00059b798f49085e, 0x00003b2eafe5b888 }},
        {{ 0x00020cf3f9601768, 0x0005c47f1f0ced18, 0x00031285e9324320, 0x0008926fa800cc79, 0x000017299d89245e }, { 0x00019c8dbe5b7b2e, 0x00054dc1c0f7d133, 0x00005341590ca39e, 0x0007e40e3ef92196, 0x00008544b679b3ea }},
        {{ 0x0006e673cd25c857, 0x000e1717b82b99b6, 0x000ce0284257ae21, 0x000cd7d6675922d4, 0x0000134e48cf8715 }, { 0x00020fa3844dead5, 0x00092c4b5f2b89a4, 0x0005e6feb94f1d13, 0x0001f55da3de9c1e, 0x0000bf9802bd31e3 }},
        {{ 0x00024c930f786d38, 0x0009e3b0eb919b03, 0x000d74ff03109274, 0x00006f10cd74ee51, 0x00001f7f74fdc12e }, { 0x000970167dca0115, 0x0003819fbac27d45, 0x000a427dd82e631e, 0x0006685b84086e8c, 0x0000417d6e1f9dd1 }},
        {{ 0x000343e877e929af, 0x00043c9faf1fc639, 0x00099f48a33c4af7, 0x000c84f4a616d333, 0x0000b1d34afcd46e }, { 0x00039a2aff6798ee, 0x000296be970e78cb, 0x000eacb90b27effc, 0x000350e1ba36cceb, 0x000025b07b3680a2 }},
        {{ 0x000613ab7466a572, 0x000a9a16224408a4, 0x0001de6950ea5159, 0x000e65293cc856e9, 0x00009b2282c9735a }, { 0x000316fb85a186df, 0x000099cc420bf2b3, 0x00065d1c833d089f, 0x0007b80c700a5317, 0x000062412e82bddb }},
        {{ 0x0007022f7ed617cd, 0x0003df59ac50814f, 0x0007a38c12a4513a, 0x000a6f8a0fdf95e7, 0x0000db96fd7cab80 }, { 0x000a0e79e834cb0c, 0x000deb1cac1fe8ff, 0x000b3d3697805f23, 0x0009cd0e68040188, 0x000080b9a53ca2a4 }},
        {{ 0x000e74c82c8bdfad, 0x0007ffc2064cfd1e, 0x000dc1f05107db53, 0x0005c1eea339c31d, 0x0000f20f060da89e }, { 0x000e4bb978fc26e7, 0x000e95ebaf28ee6f, 0x000b9f25d6af0d00, 0x0008c1b8982cecf2, 0x0000a108e862c5e1 }},
        {{ 0x000756758aa3f827, 0x0009290640bfd9e7, 0x00070b9fab7e1763, 0x00007b656d226e27, 0x00003f7891bafa5f }, { 0x000ca7e720c3e4d0, 0x00092a35285fe91b, 0x0009cff4ea1fff46, 0x00077e008edbc546, 0x000012258488f2eb }},
        {{ 0x000c20ce094f2cf7, 0x00033b817a91cae1, 0x0004ecc1b15a91de, 0x00076e6f9b89aab6, 0x00007bae3c9d9e8c }, { 0x000d1adb3d74632f, 0x000e8b45fcf7e33a, 0x0002e7421004f735, 0x000000e19e679374, 0x0000b0e04475d7e8 }},
        {{ 0x000a6b62d93fdd41, 0x00055796424c49b9, 0x0007cd56def02492, 0x000194a49f888dfe, 0x0000f9869f29b9e5 }, { 0x000f96a765fed464, 0x0004e6d179483cb7, 0x0002157b537c82e0, 0x00066411f666f963, 0x00007c51bf707ed6 }},
        {{ 0x000f21da47afb8d4, 0x000e1bf074a30ce2, 0x0006d138589ef83f, 0x0006aaad4d7f5210, 0x000072f03502bfa0 }, { 0x000998db4eb5079d, 0x000ffb4eec863a09, 0x000cfb8e86bc3ed9, 0x000747ff8d18f77f, 0x0000d2b07489cd8b }},
        {{ 0x00012874a0a79f95, 0x000f4e5c247d44d3, 0x00010e26c8dfd693, 0x000d911c71250714, 0x00005f4b31216fa2 }, { 0x000bb7940ea2bdb7, 0x000a2a6759443709, 0x000f9f7477cf3c12, 0x0000351e3d6123d8, 0x0000f7ac71946caf }},
        {{ 0x0000e639fd11e75f, 0x000ce5984ac066c6, 0x000285a62fa36766, 0x000f23eccf5954a9, 0x000013d9c6495b3d }, { 0x00065ad32f4dc3be, 0x00043257de792488, 0x00046ad4f3a87522, 0x000779f32e9ef640, 0x0000b913b812d8dc }},
        {{ 0x0002e257466c6324, 0x0000f1b13fb70b6a, 0x0000f27743e13f9f, 0x00059e42c037b44c, 0x000046e406dc7df4 }, { 0x000140e65ebb8a31, 0x00056a81da0263ef, 0x00043bd00d7832a9, 0x000681932931bfb9, 0x0000e425b1bd11e6 }},
        {{ 0x00090be4402602f7, 0x000e891b2f1ac7a0, 0x000b6d0409e7294a, 0x000bf31e5f99eaab, 0x0000fb74eb134a23 }, { 0x000a063ac1c5c13d, 0x000bee45004f022a, 0x0001ac97e034e6d0, 0x0005c6da5d838c2c, 0x000081f282d44949 }},
        {{ 0x000cc1e4250112ec, 0x000b82259558a787, 0x00090773f60a0825, 0x0003124dc5343c66, 0x0000291b4782f302 }, { 0x000581ddda3dc5d2, 0x000823f16dfbb7d8, 0x0001d5d9f7b848db, 0x00077fe3c921fa94, 0x0000f85610bf9475 }},
        {{ 0x00047ab1f32b86a0, 0x000bfcb81d73c9e1, 0x0008f5e3e5b8290c, 0x000b3cbab6d07e97, 0x00000464d3a117a3 }, { 0x0001f4e9cbfa8244, 0x0004a3d9fafdee4f, 0x000a6195a1c9102d, 0x0006ef471fab3dbc, 0x00001448eb4ee774 }},
        {{ 0x0009c341cf277327, 0x000fdd366b33f1dd, 0x000f832103f17459, 0x00020d8613b97b81, 0x0000e530424a7a93 }, { 0x000bac5ff89d496b, 0x0003c6e16efb6f3b, 0x0009a4af35445de7, 0x000b6909fdf24b8c, 0x00008e6d72360625 }},
        {{ 0x0001088fdb965d4a, 0x000d187459afccd8, 0x0003363a9676ad62, 0x000588d8e6bd4514, 0x00005899c7ff54bf }, { 0x0000a630bc8a6a3d, 0x000682c1e269375c, 0x00036d1fbb542336, 0x000557f11f9f037c, 0x0000eb8a2a124b0a }},
        {{ 0x0009609d82cffc8f, 0x000ea7eab64c31ee, 0x000a8aefd3c18eff, 0x000a9ea0cffdb1da, 0x000041dcae34044c }, { 0x000ce67dc8a145d6, 0x000af9500d24fc90, 0x0003d839364f3a43, 0x000b1e5de9c75033, 0x000024e5e262c95d }},
        {{ 0x000982daa186bf3d, 0x000d1d44763eb501, 0x0001a82832c513b3, 0x000c80bf324a0e22, 0x000032d177c5f1dd }, { 0x0001ab688e43f0b8, 0x000f1f78bbc55499, 0x0000ed8acfc02b0f, 0x000a6c7507a1783e, 0x0000b89a79b2959a }},
        {{ 0x0007e04350f65b7d, 0x00039b56d0bdd662, 0x000e3cd6ced83cbf, 0x00020b82f0a6ef32, 0x000023dea34ebe08 }, { 0x000df290bff0bdd6, 0x000b4cdc1070c828, 0x0002f2d76e2f397e, 0x000e16129fb2034d, 0x000008903ddb617f }},
        {{ 0x00014fb49f1ee184, 0x0001b33fc2e43263, 0x000d2d8e3b92e418, 0x00083abb53b30076, 0x00001aa8b3c82fed }, { 0x000ada351697521c, 0x000d0a52f7007501, 0x00035650105f08c1, 0x000dedfb02b9e3c2, 0x00008861ad3c4117 }},
        {{ 0x000ebf2c0b49920f, 0x000ddb27b1356705, 0x000f1e3cf422fc64, 0x0003faf9fe85d19d, 0x00004ef9ee33bc0f }, { 0x000f87fda813afff, 0x000ff4eabc199c90, 0x00024732b3060181, 0x0006fed54c703259, 0x0000301ad32fa058 }},
        {{ 0x00001e4bd167db75, 0x000d1fc609e4a74a, 0x00031a4833c074c3, 0x000fcbf362a44a14, 0x00002983fab4c44c }, { 0x000f18181f25f8ed, 0x0003c3273483c908, 0x000118af9d79551f, 0x000b1d37ea79c6aa, 0x000005561a58a7c4 }},
        {{ 0x00083fd5597eef8d, 0x000467c051af62b9, 0x000a73b96f3d0664, 0x000e60da1e9a998e, 0x00006cc734074631 }, { 0x000f6191644596a8, 0x00018ca65d698049, 0x000dcf446dc950da, 0x00041ba3b732b22d, 0x0000e67d9922451a }},
        {{ 0x0001f01cf87103c7, 0x00060eeacee50437, 0x0002bcc0f2099af8, 0x000403405ae419e7, 0x00002218d3d2d918 }, { 0x000ecb8ef6d99c8f, 0x000e47e8887a132f, 0x00086765f5199df3, 0x0003eea5e6751330, 0x0000a872eae8aca4 }},
        {{ 0x0002a910af444e76, 0x000d4223021926a4, 0x00092cf23b6d15a8, 0x000dc5aabb7c28a4, 0x00004beebfd8904c }, { 0x0002c1c510c876a0, 0x00026e07bd68cab6, 0x0008141545b76fcc, 0x000dd0aeb2b57fa3, 0x000028f01be29773 }},
        {{ 0x000fb2f0f13fe534, 0x0008dfbb2d09dc72, 0x0001f6b8e14a4dfd, 0x000cd49b6c3dfed1, 0x0000cba2af46aa95 }, { 0x000b67125e538c1f, 0x000817b80be73992, 0x000df438154aa4c9, 0x0000fbff7ed9327c, 0x0000e6c8e2a6691f }},
        {{ 0x00002e29099045b3, 0x00073e88670ced73, 0x000f92edaaaae805, 0x00082a9bce3dfe41, 0x0000a6fe4cc3fd2c }, { 0x0005bdd6a63370ab, 0x0001411a6eac16e3, 0x0000746d6406529a, 0x0009a52ab7644700, 0x00006744b2717351 }},
        {{ 0x00010c6b129788b3, 0x000e1b07e5c9ce97, 0x0002afa2c95c509c, 0x000dcfa4a0f379e5, 0x000038870779d124 }, { 0x000e2ff99da77528, 0x0003ef971954cf1a, 0x00025edaa877aa54, 0x00047fba8b1cec79, 0x000023185a87ebdf }},
        {{ 0x0006bf94a9ce8a49, 0x0004c3c340ceaa2d, 0x0004576efe8a0a53, 0x000fca76d2635527, 0x0000ef0d9f9a4fb6 }, { 0x000c6f98643cdd39, 0x000610ab76a9af0c, 0x0002ed99c292f6e8, 0x0005b63a052740ab, 0x00002f34b27f25e7 }},
        {{ 0x0001d6152495a6cb, 0x00076a76b2515cf3, 0x000163afb2252000, 0x000dd139771c4160, 0x000019dc05c32b50 }, { 0x000c06a0b6aebb7e, 0x00047e1fbce090b8, 0x000d6fc489a4cb51, 0x0001cce35aac927e, 0x0000dd0bd5bb9272 }},
        {{ 0x00040304b2f32575, 0x000608bafe7ddf39, 0x000d2de3e0a8e028, 0x000e121e755851fe, 0x000012305bc663a2 }, { 0x0009215b9ac0aa93, 0x0006561b0a2a8600, 0x0002055b6de6ae99, 0x0000bb796cf12128, 0x0000d6c9a199774a }},
        {{ 0x00001d3c3571f271, 0x000420537387c09e, 0x000e1021ab2abf2c, 0x000057919c5832fc, 0x000085c20ac422c1 }, { 0x0003afcdec0503c2, 0x0002c41da48ea80e, 0x000deb9fd6d517a2, 0x000ee1fc734e71e8, 0x000090527eb13975 }},
        {{ 0x00010dd399eaf33a, 0x0004ad700235e9a6, 0x00042b3eafd5b269, 0x0003ff3e50308d3f, 0x0000894a8bd16a9e }, { 0x0009738198fc9a32, 0x0007260d1ac16352, 0x000af72f3f46fc54, 0x000b016c584cbb3f, 0x0000e913d664efea }},
        {{ 0x000dd645686c5987, 0x000ed7fa6d74081e, 0x000467cc1cabbdbd, 0x0002f3efe60ea4c0, 0x0000d3d1d48d4bb3 }, { 0x00069876a93ccb94, 0x00037ec96460b259, 0x000dd2439ab5990b, 0x000acaceaa12fd0c, 0x0000da6bef5c292c }},
        {{ 0x00001c7318116168, 0x000fa17472ff5805, 0x0004fab7a07b4f61, 0x000da2392f422970, 0x00000e0d1e93d88e }, { 0x0001e184b20a7a54, 0x000b5b318464945d, 0x000ba2e4420c0b15, 0x0009fe2606d5e594, 0x0000d365f13f5f27 }},
        {{ 0x0006362953e25a5b, 0x000ec524c6a7e046, 0x000adf1ee2117945, 0x000664ea547fa1e2, 0x00008a103e0c9c66 }, { 0x000e3f0859cdc2f8, 0x0004cb15cb12a88b, 0x00096d25592eff44, 0x000ffc8c0561b6f9, 0x0000f36918054b98 }},
        {{ 0x000ab5b320711223, 0x000a2beb92041b8f, 0x00062b91c9ed202f, 0x0001a579401ae38c, 0x0000fb40e7e1d360 }, { 0x000f7ad0829dc074, 0x0000092487443e97, 0x0000199242c6d8ee, 0x0008596378595a31, 0x00001217219125df }},
        {{ 0x000ab2609e227e6c, 0x000e1b61059705ab, 0x000a2c0765f9ca9e, 0x000351fd5a3dd492, 0x000054b35ef99c4c }, { 0x000517ab66adb516, 0x000e8896e70ee822, 0x000deecf0c7cd20f, 0x0001d59caeff2c56, 0x0000ac23b6003787 }},
        {{ 0x00021487098d3893, 0x00021e0bdefdd4f1, 0x0005e1525cf8642f, 0x000032bc06995846, 0x00002ad5c774a21a }, { 0x00067f595255747a, 0x000b83318969006d, 0x000e0b7efa7ee217, 0x00097c961cb4d73c, 0x00007cde45c8bbbd }},
        {{ 0x000e56937bdc883b, 0x000a71e9e70c72e5, 0x0008f7103f8014bd, 0x0009b5cf5b2e52e2, 0x000059c31c61ce26 }, { 0x0003442dae76f13a, 0x000bb17456c61e10, 0x000aa56c75d9f1eb, 0x000b77d55bc2e954, 0x00008d57023bf8e9 }},
        {{ 0x000621dbccc0ef22, 0x000c547ceb26210c, 0x0007b259d32c7710, 0x000b4bc31983545e, 0x0000a78659cb68c5 }, { 0x0000cb3b3dddc4f9, 0x00049e959266fbbf, 0x000642d20bdea51f, 0x00008b496a3514c3, 0x00002d270b753385 }},
        {{ 0x0008c2bf83f14439, 0x000a2ce478e2135c, 0x000ef4e093be4e87, 0x0002341f67547b5c, 0x00000ad1fffb6d99 }, { 0x000c1d8afc50c539, 0x000b1af455809854, 0x00057d1dbacf8e42, 0x000b613acc68d1f0, 0x0000821512510276 }},
        {{ 0x0007e0606f082f40, 0x00083fd9eb45ab26, 0x000b6cd1ce05d312, 0x000e6558a5b234b5, 0x00000f7a741d7de9 }, { 0x0003ddf9a8ed377e, 0x0007270ef751b115, 0x0005a3127425be84, 0x0000278b4f352f17, 0x000037791e3b067b }},
        {{ 0x00051e4e6b8cb501, 0x000c279c21520b0e, 0x000ff6b7834b4d94, 0x00094c6f790864fe, 0x000042d69c0fb7d2 }, { 0x000d60d912391757, 0x0004f7f5c9dc340a, 0x00070f46ed0e8a4a, 0x000afffd05ad53c3, 0x00002565d3e375f9 }},
        {{ 0x000effdf0c11cd5f, 0x0001cdf0eeec8c68, 0x0000788a3915c4f7, 0x0003c36244a27fec, 0x00006b8aecf409b1 }, { 0x000bb395927f21d8, 0x000141c65842df85, 0x00087766136f5c4f, 0x00020dd298821b35, 0x000018965c995c5f }},
        {{ 0x000dee4dc0194e60, 0x000d316fd6f7140f, 0x000abfa03437b7c9, 0x000b47962f9fadb5, 0x0000e38b7d704c40 }, { 0x00062cdef4e7d515, 0x0005f0735d7620e5, 0x00039e5f211dd6fc, 0x000802f9a04e3c2c, 0x0000b9308ad3b24c }},
        {{ 0x00048abb8f5d2a2e, 0x000fc5f477010973, 0x0008cafe1533323d, 0x0009158dd617125d, 0x000015bd2f4c94fa }, { 0x0006588b45797dda, 0x00010b599d94bbd6, 0x000de02c461b7193, 0x0004b1a1ad51ba28, 0x00008fbf0633d6b7 }},
        {{ 0x000970de6f2beb58, 0x0006701fad097a5a, 0x0007b79da2ede893, 0x00026865649883ea, 0x0000dc66c7da4b5e }, { 0x000820bb894ec5ea, 0x0002a245e17fc196, 0x0006990084cddd1e, 0x00078883fd833c65, 0x000023fa1aafb1e7 }},
        {{ 0x000baeb23e56969d, 0x000cc70d875d56ae, 0x000b664d448127e8, 0x000719f12d6a0a9a, 0x0000a1bfa272524c }, { 0x000a8a927cea79d8, 0x000843ab99a907a9, 0x000f67e1618ebdee, 0x00088f59ef1e7454, 0x0000b1cb62e5aadd }},
        {{ 0x000e3f57dc2d6e51, 0x000a50c08191224d, 0x000b000f68b5b448, 0x000753c84c4126eb, 0x000061310eee001d }, { 0x000304ef0417f39d, 0x000a637d446a1dda, 0x000fe29988f21ca2, 0x0002976f57bf2179, 0x00002b04dfb32376 }},
        {{ 0x00051e67299da7b9, 0x000a547d19dfa5a2, 0x00035a8f31025ec7, 0x0008f5cf5db007de, 0x0000c7a0a9aca7c3 }, { 0x0004743570466a59, 0x000f8841ae98e78c, 0x000c62a876c22c05, 0x00089b7f5f818537, 0x0000034a977f0e71 }},
        {{ 0x0003e514e5a4fbe0, 0x000fc5ac242316b4, 0x000c980f9b0feddf, 0x0009eaec6abe060a, 0x000079cfa30418eb }, { 0x000756c3e6d3c154, 0x000a63807d784f90, 0x0000ed5eef704cb7, 0x000ec963d1161a88, 0x0000f21eee5e923f }},
        {{ 0x000fac0dd10881c6, 0x0003801aaa79697b, 0x0008968d0fe0583b, 0x000197fa004a73b2, 0x0000aa2663657610 }, { 0x00019b3e879170f7, 0x00043f57d3d2ec27, 0x000caee09527a33a, 0x000e416ed6fa2fc2, 0x00009abfc9b90ad4 }},
        {{ 0x0007050b07a66baa, 0x000fa966d17fbc7e, 0x00098ccda746ff07, 0x000fbbf370cd1a70, 0x000027f44964c30b }, { 0x000e2646e5bb924d, 0x000fbb61ede9046d, 0x0000b93fdf744056, 0x0006ed7c5a053005, 0x0000ad3637db0835 }},
        {{ 0x00054660dbc8b1ab, 0x000d6088cad38c0b, 0x000fb6902a066a7d, 0x000dc7ca3471b7e8, 0x0000e86a1789d644 }, { 0x00095173a8f8b220, 0x000fe11a6fd8de44, 0x000a5a2626461987, 0x000b8a17c44dbffe, 0x000097f4a165ced5 }},
    },
    {
        /* digit=9 (1,2,..,64)*(2^{63})*G */
        {{ 0x00013e2c380a9bf2, 0x000364d391c2a82a, 0x00027444dcb0941f, 0x00055fbcabcdd486, 0x0000ff69ccdcb479 }, { 0x00052b540c86a1b2, 0x000a4f04a057a87a, 0x000001893b8ba415, 0x000c36a44a876550, 0x00000e23c958c1e1 }},
        {{ 0x000f5e69622c79cf, 0x00060c516a0d2bb2, 0x000beb03cb372318, 0x0004c5d8b0d5cc16, 0x0000e18f5d34d82b }, { 0x000648f91783b808, 0x000bf0bf5a01797f, 0x000c6a4c687ad85d, 0x00067d3523d20b44, 0x00009192002a1f83 }},
        {{ 0x00028c5bd0e3d52e, 0x0007eafe60b7cf78, 0x0000021d7c9889ec, 0x000e55cc425860a5, 0x00006eb845413903 }, { 0x000c4c0a359d0286, 0x000bd5edc55ae5f2, 0x000c93c569dc0f68, 0x00021ae2ca527f56, 0x000043058b802960 }},
        {{ 0x0003271e6c831d51, 0x00077b8fac61d9a1, 0x0002d09fe01d7f5f, 0x000a2a4037d25e06, 0x000064507e385362 }, { 0x000ea6041214d255, 0x00051c0232f76a55, 0x000d1178f014410a, 0x00037fc788957c32, 0x000085a49edcb98c }},
        {{ 0x00060cf82fa14b89, 0x0000d4ed0b8892be, 0x0002acf4c3455b03, 0x00078fd269ea1768, 0x0000311da4567f5b }, { 0x00005c27534d6570, 0x0008797d4df55311, 0x00017d92c88f0a48, 0x000cafeb1235b59a, 0x0000809db71804f1 }},
        {{ 0x000f3338ee1c0299, 0x000f84bb5def9961, 0x0003040a98f9248d, 0x0003ac5761cb69c8, 0x000000beb921e57f }, { 0x00066c038dfaec8c, 0x000d10d2f5378a8d, 0x00065894c993b6fc, 0x000735690acf4c2c, 0x00003b9318cab2f0 }},
        {{ 0x00040461e09c5a97, 0x0008f4af94d70cf7, 0x000eff105d9de3f2, 0x000d95602bdb802e, 0x0000b9b80b8301a6 }, { 0x0003d814bf093a24, 0x0008ec8365e9383a, 0x00029c06e19337d9, 0x0007f32c1ea762c8, 0x000062b44e7d6f3e }},
        {{ 0x000e0341bd6ee336, 0x000102bb4d8bc50a, 0x0005fb1615b32bf2, 0x0005df468181c0b1, 0x000049acb4f25f09 }, { 0x0002c9d27e3d254d, 0x000f8cf7003e8666, 0x000a275ad19d7996, 0x00026d7c2f11a6de, 0x0000e62e628f9c93 }},
        {{ 0x000b4e10567aa138, 0x0000a3588cceff67, 0x000f0fb9dd12b3f5, 0x000abd50d8594c54, 0x0000be14cc0ee64d }, { 0x00086ed7a832f320, 0x000a0d5b5ec995a1, 0x0004fbb9c2ac7055, 0x00008b777bf4b9d5, 0x0000c5b6259dfe26 }},
        {{ 0x000a5ff674823c2b, 0x0000030614c09004, 0x000172a0cbd7525d, 0x0008894a1da98d12, 0x000030b72f8253fc }, { 0x000c72e27f2e7888, 0x0000eb81d7ea1d7e, 0x00029433ef893a49, 0x00053356e38cf289, 0x0000353f5c728ab5 }},
        {{ 0x0004cde963d2fb2c, 0x0003cf5a44f77f7a, 0x000d0f6e695842ef, 0x000f5ba097aa5f9f, 0x00008eb52caa0bc1 }, { 0x000239c14011b93e, 0x00062c0eee470afa, 0x000de8059d887af6, 0x000e9d43184137d0, 0x00000f5b3380d1e8 }},
        {{ 0x000b6b6e1ca760d8, 0x0007f19a8afd30bb, 0x0007cf16ffb18b25, 0x0004823e08679832, 0x0000efcafa52751d }, { 0x0002beaaf58337ec, 0x0006f11471f1ff0f, 0x0008cb570ffa8731, 0x0005ed83ed76c338, 0x00004c92081ca03b }},
        {{ 0x000c9c96430e9a0b, 0x000a59ec3076a27d, 0x000e0779be1a0693, 0x000d3642e5e82908, 0x000023d4c1872f26 }, { 0x000b6eb671cf9704, 0x000b48d519bf6156, 0x00059a3370dc9737, 0x000c72ba0e29ecd7, 0x0000d89e2cc152ef }},
        {{ 0x0000aaee181a2a7a, 0x000ff32b88756ddc, 0x000230d35049ce8e, 0x000b74aa4ed4e865, 0x0000261c5ca08124 }, { 0x000d55e8ca475a13, 0x000dab96c168af3a, 0x000f0644ffaace6a, 0x00053a1366563c7b, 0x00008a47ff05a4d9 }},
        {{ 0x000813e66abf3deb, 0x00033d8b76da1f56, 0x00063d76e529d81c, 0x000c8178210dfa46, 0x000088e501fc78ec }, { 0x000d7ee53868e5f5, 0x000b91e350cb901c, 0x0005834c92ea4be6, 0x000c54687b653d65, 0x000007ddaf56d876 }},
        {{ 0x000f835f9d0aad8a, 0x000272e523b8bf60, 0x000b800dac7dffa3, 0x00068eb888009eb5, 0x00005af05fd9e53f }, { 0x00029d4c6fdd401d, 0x0000c312afceb620, 0x000a13ffebd0185b, 0x0006be5b0c797df2, 0x00003ac5e2304089 }},
        {{ 0x00033878925685fc, 0x0009fc1afab4bc58, 0x000ed33fb9b57d7f, 0x000a73a05833f5c6, 0x0000c58881e99e36 }, { 0x0001b7f21a55c701, 0x000b20fadd9419a0, 0x0009ae2c0ace83ac, 0x00042a9c6bcca109, 0x00002b5c2691fab2 }},
        {{ 0x000ce9141bcb3848, 0x0004a19909523c8b, 0x000ee6b8b3259a8a, 0x000919d5aa62f648, 0x00009596a8c13395 }, { 0x000520dbd9ba1d54, 0x000298435f9b9887, 0x0001bce399dce0dc, 0x00073f36c0210da6, 0x00000d8103e648d7 }},
        {{ 0x000a389397c74a91, 0x0009bfec1ab1bbdd, 0x000e022c75a0a320, 0x00012eac4ef8a65b, 0x0000945f9752a8b9 }, { 0x000b8d4c407c4937, 0x000863674660876b, 0x0003f8d0b2f52d8c, 0x000aca39411d67c5, 0x0000712bb3bc0828 }},
        {{ 0x000ecf1276d09b96, 0x00055d1e0a1b12ac, 0x000ca83c96e8b70c, 0x0008f53cb1e4ef88, 0x0000798dffb7e60b }, { 0x00068fc88f16a70b, 0x000f62e1b39b80f3, 0x000cff33731551d4, 0x000ef56c2fd90d0a, 0x00002d84069cd845 }},
        {{ 0x000203d2c908ee78, 0x000ef61196a2fa2a, 0x00071b575e65ca1d, 0x00006d231931b981, 0x00000a20eba5436a }, { 0x000781b3c4ff64b0, 0x0002ee098939db32, 0x000c16deb83df347, 0x000bb4643e37d6c2, 0x000068ca7511af52 }},
        {{ 0x00077057e537e323, 0x000ddb07850ec06e, 0x000895c9b8ea2e5f, 0x00067909dac5a38b, 0x00009de2a5e3a8ab }, { 0x000620a4c251a5ac, 0x0001442b59e78df5, 0x0007c77739906b4b, 0x0003b1e4a4cfc91e, 0x00007e41e6a9a9ed }},
        {{ 0x000e5ae52c223602, 0x000d2aa049f842cb, 0x000456e602ff290e, 0x00090ef56ceabc5d, 0x00009d94422dd0f3 }, { 0x000f78ebd3835c75, 0x00072e6334957f10, 0x0000219b5e9c8395, 0x00039cf7b57e388f, 0x0000e66f4d445c5c }},
        {{ 0x00080ff42e6bc91f, 0x00007052b7ce5421, 0x00095c62c9b34c59, 0x00009d5dae6d4ce9, 0x00005ca79a998f9e }, { 0x0005aebf80664f4a, 0x00082022c34a1c65, 0x0008ec7d87425f4c, 0x000c55ea23803d6f, 0x0000a8295081cef5 }},
        {{ 0x000d185891700399, 0x000e4587135593fc, 0x00065ecd183bf605, 0x00088ed0d32e6eff, 0x0000f3fb39aa7eba }, { 0x0003c612df40b960, 0x000899d6d97e1340, 0x0000d9be34ef3c31, 0x0000ba273d42c6b7, 0x0000fd96d1086b78 }},
        {{ 0x000bf546da5d876f, 0x0005b1504b6c5a08, 0x00088cfb9cc92732, 0x00010fd0035216f3, 0x0000d81054fae260 }, { 0x00001d5cefaf619b, 0x000baa44cd623c3e, 0x0007700e316b2264, 0x0003f24e7366ce71, 0x00007cfc40c2c53f }},
        {{ 0x0005eccae559fb71, 0x000d4d9036520f83, 0x00015ce743b334c4, 0x0005a2904dfb3c3d, 0x0000962fbaf2e972 }, { 0x0007e09d0efe288f, 0x000a3dbf8a5a0ac5, 0x00059388f9f119e2, 0x000fe2da0547972a, 0x00006949be49f2cb }},
        {{ 0x000564c83f9203e1, 0x000b60e2fdd23cc0, 0x000c511e21322752, 0x000e48dbcb80288b, 0x0000797385f1df08 }, { 0x00012d656be5c3a7, 0x0008ba239560368e, 0x000a236ec3766f55, 0x000ddb1c3e893752, 0x0000ce0f9a1079c0 }},
        {{ 0x0003f375576cb563, 0x000f24e63e3bc1da, 0x0001a110da9fde91, 0x000d84707c719b6a, 0x00001a17e6c0b377 }, { 0x0002b1aa240c1a70, 0x0004f760700d3607, 0x00040f793abe8e2d, 0x000d78b4cbb1a182, 0x000018bc4b3b7f7d }},
        {{ 0x000b0c591bd83326, 0x000c4cabe063f644, 0x000371d820edcf2c, 0x0009d54ca9b392f4, 0x00002b614ae1ed42 }, { 0x000be79af8c567bc, 0x00021fd92f1169ae, 0x000b60d7889967db, 0x00029e6af63aeb1a, 0x0000d1f76a097e41 }},
        {{ 0x000fe8bd2ff0d26a, 0x000c04e40e50acf6, 0x00073f191935df0c, 0x0005f541ff0a98ad, 0x0000d0709fed8f27 }, { 0x00066466bcac1668, 0x0009ba16cafbc916, 0x00091ca479cfe8aa, 0x0006a974f170f875, 0x0000aea394d3eeeb }},
        {{ 0x0000b1d093f471bb, 0x0006510f5b343bcf, 0x000a023f726a643d, 0x000ed90df0f2e400, 0x00003c3843aa4ab3 }, { 0x0000bb7742e67add, 0x0003e95c70816030, 0x00093cfe99544825, 0x000be16a6e943e4c, 0x0000792d7df86acd }},
        {{ 0x0003a9e4817dcf52, 0x0001e83824297b45, 0x000001da60dd959c, 0x000f8849e9d0e558, 0x0000f7d56e3c5b73 }, { 0x0001d44293968f25, 0x0008be065eb24132, 0x00046defc5136d73, 0x000f02f92dadbbb8, 0x0000281b17b810f6 }},
        {{ 0x0002077086307267, 0x00015e63b5f1cc41, 0x000abc7fbddc361f, 0x000938b4437ae52e, 0x0000d5d97371ebf8 }, { 0x000fc8b47bfe2c04, 0x0004a491058edeaf, 0x0002b67015a14d38, 0x0000c551c827510e, 0x0000de8b56068862 }},
        {{ 0x000270352a539749, 0x000d7e61b4d5e634, 0x00025b25e620e74a, 0x0003a22edbd86474, 0x0000a57233245448 }, { 0x000a08d537795ccb, 0x00034e664cfc50e9, 0x0009842df60b03bb, 0x0007f949a50dee01, 0x000062e5de6beabc }},
        {{ 0x000ff7a5b46ad8be, 0x00056576be5f7de2, 0x0008922e96e9e3fe, 0x000f20d36d93006f, 0x0000601b5481985f }, { 0x000c615b703d9998, 0x000bbf1cd871236b, 0x000d064911e1ff34, 0x000ff3f6df156a39, 0x0000a91dfe2f0173 }},
        {{ 0x000cde4adff641f7, 0x00041e8de3eee6b3, 0x00086eb2c2b1af03, 0x0009081bd58c7753, 0x00008deb4969dc62 }, { 0x0001141f046d8047, 0x000d8d25302169cc, 0x000fe875ce874407, 0x0001659ee8ca60f9, 0x0000414e05bc5492 }},
        {{ 0x00090cfd9375a854, 0x000ec8d6dfafed30, 0x0003305917f500ce, 0x000ca1ba033a0165, 0x00006da9376c5895 }, { 0x000e293b24e6645a, 0x000f1aea3c40af51, 0x000bb3e7eeccfcea, 0x000bc22e83469ead, 0x00001e948c862537 }},
        {{ 0x00073554cc7c12ee, 0x0004f131e6c0bb48, 0x0005094218689b04, 0x000d40b221900469, 0x0000eedc75a767f1 }, { 0x000c08d7b45cd697, 0x0002f27cbf18a512, 0x000b0429e8cca640, 0x0006fec4d4d93651, 0x000034514ce6bcdc }},
        {{ 0x0007f5e2fb1758d8, 0x00077b486df2a61b, 0x00073501d1a6fc21, 0x000776affa1ed9c0, 0x0000e7124755e764 }, { 0x0003bc919d78f96e, 0x0007e6bcc2423b79, 0x000f26a75d0a01ad, 0x0005738e175ce0a7, 0x00003c4ce13c1479 }},
        {{ 0x0003e291023c1b3b, 0x000ba6f871942dfe, 0x000e94771c645656, 0x000afdf782372ff6, 0x000084b784d593f7 }, { 0x00061a89634cbefb, 0x0001b11e4e8110ce, 0x00044ce70f6b5b2e, 0x00061f4e02a2d70d, 0x0000222495a4851a }},
        {{ 0x0006cd7685d97af3, 0x0004785825808bd3, 0x00070909727c9ff2, 0x000e726d151dc3cb, 0x0000b806e9f304b7 }, { 0x000deb57a092fbd7, 0x000c605b22922a44, 0x000e8db73ec050df, 0x000d714c76494c87, 0x00002e35bd113f29 }},
        {{ 0x0009ed0eda00c05d, 0x000c259c3d12a732, 0x000c7bd5b6434166, 0x000b91ecb5f71687, 0x00008aa1f29eec04 }, { 0x0003fa2fed9642f5, 0x00066474683c2eb9, 0x0005034a4c594fbb, 0x000dd7c674956eeb, 0x0000dc5d5eff55da }},
        {{ 0x0001e6e6273a9971, 0x00099f231f63950e, 0x000032e3bca22cf9, 0x000130d62a77797c, 0x000012ab210bc6ed }, { 0x000864d0edf2e53f, 0x00039b0b3c9fef02, 0x000044fabfd0707c, 0x000e912cfaf7752e, 0x00000d5f6d1d2afb }},
        {{ 0x00024de3470dc37d, 0x000668397c69134a, 0x000648db2f4c601f, 0x0009d3de05a42c31, 0x00001832ed2bd25c }, { 0x000033860bde0c47, 0x0002daab75d49411, 0x00061d1c99e46bc7, 0x0006d2dcf9890058, 0x0000216cef95030d }},
        {{ 0x0006acbb473c0f8d, 0x0000dc82094cec3b, 0x0007898ebbc57ab7, 0x00049cc0d9976fb8, 0x000043a34cd4250c }, { 0x000d91996754db45, 0x0007f6144569d857, 0x00061a5eefb627fb, 0x000fa5c7dc3190a3, 0x00002f9e3edd2df3 }},
        {{ 0x00005c3197c1a20c, 0x0003c37dd1c8a206, 0x0007cf0c0281a36e, 0x000f1f62e7e5410c, 0x000059f6d21710fc }, { 0x00077db2a94dd7cd, 0x000a80101683d546, 0x0004f1dd5f8af1e8, 0x000a95a7f1d7127c, 0x000083f71312d346 }},
        {{ 0x00054d652ad70ca6, 0x00026aaddce33450, 0x0006fbdcde261523, 0x0004ce5dc2a6811b, 0x000053683550c6d5 }, { 0x000c91c825e881fb, 0x00056d492a210056, 0x0000c67f47b9218f, 0x000de150fad7f397, 0x0000880885c56630 }},
        {{ 0x00093e4b2fa0fdfa, 0x000b8a2ce309f06b, 0x000910e8d51c27ca, 0x0004ae92efdb27b5, 0x00004db9e3997567 }, { 0x000141c1e6fe9e0b, 0x0007fe644aa3dedf, 0x0009ced4138cc692, 0x000f98c07317a666, 0x00003ed5e5173c1b }},
        {{ 0x00060318b442f800, 0x0007de5619063736, 0x000e85967ce17243, 0x000a65c36858dbad, 0x0000f669b02efe62 }, { 0x000465311b7e9e0e, 0x0000ecd12ba4b7aa, 0x000b5462f70f3e78, 0x00014050df399daa, 0x000097efc0401f70 }},
        {{ 0x000096a8921179a4, 0x0004ac864d3779b8, 0x00062cce18df4dfc, 0x00017a0283c5d047, 0x0000bb1823fedf19 }, { 0x00068eded5c79181, 0x0002d9d23b3ebc2e, 0x000abef1f1010c41, 0x0001e9e6c90bdd6d, 0x0000b133de5a1dfe }},
        {{ 0x000990b4e4f167a4, 0x000b77799eb6df0e, 0x000089f602942610, 0x000b0008e26c3cc5, 0x0000153c35c8a372 }, { 0x000a4c3313da4eec, 0x0007f7e2586abba3, 0x00051f7d3b8634f6, 0x0009c5606239ca6a, 0x00001e8329fe8908 }},
        {{ 0x000c7bfad0211361, 0x000e706f6a3f6fbe, 0x000e9946570d08fd, 0x0009423d3ad5cad2, 0x0000a4c7595c43d4 }, { 0x00060c43d2476796, 0x00037333e23cb3ba, 0x00041f9a075abd4c, 0x000f0fb063979fe8, 0x000050ba89794e36 }},
        {{ 0x00004ca6d9af362f, 0x0006182be7cf3a64, 0x00027c49cb9a7e78, 0x0006a79a9cc86ba9, 0x00008d6173b50eff }, { 0x0000b23616d2a4f4, 0x000a85d005921b83, 0x0001143565e32ad9, 0x000d0e9c2f56297f, 0x00009e435672493a }},
        {{ 0x000a3e7d01e42a4d, 0x000c021bfe50e2b2, 0x0000b78170df66dc, 0x00081fde07ef8cfd, 0x00003567a5d00f46 }, { 0x00087981154ff114, 0x00063ece38d9d1cc, 0x000dd3e3acce4961, 0x0004b94c9b500249, 0x0000c4108a4b0ba1 }},
        {{ 0x0005187a5a610468, 0x000aa5dec3f1dec5, 0x00003bf8fe7fb8e1, 0x00045ff3807b1f04, 0x000027e52acc802f }, { 0x0000c3db2b856872, 0x000d522eb329d415, 0x000ea0acd8667267, 0x000318628f449099, 0x00009931d6113e45 }},
        {{ 0x000842a93785dba6, 0x000d1843757b3927, 0x000d31f9f72363de, 0x00009edd4326caef, 0x00004a867e26046c }, { 0x000d01979080ff2e, 0x00091965f96b10d1, 0x0005e9a735756595, 0x000f199fc373a9df, 0x0000befd7ab4c167 }},
        {{ 0x000fc047a72b6bf5, 0x000e3585499fb325, 0x000917e8670fd4f8, 0x0009cf9c57b67bad, 0x0000c3e8ea66d001 }, { 0x000b687b7d967ef6, 0x000db59e54a64bb8, 0x000b38cc1c666ef8, 0x000616a8d8862201, 0x000029b881c8aa96 }},
        {{ 0x000ca73dcaf21a68, 0x0006b678b2ecfa91, 0x000d8c64fe6d7cba, 0x000cb5c2b24dfe62, 0x00006a76f958454d }, { 0x000efc172f16fa07, 0x00031b776bb87ad6, 0x000b8346a0f01ef6, 0x0000389cd674976a, 0x0000c0f729ce4f9a }},
        {{ 0x000f232b5f42ffb1, 0x000059b92222f1f5, 0x000a0dd48310f507, 0x0005266ee5a2459c, 0x00000b4b667c3d65 }, { 0x000bd63ba4fde95d, 0x000e1c548e9e5167, 0x0006b5c1af8c6824, 0x000136d23124b4e4, 0x0000f392b3ec6ead }},
        {{ 0x0003122eee899cef, 0x000e8cc290a7f4fc, 0x000d3f748573265d, 0x0000718af6b409c9, 0x0000611f3640b176 }, { 0x0006bba5dab0844e, 0x00050504ef24d7e7, 0x0003c76eb236673b, 0x0000be0015d92979, 0x00002b8b64708c52 }},
        {{ 0x00078e66aa656cc6, 0x000a09cda94951ec, 0x000c9f2a6d6e848c, 0x0006760a44b6af58, 0x0000d198106fc9b2 }, { 0x00085099b31bdc5f, 0x00061834a53b9649, 0x0008670f897ce93a, 0x000e96997fc659d3, 0x0000918e4123a93c }},
        {{ 0x000954ac9501ff4f, 0x0004ffcc52fc2384, 0x0008216d2daa041e, 0x00084af08712b272, 0x00008cb41023b490 }, { 0x0004ccdbf9cf67dd, 0x000d0a47fb2e74f1, 0x000ea3cc4ab6b658, 0x00069d25bee8ba98, 0x0000589f94d48354 }},
        {{ 0x0009c15b35bafc8c, 0x000bc1b7bfe71786, 0x000dfb7efd32fb77, 0x0008ffa7d4e1deba, 0x00001f575658ec09 }, { 0x000e07633366d529, 0x000ed66368710884, 0x000cc4194ecbd474, 0x00068764de5e29f5, 0x0000a864508101bc }},
    },
    {
        /* digit=10 (1,2,..,64)*(2^{70})*G */
        {{ 0x000ff71880242404, 0x000f1c7a3e6fced0, 0x0005e468a56ae92f, 0x0000f9670e8cbbdd, 0x0000020c88ed5eef }, { 0x000f9526f1333efc, 0x0008b74bd70437a2, 0x00018a76b379190f, 0x0004cd66e5fb3289, 0x0000132085c0e1be }},
        {{ 0x000edc4d24ff34d2, 0x0005e5192c4d6840, 0x00056736a2a84199, 0x00067dc208b04d72, 0x000091aa355f1170 }, { 0x0000b62c6dbea762, 0x00084bf854279005, 0x00058b53b37ca64b, 0x000f4906c3de8129, 0x000017420c7a1978 }},
        {{ 0x000c11118ac1c9dd, 0x00006b4c72dfe67a, 0x0005e4d08e55132f, 0x00049e035c416b0f, 0x0000282e71d5028a }, { 0x0004097747951cdc, 0x000c718f00628413, 0x000f55e9dfad607b, 0x000fa9e0cbfe0602, 0x00007edbf6c8369a }},
        {{ 0x0004645d0c999206, 0x000f55ed005832ae, 0x000aac9fd1427420, 0x000def2ca5f5efd3, 0x0000d8417da6ba53 }, { 0x00015bfe3a91ab48, 0x000ad8bc9001d1f8, 0x000bccb573cb03eb, 0x000ed588452c8f0b, 0x0000190d99ea9851 }},
        {{ 0x0000b35e6bb694d2, 0x000723f23885e5fd, 0x0009178fdc3fe150, 0x000401404f72f8f9, 0x00006aa2e224e92e }, { 0x000c8b1a1eb06618, 0x00088489164722d3, 0x000b4beb04a9a74c, 0x0004220323e3c665, 0x000059dfb811788d }},
        {{ 0x00086a67d771ac98, 0x00014e654317cac3, 0x000919ad09dbd2eb, 0x0003350b8be600ab, 0x0000e6ed7070c73f }, { 0x0003ebd5c9545d9c, 0x000f18d4431cc288, 0x00021471ef5ad0f6, 0x000672232a7cff14, 0x0000aa1dd2bbd83f }},
        {{ 0x000fd169fdcd85cc, 0x000b4a7b1fd2b0bc, 0x00058c1ac9eaecb4, 0x000ee73b39b3ad39, 0x0000f7b13aef63ab }, { 0x0007e6961db7503a, 0x0005fd79ce0a6af4, 0x00019c46a9db1c64, 0x00073a7b255dc1cf, 0x0000917465424f0f }},
        {{ 0x000f6928ec987520, 0x0001acce548b37b2, 0x0002556adc5cbc76, 0x000befb4fb38e754, 0x0000e6f549a164e2 }, { 0x000cbaf6791e178f, 0x000242386ce6c413, 0x0004d3f2b753e67f, 0x0009e6bf901be1c5, 0x0000fea1ab006e2d }},
        {{ 0x0003452936f20eda, 0x000474550ca42470, 0x000ae49807739ac2, 0x000811424ba1aa47, 0x00001f1e4b8ebaf3 }, { 0x000ac86d5bc8d43a, 0x0000181243aea965, 0x000ebafb6389c308, 0x000c049619a2919a, 0x00002b0cb20d9047 }},
        {{ 0x000f1bcbbb3db35c, 0x0007d18c36cf4406, 0x00075a1da5e1badf, 0x0006e2f64aed3e83, 0x0000987f12991782 }, { 0x00022fe40b1a2d46, 0x00090ed60919b8eb, 0x0000d89215e03057, 0x000a894efd890079, 0x0000d06046a39ef3 }},
        {{ 0x000d57eb5760d1c2, 0x00099a2157f2db33, 0x000f50d0df45370f, 0x000e83254bda2fc8, 0x000085807ee44cd6 }, { 0x00038f14b932ce60, 0x0008ba2fb5ee9973, 0x0002ad862411b6e7, 0x00064197401648ca, 0x00007031fac23dc3 }},
        {{ 0x000afcbcddc5b4e3, 0x0003f6ad7ab9a2d8, 0x000cf2fe2c7a8c98, 0x0009cf5c29c00090, 0x00000a465e1b5bc1 }, { 0x00039efc04140b7d, 0x000aa29e99d043b6, 0x000b4b1d9836432d, 0x0007aac35ef51263, 0x00007f50574f8f0c }},
        {{ 0x0007b58f4a281850, 0x00004b366b4be7ad, 0x000dacedead3f7c8, 0x000bf0cd93f25a9d, 0x0000ff4401b26350 }, { 0x0006a8e2b1dfd7fe, 0x000fba9ff563436b, 0x000932066f358f9c, 0x0001181ecf718ff2, 0x0000bb7b63ee6136 }},
        {{ 0x0001fe1782b8ceeb, 0x0004fa8327720c15, 0x0002680b1545ef26, 0x000a5814d956ca32, 0x000014f5d4a9c76f }, { 0x0007ab1de209b63f, 0x000f5db3c832c80a, 0x000bc5f7c5d2c7e4, 0x0005a0dcc60dccc5, 0x0000de8668f74c00 }},
        {{ 0x000686123db19be9, 0x000c50b3893d123d, 0x0002dad8d4df00f7, 0x0000d5d8bf7b7578, 0x00008135b99cd691 }, { 0x000b6a20e86bb9c5, 0x000a391a7ab4116f, 0x0007a97c837120a7, 0x0002c35d76fcd72c, 0x0000531c7135280c }},
        {{ 0x0003b25760e88423, 0x0009b7d341d81dcf, 0x00083e622c18ffb5, 0x0007e4badcddb688, 0x0000becf33dd410e }, { 0x0009a60cc895ade5, 0x000e3f6a7a73ae65, 0x000e082fa7baab21, 0x000f729bc83879a5, 0x0000c301b48a24ab }},
        {{ 0x0007d6af9dfc33c4, 0x000f9ceb18762621, 0x000ffad89f4b1f3f, 0x0003a59072b0d79e, 0x0000cd702c4f6dd8 }, { 0x00002614f15b98f8, 0x0000c616d7364517, 0x0006e68aaa6917b9, 0x00000a39d43cbbd9, 0x0000899bf61d6a7a }},
        {{ 0x0006dedaf4a7ec4d, 0x000a31a59c3e9f87, 0x0007578c622162c1, 0x000f162ad40200e7, 0x0000d60b302fdf36 }, { 0x0007b6a2e807e456, 0x0002a7c4bc2e88f8, 0x00008cb732bd420b, 0x000a903e96ba8129, 0x00003121d1ba9275 }},
        {{ 0x000582b557d8da3c, 0x0005d971b4e598eb, 0x000e517b95c5fc8d, 0x000d82bc5f499ef1, 0x0000275cd05a5723 }, { 0x000af53901d82522, 0x00090ca1715cb07e, 0x0001998aeb1b5a8e, 0x00031fccd4c3de6a, 0x0000aa50e5e8ebf3 }},
        {{ 0x000f2386acf8a72d, 0x000604ac3b819900, 0x00028ee94251e622, 0x0000cfab91cb8372, 0x000019e78eaae4da }, { 0x000f8063ac90f7e0, 0x0006d1eb32912137, 0x000dda25c7380214, 0x00046a9e3a2f82b2, 0x00002b889ab4a768 }},
        {{ 0x000586ef16cc6afe, 0x000b76759ad2877c, 0x0001d2ded83e32a3, 0x000039c846f45464, 0x0000d9ede27f0357 }, { 0x0000d0e5fc379cde, 0x000d8e024bafad93, 0x00023409aa6e07c5, 0x0009598a0c8b42bd, 0x00004ed05a16b8a2 }},
        {{ 0x00086b599a8c25a4, 0x00086e1e8b4dfd4f, 0x00068a9672118f45, 0x0000495acd754d5c, 0x000052f9c3c08e85 }, { 0x00027d16d7f92149, 0x00095a799f9d5812, 0x0008a07de7d22da8, 0x00003bca50473b1a, 0x0000faeb0de17d42 }},
        {{ 0x000721dfee849503, 0x000f43cad3d97f9a, 0x000477f19466ae81, 0x000004b7c52bf3bb, 0x0000ff72b1d2fc85 }, { 0x0004e7ef4acdcc1e, 0x0007a21141d4579b, 0x000bf553e4eae6c6, 0x0002e42153411321, 0x000058c897845f4a }},
        {{ 0x000a78b173431ac6, 0x000037c593710000, 0x000074fceaa51d30, 0x000c0a565e1c48c7, 0x0000087805015262 }, { 0x00044c6adeb5098c, 0x000f59dbcc8ad945, 0x0006844fc921b6d7, 0x000aaf80a70dc04f, 0x00009ea6e278d5f1 }},
        {{ 0x00018b6ba74db2c9, 0x000294e907a544b9, 0x000a9fb236691c4e, 0x00031d04ba7520dc, 0x000017babaf8cdc4 }, { 0x000818ebab294990, 0x00050fc644be8924, 0x0006b5968097aa9c, 0x0001283fe707a981, 0x0000f54144f9bcc8 }},
        {{ 0x000a2e6ab0d7e737, 0x0009cad58d2f767a, 0x000d2db95cc250c3, 0x000c89598dc5267f, 0x0000fb76d298d7eb }, { 0x000b5ae1ce39226a, 0x0008204d0c63eaaf, 0x00041ea4c6649b4d, 0x00017f91fa1cb92c, 0x000080a55559aaab }},
        {{ 0x000d96332d46ecf1, 0x0000a906e77de048, 0x000062d7886a21c8, 0x0002acf1d7992bcf, 0x00002e42c527e67d }, { 0x000072690f62575b, 0x0006e3a413aa2b94, 0x0005bc265f962dd8, 0x00007b63799a4d91, 0x00009074a785f7f4 }},
        {{ 0x000cf86624af5d13, 0x000f4cf02ff6072b, 0x0008cde126b78788, 0x000985309a47d2ca, 0x0000c4b1abc9fa46 }, { 0x00028fb8b07c6947, 0x000bf87aa9e5cb6d, 0x000144b063bc3aa2, 0x0003ccd35c98124f, 0x0000562b62f1af58 }},
        {{ 0x00050467db792ce5, 0x000972a40c251f81, 0x00078dd5dce34079, 0x0006e6f2388cd5d8, 0x00009ca7a668bb67 }, { 0x000b3e67f86537e5, 0x000f9bbcdabb8397, 0x0002764532467c29, 0x000f53b2ba79e974, 0x00008ea58750c9ff }},
        {{ 0x00068bd1f9103bc2, 0x000158fb1cb6ac9d, 0x000df3913b4c4a6a, 0x00069fd95ffa13e8, 0x000056704285e41b }, { 0x000990a0579bee8e, 0x000f8f926d783f2c, 0x000332aa0f6a131e, 0x0008f1b7c543a59b, 0x00009d122d6f7adf }},
        {{ 0x0006ba4245768c48, 0x0002db8519cce2c4, 0x000f6759961dc7db, 0x00035185cfc9af71, 0x00002da71c5029d1 }, { 0x0009e5918edc1491, 0x0005da4d3fd524d4, 0x000123fc1b33c45b, 0x000399a421e72464, 0x0000b38767166257 }},
        {{ 0x000915ee22d46e27, 0x00009b665c7322db, 0x0000faabc5d86f27, 0x00004cf85cf12cc0, 0x00003fe2633a632d }, { 0x0000a1c1ed8106c2, 0x0002e3ba317db248, 0x000897173dd719f7, 0x000cd0e5087dec65, 0x0000d47fab598e63 }},
        {{ 0x000caeecf58691f9, 0x000319678bf030ed, 0x000973a71452fdfe, 0x000a42b26fb5e9c9, 0x0000351791169cdc }, { 0x0008168ffbb6faf1, 0x00096bb518d0c6b1, 0x0002b381cba209dd, 0x00080d2819918115, 0x0000fd4bfcc99d38 }},
        {{ 0x00083365828a1975, 0x00058aa2502753c1, 0x0006390d16bf0feb, 0x0001befe5bb279e2, 0x0000995cb1132868 }, { 0x000733edbe3a75bd, 0x000fbc4eb8098772, 0x000dc2da79704c0e, 0x000789b67b336e18, 0x00009002f7d131fa }},
        {{ 0x000791f21653a4b0, 0x000d09a26025c3c4, 0x00064bbeed725db9, 0x0005e4d8ee6e981a, 0x0000b19e4a202131 }, { 0x000266ff7fb4c307, 0x000608925f02425c, 0x0003b83044c40262, 0x0007ec552ec86319, 0x0000716e68f00346 }},
        {{ 0x000c27ebc50428fd, 0x000abc3017025f39, 0x000c5f6d8def3924, 0x0000e19ecefcf628, 0x000032c44e11511b }, { 0x0005c3dffed38017, 0x000f7a11da1767e7, 0x0001d13c6c588b39, 0x000a35cb7fe15145, 0x0000aa988aa4f7b8 }},
        {{ 0x0009889769d0f731, 0x000f973ddc925fcd, 0x000055dd56a16ccb, 0x000258eebb679c84, 0x00009268c8cc2b74 }, { 0x00097b2662417e26, 0x000a87fb28ad2470, 0x0003f70e2717aaa2, 0x000e018a7e99146c, 0x000093a9487d1a5d }},
        {{ 0x0003e3172fc44d4d, 0x0001de6cf9f82a81, 0x00092fac300b1732, 0x000eb09c3319decb, 0x00004b56a4e95663 }, { 0x0006f61f75789888, 0x00070dc7d5a006c0, 0x00066f0587dcb0c1, 0x00029a49e2dbc27c, 0x00002832c4e33d0f }},
        {{ 0x0005c14ee801bced, 0x000cc7104779ce2a, 0x000164ec7c3519ae, 0x000edc5d5aca8381, 0x0000fa85767ed4ac }, { 0x000ab6c37ae9b82f, 0x0009683f17a62c70, 0x000c0c80967bc048, 0x00011798f39eb962, 0x0000d8fded0d599d }},
        {{ 0x0001e7018d266862, 0x000e5dba28a0749b, 0x00063c6a8fd87c38, 0x00018a515246c20d, 0x000045c4ea3224ad }, { 0x0003ddb70877d76c, 0x00045faf380ff628, 0x000d6fed75158c1e, 0x000ba3bdaef7fb1d, 0x000069deaeeea084 }},
        {{ 0x0005f83f869b4f81, 0x0006111fe7df7a4b, 0x000150472fba2a01, 0x00027069ca7cd07f, 0x000060963577ae8e }, { 0x000e155d967e0f8b, 0x000475ef3d8ac3d1, 0x000f446c416bd09a, 0x000e51a646fe2066, 0x000079aaace72c50 }},
        {{ 0x0005c34d6e4e9113, 0x0008e9bc3a72d004, 0x000b3f5075acc01f, 0x0003533374037665, 0x0000a738807c50b6 }, { 0x000a94ea00049bb0, 0x000310ff01fd7967, 0x000183a17db85256, 0x0008620c04dfb810, 0x0000ab6af468538b }},
        {{ 0x000fa6b43cf5d930, 0x00080d9bb9aa8822, 0x0005bc5f48d33762, 0x00043764eea20e4e, 0x0000fab3b1672224 }, { 0x00016dd2c48181ee, 0x000f5cf3aa91121f, 0x000948c634d733f5, 0x0005e41c08c6bd5e, 0x0000bc67f9d574df }},
        {{ 0x0003b606436178b1, 0x000c5d980101b980, 0x000db479092353f9, 0x0004b3ef3760883f, 0x000083e79a8931ea }, { 0x0007ca7221dd1d3e, 0x00015dfe05beeac9, 0x00022468b3abf051, 0x00031c5c3f2f1e86, 0x00003d226244ff9a }},
        {{ 0x0002738eddf0b5e3, 0x00087fe1cd19f723, 0x000cbf106d0d6957, 0x000765d3ca20915d, 0x0000205e1c9f6292 }, { 0x00018be236919468, 0x000b5f6b5eccf1aa, 0x0001872fffb9b505, 0x0007994934b5aff4, 0x00005e213e2348c5 }},
        {{ 0x000cecf93e151784, 0x0008bbab00ed3a95, 0x000416e0ff80909b, 0x000ade39734c6137, 0x00001fe486ea0896 }, { 0x0002d754720a1bc6, 0x000f841a968fbf0c, 0x000257852acb7321, 0x0006be3c7d23d458, 0x00003c39d3db2eb6 }},
        {{ 0x000b84a8eb836754, 0x0004b4a4eb732ec8, 0x000d2645c301b385, 0x000de24926c943bb, 0x0000f1ce27e0f66e }, { 0x0004a527b9540d18, 0x00054d32499dd5f3, 0x000dda2b269e8c86, 0x00023459d12053d7, 0x00005b6b44f0bd90 }},
        {{ 0x000c957990ece73d, 0x0000df09cec46eaa, 0x0000b8f62236fab3, 0x000c7f275b15347a, 0x0000c5ac243cde48 }, { 0x000f0546734a7a95, 0x00048a3c33a655d3, 0x000e494c042fdbef, 0x000d948415d503c2, 0x0000ebeb1661c3ec }},
        {{ 0x000222b6eb506da1, 0x000b577496125bd4, 0x0007f55249816eaa, 0x00095d799fb0023c, 0x0000430859f143e9 }, { 0x000dfb4f419b49b7, 0x000a7767866d25a6, 0x0007da3c5b2da294, 0x000a816b8b6eb04f, 0x00007d8063c0d891 }},
        {{ 0x000b45f1f50f92af, 0x0008fa448ee1f3a5, 0x0006aa5cbde06eff, 0x000579a489604d9f, 0x000024726a073e31 }, { 0x00065f33c2e91ce3, 0x000a0764acc757ed, 0x00078d775c347692, 0x000f15d4ba7cf9ab, 0x0000a19d93dea6f5 }},
        {{ 0x00095867f0c050ab, 0x0003315a377b14ed, 0x000ab4e18e1c0315, 0x00095d6486bf5463, 0x00009848fb503c6c }, { 0x000d4e25b124b432, 0x000d044ef5ab0172, 0x00011d673761aee5, 0x000af5f690bd2d8c, 0x00003bc0b64a021c }},
        {{ 0x000ab18bf1911b77, 0x0003cb5e224c5d71, 0x000c4dafdd27e4f1, 0x000f8ae52fa3baf8, 0x0000e716268c21fb }, { 0x000e9459a90edef4, 0x000013e9bdf25ddf, 0x0002ccd877ab4708, 0x000099c8004cc0bf, 0x00006969a689a087 }},
        {{ 0x000150ff364d5f18, 0x00071cacce9bb324, 0x0001b68d834f5fce, 0x000f2a54bddea65e, 0x0000dd6efba4ff65 }, { 0x000d6aaf6a6490d4, 0x000f401345f4e475, 0x0000c2cf77022252, 0x000ca4b133bf21f7, 0x0000818207356a20 }},
        {{ 0x000e990b48335965, 0x000ade1813d28f11, 0x000d5c3e99daeab5, 0x000eb3c7b6000e11, 0x000048dc9938a793 }, { 0x000bccfa65f60763, 0x00077ee93e976d58, 0x0000ba9412fd72f6, 0x000367788696241d, 0x0000ac1c3f505d93 }},
        {{ 0x0007de5f15b7c4ec, 0x00039b4c5cd14112, 0x0002d757f99bfdcf, 0x000bfe393701b404, 0x00007673cbf8b48f }, { 0x000d5487494f9df8, 0x000d7cef8d53f19f, 0x0000a6b5ebc3fb1b, 0x00050128addb1031, 0x0000e5c1f0ad87ff }},
        {{ 0x00098a932a972998, 0x000cb16a8123ef0f, 0x0003ee6ec10db2ac, 0x00074e05473c32db, 0x000030cbcdd6de76 }, { 0x0004ed4bc54db179, 0x000c3f60c6aa20a6, 0x0004103a4fed7821, 0x00098fa079a32ba1, 0x00008266f6300629 }},
        {{ 0x0002377007b82c41, 0x000d906d4caed0d9, 0x000684e5b2ac09e2, 0x0000627d7b802140, 0x00002dfa1b9afd19 }, { 0x000deb5a5140a60f, 0x000bcc371f21a855, 0x000a9a380f69015e, 0x0002249f918461b7, 0x0000baa37b31dfff }},
        {{ 0x00024176970787f9, 0x000a446d5eea89e7, 0x0003960208b4ce0d, 0x000762c4ffdff842, 0x0000b88756f44d35 }, { 0x000fc0da3195c361, 0x00027b9102cae3e6, 0x000d81d583cdc7bf, 0x000b577472d143e3, 0x00006226cc2a450a }},
        {{ 0x00019f54522deaac, 0x0003edfe45083a2c, 0x00079cbbf77b74d0, 0x00060e27f66bc721, 0x0000bc98cdbbf0a2 }, { 0x000fb17de31b650e, 0x0001e495b7eeed57, 0x000cfaf861be44b3, 0x000d376add9954ec, 0x0000243fb73909ab }},
        {{ 0x0007b8aeb5cd1941, 0x00023c5abb020e8c, 0x0003520c40119c32, 0x0002e52e53d18c18, 0x00009ba1a706394f }, { 0x000fe115bb0b6f18, 0x000b243f0f8c16bb, 0x000c16989d3b2f32, 0x00032edc9224ed5e, 0x00004558e4f0f44a }},
        {{ 0x0001277b7939f433, 0x000d5493927f4fec, 0x0003b4732c4c5df1, 0x000f4d751f91fbf9, 0x00009c1127aef031 }, { 0x000ec971f4c96c45, 0x000206ee7a759524, 0x000043c434006fef, 0x00038004f00df7de, 0x0000b58465048083 }},
        {{ 0x0000d60da1b917d4, 0x000f107e75d9ccc5, 0x0009ecf2f5648cab, 0x0009c77dc63b7be8, 0x000043a02262c4a4 }, { 0x00017f5de960c25a, 0x0000bee08df1b120, 0x00097e712bf60ed9, 0x00025a73d3873487, 0x0000b1fb76167cdf }},
        {{ 0x0005d84aa3afdbbf, 0x000b966e8e575672, 0x000ed4249074932b, 0x000372274c9f781c, 0x0000cdc2169236eb }, { 0x0007c4b46d4fac70, 0x0002e094d0c0ce2b, 0x000adafa0737f634, 0x000c614c71564008, 0x0000178fd068c458 }},
        {{ 0x0002aabc59c94f12, 0x00095b397acf4ec2, 0x00003bfdf9e674d3, 0x0004780850426c40, 0x0000ca04d6eb4c85 }, { 0x0006908d03fd22f0, 0x000814c5f5942bc4, 0x000db3a0f5e9bc06, 0x0001e62ba9a6168b, 0x00006bd8a9096113 }},
    },
    {
        /* digit=11 (1,2,..,64)*(2^{77})*G */
        {{ 0x00009d21104a01b4, 0x00024745a1251fb6, 0x0008668a04b75ff4, 0x000fe349e967747a, 0x00006e02c0c6d4e7 }, { 0x000084a333d89913, 0x00059a82fe122271, 0x0005f3a1a369f264, 0x000fa053caf9b5b9, 0x000022f6b350187d }},
        {{ 0x00048ff65b341317, 0x0008233aed1d1f71, 0x00033bc63a356251, 0x0001b22cf566eaff, 0x00007f53d3789579 }, { 0x00086fa303582c2a, 0x0005fd4cf94cf0dc, 0x0003899cb5afac02, 0x00035800d319bec8, 0x0000afb9e2fd22fb }},
        {{ 0x000f31d817bed07d, 0x00055ab5c8b06d5c, 0x00022f1cd42f3db3, 0x0009efa79b1a785a, 0x00001ca931baa4c5 }, { 0x000b66d8a740682e, 0x000d06a15d7dc85d, 0x0003d89a928be329, 0x000b486c72f132b0, 0x0000478e55d53bd7 }},
        {{ 0x000e23139482dcb9, 0x0001ee1adc1696fc, 0x00023caa5f3be88a, 0x0003ebb3598ebe59, 0x0000801aabeabc49 }, { 0x000560a9c8a8fda0, 0x00082d4db067df91, 0x0005b59ef1377a4c, 0x0000329c198dda09, 0x00005daf596bd7b8 }},
        {{ 0x000029268430e2db, 0x000923e99e0925f5, 0x000612f910d69e1d, 0x000881aba2dd74a9, 0x000075a644f8e684 }, { 0x000b66a21626e638, 0x00014b41aeaffa32, 0x000f702ee5abbbdd, 0x000eafd0c8b17203, 0x00001a68d527f1c3 }},
        {{ 0x00089de4e0f889d4, 0x0005e5efe51bf748, 0x000dd5a81bb30fc6, 0x000505b259c4f579, 0x000013cc73a8497e }, { 0x00057568b82d7835, 0x000b706646083ce7, 0x00054a6bbebb3b88, 0x000bc0c10be2a033, 0x0000d865d4a67d97 }},
        {{ 0x000e11a1d8d399bf, 0x000d87900f7e6c85, 0x000eb19b3117a1bb, 0x00064247538e8b94, 0x0000d5742a10d630 }, { 0x0007b6a724c5c32f, 0x000812ec1e2f2ef2, 0x000b9e4effa0ede3, 0x000d6dc2d193d2bb, 0x00006ad06c86c96d }},
        {{ 0x00064232b86345c6, 0x000491cf1c367cab, 0x000b12cd89c580bf, 0x0005b4a329bc85a9, 0x000061507757ea7e }, { 0x0007fbd8ddb9611c, 0x00062b0167dad297, 0x00011cbbb1d53bf6, 0x000c8188b1604f30, 0x00001f4d0fd7d22e }},
        {{ 0x000307a8c0590774, 0x0009aa5b964e39a7, 0x000c5dcc1eae81ad, 0x000c9a564d4c29e3, 0x0000ef66dab1bbfd }, { 0x000139e05cf3cde4, 0x000ddaa91708c3b5, 0x000bd7852e8e71ed, 0x000d569b7a151e62, 0x0000050f4937247b }},
        {{ 0x0001e893a8e9fd20, 0x00031b494a7cd2e1, 0x0000f4b09fc385f9, 0x0009d696a5671ffd, 0x00009847745f23ad }, { 0x00023ac856040d5f, 0x000a7a165cf7b104, 0x000882d2bc2f947b, 0x0002854de81b6717, 0x0000ef3de081df86 }},
        {{ 0x000b2c9190e12088, 0x0008587dac50b74b, 0x000e9d3ce1bbce37, 0x0002da0fb28b2b89, 0x0000cabf97afa5a6 }, { 0x000f9f3f579f2fc2, 0x000a71884a9d5b7c, 0x0004aed7c41c7a6f, 0x000092de27ebe232, 0x000029c3d363af68 }},
        {{ 0x000aec5d6d717771, 0x000f8e1096187086, 0x000456979c440027, 0x000f2b0afcc9e836, 0x00008cf2ab4e27c4 }, { 0x00007dfe122baae8, 0x0000f8a994f04e95, 0x0007d694479ec283, 0x00039574fb39ccae, 0x0000aa93a28e976c }},
        {{ 0x000a7d33d6e39217, 0x0006acb489b03e9c, 0x000aa446e1236c2d, 0x0008923e5db86ec5, 0x0000d513e9282311 }, { 0x000832e75dca4e42, 0x000ed5c0ae6c4d0e, 0x000d8b63ae0adf74, 0x00059047ea9ca1d1, 0x0000237a1f96d350 }},
        {{ 0x000170458c664431, 0x00049ea5232123db, 0x000bb43e414ecb07, 0x0000237ed08307c8, 0x0000c4bbbb01d044 }, { 0x000c7f7317b92a8f, 0x000afc6c19bb9866, 0x0006418391d59403, 0x000750822c5f3484, 0x000021de21b188b3 }},
        {{ 0x00068d834bef31c4, 0x000d00e9d6d2e8b4, 0x00010e11f94b7363, 0x0003f7485d318b8c, 0x000049cae972b8a8 }, { 0x000b12c61b315467, 0x0007eb2bfba796e0, 0x000b625f2afba6cb, 0x000620753152364d, 0x0000ade1ce1d3e84 }},
        {{ 0x000eb572d99661c4, 0x0008402fcf0a7fd8, 0x0008c5d33d449b8f, 0x000f0f09842fd078, 0x0000da7983de9402 }, { 0x00076f6bdb01f785, 0x0001f4c4d194a88b, 0x00090292128bdeab, 0x0009129fcd2b63fd, 0x0000ebc377dfedf6 }},
        {{ 0x000ba4521833ac99, 0x000984bb46c2397b, 0x0005a248d600e4d2, 0x000c9b434004cf56, 0x00002121f29dec04 }, { 0x00078cd8c6df3a54, 0x000f1711171121e1, 0x0001d03cddaa270c, 0x000d52107ad7b7eb, 0x000098a27d377993 }},
        {{ 0x0008ca59bbd2a43f, 0x000a201b2a672385, 0x000756d1bb36ec67, 0x000c1e1753556d36, 0x0000c58a255d75d2 }, { 0x000aed0cd5454528, 0x0002b92a569a73b1, 0x00083babe9f4067f, 0x00068af55517a52e, 0x000059afbc58d069 }},
        {{ 0x000c3323e86fa8e0, 0x0001f8480af13722, 0x0003c58739374a0c, 0x00050ad31244c311, 0x0000126dd368d686 }, { 0x000979d546fc2f7c, 0x0002f4a5594b1dd8, 0x000b453aa4371b74, 0x00058ee19a1928bf, 0x0000915af098fee8 }},
        {{ 0x00033084d9c7276b, 0x000bbbc19f9e9676, 0x000285cb3a74d9cf, 0x000491240d9d0315, 0x0000ba33cddb43f0 }, { 0x000d408f6f73b28c, 0x00010dc55667eaf7, 0x000ab10446007a1b, 0x000e39f1473b7f92, 0x00002ef1a03646da }},
        {{ 0x000103f6bba1307c, 0x000d1dc164a89bb6, 0x0001ddefc64a2b32, 0x0005d70bb74a42bb, 0x00005152ad60f6a1 }, { 0x000e4781a0271b2b, 0x000ca81355a5752f, 0x0007024ce4d4d310, 0x00084f73b562f96a, 0x0000c477204acd46 }},
        {{ 0x0000757b5f4712e4, 0x0004c75f8dfcf8a2, 0x000a74f11cc657d7, 0x0006a5bd8a284f0a, 0x00007dda9a0af13f }, { 0x0002a3e7718f7a6e, 0x000d22093b8a5ded, 0x0009191c86356347, 0x000f92e33d3f9335, 0x000055ce4eafcd22 }},
        {{ 0x000c17e23aa997b3, 0x00096f4cd941534e, 0x000af7af39d3fdd7, 0x0000044e30d37406, 0x000017d4ad645c92 }, { 0x00052737e78e08ec, 0x00019516f496ba27, 0x0001c18619005520, 0x000df03f56a69317, 0x0000b6ddc6edbaf5 }},
        {{ 0x0009a1904af269a5, 0x000ee2c68af43eec, 0x000da69b7072b08d, 0x00046bcf55502468, 0x0000f80ea4ac443d }, { 0x000a17081f1c5679, 0x000da8d470c56a4f, 0x000e1f2684ceb758, 0x000ba15f3159abc5, 0x0000964c76954003 }},
        {{ 0x0008102033e8adcd, 0x0002e2b10c8c0f59, 0x0009a854e5133f03, 0x00015f2a72102c3e, 0x0000c9a9e311aba2 }, { 0x0007a89342969c71, 0x000f33ff1d18a13b, 0x00012053c7add2ec, 0x0004e3ed835eebd9, 0x0000d9e680e72864 }},
        {{ 0x000ce7c67c39f8eb, 0x000838decf2a73a0, 0x0007816d5317f027, 0x0009468a2066a639, 0x000026677b2d0e7f }, { 0x0005a428ef86e036, 0x0009071759a113ce, 0x000269cb3f0ac1ee, 0x000563c7be3b373d, 0x00006ff591e18a76 }},
        {{ 0x000b6b40b404a661, 0x0007ac81c86682bd, 0x0006dd64ea76b3ad, 0x00015d19ab78d272, 0x0000fbd431bc398a }, { 0x000802739262b87e, 0x000297db8c4961f6, 0x000bb8e36d198176, 0x00030325eb1a12f7, 0x0000129ba25ce217 }},
        {{ 0x000c873312835882, 0x000685f16b1bd5e5, 0x000ff15bd7c0e8a6, 0x000dcb99ef957e41, 0x0000fdf280b53be6 }, { 0x000093ef1a573f41, 0x000b88cb65f57cb1, 0x00014d78ee26f748, 0x00048d340db60a62, 0x0000c4ae6cba4b4f }},
        {{ 0x0005f9a99adf6ad0, 0x000dfb53f4ece64e, 0x000d900aab7fe12a, 0x000bbe8cd6a6bb02, 0x0000244f4571b32b }, { 0x000cb9c4a0dadc5c, 0x000682511a7de841, 0x0005e0eaa53ce5b2, 0x0009034a28356809, 0x0000d2319e930091 }},
        {{ 0x00070b941f431f96, 0x00035babb2eec644, 0x000492f8199f509d, 0x000c8d4c4a0c23b6, 0x0000055b8ea1eb9d }, { 0x00073291adbc3b63, 0x00073799df435742, 0x0008e83346629e9d, 0x000445a814ccfb8a, 0x0000b7b0d75235ee }},
        {{ 0x0005efcdfd123b2e, 0x000866b42a92806b, 0x000bf27ae64de389, 0x000d09bf8810684e, 0x000050a1dcc40eb7 }, { 0x00050352364d1ab7, 0x000d2076e9edd128, 0x00004fd440c2c9e1, 0x000e85a6de29d0b5, 0x0000da6c7734e64d }},
        {{ 0x000943b9b2fa6dc9, 0x000e0b2b4b4b67c8, 0x00005ebe4c512574, 0x000128fd8c1d10df, 0x00006e0e8ebb49c2 }, { 0x000b34a5656dfc03, 0x000d6143b3def048, 0x0001c1b2d09a22f2, 0x000c08d4085ab3aa, 0x000060432732650d }},
        {{ 0x0009efab06795cba, 0x0000f4a10358560e, 0x0001b46c7da3c8a3, 0x0006b31152d8a507, 0x0000637882aff5d3 }, { 0x00020630df6bfbd5, 0x00072a9d19e9b915, 0x0004b06fe46d6faf, 0x00003090d8cfed74, 0x0000ce2f38fbcec1 }},
        {{ 0x0004f8e4921219b0, 0x000ef7ce382360f1, 0x000973c673c5135b, 0x0003f246cb6db05c, 0x00000c2638a18b5a }, { 0x000d76ff4b38f5e9, 0x000897a11c203407, 0x000f3205cc6b7d94, 0x000eab1200386f86, 0x000080180ab081c5 }},
        {{ 0x00082444682aa7c3, 0x00028e7db15be7ac, 0x0004107abe9a7d16, 0x00071706467a0805, 0x000025b9428d4196 }, { 0x0007714718a7a5d9, 0x000b58b20d841b88, 0x000854475e368b17, 0x0000c2936127a054, 0x000012a411de8dbc }},
        {{ 0x000cc20bdfcef420, 0x000aaaaaa8a5288b, 0x0009f26195e97eb7, 0x0002dee0c91cc0c8, 0x00002946f9408a70 }, { 0x0007409503d2975e, 0x000e4daa20ede1b5, 0x000c1d5f70119b97, 0x000b42e4bc441f00, 0x0000a83a9d6eb2a0 }},
        {{ 0x0005f1068db22608, 0x0004a61e4585d453, 0x000d3143dfb4ed4d, 0x0004ae66c9231fae, 0x00004c8d8da2e90c }, { 0x00009002a8263ee7, 0x0001200a2d261155, 0x00029e7afc3b45b0, 0x000e539a1649406f, 0x00005414f28a6295 }},
        {{ 0x00027abb0cc2f99c, 0x0009aa53ff3571bf, 0x000790fe6cb4f16e, 0x0003a2b1624f98b7, 0x0000a232d3065d82 }, { 0x00072b0da96f4899, 0x000cbfc5c6d5bfd7, 0x000179a26cce6886, 0x000505c346915c75, 0x000008d2131b6918 }},
        {{ 0x00071a4c560e39dd, 0x000490c2e03fa69a, 0x000994c7571e040a, 0x000ccdc6036cf0e3, 0x00004bd59ba6dde0 }, { 0x0005e78c885a065c, 0x000b65133c673db2, 0x00004e0134a9208d, 0x000f06b34c0bea51, 0x00008995c776117a }},
        {{ 0x000d702cfbe0903e, 0x0006ef88e2f7d902, 0x00074a4e05d17b88, 0x0005b20e9fdbf33d, 0x000010bd286ddbf2 }, { 0x000dd0fd8fb34fd8, 0x00013db85c88ce89, 0x000b8030d8880391, 0x000235bac23a3ce3, 0x0000fa45c03bfea3 }},
        {{ 0x00047e2ad1ea2431, 0x0001602b57872b88, 0x000ee0ff350f3dd3, 0x000ab0c402ef26b4, 0x00004bfae875386d }, { 0x000c3fc966771b4f, 0x0008aa009996a74c, 0x000c5de647f65d59, 0x00053fe4316a56ac, 0x00001cad819b5d5a }},
        {{ 0x000c2f5e8ebb9875, 0x0006573ce07d1b0c, 0x00018c7de2e97cd3, 0x0008da2f4d82910c, 0x0000fe01b8994111 }, { 0x00085928bcf47e0c, 0x0001950b8c30ccc0, 0x000a12352e9d8df2, 0x0002d88597b856ae, 0x00004ba7a71332bd }},
        {{ 0x0001d44d0c939385, 0x0007decba8f4b4d6, 0x00099ee0887ec970, 0x0008ef97ac07cd4b, 0x00007b8f4d37c895 }, { 0x000aa760774d89e6, 0x000f62b135e6f4d6, 0x000b7cecb07896fc, 0x00076a2b5680962e, 0x0000fd88fe58cca3 }},
        {{ 0x0008eb579c89ba85, 0x000b8118fb98871c, 0x0002f2dc1362f68b, 0x00025b114cb26f5c, 0x00008a957abc4172 }, { 0x000dfba9354cbe2e, 0x0001e15c93bae41a, 0x000c8cf0fa7d7f42, 0x000606cb146fbae5, 0x0000f605d4ba99c2 }},
        {{ 0x000697b7814fba90, 0x0004013f216c980b, 0x0008410654403924, 0x000dd1237c8ac4fb, 0x0000683bbe6245ab }, { 0x00033aad090c72fd, 0x000c939ddb945b56, 0x00077d5a16a024cf, 0x000b36f90447b06c, 0x00004f7703863965 }},
        {{ 0x000d40e0825f1a07, 0x0008c6bb57c64779, 0x00044c0cdc444452, 0x000779f7b871f8b6, 0x000076eaaa21e046 }, { 0x00087b30ac366f12, 0x000d8f6f2d84b5af, 0x0008acea8597faa2, 0x00014068b228993f, 0x0000424abfd6e5f6 }},
        {{ 0x000ba6b8ef40499b, 0x000f0317642bca8e, 0x000e66d95628781b, 0x00075f9451a032d7, 0x00004159c0acf639 }, { 0x000ec80051301541, 0x0005badd44856928, 0x000dfabee96d2c76, 0x000f8734884bda40, 0x0000d972ba5752fe }},
        {{ 0x000be6868dc83e03, 0x0005e0798fa7aaac, 0x0004248d23c0ea74, 0x00098ae9b41209ee, 0x00003cedae7fcc2a }, { 0x0000a5ccc7b9ebec, 0x0005fdc32c04dd3e, 0x000def4916cf5e80, 0x0009ad4cfa6c35fa, 0x0000a98de7d24b36 }},
        {{ 0x000b2d7405bd794a, 0x0002eaa1536189f8, 0x000f2f40816c05a9, 0x0001b0c0e1126c55, 0x0000e641bd7bfcf4 }, { 0x0001863ad3cc6a83, 0x000109e3738be712, 0x0009f8c75cd5b817, 0x000f12615b5090b1, 0x00007dbc50add023 }},
        {{ 0x0003c1abe2e2289b, 0x0008b79a7aae5d37, 0x000c9a2415e1f025, 0x000dd1b297076013, 0x00009d9bb7e7d7d8 }, { 0x00022100092b6697, 0x0001d22f2488af76, 0x0002041aa56cc46e, 0x0003ffd57e4105d0, 0x00004447fba18228 }},
        {{ 0x0000d8ea16b68aca, 0x0008a7877a50ec69, 0x0006795ebd16449a, 0x00056c47abf0392f, 0x0000c5c659818595 }, { 0x00060dbb7c6ed91f, 0x0008393ef953f575, 0x000567b78ade06a9, 0x00077e81790a64d0, 0x00009be345281b73 }},
        {{ 0x000d3b3730c3da90, 0x000a2864b66abfb6, 0x0004ebc7d1cf59b3, 0x000aa041ad26f52e, 0x00008875834f1291 }, { 0x0008f625c182ae12, 0x000942f4789dba54, 0x000529bfa3f61d44, 0x000e1880e91fcb81, 0x0000772dfeb7bbe9 }},
        {{ 0x0006ee1cad7e490c, 0x000dc011c6219b80, 0x000921be12763c71, 0x00098f988e7a562d, 0x0000ff590f7684a8 }, { 0x0009c6ab4f30f5e0, 0x000ce43faff79f72, 0x00023b683e91a933, 0x000696bd0c141a41, 0x0000472541b41fe8 }},
        {{ 0x0009a4a020384d2c, 0x000e87730448112c, 0x00011afee636432e, 0x000e4241082b51af, 0x0000f39a9a8e69f6 }, { 0x000d49d0784959b5, 0x000e0d8257e5818e, 0x000fc707ca9eace4, 0x0004f5bbfdd8efa0, 0x00002149995867c4 }},
        {{ 0x0002daa6890484d1, 0x000db5eb7caee1e4, 0x000d651a33cea7b5, 0x000e198b91033898, 0x00002903be868557 }, { 0x000107f8ce029e4d, 0x000745c1a1d885a8, 0x000073af6f495ffe, 0x000f1422825572ec, 0x00005e04a9386799 }},
        {{ 0x000fd321053b8bde, 0x00015a36fa6110cc, 0x00012bc0351a15e4, 0x000cb742b74fb1eb, 0x0000cad7c54d4248 }, { 0x000d56251a7ec868, 0x0001ca2cc80ad571, 0x00056953911caea0, 0x0002c80cd7c27fc3, 0x00000a95d3fdd32f }},
        {{ 0x0002503375e361ed, 0x000e509c955bef4d, 0x0001345d7cd1281e, 0x0006562eedf02cad, 0x0000ba01e9851b7b }, { 0x000be946c753683b, 0x0003a5edf53cb366, 0x0001c77f0e544309, 0x00037eaa454dcb61, 0x00000c4e558ee575 }},
        {{ 0x000541c7c1655fac, 0x0009ad1a995e0941, 0x0006cfdc3d170c39, 0x000be851365e848f, 0x0000c5364d173f04 }, { 0x0000fc20ed612d22, 0x000a7faf4acfade6, 0x0006e3d0ea83ec7d, 0x00024ab770e58589, 0x00005a5480d97ba9 }},
        {{ 0x000018f5e5fc5e81, 0x00083eca42768678, 0x000bb4da53ad5c55, 0x000d864bd46f6c3c, 0x00002cc3c22b21c9 }, { 0x0005acc0361b36ae, 0x0007470e47b668f5, 0x000d986831cd85e9, 0x000c8ed4dcef48cc, 0x000012e983797847 }},
        {{ 0x000dbf24eccee72c, 0x000e6af64db44442, 0x0004910eaf9370ff, 0x000496b8a19e9d63, 0x00002081b3c75ada }, { 0x00086167eacf78d7, 0x0000c8c2cd706b27, 0x000abd2eafce6c92, 0x000681eae5b012a5, 0x00008d905b32a1a3 }},
        {{ 0x0009ad7d324559d4, 0x000b57d594881dc3, 0x0008960aa38fa09f, 0x000f644de6b5efad, 0x000000bdc3e7618f }, { 0x000a309b4ea8cce9, 0x00039e02cfc0e816, 0x00018ca8025c0c84, 0x000981109117c466, 0x00002b74c94f9322 }},
        {{ 0x000b6614f982ece8, 0x000b7b7abea49b1e, 0x000c257cb1cc77aa, 0x000c2f67ab4b1d27, 0x00005154e895caa0 }, { 0x000ea5d68b9993ba, 0x000c76b7f6f7459c, 0x000e3a9037efb0c9, 0x00075d74ce431a44, 0x000063571583a729 }},
        {{ 0x000102c8a0fb0bfe, 0x000f8653a0a3f957, 0x00060be8d9da3b5c, 0x0006691e98936941, 0x00000313a1550ce9 }, { 0x00067e106b240330, 0x000010e235d21c8d, 0x0000e87d6f10977d, 0x000925a67505e55d, 0x000059ee2e27a5ee }},
        {{ 0x0003858b5c81bd3f, 0x000d2a9431dd80e8, 0x0009efaae17d3682, 0x000f67830eb7c7cc, 0x00002f1f3290cd4f }, { 0x000990a639034a2c, 0x000a593b6251d5c7, 0x000299c23319bc02, 0x000b194511cca1fd, 0x00006f501add6b8d }},
    },
    {
        /* digit=12 (1,2,..,64)*(2^{84})*G */
        {{ 0x0006beb87707b7a2, 0x000c72a87dec0e16, 0x000d90f4e489ddf2, 0x00017feb5010ded8, 0x00009f1c146a514d }, { 0x0002ae989277487f, 0x00076313476cd0e8, 0x00052ddea0b6d98f, 0x000ff20c6a63d0e6, 0x0000d40ea3d516db }},
        {{ 0x00019c667d0d91d1, 0x000e8105ca7d8669, 0x0001a93ed4b79dc6, 0x00058efbc582967e, 0x00007205a3aecabf }, { 0x000187f1f85aef05, 0x00012b160f5dcd7b, 0x000f2c42bbc43ffd, 0x0004562f5ec697b9, 0x000026b5000648eb }},
        {{ 0x0000d41a77c52336, 0x000441d214aeb181, 0x000c6340187fcbdb, 0x0006e6ac41506af3, 0x00003b6fa4818220 }, { 0x0006bdb65cf1fb29, 0x0003ce4a84bde96b, 0x00083b4cb3bfaea2, 0x0008473e742f060d, 0x0000fba067aea100 }},
        {{ 0x000b6c2d46254ea3, 0x00039b6ec7fae9f1, 0x0004d44a4114c60f, 0x000f5ba52995f271, 0x000066e8cbd34843 }, { 0x00062c42a011d210, 0x0000c318129d7161, 0x0000f32c7f0a2090, 0x000229f63b03909f, 0x00009687ec5c5909 }},
        {{ 0x000507db0a04df74, 0x000af43753b9371c, 0x00099a17c1cd2a88, 0x00066679629cab45, 0x0000a296edbca1ad }, { 0x000519b397e39c16, 0x000e052af036c326, 0x00079fe7dac46a92, 0x000efcd5086f0cc7, 0x0000bf3f8cd63cc7 }},
        {{ 0x00042e43a80c6fad, 0x000b1ef9c053df72, 0x00078ed2a6c7dc5b, 0x000da22fb8de25b3, 0x000063c34563eabb }, { 0x00066648e3f185ac, 0x000a5f4dd6f958ec, 0x000f2dde11a9f374, 0x00087dd496925a77, 0x00007412068d6cf7 }},
        {{ 0x00005e399e662c0f, 0x0000a57173e460c5, 0x0004e0120bf24c7f, 0x000f062621bbbce7, 0x0000fbd676e31f74 }, { 0x000bef99ec94a32a, 0x00023cb57797ab7b, 0x0009ae3d0efbd3a0, 0x000900cc160ad35b, 0x00000124b141f449 }},
        {{ 0x0007c8bdf49d7f19, 0x0007df31711ebec8, 0x000f46d03fcfcebc, 0x00035281f2da40f1, 0x0000aacf4dcfdeba }, { 0x0004907c5d800621, 0x00068e3c2eef12d5, 0x000ae7e3f5965a34, 0x0000ca494de95bb9, 0x0000c88b84c6fe58 }},
        {{ 0x0000c1e7da0cf5f8, 0x000cdf08db60e357, 0x000689c14b3f9c46, 0x000208bc9743e3cd, 0x00005819e16c67eb }, { 0x000b4a07ea1a52ab, 0x0009f39a6be09070, 0x00057b86afe8f489, 0x000d94f6036d4703, 0x0000fe9b7f7d190c }},
        {{ 0x0003057bc03a43d5, 0x000782945c1dd539, 0x000790186ff52f5a, 0x0003b2be0cadc589, 0x0000ce868856076d }, { 0x0009d9be5f0800fd, 0x00081cd027a4cd18, 0x00033c3a42bb278a, 0x000667f68114734b, 0x0000ae65c75c585f }},
        {{ 0x000361bc67e9e34d, 0x0007e2e089808ef0, 0x000dd23c849fe842, 0x0002e6cc9f1f7a27, 0x0000c13a3157f998 }, { 0x0001b02bc1e235bd, 0x000a00b7f071426a, 0x00004556f9a54850, 0x000360dbd6a3c181, 0x00007d7ec15b5c2b }},
        {{ 0x000f7d41e8e79665, 0x000895931fba2393, 0x000359404fe8f948, 0x00083d71df47424d, 0x000023b7a1aa0b1f }, { 0x000ef6eea564a053, 0x000d0c2cf6eb9980, 0x000f51a339da0be5, 0x00028b46286bb49a, 0x0000f0e9d516a3ff }},
        {{ 0x000b44589e394e0d, 0x000207236f5defe7, 0x000bdbb538d8261f, 0x0003ed3c7fa9922d, 0x0000fc2a961330be }, { 0x0009acb43064282b, 0x00010ca9ea837700, 0x000f210a61177cf3, 0x000f5b1c0d7fa2c7, 0x00003784b94dc9c5 }},
        {{ 0x000028746afacb37, 0x000132e649d4e574, 0x000fd8eede035012, 0x000e913c4cd53a2a, 0x0000887fc8e5c525 }, { 0x000c3ac1b0c48f95, 0x000ee9064291d4c7, 0x0001722017588ca1, 0x00088c181bf48f15, 0x0000f3153a52a8b6 }},
        {{ 0x0008e6567629384f, 0x0002ec3c093c171c, 0x00035c0dd4ef3c20, 0x0002147cbae7acb2, 0x000090ba8583656c }, { 0x0009e2f0802b74dd, 0x00042ba28b8d0b1c, 0x0006ecde1a103afc, 0x000ae4d8015d7413, 0x0000a3d25fc46b0e }},
        {{ 0x000cadd1a806623b, 0x000e2b6f79588c00, 0x0009a8a99724a1aa, 0x000f2088afa52fc6, 0x0000f705025b0678 }, { 0x0009b5b7e0f923c4, 0x000e2b31803dd6fc, 0x00048c34f654baea, 0x0000a8a16488e4fa, 0x000078ce7289743f }},
        {{ 0x000bf0e7f5c451a2, 0x00030d3957626499, 0x00072bbdbc00a6ca, 0x000755696a9604ee, 0x0000c65ea7b931a9 }, { 0x00082e474cddae64, 0x00064217cb44b924, 0x0003667b7a354fe4, 0x000a641fda2f5ce6, 0x0000f36622cc64b3 }},
        {{ 0x000c9dd0ef759995, 0x000ddd0b611c24be, 0x0004740b22624bff, 0x00035cf211d468e6, 0x0000807471447a06 }, { 0x0007b4f8f0064eff, 0x000d3bf956c2e32a, 0x00043529005e550a, 0x0009f776573dc686, 0x0000b45bf82931b6 }},
        {{ 0x000d2dba29a420a1, 0x00077dd7280f8553, 0x00004e810c5f5788, 0x0001f6e42bbaac26, 0x00000e0aa6d6e58f }, { 0x000163cfe5ae3f65, 0x000b594985093755, 0x000f7d4053e708b7, 0x000c881fb8f33031, 0x0000315d7722ea99 }},
        {{ 0x00023d24bafda93d, 0x0005217cf41c6e88, 0x0009813e6da10623, 0x00029057cf1f03c2, 0x00009f3f08321169 }, { 0x000891e4e39dfe78, 0x0001813901f0b4f7, 0x00018002374a94f1, 0x00029912e684360f, 0x0000123427926e8e }},
        {{ 0x0007ced2848ab4fb, 0x0004819ac445e3d2, 0x000766d870fcc671, 0x00034ca0a553432e, 0x00004c49d4d2e4dd }, { 0x000df18da1030901, 0x000d2757a4afa571, 0x0003c69333af3c8d, 0x000b49823de0a14c, 0x0000c31b61984a35 }},
        {{ 0x00069041a0c93e91, 0x000fe82d8d38a9b9, 0x0000514dd1f04e0f, 0x00082e4dd2e97c60, 0x0000d5f81ee7afd8 }, { 0x000a8b342d40d5ba, 0x000993b76a2086b3, 0x000b9d37b4d18b27, 0x000c84167d6bba71, 0x000006b92014b8c8 }},
        {{ 0x000a7dbf4441e65d, 0x0001949522f9be32, 0x000bb4a18a63f9da, 0x0006e8f42690c075, 0x00000b984a4ca3f6 }, { 0x0000aca623b9e392, 0x0006a0e3faf40783, 0x000eae64e0090621, 0x000be25b3b2f02e9, 0x0000a47a4149ebe6 }},
        {{ 0x000e79d046d6c0d9, 0x000e60502f40d9ab, 0x00076db7ba07217b, 0x000db4d744681c46, 0x000074b09e5ed137 }, { 0x0005cee5b6dd4217, 0x000310a78bd01e0c, 0x0003acd63802d1dc, 0x0006da3b66dd359e, 0x00004f7a41696b06 }},
        {{ 0x000a2e4e3632e762, 0x000ced46ffd9fbbc, 0x00086c97c281ada3, 0x000fb00cf6e5bc69, 0x0000f75bfce3bc69 }, { 0x00075117090a5ff5, 0x0006fc016de36c5e, 0x000c5b91a11ff6bd, 0x00025b5f8e0b10fd, 0x00009f9ba307785c }},
        {{ 0x000ff3974d378756, 0x00082fc3b922bf8a, 0x000f9c3619e62f97, 0x000e743c4e4d8c09, 0x0000070f7ed0ec03 }, { 0x0002c29a21b92463, 0x00031052c18df9b0, 0x00038e3dd7174a50, 0x00055a895c5c3a29, 0x00008d20d8ec2e2a }},
        {{ 0x000b0838226e8425, 0x000ccd6141ecf611, 0x000e20a9f7315526, 0x00050ee579195509, 0x0000827cd560e1b4 }, { 0x000ee0b7e6ba0f71, 0x000fd35e07289f09, 0x000bcef7b719136d, 0x000002bfa1f94c48, 0x000063e3e9d89c31 }},
        {{ 0x000fbe8ff0b58802, 0x000fc2e34d74e31b, 0x000f7c390ac5a4ca, 0x00076e98fa352c30, 0x0000f58dc8de69f6 }, { 0x0001022c67c9035b, 0x000c1e6a79046089, 0x000a33fa75e20d04, 0x000e287001f812f3, 0x00004fd325eb32f0 }},
        {{ 0x000f7a6bb4ac623e, 0x0005e5d30a713a1e, 0x000f9574a3df3bdb, 0x00099bb284876e3e, 0x000058865f427515 }, { 0x00067d2e36052baa, 0x000c828dfa2a659d, 0x000801fa7659b2da, 0x00031653fd5cd339, 0x000060ed537958fd }},
        {{ 0x0004a96feb09a9c1, 0x00026d7cc5f43941, 0x000c308d71570ea4, 0x000a0e50ab5551b5, 0x0000c0238a0327c9 }, { 0x0009f80382b52a70, 0x0007f740d12bb0b3, 0x000026c163fb98f7, 0x000cd031dc522f02, 0x0000b4064550e478 }},
        {{ 0x000c58e2b0841de2, 0x000810c39024a940, 0x00088ad8b1a25b9a, 0x00085a47caacb96a, 0x00007b4e4f38b169 }, { 0x0008ab30080ea35c, 0x00001634da919fac, 0x000a28b2f2812eb1, 0x000e84f10ef79b69, 0x00003e9ab4693571 }},
        {{ 0x000c047df2f3ac76, 0x000c2b67c4a658ad, 0x000864e2a38fcd7a, 0x000ec6e4fb3c4763, 0x000051531fb65393 }, { 0x0005e4fd59db390b, 0x000c9c55e59d92d3, 0x0005b30150334900, 0x000919016cedca47, 0x0000584c78dab3ac }},
        {{ 0x0006a76dd7489981, 0x0007e18e4bcf6b1a, 0x000482a8ab23b028, 0x000ca4849856d628, 0x0000dfe5317984e9 }, { 0x0000d6fb0302b210, 0x0001334cc7b4f79f, 0x0001159b13471521, 0x00027aed5a1d0312, 0x0000d3bf917b1a7f }},
        {{ 0x000739327907285e, 0x000bcd51fb747d2c, 0x000868fa43b278a8, 0x000c7515aab50f95, 0x0000376a6d9ca2f4 }, { 0x00038542d43e54fa, 0x000a8440df6f9be3, 0x000b3e6a8f41a44a, 0x000e875d99092757, 0x00005ff9309a3d10 }},
        {{ 0x0008059420aa911f, 0x000200838961a0f4, 0x000e5063f3a99a11, 0x000f51dc61426ead, 0x0000999972790d9f }, { 0x000bd6e76938fb53, 0x000e7d87396e1dea, 0x00078ceec367f734, 0x00093fd665c8bdc4, 0x000025cb654cb623 }},
        {{ 0x0008dd4397eab620, 0x00020bb23330a011, 0x0004aa716c087df4, 0x0002a0c21d23f5ee, 0x00009e0d97bf7d73 }, { 0x0006bba1a6ca60ae, 0x000e59a122d7b44f, 0x0009ac7ccf2f7140, 0x000ef35f8e6d3b40, 0x00006562071bc951 }},
        {{ 0x000eb4a728dfb0d1, 0x000ece5cfc0aee0c, 0x0005f1d7a9946ac7, 0x000b2f8b2847edc4, 0x0000f3cb9e6d73cd }, { 0x000fd7e0f19cebb2, 0x000320e26da033d0, 0x000cf4084592072e, 0x000e145256cc3889, 0x0000f9a9e6566860 }},
        {{ 0x000e87665c992681, 0x000f23b1f0df9d48, 0x0003142e7e966892, 0x00009ced48cc2c5f, 0x0000034da4569da1 }, { 0x0009723443b0241c, 0x0000decd73dc965f, 0x0004451def97585b, 0x000aa0cb415ace45, 0x00008cda22cac57b }},
        {{ 0x00036cf77e985285, 0x000e12eb72d9f264, 0x000ead77798b269b, 0x000e004d0b19396d, 0x0000b0da9787f548 }, { 0x000ed049715c4ef1, 0x00042b3e990ef77c, 0x000c4c41b2b3b279, 0x0007441ce7373e3e, 0x0000c74b8d8215b3 }},
        {{ 0x00087b9090eee8d9, 0x0001e857e977ca05, 0x000a8761cc9b0977, 0x000721b61b66ee8d, 0x00000d8c50c1bfa1 }, { 0x000f5f64b5366b2f, 0x0005b1b32ba0127e, 0x000624b6b0f733ae, 0x00003c3f2a7720e0, 0x0000f1e94afee463 }},
        {{ 0x0007d52107341dcf, 0x000da6292b8c5599, 0x000fd19ee9087108, 0x0009ff1a50c14aab, 0x0000faa98a57dc72 }, { 0x00064ed85eec4469, 0x00054303e7c253b2, 0x000a21af1562e2d2, 0x000f6422ecc834b1, 0x00005e395857864a }},
        {{ 0x0003b3e080e596b9, 0x00020c066d41870b, 0x000093d3d98894b0, 0x0003d8d35a50c20b, 0x00005c4a4156d0fe }, { 0x0006900c54967769, 0x0007c58a179e9d94, 0x00092e175b2d285a, 0x000c91d4f2daa3f7, 0x000009935a179da3 }},
        {{ 0x000e2ddd1a219abc, 0x0005b1ef76765d05, 0x0002379a97e897c4, 0x000e1ddbd4ad726d, 0x0000c1f23448c02b }, { 0x000809d5edd717e7, 0x00030fca93e6dea4, 0x000b4b6d3b35eb09, 0x000f0bca2e8a2583, 0x0000004d3d719c22 }},
        {{ 0x00066b19648d9dd3, 0x00077a3da7a095ed, 0x0000508a9178ada3, 0x0004ae54a9049acf, 0x000017300723e8de }, { 0x0006ccb81ee1f40b, 0x000b7a0de6e95555, 0x00085ef0cf517a36, 0x000de7e21cf52d09, 0x00008ba0c8b2872d }},
        {{ 0x00094348a4e6b6a4, 0x000e6b89f4ccbb8c, 0x000211dece842856, 0x000082d0ced5f44d, 0x00002f608774e31a }, { 0x000c51779a9eec3d, 0x000acc429ef8479b, 0x0002a4faf1a4db07, 0x00027fd144c13a4e, 0x00005e156474bbef }},
        {{ 0x0000f4728b20513d, 0x0003c922774856f7, 0x0004fb2782d3e590, 0x000a061166e52fb0, 0x0000b15e370c9ac4 }, { 0x0005ee9fb249de3d, 0x000dbd1b2947f224, 0x0005e22ed0071342, 0x00012adbdc827ea1, 0x0000818801cfdb43 }},
        {{ 0x000537c1f02c6851, 0x000b321bb937d636, 0x0005671375b66207, 0x000852a68df8ff2d, 0x0000496bf164c230 }, { 0x00045b0e55c95c63, 0x00007263b37f8e84, 0x000b9a90b8785c72, 0x000cc742c170e9a9, 0x0000126204cdfcbd }},
        {{ 0x0000d9a50b845667, 0x000a70683a7337bd, 0x00042f134d3dc726, 0x0004c501f1e3416d, 0x000077d800d0f3e2 }, { 0x0002ade2f283b9d7, 0x000aa506fe28ef7e, 0x00054084698d5e5f, 0x000e17b633cc5ef1, 0x000066c31b2862f2 }},
        {{ 0x000e6fbcea358249, 0x000d7fa458d36710, 0x000c1f54c790f34a, 0x00096f8e31549722, 0x0000b594e2aa6b23 }, { 0x0007b2736a31e704, 0x0001a6b3bef0262d, 0x000bb7a26d997c33, 0x000d94eecfa8c472, 0x000052cf6e513e40 }},
        {{ 0x00075c9c88b57072, 0x0006ff1716843491, 0x000e1b84ff0e5ecf, 0x0002798ff36b4722, 0x0000fd7a19679edb }, { 0x000cf60d2f9f4605, 0x00056226e6bfdb1f, 0x0001c1e53e170898, 0x0000469445ea5b7b, 0x00001577a6f96234 }},
        {{ 0x000ca2e2a9f36d95, 0x000a42d677819e1a, 0x000dfdb7473e6ddc, 0x0005aca3f7be74c7, 0x0000e0f21bb12470 }, { 0x000f16207d6ca7e8, 0x0001b88122a6fe3d, 0x0002a913e42b12c8, 0x000d30557db69f70, 0x0000ec28095ae062 }},
        {{ 0x000023dda39ee7d8, 0x00071ab63ed9998a, 0x0004075a01f290e2, 0x0004c6872a3125c4, 0x0000f904a184e285 }, { 0x0002370cab7cf923, 0x0001efc0c743032b, 0x0000ac0c798dc522, 0x00042f8074f2120c, 0x0000c5c63ec4a64b }},
        {{ 0x000b922caee9c36d, 0x0006c626cc820fb6, 0x000c4cc8181044ef, 0x000a22bf059feee5, 0x0000001293501b71 }, { 0x000509216ce29c86, 0x000bfad106bed4e9, 0x00054a98c83f1cea, 0x000260e87546e71f, 0x00009ce0ef82db3c }},
        {{ 0x0009536e2465169d, 0x000936785c9a2dbc, 0x0001a294f24b62b3, 0x00035c81dd6ae515, 0x0000b9a59ef9c27c }, { 0x0000a8075f1088c7, 0x0006ea934521b032, 0x000c66df4fa0987c, 0x00065293cf39c526, 0x0000932812328d7a }},
        {{ 0x0005b9502ede5c00, 0x0009c62640264225, 0x000fcf00804813d3, 0x0009329508dd1077, 0x00003a1cef23fb5d }, { 0x0007fcafd4d1e2ef, 0x00025b486a098170, 0x000c3fbe909cd3bb, 0x000fd6a2fa39dd3a, 0x0000301ef9be9169 }},
        {{ 0x0000d74541238bb1, 0x000a1ee346eb2263, 0x0005bb58677672aa, 0x0008f2c267dfab08, 0x00008519f05e7745 }, { 0x000b740ffbc988fa, 0x000ed3ef2c2d065e, 0x000fe8badacd3c5f, 0x00097a322ff9b89f, 0x0000270de0bc09cd }},
        {{ 0x00021b2cceb112c7, 0x0003213a8aef5181, 0x000bedd9a2d4160d, 0x000a099487bfc74b, 0x0000bfb60e023e05 }, { 0x0007a59316c50a89, 0x0003f36a01eb3d34, 0x000a17d37464fae9, 0x000fa32cf50eb1c6, 0x00003421f128bfda }},
        {{ 0x000c2da0c5ea518b, 0x000897f931020167, 0x000418a384b07616, 0x0006ae09112c5f65, 0x00005b5dd45be1df }, { 0x000c2e94a72326bd, 0x00088b6d4a1dd5d1, 0x000674f41c95d1d6, 0x000cebd483e7ad9b, 0x0000363b24c971c3 }},
        {{ 0x000e5d95c2d82265, 0x0002a075a97d231b, 0x00068f25f195c1c4, 0x00064c2277c80de9, 0x0000d8d6a5c50d2d }, { 0x000e26d6207fa264, 0x000f8b19c2492eed, 0x000062269993127a, 0x0005724d342192dc, 0x0000a4ee006ab0b3 }},
        {{ 0x0005cdcbbff59de0, 0x000fcb6137c7408e, 0x000550d49b1acec8, 0x00020df1b9a38d7b, 0x0000a28ec5df468e }, { 0x000827dcdbc769cd, 0x0006a6df65965a08, 0x0002638c0c826940, 0x00084a22060fe88c, 0x00007b733efb859f }},
        {{ 0x00066d1dfef7b4aa, 0x000f3240c56c6904, 0x000d974a461ec7d5, 0x000b6bc2b40265da, 0x00009a037291c042 }, { 0x000effe346803700, 0x00069621484469b5, 0x0008a3dfac592a2a, 0x00081bf510db6054, 0x00002ee4b2e6bbbc }},
        {{ 0x000c3f2c5b8478f1, 0x0006164ff112c32f, 0x000c7f57f357647d, 0x000fc170d8a9c736, 0x0000f62d2b5914f6 }, { 0x000853ac70c175fe, 0x000797e5aeb49c20, 0x000dd1c7318d5bf6, 0x000bfb1fcf3db034, 0x00006e90923fe993 }},
        {{ 0x0008ef1d90bd82e4, 0x000f1948b2a29d5d, 0x000674f0f04a7a60, 0x000a94e8b28f1f45, 0x0000cb6517882536 }, { 0x000ed48d66270e5a, 0x000086041dc61bde, 0x000c3919a2246fed, 0x00086105df74e8a9, 0x000062b49a271a05 }},
        {{ 0x000fc8a61448916d, 0x000c74f3a29467a3, 0x0009855614595002, 0x0005455f2de81e94, 0x0000c7e3b2cd575d }, { 0x0001458e271cf38c, 0x0006c8f06d0de9fa, 0x00049a303fe35dec, 0x000c8fbd5bbc11cf, 0x0000091b6978a5de }},
    },
    {
        /* digit=13 (1,2,..,64)*(2^{91})*G */
        {{ 0x00072002d0deafc1, 0x0007568e039c2560, 0x000b74f7fa8c3e04, 0x000fa452b5f26fb8, 0x0000d5c673e4de9a }, { 0x00094308345d1eb9, 0x000e937e84fb7e3e, 0x0000233f0b08ef7d, 0x0005f8881b401d8a, 0x0000861e80d65e10 }},
        {{ 0x00076ddafe513028, 0x000b231be24319a2, 0x000bb927cde9a7fd, 0x00047f98503f7d28, 0x0000bacef2354247 }, { 0x000e4d52a90363e2, 0x00072961d7a64eb7, 0x000900d06b997d69, 0x000d4f0d5e436088, 0x0000deb49837ce80 }},
        {{ 0x000bab8085f68f7e, 0x00057b11d59715df, 0x0001dbdd17b82354, 0x000627448c7eaa76, 0x000008e08b2e2ab5 }, { 0x0009f8090d0bada8, 0x000652153765b2f5, 0x0003a01a76305a37, 0x000cd7310bdd4e08, 0x0000e04ea95f672d }},
        {{ 0x000bb2d0d12e52d0, 0x0002d9d615faa8f2, 0x00006ad2841c4cc3, 0x0005178828fa1eb4, 0x0000037e5443ad4c }, { 0x00018eaef0ca0cd3, 0x00057646caa6d2f2, 0x000e0158862a3d51, 0x0004399628eb879d, 0x0000cd6553865dd0 }},
        {{ 0x000c0571a1474d7d, 0x000eb4744ddfa350, 0x0005ceae85a5f097, 0x0009beff49ccfc5c, 0x000030d609ea9dd7 }, { 0x00057759b5198b47, 0x0006cd54b3a7b3cd, 0x0008885661e41633, 0x000b6143a35c0366, 0x00004e96b4e32798 }},
        {{ 0x0000a94af9ea7cde, 0x00069d83d30a2c5c, 0x000e36e136e23569, 0x0009d6c9b0857f77, 0x0000ee3ba9363a28 }, { 0x00091c4d690c1482, 0x000da0bdaa6ec1a9, 0x0002efb68a84b025, 0x0004cc597ba9fe49, 0x00009c2e3b6a4baf }},
        {{ 0x0001e48f047ae313, 0x0007928d2f01cb3a, 0x0003357921acd842, 0x00099fab493d6eaa, 0x0000971a98af096d }, { 0x0007186cef6f15a9, 0x00051f6c508b23ac, 0x00096ebcc85a8832, 0x000e40f38f137571, 0x0000e4475b94f38d }},
        {{ 0x000f0606c39967da, 0x0008ecee978a1d35, 0x00031bb32640de50, 0x0003dde62e3a68b3, 0x0000e08b646ad0ee }, { 0x0000f5f9d3887578, 0x00046276f9326f11, 0x0004b608425a9f9f, 0x00069fb512f521c1, 0x0000178ec5cbaa24 }},
        {{ 0x00015759ba88c80e, 0x000536b269fb37d5, 0x00042c8ed42b45c6, 0x000153353aaf7380, 0x000055e311c5bac7 }, { 0x00087e91e457b502, 0x000e9f8aa423a61a, 0x000124c369ae3354, 0x000978736592245a, 0x0000f1cbf0fedd43 }},
        {{ 0x00019c380abd55aa, 0x000a2e43d34a2e3f, 0x000dc29f25566282, 0x0003bdcc7ee3759c, 0x00006455153981a4 }, { 0x000fab2b4f7c81ad, 0x00074aade744b1f6, 0x000ff14cda1c443c, 0x000bd5111a7d222a, 0x0000a28194e30835 }},
        {{ 0x0004f489bbe9959d, 0x0004d5f4bf1d7255, 0x0001fa780b53c39e, 0x000e212069c1d8ee, 0x00009ce8effaceb9 }, { 0x0005945723762269, 0x0006d34992fb7e89, 0x000a0d0fec2418ef, 0x000dfa5000169a7a, 0x00002dfdd7e68c1e }},
        {{ 0x000876812dab14b9, 0x0007f78a4d32282f, 0x000c4f8b89ba0bce, 0x00094fe50e36e029, 0x00004692f67fcd8c }, { 0x000d3eea24df7225, 0x000ee8ed23a17f08, 0x00006374aae9a53a, 0x0006455e06d7b448, 0x00001b31dbe9cf50 }},
        {{ 0x00008ccbe2c9375c, 0x000c475b15d5acc9, 0x000f2cf48fe143cd, 0x0006c10f2c6d9c79, 0x000052c6758da616 }, { 0x0004e7f0deea926a, 0x000b8c4ba35afce8, 0x00007882ce2c78f4, 0x00030cbc15a3da8b, 0x00007993789e4ddf }},
        {{ 0x000a821ab5d22890, 0x0001522b99a72cb0, 0x00077cdf65604ed9, 0x000cad53e06de6e6, 0x000000279f3d1814 }, { 0x000490a31fb979cc, 0x000d92b7cb0b5a2b, 0x000c1473e470c4c5, 0x00054393aa7fb121, 0x0000cf1f5b79004d }},
        {{ 0x0005647406b4d83d, 0x000a77d62dd61ad3, 0x000c367b9f6b2ca3, 0x0000b8eba781a961, 0x0000c08fd788cd26 }, { 0x000f2bdb2ab6d0a3, 0x000f27b2736a129c, 0x00021de8172c3e3d, 0x0004959c5a631370, 0x0000e07bdbb63a83 }},
        {{ 0x0007d51e7e42a51e, 0x000f444c5f95cd80, 0x0000ecd677766dc5, 0x00040656dda8c8af, 0x00007a567e594477 }, { 0x0001a1b8848c495f, 0x000ff6ac2ccbcda6, 0x000f23870b5b7597, 0x000568bd43bc1923, 0x0000956dd9ebf318 }},
        {{ 0x000e676223b52ac3, 0x000ab08700230ef6, 0x00014eb24bd5447d, 0x0003015f20af585a, 0x00006fbd09543e40 }, { 0x000716998cd613d8, 0x0004d7c4e774d3fe, 0x0002c4f9938e4db7, 0x000fe92a582a3851, 0x0000708acb1bfba4 }},
        {{ 0x000ce0a3b86fb6ef, 0x000e671fb1675839, 0x0008348bebd1d282, 0x000eb8619dd011c7, 0x000021f9f273dcf9 }, { 0x000c00295b68f0cc, 0x000a689d5ec67385, 0x000a2c40c61023a6, 0x000546819a3b49f9, 0x0000308d7cb76e31 }},
        {{ 0x0004ada5e11c599f, 0x00075fb1d4bcc766, 0x0001e0ece31035ea, 0x000b3720fb07691d, 0x0000d42fa02b82da }, { 0x00097f2f7329b17d, 0x0007c4e5b544cf50, 0x0005f3a7e48b861f, 0x0005af7f4b4a13a1, 0x0000b8d9cdcafafb }},
        {{ 0x0008e1b8c2d74b0e, 0x00029d8affa8208d, 0x000b0347f239c200, 0x000ef527fc3c31b0, 0x0000983c109b8ea2 }, { 0x00039cab167742fb, 0x0001d15c8418682f, 0x000e8a2c7280fa2b, 0x000566e82541d487, 0x000043cc4696534f }},
        {{ 0x000f6985d44acee8, 0x00019a0e68db0554, 0x00086ac9b300b789, 0x0005c1284f356975, 0x0000a5ac2e22d6a3 }, { 0x000c9d95d6ef7c99, 0x0008615132ed17ab, 0x000403a5d02745a2, 0x000355766510ed0b, 0x0000b2d6d88461f6 }},
        {{ 0x000ddcc3fd82271c, 0x000b36e19abd09e2, 0x00012214bade2615, 0x00022eace0b4097c, 0x0000faacf01c7b54 }, { 0x000b860163208096, 0x000ba3d9e8153b8e, 0x00018bee1638b3c8, 0x000587c0c3324fd0, 0x00003de457dc65bb }},
        {{ 0x00039ed2562fdbc2, 0x000fbd1972f07f4b, 0x000e1edcea95283b, 0x0003623a6e5579e2, 0x00009e02c169c92f }, { 0x0006d17da40df3f7, 0x0006d17a2f62e55f, 0x000e24fe04fed313, 0x000f732f96b9ab38, 0x0000e962da3c756c }},
        {{ 0x00012c0401631864, 0x00079bd0f968c006, 0x000b9ee0840a483b, 0x000826e4da03da7e, 0x0000d17d0ff62307 }, { 0x000c74ec710cf280, 0x0008493109f5df10, 0x0000743a88df63de, 0x000c729b3fbda403, 0x0000893136d324b9 }},
        {{ 0x000c593df45e820c, 0x00049646edb951fb, 0x0007c2c9848e3595, 0x0000b05965f4a9d7, 0x00007059afd6a155 }, { 0x00047ed87de67e90, 0x000cbd46af3641a8, 0x000ab91c654ee10d, 0x00064a3047802200, 0x0000176fc6f6b4a5 }},
        {{ 0x0008923bbab39469, 0x0002e4e6d6711921, 0x000ac584ba5544c8, 0x0002f1e971e14100, 0x00005ee0cd3a9fe2 }, { 0x000af777f11421a3, 0x0004599bbe7e86e1, 0x000e0668efcedb0a, 0x000b42b3e4a1510e, 0x000081d2b6ac6370 }},
        {{ 0x000f3a93c5729e80, 0x00034004ef657241, 0x00091cd213392d1e, 0x00061a9943e04a4c, 0x0000714ea218efe2 }, { 0x000dbd736683a3d4, 0x0003afe3923555b8, 0x00089059de839c3c, 0x00012bf57744bedd, 0x0000f135993bf2e7 }},
        {{ 0x0002c8361b7a2aca, 0x0006699df1ea40e2, 0x000b350de8491faa, 0x00043ee12b3f24a0, 0x00002727eb357cdf }, { 0x00035a753655680b, 0x0001497f9c0d95c1, 0x00027ae958781188, 0x000c30ac38846117, 0x0000025367304961 }},
        {{ 0x0004c10cf8896213, 0x000ddf75f7ae2f68, 0x000d533960a9a46b, 0x000722eb615fc6e1, 0x0000ae866b88a248 }, { 0x0008a74fcafc168d, 0x000df95a73fbbd0f, 0x000a54e81f1dad9b, 0x0000f13437f24e0c, 0x0000c84c23966c22 }},
        {{ 0x00040a120307c01c, 0x0004985c9f842969, 0x000aa7b866ad1714, 0x0008fafd07fbaa8f, 0x0000621c9f76d164 }, { 0x00092f8868ee11e7, 0x000fa2055a4aec84, 0x000492a7ba874698, 0x0007a81ab550e9a7, 0x00002f1e26b56e34 }},
        {{ 0x000b6155609a0534, 0x000e1adb2d8f3984, 0x000c102373f2ce23, 0x000b3e3d6f13b51c, 0x0000421c4ce957c1 }, { 0x00020f448ada30a5, 0x000a9850a52c1c82, 0x0004fea81b397d5a, 0x000db0ec6eab1a69, 0x00001f18e2afb1cf }},
        {{ 0x000589fb928c5f43, 0x000ab31b0e63d348, 0x0002245b5418c388, 0x0008872a4f460057, 0x0000c3c71e7aa249 }, { 0x000bb0696dee0485, 0x00070cc6583553ae, 0x0009f6a5e077bc6c, 0x000ffc1520879094, 0x0000bccee4609921 }},
        {{ 0x000c457b0b8cd23b, 0x000e4deb9718710f, 0x0007f72f2ee9cd44, 0x0006782c138f1bed, 0x0000d9595242d62d }, { 0x00095c766178c8de, 0x000a1d4883841e20, 0x00092f15dd20214f, 0x0002f65958ec25b3, 0x0000f894c054a41e }},
        {{ 0x000a9fc894303e0f, 0x000b95c4bdc56104, 0x0003ccfeea6d613c, 0x0003aaeccb2885fe, 0x00002d45649bb35b }, { 0x0005fcc99d08171b, 0x000d5cd3ce299af5, 0x000357f6c183b368, 0x00076c08f195be9b, 0x0000561cacedeacb }},
        {{ 0x00098ba31d719d2c, 0x0002f4c721a4593f, 0x00047d0307a4b9cc, 0x000eccd14ed1e9a1, 0x00002d73755fff02 }, { 0x000ffb3c0ec394be, 0x00055739b128eb4e, 0x000de44d733bbc94, 0x000accfd37801214, 0x000050af6c641c91 }},
        {{ 0x0005f825c4f07a21, 0x000e67cc7bf3827e, 0x000a61fffdcaef2b, 0x000eab454f2165bc, 0x00005d08b9caa394 }, { 0x000659f053bb7164, 0x000eb67ffed4d6f5, 0x00085eb0445c885a, 0x000d096f1bc60957, 0x000009efc070a594 }},
        {{ 0x000c64a71a41fd04, 0x0007ec36044d63bf, 0x000810766586a678, 0x000b5b8f6a608940, 0x000015d2319af57d }, { 0x0002002ddd63570e, 0x000cc8083dbedba1, 0x000219e65019b26d, 0x000e6e1de804ae11, 0x0000b12ba3767b3e }},
        {{ 0x000d0a9304441d8a, 0x00024895dd21b96d, 0x000af239cf49a271, 0x000aaff7a8b99f28, 0x00007171f2ff03da }, { 0x0001f37bf44ca0c6, 0x000af224e233de42, 0x000109432cb14982, 0x00098cdd9192445b, 0x000083d6a7497fb5 }},
        {{ 0x000a5ab0888caa4f, 0x000f449f8bd2bbd7, 0x0008ce0d68d4a75f, 0x0005367f11d902cd, 0x0000a44f775a6959 }, { 0x000b840d135a5237, 0x0007406f7a09e03c, 0x0006666eb4106a14, 0x000c92f572f8a197, 0x00008f690387b3a9 }},
        {{ 0x0007753ef58b6176, 0x000379f7db326750, 0x0003002b30262a9f, 0x000bdea9478fea13, 0x00009fe0ef597427 }, { 0x000109beb4404306, 0x00054ed9c0bea786, 0x00095b35866fc305, 0x0003c6c6d41bb398, 0x0000d3e8797ead29 }},
        {{ 0x00025549467a0bf6, 0x0001bea3b6d1145a, 0x00090602dd0e0635, 0x000da6f62b6271ea, 0x0000e2394cc29faa }, { 0x000777d45fafe2fb, 0x0009d5e99bfe35f8, 0x000ff3d9f99729e1, 0x000a51d329770ccb, 0x000030a49d96b3e2 }},
        {{ 0x000b3786e5ed1d59, 0x0008122d847999db, 0x0006081ad64caea1, 0x0008c00a103aa33b, 0x0000197d6168ee3b }, { 0x000f6688f1378163, 0x00070ce6961f247b, 0x000e4392b44e7734, 0x000264b4f20aa85a, 0x00009914ec874ca0 }},
        {{ 0x000e24bf90175fbf, 0x0004f867469ceadb, 0x0003727b0ab89a54, 0x000f4fcc71904c52, 0x000031f4ae121f33 }, { 0x000f170db279ffb1, 0x0005690bd98764be, 0x0006b3571bec445e, 0x0009033c7a0f45f1, 0x000097d5d9760ff1 }},
        {{ 0x000ee1767e237834, 0x0002da029e0ccc17, 0x000de1cbc04beb15, 0x0009f0422cd53e87, 0x0000d2d72323a5a4 }, { 0x0003f6bc86ce1e10, 0x0002259648945af4, 0x000ce1d14415bbd3, 0x000439b8c9e45a5f, 0x00000c39573f5251 }},
        {{ 0x000f428fce95e48b, 0x0009848f2142e791, 0x0009a7a393bbf194, 0x0009f129b17d8925, 0x0000aaebd8ce27b4 }, { 0x000b5da0d65b712d, 0x00069010db06dfe5, 0x0000f54602f5c09f, 0x000bfc71bbfc16bc, 0x0000a4f9df66163f }},
        {{ 0x000e43c42cf85546, 0x000c44856ef96a74, 0x000849e83415f3c7, 0x000ea2a0db47eb84, 0x00000d10f03005f3 }, { 0x00020e52e37d20d6, 0x0005431d1a4aa2a8, 0x00083df6a0eb24b6, 0x000ddb47a440d466, 0x000048087cb057c8 }},
        {{ 0x00097c4c88758f6e, 0x000f1cf25e51b09f, 0x000e26a784d0b41c, 0x0005c8b0e180cd2b, 0x000009259c42e92c }, { 0x0008e2f4d6cd7faf, 0x0001824c03dbb7ea, 0x000458a9c70dd19e, 0x000e82253ae04376, 0x0000c21fa7e45f39 }},
        {{ 0x00099cfe2f0814de, 0x0007447d95ea1682, 0x0009ffd3781b0f15, 0x00082a3691758fad, 0x00004a8af81a2215 }, { 0x0002e22a81992dca, 0x0005417b0a98927c, 0x00005d1f1a13cbfc, 0x00047eb6a2e26e1d, 0x000016512c4f1dd8 }},
        {{ 0x0008cdcef72b2cef, 0x0000afbd90a060ab, 0x000b0225b18e21e3, 0x0004a7ccf496a27d, 0x000094a48f740aaf }, { 0x0002d4d14b785966, 0x000a937436a0bbaa, 0x0007d3b12efde960, 0x000dbdccf263fa10, 0x00008368434e4361 }},
        {{ 0x000e4e4a56705d8e, 0x0000f6812e9a1bcd, 0x000b811fe812283e, 0x0004f583861d9075, 0x000054c73bae5329 }, { 0x000eb36201e51c99, 0x0007b4ad673c0175, 0x000e10baa3e900c7, 0x000b52456667417a, 0x000012e79d4dacf7 }},
        {{ 0x00069ef68c67b87f, 0x00046043a1aa9145, 0x0009203d62e8c5c3, 0x0000f935c175a129, 0x000001fa1a6965d8 }, { 0x000d9e2a0a581188, 0x000536363a113b4a, 0x0005eae52d5c3801, 0x0002fa5eae3bd3f5, 0x000066503ca8481e }},
        {{ 0x0009192ffa7d59dd, 0x000bb4c508b65e9e, 0x000084d870d6c300, 0x00060a3dabde4a1d, 0x000027aed4e0a762 }, { 0x000479a99a505811, 0x00085965099fe925, 0x000bb6adfbb04f0c, 0x00038c96125ddb65, 0x0000b41360e14375 }},
        {{ 0x0004fb15b0510f55, 0x000f905013fe1620, 0x00025e32316b7fd8, 0x000feb39fbe11dc9, 0x0000d62b1f4b95aa }, { 0x000a88e26620cd37, 0x00028c213c999279, 0x000fa35a08cf8550, 0x0000d94b4a033e9d, 0x0000037e5cb1f698 }},
        {{ 0x0001d933d32c017b, 0x000346254671ff6c, 0x000a9dcc5230cb16, 0x00075f8c04bd4241, 0x0000bb7123eb38ba }, { 0x000c6641dc5f7a5b, 0x00051f3689e2c70c, 0x00067a54bb55957b, 0x000f99be592aab0e, 0x0000100ca18fcdd4 }},
        {{ 0x0002150673a6e180, 0x000f757d31e535ec, 0x0005c5ade79e2400, 0x000211ffa04cc43b, 0x00009f71ec100187 }, { 0x0005a52faef35750, 0x00071e08d68b0139, 0x000f584d694447ac, 0x00028f464a9d6816, 0x0000d379add57aaf }},
        {{ 0x0006293a2dec1ae3, 0x0003e600e1cb64eb, 0x0005c8568fd06998, 0x000961893004828f, 0x00004d492af4542f }, { 0x00028e52e6b40b4b, 0x000a421d145d2d9a, 0x00031901531d2a2a, 0x000c3da2dbe6225b, 0x0000a8127987847e }},
        {{ 0x0008303593dd6e66, 0x000629b4c371e6da, 0x00048ebed994bfaf, 0x000bac981cebe067, 0x000016931e8d75c6 }, { 0x000130cefca57500, 0x0005b0a7e37a9bb0, 0x000e2a5edfc6fbfc, 0x00054b1028c56f61, 0x0000ecf694a5348e }},
        {{ 0x000752496e016bd2, 0x000cda0bd865cfbc, 0x0007f50cd4695497, 0x00052a0244b03e57, 0x0000637b077b76b9 }, { 0x0005ae9668f90345, 0x000e83b564beb6b0, 0x000033f32f9fd853, 0x000f2bc260db1d11, 0x00009c49e1c7cb1d }},
        {{ 0x000633e728e450f7, 0x0007637c92544f2b, 0x0002ed2450ac358d, 0x0000057cecc08435, 0x00008ef9d3967336 }, { 0x0002ca7d5cb05be9, 0x000ed9f936c581cb, 0x000dc16f479d4101, 0x000e8a15b96e7b8c, 0x0000175bf8daeb7d }},
        {{ 0x0003f3086f736b1f, 0x0000641f50240258, 0x000960cfd0c861ef, 0x000f6f99f47bc576, 0x0000686ca9cfba8f }, { 0x000b7e65f9f1ce62, 0x0000b94b587579ab, 0x0007756906660826, 0x000329dcfec2d22d, 0x0000255e257067be }},
        {{ 0x0002c8ff8f6efa40, 0x00054776652a220f, 0x0008e3cdc9f20d82, 0x000f5bff361618f8, 0x0000f1cc52b08e1a }, { 0x00062b9b103f16d5, 0x0003ead8efb4d3e2, 0x000fa7ebd284ee8d, 0x00051001c937b5ad, 0x0000d93b71b723eb }},
        {{ 0x000588993b473591, 0x000bee4c43ff28b8, 0x0006e4ddb53f06fb, 0x0006fd4c27647662, 0x00006c003ddbd070 }, { 0x000832749a9bb5b3, 0x0004ed68f938829a, 0x000d96ab08b23bff, 0x0001781c7996627e, 0x0000272b5788c7f5 }},
        {{ 0x000ddbacedcf37d2, 0x000bf7d4bf503533, 0x000b178bb90d5ee0, 0x0009dd96685dd726, 0x0000ebe65a8ae8c4 }, { 0x000da2828c5f8f65, 0x00074a412f22e85f, 0x00066f647430480a, 0x0003ab137e2a2486, 0x00009656ded0c9e4 }},
        {{ 0x0005aaf668ac79d7, 0x000433ea2dee7a63, 0x000c3da02c1bc912, 0x00031cfaddcde2f3, 0x00001022732669a2 }, { 0x000b89dea0558e1e, 0x000e6a9e5aa049a6, 0x000d25865fe4287c, 0x00021cec3e74083a, 0x00001aa8b5c32deb }},
    },
    {
        /* digit=14 (1,2,..,64)*(2^{98})*G */
        {{ 0x000be00843e7b7e6, 0x000b723fe67ba665, 0x000037ef26f02671, 0x0004513139145076, 0x00001e80333c65b0 }, { 0x00067e7cf2b69e56, 0x000e805d53ff04d3, 0x0003276aa2047eed, 0x000006ba0bf7ccdc, 0x00008a2d8826cd00 }},
        {{ 0x0009356a79debb8e, 0x00065b7b0dc85957, 0x000825e834b42de6, 0x00021f4a727de460, 0x0000c18079e2bfdc }, { 0x0003b0bfe5e20c23, 0x0000045f5f9a0529, 0x00087fe98313de54, 0x000411dfc1a8b0f8, 0x00004e039a515ca2 }},
        {{ 0x000607c8357f525c, 0x0008102c024789cf, 0x00094badb3c6d4e2, 0x000e13cdf90dca9e, 0x0000f013f32caca4 }, { 0x000259ced8a8fa9f, 0x0001f97c2c99b76f, 0x00062668c42a120d, 0x00009098c6576ba6, 0x0000ddf41abff43e }},
        {{ 0x0009d20f961c2e82, 0x000b85dc8de610b8, 0x000ade101437f35a, 0x0007eebc5e8c515f, 0x00002509d1321032 }, { 0x000842e3dac8ba0c, 0x000fd66098583ce0, 0x00048bafc3fcb163, 0x00076414aa2eedb9, 0x000007db9f48bc83 }},
        {{ 0x000c04346f8b4a53, 0x0001fa33b9e2d7b5, 0x0007687de490598f, 0x000ca4a917cc60b2, 0x00003bbf0d26c8f8 }, { 0x0008cace3b4eac97, 0x00066f2614650a98, 0x0007392b98719d3d, 0x0000e25835e35b53, 0x0000792a37dc963a }},
        {{ 0x000a9c2a39c1acbe, 0x000e5753de92e23f, 0x0007651f8ca95eef, 0x000048a7eccb03cc, 0x00005893c9a2ca16 }, { 0x000cba2866a3fc9d, 0x0009b236724ceedd, 0x00073c72dabcd16f, 0x00061668e3a90225, 0x0000f7789359b346 }},
        {{ 0x000a35b6a1ba7e68, 0x000bb7f788907328, 0x0008c9c6fe382aca, 0x000b618d6c443bef, 0x000027c8cd8a0530 }, { 0x00044a35a0bbe471, 0x000d61665e9a90a2, 0x000d40e5d11ff8ff, 0x000053d3b45bf380, 0x00001226d9bd6c94 }},
        {{ 0x000595c4b3f8dbd9, 0x000359d74cd06ff8, 0x000f270e29f8a825, 0x00087d12b9c17c7d, 0x0000b2e80e811f87 }, { 0x000db27ce43a86c1, 0x0003fca990f62ccd, 0x0002ec59bf016957, 0x0000628f5d9bed21, 0x000027d55a4ba6a5 }},
        {{ 0x00088b9aa14b8da6, 0x000a46658926ddd8, 0x000907f47b17a1a9, 0x00091d4b8138b2d4, 0x00001b4b6e70d892 }, { 0x000f508ea42a5543, 0x000c04e3a96e3551, 0x0008d3bf2ff6a996, 0x000cbf37aba65b0b, 0x000012115142acd4 }},
        {{ 0x000f55cded9f6e0e, 0x0000c49747c866c7, 0x000d85c2d274f8e8, 0x0006e2808bbb1e5f, 0x00001441178e9e4d }, { 0x0005bd1a0a03f0a8, 0x0004495f38c85f0f, 0x000e96069eb98983, 0x000220e6f5b589fd, 0x00000accc96b8e0b }},
        {{ 0x000a8761f7872bd9, 0x00061e55e915918c, 0x0008bc714273ddea, 0x000c0209c6158d6d, 0x0000e6410b2c4167 }, { 0x00091666024d7e90, 0x000b9c29ebbc601e, 0x0000f7ad482d61b5, 0x00067a752ef3b0f4, 0x0000530ed4817b10 }},
        {{ 0x000d03f718913dd5, 0x000c2befc15aa1ee, 0x0007c847102cc8f2, 0x0005c8a1240d1254, 0x0000fdffe724edc0 }, { 0x000f68ea1cc2db7d, 0x0003e6c4a02c4997, 0x000d509587b544b9, 0x0005f5faff725083, 0x00002c702007fc4c }},
        {{ 0x000b3261f5dfe89f, 0x00029c7b7ffd977f, 0x00048543b82d075f, 0x000575c83ec104c3, 0x0000ac684c28b97a }, { 0x0003997db2fbb677, 0x000d260d883e5448, 0x000009297105579d, 0x0001fe89913520d7, 0x00000d81132cc8d8 }},
        {{ 0x000236a324c0abe2, 0x0000f060dbd2c016, 0x000abd6646100dc4, 0x000ed57214f8eea6, 0x0000ebb2e4bb264d }, { 0x00004281091086cd, 0x0005284a16022e9a, 0x0007a38f5aa9551d, 0x000c6229c0c18d87, 0x00007ce409a2371c }},
        {{ 0x000ceb5971a6fc75, 0x0007c5e9afdfb3cb, 0x00029ef45d6dc1e8, 0x00085c89749a5614, 0x00009b93cd3762b8 }, { 0x0004b4099139dd59, 0x000ec5fb269bbc4a, 0x000e450853d0fc8d, 0x0003b57867cd85e1, 0x00004a5c6d138b15 }},
        {{ 0x000d112927153319, 0x0003f1d26ee83821, 0x0005188c3a678749, 0x000f5953dcf17dcc, 0x00006ee6a46384a1 }, { 0x000b8bdb94fd5a59, 0x0002b7750491a746, 0x00019255042413ef, 0x00049dcab687fa76, 0x0000d3bdf3fd8d81 }},
        {{ 0x000d8811182f36a0, 0x000bdc93bf834aa0, 0x000941e46d331d24, 0x000001cad0437181, 0x0000a5e4336ef208 }, { 0x000560a8b653f914, 0x000fb82f48c8e33c, 0x0004dc5e9825a5d5, 0x0002e1fe5a11fd07, 0x00008d6b831c2888 }},
        {{ 0x000054303f46b6d4, 0x0004be23929d0979, 0x0000029d58b20226, 0x000b8f9ac3885b29, 0x0000aac58d89d493 }, { 0x000f3d6452261dfd, 0x00021440738042ae, 0x0006ce71fff79069, 0x000abe21b98e6a97, 0x00004b0f9c9efeab }},
        {{ 0x00018d761d16f241, 0x0006d82b6d72d285, 0x0001a44aca62ed09, 0x0009cbb1c77ca9db, 0x0000674f75f7e28e }, { 0x000e54ba56fa1c83, 0x000f10ef0d1b5ce8, 0x000c5b965d2e6f98, 0x000d8b082717039c, 0x0000b2abaa683928 }},
        {{ 0x00073eb5de6f16ea, 0x000838af7f91d0f2, 0x000180eba14a2d69, 0x000c07d543b61b0f, 0x0000a48e18374f25 }, { 0x0002413c252f7b56, 0x00055f77b4e4d538, 0x000f5bfedb51fa0d, 0x000d51bb3b7ce66b, 0x0000bd4cf4850d41 }},
        {{ 0x00058897dc9c0e4c, 0x0007cc4e0e8ad934, 0x000e1a22eeb0bf71, 0x0002afaf3b5679ae, 0x0000e7087f871d8f }, { 0x000917072350b58a, 0x0003f3089d0caf3f, 0x000e7f4d276ecde3, 0x00008251839564c5, 0x0000087315f576bd }},
        {{ 0x000446bca9aa0ee7, 0x0001c17e4515fa57, 0x000486c5bc65bdfd, 0x0001f2065f85cff8, 0x0000dbfd705501c3 }, { 0x0002113ffe594d91, 0x00076aa6d63184f9, 0x00079416c3a80498, 0x00070ef357c1a46a, 0x0000f69954823f55 }},
        {{ 0x0000c22f68309ca1, 0x0003f8312e7e46ca, 0x0009b5503c6d1a9c, 0x00023dd0469d0a37, 0x0000cf8b19a3e343 }, { 0x000c392312daf2bd, 0x00043514d2c7d8fb, 0x00030fa9a7d2b439, 0x000fe3577bc45ced, 0x0000ab008068f4cd }},
        {{ 0x0005fe87134cd0da, 0x000a17e3cc8ac855, 0x00078703ee282882, 0x00067052f8d725f5, 0x000030f3d852516a }, { 0x000242d3ae1ef785, 0x0007fa96ab6b01c8, 0x0008fa96637638d6, 0x0008ee49b69a02a8, 0x0000dfec375d87b0 }},
        {{ 0x000c24d8d4d1fb51, 0x000be65d62105e5f, 0x00093104e58c9f41, 0x000009e09bb222bf, 0x000003c8d01fc99b }, { 0x000f5d057a938685, 0x000eb85d41299673, 0x000c9cdf24c8ca30, 0x0005254a3db81c3c, 0x0000a0f18edf4572 }},
        {{ 0x0000a412c93e0e06, 0x000ef6d88c546946, 0x00053a654474d349, 0x0009ff9d1b26baba, 0x00004c0a14f6e6e4 }, { 0x000e419776983108, 0x000e18b54b25039c, 0x00029a3cd9d75545, 0x000cfad94d4ee67d, 0x000011516306d1e2 }},
        {{ 0x000a4bb3d86217f3, 0x0007c851c203a40e, 0x000600b1dbb8dedf, 0x0003f9ad230d352d, 0x00001526e5ac6533 }, { 0x0006ec4cd010437b, 0x0004203b98e782b8, 0x000f7f38453d3262, 0x0001381c8ac958c3, 0x00002e418f417cec }},
        {{ 0x000f921d710cc9ab, 0x0008a269be47be0e, 0x0007db96e1305cfa, 0x000a2bce5323b7dd, 0x0000b7178cc6492d }, { 0x0006c2921097b191, 0x000d0fc3c4b880ea, 0x000df1d178576177, 0x0003f6e393f6f914, 0x0000ac9a7fe9fa67 }},
        {{ 0x000e1999144d70e0, 0x000616651b8d9a36, 0x0008a1e1749d7d0f, 0x0001207ce42531bc, 0x00005552258e045d }, { 0x000453c20f47dfdb, 0x000d63dd5d369463, 0x0005c23b3f858e11, 0x00030f7308255dc6, 0x000097f02a0dbe3f }},
        {{ 0x0000448da6108dbd, 0x000d0c089a6ef014, 0x0000581351df8a6b, 0x0008d1c244dbcc3f, 0x0000ec04c9855b4b }, { 0x000882a424d7120b, 0x000876d4ee343e2f, 0x000c9865fedd1f17, 0x0006f503f6343f1b, 0x00001fce589b6c69 }},
        {{ 0x000b6c5db412e00f, 0x000c6f4e7db9f57e, 0x000950cbee732732, 0x000e651b17dfba38, 0x0000cfa203fd495c }, { 0x00067ce8c320f904, 0x000ea661cdfe0491, 0x000e2dd72666a257, 0x000f366ad9baac0c, 0x00006dc6d3d1e884 }},
        {{ 0x000bc95725aee3d5, 0x0002ea553fb2b560, 0x0007423c28530356, 0x00000bbd96ce141e, 0x00009b1c7fa39ddb }, { 0x0007326ca7661923, 0x00044911220fd06a, 0x000b99589f2eb8ac, 0x000d2c8a9716e3a3, 0x000016f0b4225082 }},
        {{ 0x0001690ce1657980, 0x000051ccd090c436, 0x00038fc32ca6d826, 0x000d1fbe4561e8f1, 0x00005c54597e892a }, { 0x000bc9fb209cabb7, 0x000e3370f009052d, 0x0000145c0e0d8cf2, 0x00051250f258758f, 0x0000f2608411d361 }},
        {{ 0x000532da0afeae19, 0x000c46e246518404, 0x00046092c0b1374b, 0x0001c3038dca45a8, 0x000033c9b87864ee }, { 0x0007d1a0f94114b6, 0x000a8a51999a6bf9, 0x0009c69a51f582fe, 0x000472a933aa9505, 0x000027a0c0040cfe }},
        {{ 0x000422b0ea14fa4b, 0x000a0ba2e1c21dd4, 0x000be844e6da17b6, 0x00005c51ff41b29f, 0x00007397b90d379e }, { 0x000a349d39ffa32b, 0x000651c92f1bfc4c, 0x000f17ac719f4a41, 0x00085fbb806ab1f8, 0x0000fbba41e33ddb }},
        {{ 0x000e959ff7b2ee8f, 0x000ae32343bf1a9e, 0x0007493c06b9bdc7, 0x00079d6b48fd3bd2, 0x000098c8ad76a47e }, { 0x0004498bbccb8f72, 0x0006af89706dc3f5, 0x0001c204bf7138d2, 0x000f8d1c8737db0e, 0x0000b2806c0cc363 }},
        {{ 0x00065b514edf386a, 0x00009faa29e24afb, 0x0003ba4f3652565c, 0x0001ed0cefcc3c45, 0x0000b4260964a621 }, { 0x0003d84d2f4de65c, 0x0008bcf2193e3934, 0x0001b16c00689684, 0x00018dced60e9a4e, 0x000031f1675d66b3 }},
        {{ 0x0000da1a5636e5cb, 0x000e641ee398a212, 0x000ec2e2ab685487, 0x0004e95eab88a56a, 0x0000f1202655d900 }, { 0x00084fcb18da2d59, 0x000409d65bf0384d, 0x0003850a4cf62127, 0x00046220471a14d2, 0x0000aabe22696247 }},
        {{ 0x000c58b582f24ca4, 0x000ef7bfa44b4794, 0x0009104047cc8e91, 0x000a3e3275350549, 0x0000d5c4cf012836 }, { 0x000c293a547b773e, 0x000f8d88327f246f, 0x00095154b8cc443a, 0x000e8e6f2c3a3bfa, 0x0000876c2770053d }},
        {{ 0x0001a4c61890bb48, 0x000ed2b5acf6e10e, 0x000cce418caac285, 0x0009c327fc50343f, 0x000054fa09fb1487 }, { 0x0001219614c932a6, 0x0002acd33425d0a3, 0x000d6a5a2e6ad1f4, 0x000390c0b306399e, 0x000069c28a3f1ea7 }},
        {{ 0x00080094dc4fe82e, 0x000926487f5d28b0, 0x00018e21e3213666, 0x00082bf5f11392d9, 0x0000b5fd9212666b }, { 0x000eebe7f030377c, 0x000dc8c208ba19bd, 0x000ea6270a941ac2, 0x000bdfd9ad3e7570, 0x0000821cf9f8f5d2 }},
        {{ 0x000233c3bcb79698, 0x0007798e5c298ff6, 0x000dd44472627899, 0x00035021617f561b, 0x00004d4a92c755e1 }, { 0x00039704dff5bf49, 0x000cca192e3a2cb7, 0x0001729f4c39863a, 0x000cd0ab1961bcfd, 0x000099be056ddf36 }},
        {{ 0x000983e996aeb001, 0x000fa587e7b754b2, 0x00017e82e022cefc, 0x000a043522336079, 0x00001a6d17a03274 }, { 0x0000980f9163017c, 0x0006929958d881dd, 0x000255946bab7a49, 0x00009d6d1026bae0, 0x00008bccd8fe0233 }},
        {{ 0x000d2d5a30705fba, 0x000888cd5f30851d, 0x00064d220bffdbbf, 0x000fd8ea494dfed2, 0x0000fde12797428b }, { 0x00094c486a8e2a7a, 0x0009bbdd44cc7a57, 0x000d43c60acfc916, 0x00005ddc94ba6b20, 0x00007845c76e4d96 }},
        {{ 0x0000a4725a093707, 0x000cdcef21496e4d, 0x00081bc4f0d7dfee, 0x000567a2ff951d19, 0x000004d0a3778435 }, { 0x000aeb8750fba3d2, 0x000d0b50e3bab661, 0x000590c1a91e5597, 0x00060f29e462caff, 0x00005cc394b808b3 }},
        {{ 0x0001170c04952602, 0x000ef2f2f7d0ca2a, 0x00083f53c32b2b0e, 0x000d4ca2f6d2e812, 0x00000b742e457e8c }, { 0x000c212c2c471b8d, 0x000904f7805a99ca, 0x000620057f79b6ee, 0x000379240bde56a3, 0x000033f6108aa3ee }},
        {{ 0x00052e379c819e4d, 0x0001ef315697be5d, 0x000b57359024af02, 0x0002f60af6f0545d, 0x00005709fccda5fb }, { 0x0001cedd8c6a019a, 0x000c747f462c23e6, 0x0007e4ed47245af3, 0x00083d34fd44a429, 0x00006bc4a68eae78 }},
        {{ 0x000f0076f4926fec, 0x000495699f241d77, 0x00068119bda8ce25, 0x00019f891ee4adbd, 0x00003a1ea5347618 }, { 0x000550518bb90e9b, 0x0007dba9264da3d1, 0x000d4d575100188b, 0x000532e716eb2d2c, 0x0000e97178dd93ce }},
        {{ 0x00024db6c372c507, 0x000dbf057d0b70c4, 0x000e8ac584e05e23, 0x0007c63eceea8612, 0x000024b1bdc12cd1 }, { 0x000dd6f3a8ca9d0a, 0x0002fd116213fd1b, 0x00045acc53e70ec3, 0x0006f4621c61e7fa, 0x000085b4cbd42400 }},
        {{ 0x00099a3c67370e5c, 0x0003a675d97f7e2b, 0x0000f571b84ecef2, 0x00083067a31db0fc, 0x000017a0dc2e6d1d }, { 0x00085346d4a4a2c2, 0x0009d5536b1ff923, 0x000718d7f1b94518, 0x0005a30632973857, 0x00001c47d9a812ac }},
        {{ 0x000f7d00d2868ab4, 0x00011ef31631f9e0, 0x0000f428795737fc, 0x000994421bf91da2, 0x0000273abfaac3f4 }, { 0x0004d97e8ccbeaa3, 0x000c93bd8fe79da6, 0x0000704c4b09c411, 0x00014c02acc5e8c4, 0x0000b4ad51bd9104 }},
        {{ 0x0002ef029032eee0, 0x000f97302f1cd1e1, 0x000b0ae310148f43, 0x000a922f7ce87eac, 0x00004d8e4c69e3b3 }, { 0x000bf1615a2e8a8c, 0x0003b4e1bcea6e23, 0x000400c7cb616594, 0x000051105d0b5402, 0x0000087338e47a50 }},
        {{ 0x000f0469962cc718, 0x000da79f8a7a838e, 0x0000165cecca8079, 0x0005857a111aea63, 0x00005efd15955b73 }, { 0x00066e898fcc53b0, 0x0000ec9d8080711c, 0x0009d97055e6ac0a, 0x000ce1126e1aec60, 0x0000c1f2ccbe262d }},
        {{ 0x000ba8ad1ed135cc, 0x0009dc408811fdbf, 0x0007341f43636885, 0x0001975388bfe1ef, 0x0000b69e74c27446 }, { 0x000526bed9600a55, 0x0007bcebfd2fd5bb, 0x0001b9bf4e885c17, 0x000c7cd530b839a8, 0x00008c55b30c7f4f }},
        {{ 0x00038b8ee0eac787, 0x00034c2385644641, 0x000778ba2d92e1fa, 0x0005103789e3ff94, 0x0000d8a525b4c26d }, { 0x0005fab8823fdd61, 0x000c24c4123c4899, 0x000864697a151ca0, 0x000c3fea366e6936, 0x0000a8b00c976bf9 }},
        {{ 0x0002f26699c531ca, 0x000a607eafbb1e1a, 0x0006446913f9ed17, 0x00084c378d75c9ce, 0x00002f579798942a }, { 0x000be4b0873a045a, 0x000c4cfe469b17ae, 0x0008fb488e3bb18c, 0x000c6b73f6254577, 0x000013dcc4276048 }},
        {{ 0x00017c8bfcafa626, 0x00038d78cdfedb1f, 0x00070ca48e783c87, 0x0002c1ec5a535c07, 0x00007254a1aa9e62 }, { 0x000e719e57fc7500, 0x000412efb11fbc42, 0x00026639a10db9dc, 0x000828243ea0b548, 0x00002ca6e35f3d10 }},
        {{ 0x0005b95cd967e18c, 0x0006485304ddc5b8, 0x000cc7f246112826, 0x000bb9ca9095e8c6, 0x0000dc3c5bfcdad7 }, { 0x0008349cc1a3f2ac, 0x0001c19b97447336, 0x0008944abac218e4, 0x0009d8ce429c5f53, 0x0000d9a807c23931 }},
        {{ 0x000adb13a8f96a40, 0x000cc1d21830ee5a, 0x000e3ee48eeed2de, 0x00073da9436f9c4e, 0x000084a78c9eeb4f }, { 0x0004d8c4d2962386, 0x00024a081313dbae, 0x0007dd3c6905bb3b, 0x0003055f8c76bb46, 0x000014e6ca4ea310 }},
        {{ 0x0006544814c3dd38, 0x000ae75b055cb405, 0x000b41270952f9db, 0x000528063898f8e7, 0x0000345a4f463cf4 }, { 0x000d3837fbd3e57e, 0x0001095263a75a91, 0x000efceac4dedcad, 0x00056557ccfb6836, 0x0000601bca5adf6b }},
        {{ 0x0008cfb7c7648894, 0x0002138ff4f94ab1, 0x000254b0012b17d7, 0x000d3be877243c71, 0x0000e33accb24543 }, { 0x000bac55e7f80d33, 0x0008ce6729082196, 0x000b31a4eec38aec, 0x000e7812f35c4592, 0x0000abded42c4fa4 }},
        {{ 0x000c639ea67a665e, 0x000231e94dea0f66, 0x0004752922fd3b67, 0x0008002f8259ecae, 0x000098781142c79b }, { 0x000839e6a69aa2dc, 0x000dccb7e1026fd1, 0x000b4aec329bec7e, 0x0004d41a39e73b50, 0x0000263b2ffecddd }},
        {{ 0x000014c5132e700e, 0x00086155150b8057, 0x000e8917ab8428ab, 0x000983d951892389, 0x00006f4bc50a7b1b }, { 0x000b4f4f21091832, 0x000a9361b07523ca, 0x000d80cd15d93e68, 0x00013de5a2f6219c, 0x0000812154648f57 }},
        {{ 0x000a527e6b978cee, 0x000db82cf7d62d27, 0x000ca96802e48946, 0x000e9519e1f36e29, 0x00007265e87b84fa }, { 0x000802786e991660, 0x0003097581a85e38, 0x000f541313771b92, 0x00067126e1f1a520, 0x0000d10aa19b0e7d }},
    },
    {
        /* digit=15 (1,2,..,64)*(2^{105})*G */
        {{ 0x0006f0af00724aea, 0x0002a9e36f09ab05, 0x0009f3200c5b7d67, 0x000971f9803f163c, 0x0000abeb83ab4494 }, { 0x000dc4a79b89e736, 0x00099ec70880e0de, 0x000ed078e9a3f10d, 0x000e77ebd7332a66, 0x0000efcf0956aef4 }},
        {{ 0x000dce3782cc4229, 0x00054d0c55c44969, 0x000687ed744ff73a, 0x00095ca63a6a9635, 0x00006189f2556fe1 }, { 0x0005e2f1f92bf8e0, 0x000c16ab2b91f71a, 0x0002b5bd2443b5ec, 0x000ec35ade448982, 0x0000c8788b8a39ca }},
        {{ 0x000392b0b18c6d03, 0x00098e7abded5516, 0x00086524935ee7e4, 0x000a3d6a903d358b, 0x00006875db035981 }, { 0x0006fc5d51954294, 0x0000e980b75c904f, 0x000db1fb9c1ad202, 0x000ea6532c305b0a, 0x00001aa27478cb82 }},
        {{ 0x0008f862ec09473d, 0x0000ac9d64cc78c4, 0x000b52744f8f7011, 0x00038e5ab6c50621, 0x000008c758760cd6 }, { 0x0006d3b2c6ac19b7, 0x0007bede1603c166, 0x000ef5c6e18a250d, 0x0000ffdfc19a80e0, 0x0000dc276b838e08 }},
        {{ 0x00006fa37070bc95, 0x000637e822e37bef, 0x000334cc36ddfbe8, 0x0005c75eb73f237b, 0x0000f47794e531d0 }, { 0x000fa0c354bdcde3, 0x000d86efc621c127, 0x0003c1174b714c9d, 0x000d773f35d624b6, 0x0000fc893443d0bd }},
        {{ 0x0003d69f745fb237, 0x000c331e3229d41f, 0x0001a07c7c60896f, 0x000214b09a929c65, 0x000030eff0f7b347 }, { 0x0009ce6bfe978f9e, 0x000339a168336319, 0x000d6a06afe4f177, 0x000c17ed868a34c2, 0x0000a6c91a8a3c76 }},
        {{ 0x000567bcf9802012, 0x000c8aaffcdb4631, 0x000a1f3bce0ebf8c, 0x00088f58d658bbc1, 0x0000d77d6056ff71 }, { 0x000c08728ddda15b, 0x0003de03572359bb, 0x0004ca5173fe02e3, 0x00082257ccf0353e, 0x00002fdd1fd23a9f }},
        {{ 0x00011cac13161f9c, 0x0001bbc453cadd69, 0x00072aef15e577c3, 0x0008c37af203900a, 0x00000e41db5e3490 }, { 0x000b87d44a487f26, 0x000dc42ec965469f, 0x00012e582d33e4d5, 0x0001872850e9f769, 0x000038b03659451d }},
        {{ 0x000cc69f23d9740e, 0x000004bfc25419a6, 0x0002df389dda0805, 0x00042da3f3646720, 0x0000f63588d2624c }, { 0x000b9eea3275aa7e, 0x000665a6bc68bb94, 0x0005dde089018c6f, 0x000b34b7ce43d79b, 0x0000cca74fe735d0 }},
        {{ 0x00075c28e4f6afb1, 0x00015ba6102d0212, 0x000807e8390524e7, 0x000ef68a8eaafac7, 0x00005aa40be45ccd }, { 0x00072a4f8c8e1dc1, 0x000599ddd2e54e03, 0x0004cedb11fea727, 0x000190c246991adb, 0x0000a2b27ebd40ed }},
        {{ 0x0009ee1edba4daa1, 0x0001c73a6a5da60f, 0x000fe25e811523b6, 0x00078cd576f083d9, 0x00001e9fcdcfb94b }, { 0x000aaf21de46d633, 0x0000eec8bdcf8e7c, 0x00004054fc48103c, 0x00029bd40a5db827, 0x0000e01632abbb9d }},
        {{ 0x000759b8001f79c6, 0x000ece14f0dd2e8f, 0x000e12ddc2c79568, 0x000005b98e77e7c4, 0x0000b791403a3eee }, { 0x000aa7ff8c7ea4a3, 0x00083eddcfe314e5, 0x0003f92dd00d49d0, 0x000ac95b6eb84c52, 0x000080d5b1ce41d2 }},
        {{ 0x000baaaaa42eba90, 0x00083c3b9791a263, 0x000a922ae072d050, 0x000f042de960022d, 0x0000bdcee541ee28 }, { 0x0003bb68bfa63380, 0x000a3460d881efdb, 0x000f9276f56af417, 0x0002898d4506416b, 0x00006f83b035b65d }},
        {{ 0x000fd6b0c5230ca5, 0x00013bf5116f9379, 0x0006214fd87df2b3, 0x0000cfdc87c64a58, 0x000027f51850fe61 }, { 0x00092d51304b9476, 0x000443ce8ee0e0e2, 0x0000005ed0ca005c, 0x000d248e06f6fba6, 0x0000747a3ca00962 }},
        {{ 0x00048c919c326cd0, 0x000405dbab093b30, 0x0008fdf707eac1e4, 0x00007d89c012f59b, 0x000033b871966602 }, { 0x000eaf517a2fb4d1, 0x000d1b591071cd58, 0x0009749a1f2b94c1, 0x00002c84b791ed89, 0x0000772ac30fc8b2 }},
        {{ 0x000da8e4f1aff818, 0x000caa2d551ee106, 0x000804d524b4b06e, 0x00008fc8d87a8f1d, 0x0000ef431fbeb97c }, { 0x0000c0d9b00298e4, 0x00033f0cf9a4e718, 0x000fbb34f16943d4, 0x000ca08c0ff50200, 0x0000745ea7a228cd }},
        {{ 0x000d1395ccbc828b, 0x000e6d8033f28764, 0x000630492e0f2bf4, 0x00047311db0186ec, 0x0000dac857b0b37f }, { 0x000038d71b0c1915, 0x0004dd6186244a03, 0x0002e0d184389b23, 0x0000269ab918e5f2, 0x00006cc928b6af68 }},
        {{ 0x000de6ad0f11bc0e, 0x000f6b7a93c32461, 0x000982d307be7245, 0x0001a341821ab606, 0x0000a0e9cfcf1ca0 }, { 0x000b972fd6728dfb, 0x000af88ff1929c78, 0x000ad87757dee47c, 0x000d111ccc6df4c6, 0x0000d7c581401c53 }},
        {{ 0x0001f7f91b4ffc66, 0x000d0b60b846531f, 0x00051eadeb59f002, 0x0004950c5ba7e45e, 0x0000b7e2a63df2f1 }, { 0x0004c828296f804a, 0x0005c9244de90d77, 0x000a7cd199349397, 0x00056f3eb951a1d4, 0x0000788cafa21d5a }},
        {{ 0x000ce2e57adaafd4, 0x000a4fce7560b907, 0x00012b053f53cfb7, 0x0009317a8b4073e6, 0x0000b35cda57c1fb }, { 0x000b6dd71e0124a6, 0x00021891df1ad467, 0x00018e156fdf889e, 0x0000b1b7783a0692, 0x00003ab3282c54d9 }},
        {{ 0x000182d411742393, 0x000da0a73fa9cb0d, 0x000d6109219851e6, 0x00008f982edd3992, 0x00006166219cbf0f }, { 0x000cb8df03c5c3e4, 0x0009038d9a528dba, 0x00091e3c5a90b7d9, 0x00055b47ce8b1a0e, 0x0000a0f6e1eafa48 }},
        {{ 0x00012fcc62fd2575, 0x000c5328300129f8, 0x0001e7946f9a564a, 0x000bfcb511f63766, 0x00002838b2c1b003 }, { 0x00017bbc879d576e, 0x0002f5f29fb85da6, 0x000a07cb9d60a23f, 0x000afb8b772f4e00, 0x0000907e471fcadb }},
        {{ 0x000682238fc10ebe, 0x000cdb997aee3170, 0x0006e19313189010, 0x000b1dd9b812a409, 0x000009ce0a98c979 }, { 0x000b75351318c260, 0x0008ad6aeccb4bdf, 0x00096db6712057b0, 0x0003d3d336a34e9a, 0x000074e57bd9af57 }},
        {{ 0x000f1e0fa64b9ba1, 0x000a4ff1d7aadabe, 0x000fe6d896d5ce68, 0x00034116f65d3825, 0x00007bcae08c6246 }, { 0x000ac54f612c30f9, 0x000ceba33db72cdc, 0x00092fb725511bb6, 0x000d9573c017cbc0, 0x0000a29dfebf2fc8 }},
        {{ 0x00044c4ab2167d7a, 0x0005980b2369f0f0, 0x000d1f7b169f84ef, 0x000e52fc697d3a70, 0x00000371cf064a81 }, { 0x0002f2497b167080, 0x00020fd834a2ddc6, 0x0002698938f1ecee, 0x0008c649049e6b86, 0x00006463bb9f084c }},
        {{ 0x00070d2d35f711e1, 0x00024b6c64582934, 0x0006a1c3ddc10d9e, 0x000d436b836bb527, 0x00000a5cf78fdd97 }, { 0x000e797ecd201480, 0x0008832feeb1b77f, 0x000886ec00cee664, 0x000e180042bb75c3, 0x000057f8a6d620a0 }},
        {{ 0x0008abdd527350b0, 0x0003c2a753aebcc9, 0x000c308e0f53280b, 0x00044fd09daf9f3d, 0x00004eb1613ae15d }, { 0x000462e7e6da22c6, 0x00044318d168be52, 0x000ffec2ba55ff61, 0x0004c09afcf7c10d, 0x0000ff5b8008b9a4 }},
        {{ 0x000d3eea67fe0e27, 0x000638ba0cc3f125, 0x000d5f283fc1ad1f, 0x0006105d368dd485, 0x0000f512d2d3ea5b }, { 0x0004dc21e92938f6, 0x0002481e6602671c, 0x000f4f264a2357d1, 0x0007c19858f144a3, 0x00008d20d20e844d }},
        {{ 0x0001db64b60dcc0f, 0x0001eea521d03782, 0x000de325b7b9b75a, 0x000800aea802b8e0, 0x0000a207924704fd }, { 0x000e7c824e8b91d0, 0x0007ee970342f9d6, 0x000dad47c7c593a7, 0x0006b3baa93a10ae, 0x0000b3c50420ccb3 }},
        {{ 0x00044dd196d3754c, 0x000ae819966e7eef, 0x00071169c0771aae, 0x0001b5ed464ec4aa, 0x00006b0c6bc0965a }, { 0x000ca9d721b441b9, 0x000b8f1bb46b6a1a, 0x000350de796ed922, 0x00067beb41e65d31, 0x0000103b3fe2d35f }},
        {{ 0x0000c504bf8e9753, 0x00032a0fcff6a17e, 0x00023e05782f0bc3, 0x000b8f8e39970a3f, 0x0000a169715bac3a }, { 0x0006fd0c5dcd460e, 0x000cb5677fcc2f46, 0x0006febe5366b0c3, 0x0004c78aaef1a522, 0x0000e87aa65153be }},
        {{ 0x0000903c4cbe46bc, 0x000c0eb1bf4581ca, 0x00060c177519da3e, 0x0005066ace635e18, 0x00005345445cd5a7 }, { 0x0007197ca469b8c0, 0x000858948f214029, 0x0004225140fdccb5, 0x00087719031b49c1, 0x0000a2053b8e6576 }},
        {{ 0x000332d0815fb0b2, 0x0002b1e274983d81, 0x000e423eac0e20c3, 0x000b147d1da1a3bd, 0x00005049bb2f20a3 }, { 0x000aa2f407901242, 0x000b5a3aa6d0e10a, 0x00034aac28430b6c, 0x000c1136717db2e7, 0x00007894ad4da6bc }},
        {{ 0x000fe4ca540470ce, 0x000a79e7ace3fc7a, 0x000e95c40e7284f5, 0x00012800e9594eac, 0x000074e26fc1378d }, { 0x000ab186fc36c333, 0x000b556e6ee81c69, 0x000ad0d7933ec714, 0x0002305298f671be, 0x00009b93b73b2fc4 }},
        {{ 0x000df0fefced17bd, 0x000f66910b5be761, 0x0009dd130cd0a056, 0x0005c2188046b773, 0x0000009bdb10a2de }, { 0x0009ee86e04a5ad7, 0x00080a6bcfcd0279, 0x00052142deea631c, 0x0003ecdd9a8786c7, 0x00002b9a476e5434 }},
        {{ 0x000074d3de5dd49a, 0x00093ddc15c9f0ab, 0x00039220c4bffda7, 0x000fbf5dadf87a73, 0x0000e0074caa9da5 }, { 0x0006df62f75d4dc7, 0x000b20351a03776e, 0x000a43811d0fc2f2, 0x000c76a852bb1f77, 0x00003d9c33f35883 }},
        {{ 0x0005c11927d7db11, 0x000c0beed51275ad, 0x0006594e2f1817d8, 0x000e7addf5de47dc, 0x0000c87b4cd166bf }, { 0x000890cd3245ae9a, 0x0007800a73a294ea, 0x0009f531344057c6, 0x000c0b66a5f88434, 0x00007a6fab6ca15c }},
        {{ 0x0005929d26ae4d60, 0x0004ec170f4d3b44, 0x0008ca69efd95c9e, 0x000881838fd1a136, 0x00008822948519a9 }, { 0x000418f5d76d3725, 0x000ab6025b15ca6a, 0x0000ca2e4e6eecf5, 0x000e72c344b3c83e, 0x000017e3caa4a180 }},
        {{ 0x00041b28983194f2, 0x0000a8a213ed68b6, 0x000cecef298c37a2, 0x0005ba953860ae7e, 0x00002530dfcc9dad }, { 0x00085a2310dabfe5, 0x00053e101110f35d, 0x00018c30f6264efd, 0x000d17ff479c26f4, 0x0000836740a708c2 }},
        {{ 0x0002dda543915e7e, 0x0005f5d4b07e2b19, 0x000afac0fcdde015, 0x000a72d14d144c4c, 0x000034e73127607f }, { 0x0002db773ecd3720, 0x0000fa8cf03dce66, 0x0004b74758be1c61, 0x0007668c48b5ea96, 0x00009b6250934072 }},
        {{ 0x0006f31c01a3dfbe, 0x00044fb93f245674, 0x0009ea5f3bc92e2f, 0x000eafb87fd46c69, 0x0000f667b3eec5e7 }, { 0x00083cc1ece5f671, 0x0006a59888b219e5, 0x000789c00b155ab8, 0x00029106ce50ecc3, 0x0000f8a9874cc0ab }},
        {{ 0x0009a7ffecf51501, 0x0002d66a483bf119, 0x0005180ac17cfeb8, 0x000d4f928fa08a6f, 0x00000c4ee2e35856 }, { 0x000cba1424521b06, 0x000b7dbc4ae62da7, 0x000dba90db391823, 0x00031ffbc6cea00a, 0x00007680c0ab694b }},
        {{ 0x000ff63e0965f49b, 0x0003121208490def, 0x0002bd04ea6fe3be, 0x0007b166f1ca66ec, 0x0000bbe50fc3adb5 }, { 0x00010d8a6a559d7c, 0x000debe8b2610bed, 0x000f68d0b38a6296, 0x0007eb6b4b732ed7, 0x0000e692b4dee5cb }},
        {{ 0x00046bae92221adf, 0x0006498ed4de248b, 0x000d5bdcda959eec, 0x0004c5f13d7cedac, 0x00004112599c6663 }, { 0x000b84c4c8c12811, 0x0007372fcef7ad8e, 0x0007e92e99ce45e0, 0x000ba88866115160, 0x00004192a1db5e1c }},
        {{ 0x000f01e9b964c398, 0x0002d5c164e1214b, 0x000ce245d14a5d2b, 0x000e5d035891be86, 0x000058456cfb56a6 }, { 0x000ed62a7c96800a, 0x000d739b186ebc0e, 0x000596e2f14dd4cd, 0x000592e54d5c74c2, 0x000037f83089202c }},
        {{ 0x00069dfbf102d869, 0x000846f76b7618a3, 0x0004cd18647184a0, 0x000791ddbbff750a, 0x00000418f22b31b3 }, { 0x0008af82f7454899, 0x000196c7191a321c, 0x0008783dee978292, 0x0008738839edac3f, 0x00008a38ad413f8b }},
        {{ 0x00018d12afd54f4c, 0x00098d897ce0b2a0, 0x000fc2184a2fde82, 0x00068cefee6d7c87, 0x0000880946a4e593 }, { 0x0008157478782d1a, 0x000480801206e150, 0x00039ba30e4db2ff, 0x000d448d5e0e4245, 0x0000a2f55c162b2c }},
        {{ 0x000d7803581ab555, 0x000eab4820357c72, 0x000e419c5554fe07, 0x000fef5a1992039a, 0x0000d9875b456d0a }, { 0x0000ee0ea7058063, 0x0001b4dc77ad4d4a, 0x000858f1411d534f, 0x000f0bd5ed49a799, 0x000050a8f7fd0e17 }},
        {{ 0x00033d6d545551ce, 0x00020cfab7bd7712, 0x00031284f564f60d, 0x0004b93e76c87307, 0x0000270e7036f3e7 }, { 0x0004b690d19b6f88, 0x000ce26fc195bcc5, 0x0007b020fa92369c, 0x0007d06f32544595, 0x00005326f8c26b7c }},
        {{ 0x000a5acf73de976d, 0x000bf8a212306466, 0x00089df47218a462, 0x000994d43f51aabb, 0x00006b738c5c4d3b }, { 0x000ea5938bb50b53, 0x0006057d635b4d5d, 0x000d0e302e4e6897, 0x0000f9dddc3de68d, 0x0000247ceec78021 }},
        {{ 0x000446bc5330e6c4, 0x000131afe637294c, 0x0005e1616084d14b, 0x000a79fb501968aa, 0x00003b7dc534fcd3 }, { 0x000178d549b09def, 0x00061a696d6134bb, 0x0004ec90cb133eee, 0x000053a1a9361fbf, 0x00006af0a49f5d2c }},
        {{ 0x000b745f8fc43a35, 0x000024ba4df39156, 0x000ec4617678c3bc, 0x000a91c02566ce1b, 0x0000719b228fbdb9 }, { 0x000f385adab6fcc4, 0x00066f663122edfe, 0x000c4c603faed143, 0x000bd6d563190f9e, 0x00007f12983cad34 }},
        {{ 0x000f1627dec28dca, 0x000b7de3f642c293, 0x000851f9d3a9edf4, 0x0009b5b4620be088, 0x00004176cb7f6c12 }, { 0x000787c0dcf777ac, 0x000f413eefcd6385, 0x00061d44b6e6ae82, 0x000d1495e8f3bf62, 0x00007ad2517a88ab }},
        {{ 0x000193cecacc8855, 0x000e5a2d9a23e449, 0x0002bc7c9a4324de, 0x00070bc0888dfa99, 0x00001357e1a16b5f }, { 0x000cdf57355cc9f8, 0x0003a7d2d738fc52, 0x00076a8097379e6a, 0x0007681a56697b03, 0x0000e15f32fe6b63 }},
        {{ 0x0000441aedd1db0b, 0x0003920bd55659e1, 0x000b3331ee1ea4b9, 0x000604e1a7973e32, 0x0000b848523ff344 }, { 0x0006429320a3e8ed, 0x000324a045ad915b, 0x0006ee6d1990975a, 0x00014ced86c11b43, 0x0000ed1e5721bace }},
        {{ 0x0003cab91b752141, 0x000a5e89c92b2350, 0x000e676eda85d689, 0x0001c8f31a73bbd0, 0x00003841c3e8d0f4 }, { 0x000cd61ad4ee96e2, 0x000a42f1db3574b2, 0x000537e4eb45e77e, 0x000813c06485b018, 0x00002be90410b3b1 }},
        {{ 0x000fbec0cb6f0fc0, 0x0002d5a2ff9f403a, 0x000ab264cf5dcff0, 0x00007f70d8dc98d8, 0x0000dfe1570152b0 }, { 0x0008d1e1d5d329b1, 0x000c06480130dfa7, 0x00051130fb1fccc3, 0x000875ec0306dc2b, 0x000026a348354b2b }},
        {{ 0x000e31318e23bd1d, 0x0006f5a192ef7106, 0x000acd43006b8a32, 0x00067ef9e88afbd4, 0x00009533ab5f7940 }, { 0x0002a21cf2ee4b63, 0x00061b824f89b4e2, 0x000826214d380f04, 0x000d957a765a5dd8, 0x0000431ac8130d01 }},
        {{ 0x0009ea88981c66a0, 0x000fdee42a554f77, 0x000486443c22796b, 0x00047173d828983a, 0x0000d1238eb95f6f }, { 0x0000e913e1673541, 0x00064a7d065fd831, 0x0001a1798cdb2234, 0x000f61027f49af79, 0x00009e532ad5d547 }},
        {{ 0x0006b46e1f443280, 0x0001d14bc21ef518, 0x000ac63104730cf1, 0x000894c3be51c3ce, 0x00000536de024903 }, { 0x000be017a2ce58da, 0x000e02380041ef4b, 0x0004a6821c8b404d, 0x00079bad4d72fe50, 0x00006b9d35305a63 }},
        {{ 0x000b652edcec4a5b, 0x0009f09b389328ad, 0x0001f4b242d8c5b2, 0x00065eabf322fbc6, 0x0000dd15a5fc8674 }, { 0x000ea672143089ac, 0x000e72c03ec6627f, 0x000cc347decf2065, 0x000adfd0dde1b3b2, 0x0000a98c9af7b1ab }},
        {{ 0x0005cdafe7faf119, 0x0009ccb4f2999aa1, 0x0006a2fe40f56991, 0x00042ecbb38476cc, 0x0000d4790f63e31c }, { 0x000bf2c38ae9c943, 0x00018a4d935b7db4, 0x0003d0a6e142ea64, 0x000ec115b648b600, 0x00003a16278132b9 }},
        {{ 0x000ee635fb2543bc, 0x00020634442449cd, 0x000954c719bd8c67, 0x00024b182e0e9921, 0x000066a2b2a91885 }, { 0x00074ea6ed7740f1, 0x000314a18ee6e8a7, 0x000675f4394244fb, 0x00033dc08eed748a, 0x0000cce701dd1d5d }},
        {{ 0x000add5f69c02408, 0x000d61cf3480d4af, 0x0006c1ad6fb043be, 0x00005529cf0d8edc, 0x0000e6b784db99a8 }, { 0x0005a454130bbd68, 0x0002fad96f8acb1a, 0x0004c17786cf7d98, 0x0001fdbf4bddd49d, 0x00002c7d15e91516 }},
    },
    {
        /* digit=16 (1,2,..,64)*(2^{112})*G */
        {{ 0x0003c0440fd7f2f9, 0x000bdbfc360e25a8, 0x00048399d668b4f6, 0x0002439f4e642519, 0x000089f1fa0b9870 }, { 0x00097b3d282ee427, 0x0008cf1d720281f6, 0x000e67baa329978c, 0x0002104e72205910, 0x000061a43a34e6ac }},
        {{ 0x00071279a45648bd, 0x000de9afb3ff9ed0, 0x000b10e3d7b7d810, 0x00004e74928c2e0a, 0x0000f66bf52858f1 }, { 0x000e9fccc7111eb0, 0x0004f6601fbe8556, 0x0007f13a84fd868d, 0x0003acc0b3ee394e, 0x000011bb82692f56 }},
        {{ 0x000f57841f3144bf, 0x0001f9314092ebb4, 0x000cf3369fecda45, 0x0008e3164f17256b, 0x000033b176e2d462 }, { 0x000333e4efa6df27, 0x0001d05e708d8553, 0x0001d0bddee0a802, 0x000f8c7d5856aae7, 0x0000603c90846016 }},
        {{ 0x0007512e2eb6a7c2, 0x000b8dedda492f81, 0x000a60d843d6851f, 0x000eb594cc11a3ae, 0x0000e9fde037b87d }, { 0x000cf109a400a483, 0x000c6f3177bf8b6c, 0x00093bd6f59a5d1e, 0x0006c915d232ea4b, 0x000066d0031672f4 }},
        {{ 0x00008a54ec2b1382, 0x000f888a1f258972, 0x000f812b7fecbc4a, 0x000e06b5c7ac6961, 0x0000e7ee6a3486dd }, { 0x0003eeb9a9acb8ed, 0x000811978660710a, 0x00067f8f391f11c9, 0x000b760a50cf70a9, 0x0000a64a54a69740 }},
        {{ 0x000869d0d5ac68e0, 0x0008233169bca968, 0x0003a53cda259d70, 0x0005a1a9404d286d, 0x000088e4951616f4 }, { 0x00052a733ab82011, 0x00026e5d0150d651, 0x00030ee1a18ef179, 0x0006e1c49a92e250, 0x0000660861970a58 }},
        {{ 0x0001843074252dfe, 0x0003f588434a920a, 0x0002c09cd3518f04, 0x0005fc19ac0af8e9, 0x000065ea1ee67b6e }, { 0x0000581fc169a790, 0x00004aa5447ab801, 0x0005b7021d9fcb63, 0x000998a9ee5e5a32, 0x000086ce09c0bdfe }},
        {{ 0x000b0da855c77138, 0x0004d7080eb24a90, 0x000d3d8064a79971, 0x000e895b87bedfb4, 0x0000ff2e824b81fa }, { 0x000939b48c4a16ef, 0x00015dbde314c709, 0x000bd2f933b9e136, 0x0004c0d9ecdbf1fb, 0x000042e460791f67 }},
        {{ 0x00031e066075581e, 0x0007aa7cb0c9c8cb, 0x000fcea630fbe552, 0x00099ab0f75105f2, 0x00001f912867f6e9 }, { 0x0004564766700a62, 0x0007075d4b6d45a8, 0x000cde96422fe35a, 0x00087645bd363f3a, 0x0000ada1862cd2d3 }},
        {{ 0x000f0a0c47a9219b, 0x000f3cc37107c78e, 0x000a68021e1e77af, 0x000af3d45acff3b6, 0x0000e6dd110491f2 }, { 0x0006145e628202b8, 0x000de5176b6028e0, 0x00041e71b471983e, 0x00035f755c49ec67, 0x000098d7591e8ff1 }},
        {{ 0x000bedcf6b007f50, 0x0000d07e75746b54, 0x000d781ea953336a, 0x000b0a8e21c1e1f6, 0x0000184b09b48421 }, { 0x00054ff226110b80, 0x00069c13d7a4c8a9, 0x000bd88ff53a5a2c, 0x0002edbbc2748887, 0x00009a72c0defd99 }},
        {{ 0x000775780f973b43, 0x00047b197c3dfb92, 0x0000a7a109452cf0, 0x00079c39255a549b, 0x0000db1d2d13d82a }, { 0x00002f1085a734db, 0x000a5b8e1c9d7c80, 0x00013facbcd0a776, 0x000350fce8c69dae, 0x0000dfeeec7f75f2 }},
        {{ 0x0000deb4ca3a9977, 0x00053446d3fea556, 0x000693a2509e138a, 0x000c97b7db4ae9c8, 0x000056217a7b50ef }, { 0x000b113eaae38142, 0x00096947de6c34cb, 0x0008898319d2c511, 0x000be485b4ee6825, 0x00002fb9bee4a336 }},
        {{ 0x00009ed751690d07, 0x000fcb92115b4c9e, 0x0007a869f38018bc, 0x0000b109a7c163d3, 0x0000d25180d0c741 }, { 0x000626f43012e59c, 0x00047fa26f516cac, 0x0006c84c3b619391, 0x000580cb4753cacf, 0x0000d99e17d2ba70 }},
        {{ 0x000d31e20e626aa0, 0x0002899a7b56eaf9, 0x000b97bd91088d33, 0x00016ea8600c0e52, 0x00005520570c169c }, { 0x000e6b9cffa1bae0, 0x000b32d3ed689748, 0x0007ede2eac48130, 0x00023baefe44bbc0, 0x00004454704b6e6c }},
        {{ 0x000c8ac9b3f30faa, 0x000821f711b0eb9e, 0x00078b7fefa0a379, 0x000f7e8dbb905f2c, 0x0000d9b3674355ae }, { 0x000e568622b92879, 0x0005fb6b40a24474, 0x0001518d75018f42, 0x00065c23f60121fc, 0x0000b6c0f8efac61 }},
        {{ 0x00029b6e563d4ca5, 0x000090d196aa5a1c, 0x00064da647a59d2e, 0x0006673b7e0df8c7, 0x00003ebfc59c82e9 }, { 0x00077c02963ba3d8, 0x000dc8b711484633, 0x00032cc947dff086, 0x0005fd3e4b5738de, 0x00002458f4cf0a1e }},
        {{ 0x000f18fb2162c8d8, 0x00018b92bb39c1fd, 0x000c06efdda4112d, 0x00032e00514373f2, 0x00001c606faff29e }, { 0x000b75dfb7307746, 0x000394242b3523f1, 0x000d6c807144512e, 0x00093377ce958532, 0x000075bf8f2eff5a }},
        {{ 0x000f8237e184aa56, 0x000a1c92d7d1413a, 0x0004e8b2bdb60c3a, 0x0005e6de867e2d6b, 0x0000a224fd131823 }, { 0x0003b73e5c38837f, 0x000e0dc84687cee2, 0x000181fb2daced27, 0x00066f750c5ded6b, 0x0000dba3169b9090 }},
        {{ 0x000e01470209ad8f, 0x000b4183037bf522, 0x00054ac5ded5224d, 0x0000656d7b61e6db, 0x0000906aa9885eb2 }, { 0x0002c37583833f85, 0x0004a9a360aa449b, 0x0004c69fd7852936, 0x000bb90efb70cfd0, 0x0000ef760d62ea7c }},
        {{ 0x00045a8ff05fe573, 0x000ccdcf50225f97, 0x000c3c869b135443, 0x000c34c64c52e175, 0x0000b6e1d4a6ca70 }, { 0x0007a71cdf35f6f4, 0x000f1a358321bc3f, 0x0006141dc268fae0, 0x000f35c58b1732be, 0x0000235d7150b4b3 }},
        {{ 0x000eeee592f8a112, 0x000413e894c5bbba, 0x0008bae94aef8e2b, 0x00074b0446915c7d, 0x000077f05c59725f }, { 0x000df69b21ac3da5, 0x000f813781354864, 0x000ec8c0d40ddda5, 0x000f943f53a275c1, 0x00001739b2d5fa33 }},
        {{ 0x00040e9f29ec6f84, 0x0002e502d0f39aaa, 0x000a7199ba1db519, 0x0007170fa622118b, 0x000062772036e611 }, { 0x000754b54fc3aaa5, 0x000aceb22b57f0fd, 0x0008a72baefab7bf, 0x00045f0fe849a33b, 0x0000513d3bc11f59 }},
        {{ 0x000b33f80501f2c1, 0x000fa22236f4a982, 0x0008db3408b65b08, 0x000a7b6740b0c59e, 0x00004d2185638a58 }, { 0x0004d80f0e06a637, 0x0005b9532e6c4662, 0x000f29818d65accc, 0x000b5facc68956d0, 0x00005c86484c910e }},
        {{ 0x00093cca39225886, 0x00097fb988baf015, 0x000d82749c0eab29, 0x000f43360e6331a7, 0x0000dde7d468f9e6 }, { 0x0008a7318da83916, 0x000968d265674566, 0x000957a433029011, 0x0007db2f640fa030, 0x0000ecd715314932 }},
        {{ 0x00048c5860d99716, 0x000ba4efc4d31579, 0x000197676589c678, 0x0000e5b72b56f8ab, 0x000013465a13e776 }, { 0x000bbbc6da512eeb, 0x00072f00ebd4f086, 0x000ca2c6ccb3e2fc, 0x000b072d03e5168c, 0x0000f06060e5f45f }},
        {{ 0x0005959cbe186e9c, 0x0003886605daaa67, 0x00074604cfe0af51, 0x000b65f8946fd72b, 0x00007a875d7ef388 }, { 0x000a73bb803de3a1, 0x000fe65c8a588eea, 0x00060d978e1d0473, 0x000b6bcf1368c92e, 0x0000847dbedc4d5b }},
        {{ 0x000eb1b6c88eb6f6, 0x000b10d6457d188b, 0x000accc881c36490, 0x000166a1386eda86, 0x00007a87d33e3d7e }, { 0x000eb5009735b4ad, 0x000768799240a9f2, 0x000f48ac891ad54d, 0x000b16c3a1bbb384, 0x0000158d9593875a }},
        {{ 0x00040b8b0c182e20, 0x00089c04a7d2caa5, 0x0001519f1783ee26, 0x000194e8f6f3fb3d, 0x0000f047804535e7 }, { 0x0007a5b479c3334f, 0x000ebc905e214f55, 0x0009fefb01bee293, 0x0006b190b9a74075, 0x00007399d8ab170b }},
        {{ 0x000505be1efb2b17, 0x00031e726bfe65cf, 0x000b7d8b04e8129f, 0x0003ed02e097e391, 0x0000ca6e619f0f6b }, { 0x000d838a95ee8071, 0x000cda1e3a3035a8, 0x00060f518312713d, 0x000e396f26431ebf, 0x00004e5e1383c630 }},
        {{ 0x00083bf494f1922a, 0x000d07f650006f01, 0x00033885071d3899, 0x000678396dfd7dad, 0x00008e6b9b5a00db }, { 0x00061e1a6d262a4f, 0x00090434a7bd989d, 0x000bcf24f627dcab, 0x00019a63e505385c, 0x0000162da51b10d7 }},
        {{ 0x0002b23a10647092, 0x000b4c64a8a3d626, 0x000fc1c509c1f5da, 0x0008752338469c4c, 0x0000592d71f92d24 }, { 0x000e65e5e66ab21c, 0x0007ab4a63d1a4c4, 0x000a2eb259587d83, 0x000fd941c454e7fe, 0x00001e808c047aac }},
        {{ 0x000985fe38c5fb2d, 0x00031a1ef74c45bd, 0x0005d4ac86cd7bdd, 0x000df50a92cfbfa6, 0x0000f38525a7bc74 }, { 0x00027e0dfe332909, 0x0005ffe5a0148b4a, 0x00068afa056fbba5, 0x0008e20ce6f9a55c, 0x0000c1b3d8c7c69f }},
        {{ 0x0002ab75b80ff311, 0x000335212b4603ee, 0x000088c4f75c8654, 0x000b6c312a876c55, 0x0000ec0c5bfb1759 }, { 0x0009a79cb22aa461, 0x00039761a23ff976, 0x0001f38e005de83b, 0x000defe22c613899, 0x00008b2f4e8bfccf }},
        {{ 0x0005afec02a30992, 0x000a30d594cc7e13, 0x0002fda0bf775275, 0x000f5c74a3079ca6, 0x00002d8fc95c11f7 }, { 0x0008853e7d3553e5, 0x00061f406ed2c6ed, 0x0004a50e5bda669a, 0x0007d947e58da173, 0x000046a193aa855a }},
        {{ 0x000e471b95b01638, 0x000188da35fcd5fe, 0x000b5b15af55904f, 0x000cb71628965df8, 0x000058d0fcc4ba62 }, { 0x000d74938578142d, 0x0006ec80a08fb758, 0x0004f469a20c56dd, 0x000db2288704958f, 0x0000e41307922cc2 }},
        {{ 0x0007a222bccd2e68, 0x000ed378c9049f43, 0x000343f0b5f94e09, 0x000b4556b5be948f, 0x00000a6efbb7a0e9 }, { 0x0004a461c9039832, 0x000e8bb8809bf618, 0x000b8ea28fc4e42d, 0x000e3db1c46f07fb, 0x0000437a4a4ab1e4 }},
        {{ 0x000e03fdde9732a4, 0x00038e6d9278d982, 0x00036ff876027328, 0x000107dc324d9481, 0x0000e4b863ced76e }, { 0x0002045a740a4e7f, 0x000871947b533c0b, 0x000e7a9f65e10697, 0x00067ac8f1a8bc73, 0x00005706a2f93e11 }},
        {{ 0x000fb693dc0269f5, 0x000e0afbd76e195d, 0x000c898214343849, 0x00079e0cb478dd1a, 0x0000bf92ae14b76c }, { 0x00050e61bd7654af, 0x000959651dbf1a78, 0x000e839485a58314, 0x000470b9acab4b4e, 0x000053da0f7c685d }},
        {{ 0x0008bb92b4197f71, 0x000bfe3dc09508b4, 0x000f2b75e3a19e68, 0x000e99255faf6d2c, 0x00008a43244b37c5 }, { 0x00012686517eb89e, 0x000b6845c7cf8929, 0x000944af69cde6fe, 0x000877f69f6cd7ac, 0x0000c9b055220817 }},
        {{ 0x0008da07e4a739f9, 0x0000013b78c558ff, 0x000acfd5d2c490d2, 0x000b770d343c0bae, 0x0000eee5df52f0ad }, { 0x000001122fea8b0b, 0x000b99da8a90ea83, 0x000b53a188f1cbeb, 0x0004dc77681c1ff2, 0x0000b1f2bc907c40 }},
        {{ 0x0000f5091f08fba5, 0x000efbe6718e4488, 0x0001fb4964f4d8db, 0x0002b10ef246c148, 0x0000fbde6b0ad954 }, { 0x000f886488030d36, 0x000f82f724bdd38b, 0x0003c4e9843fc28b, 0x00002bca902e915b, 0x000027da8403820a }},
        {{ 0x0002c8cdd5a84a71, 0x00075a6c93f72d63, 0x0002980f33151397, 0x000628017d5f75d1, 0x00002b6fc243ca3f }, { 0x0001bcaa034d4c38, 0x000e1927023c4994, 0x0006f72f8efe26d5, 0x000c24bf0a8afe5d, 0x0000af62213dc85b }},
        {{ 0x000bbee6e35723eb, 0x00074f98d2668d5c, 0x0009af1932828c67, 0x000a2e3110715281, 0x000018a72e0098ac }, { 0x000b8bfd572fd9c8, 0x00047e80cdb9808a, 0x000652b2d5bdd6d6, 0x00031881817d6e06, 0x000045b2eb036f1d }},
        {{ 0x00010e2fa9c3473f, 0x0000b59513aed8bf, 0x000c403da0d07f42, 0x000fb7682f77f3b6, 0x00008f8e17dae5ca }, { 0x000f98a2d0c72d35, 0x0001b2262a2f201c, 0x000f81558fe0288c, 0x0007a54a84d59ea4, 0x000094d0140957a6 }},
        {{ 0x00034de6ed85c687, 0x000eb39720943c28, 0x0004b642d2e37390, 0x0002c014aba044cf, 0x00009c5d21883610 }, { 0x000c0c6eeb0043f2, 0x000fc9a5f25bd125, 0x000a136e4386c89d, 0x0008c19769259cea, 0x00001522a51ab777 }},
        {{ 0x0008f8514711e5e4, 0x000de0f5606b81d7, 0x000494a0cd9e404c, 0x0004cf551abd7b37, 0x0000aae61990ee74 }, { 0x00069b8f8064759f, 0x000d4f0f30c3282a, 0x000000ed938b30a2, 0x00022f77f0da8ac8, 0x000034f8a1881bef }},
        {{ 0x0003840fe88767bf, 0x000b8176add85450, 0x00066f408b7e5e73, 0x000da5e771b71cb6, 0x0000d35c39b650e0 }, { 0x000174cced9e5822, 0x000b8d51ec699775, 0x0008df0a9a391539, 0x00071f40cae243fb, 0x0000e2156e1d8e4a }},
        {{ 0x0008e419d542ad83, 0x000d666e58ba37d6, 0x00020f33198855f5, 0x0004e44bffdcc8da, 0x000059353e915b41 }, { 0x000e22cd180b3421, 0x00018f402a403801, 0x000f01c6aaa43566, 0x0005abda688940fa, 0x000095b351fd3384 }},
        {{ 0x0006fb438a17556f, 0x000f9ce59ffc3e20, 0x0005bae3a15dcb2f, 0x00027446360aa1b5, 0x00007cd6cec33e6c }, { 0x000544607e7d51cd, 0x000b1c4500d8e20f, 0x000c112c7a91b2e3, 0x000e9affd1ad9095, 0x0000a60766068749 }},
        {{ 0x0007d49c6cff6daa, 0x000e9b1a20ef103d, 0x000f93d1bb0e184c, 0x000346aa0ad85912, 0x0000118d622a8970 }, { 0x000916a1f7d47a14, 0x000d88d20bb569d7, 0x000275e91a3c185f, 0x00072f067bd4b250, 0x0000837bba3c6297 }},
        {{ 0x0002546a940f5586, 0x000c37d53a4f5668, 0x00041560fd7cc5a3, 0x0003f20218973466, 0x0000fc85e2f5dff5 }, { 0x000a16ad6208772e, 0x0009d19c61ac1625, 0x000c51965fdc16c3, 0x00040bc4352f8fa9, 0x000086c395e5c1ac }},
        {{ 0x000644324f2ef32f, 0x000897f4faed184c, 0x000c89e5c21081c9, 0x000d382fc5f236b1, 0x0000323cdb72b7dc }, { 0x0000ed6d6bc3b8ef, 0x0002e595c5d84462, 0x0006f2c0ac0dca60, 0x000ecadc7c020807, 0x00002650966fe977 }},
        {{ 0x00067238b80e6e38, 0x0008219b6efcffc7, 0x0008214e7fc3fae4, 0x0003bf8bc60f12d6, 0x000005e6dec70c8e }, { 0x000a94dc7004d5b7, 0x0008221501b5d92f, 0x000ec80e9ac7e6f1, 0x00043be433279e3d, 0x0000bdaa02065868 }},
        {{ 0x000ae57d7389dc14, 0x000a01a530ccbbd2, 0x0006bf9cb7b1e9b6, 0x000fd790c87c8214, 0x0000c004262ed87b }, { 0x000d81a25c341115, 0x000e2bf3042a6dd3, 0x000fdbe2d8085f66, 0x000b98267b1a007b, 0x000031f1387a3dd8 }},
        {{ 0x000030f0a67c1e0d, 0x0005af8ff81578e1, 0x0000801453d6f902, 0x000e79d6f7862215, 0x00001e2caf4c2888 }, { 0x000a817bd311699b, 0x00004151f0bd7390, 0x0007dc5d42b22089, 0x000fde3982d7989c, 0x0000a862ef2d9f7c }},
        {{ 0x000a570c2107ac75, 0x0006330b0f2ac952, 0x000a80c9da2db964, 0x0007d589a48d0995, 0x000066ec9369c06f }, { 0x0005135ae5c22331, 0x0008bfd4e86214dc, 0x00020a4df9b2cf0a, 0x0000da66e2d014ee, 0x0000b652f2880073 }},
        {{ 0x000ba378872d98f6, 0x000313ad4f3049fd, 0x00030b64fbefd2cd, 0x0002deffdbb8e989, 0x0000f27faf9b53e7 }, { 0x000726e0841b2012, 0x000cea65a72f2cfd, 0x000ea4c375be3f8e, 0x000671a371208bfb, 0x0000a839bfdbb74c }},
        {{ 0x0003b681d32f73b1, 0x00029a5061f16587, 0x00016b5650dfe8e1, 0x000c99fdec0016df, 0x0000ea7e61216b53 }, { 0x000fc9eebda36ba8, 0x000c5c43ac343c03, 0x000d45a775638b98, 0x000ac46d49c847e7, 0x0000844582f5d365 }},
        {{ 0x000647b7837876bd, 0x000ad727fe02299e, 0x000143077b95970a, 0x000cf08be6335ccc, 0x0000f0ca4f1d1bce }, { 0x000627278ca4b059, 0x000f0de02cafb8aa, 0x000f569bc5401d7d, 0x0005e2982299bc2f, 0x00009199192015c9 }},
        {{ 0x00058f5c2f9e22e4, 0x000b3337933e47b6, 0x0002c8e707d309be, 0x00021299ff4ff6b1, 0x000037dd6c45dafa }, { 0x000ca95dd4570da3, 0x000b377c95b3b3ec, 0x0008648e592b39a9, 0x000b6d929fa04426, 0x00008591df48c6ec }},
        {{ 0x0004d3be78cdc4c0, 0x000cba42333fc4c2, 0x0008d262e3383d84, 0x000588b2e274f8e6, 0x00001215944a8491 }, { 0x0006a7c58ef9ec49, 0x000af58ca14f530f, 0x000264f068c28b49, 0x000f57662a6e7f16, 0x0000a687c108833c }},
        {{ 0x000f323c80b194e1, 0x0005603f9374ab69, 0x0004cf2404c4a6aa, 0x000bd8c50864f919, 0x000016461ee3c770 }, { 0x000be0fe51bd269a, 0x0000dae3f046a9a9, 0x000e43a90548ed71, 0x0008371a670da183, 0x00007e7cae92ca95 }},
        {{ 0x0003c2f435522de5, 0x00099b6cbc613e57, 0x0004d5bbb1e76fb3, 0x00050f9d533131d5, 0x0000500dd4c4695d }, { 0x0006451f5801b985, 0x0009f93a4a375fa7, 0x000c41eea66a4aeb, 0x000b4eccf5f06787, 0x0000be0cc26ec73c }},
    },
    {
        /* digit=17 (1,2,..,64)*(2^{119})*G */
        {{ 0x000274dcf1b355e8, 0x000d5ef694db7e04, 0x0007778800d4cda3, 0x00062f4e1edca878, 0x000092c279274250 }, { 0x000be2b2abb3de8a, 0x000b9b86def6d1a2, 0x0000bff865bb4348, 0x0002de558d25b167, 0x0000a14c44f5e051 }},
        {{ 0x000da4a7963ed790, 0x000cc9dd111f8ec7, 0x000d3a203fc9ff36, 0x000619a51bcc33f8, 0x0000c3316282fb7e }, { 0x00058ffec2ca8e3f, 0x000cbe30bb1151be, 0x00071d53238e4f7c, 0x000c05f7854febac, 0x00007acf3890bf9e }},
        {{ 0x00074fc72746a6d0, 0x000c2fcd1e0cf9e1, 0x000da997ebc1b1ae, 0x0008e884edea75f0, 0x000051ef991ea87a }, { 0x0007fb62f575580c, 0x0006c77eb930beae, 0x000a76b4c76571cf, 0x000cc0f9b504cacc, 0x000049875ab2027f }},
        {{ 0x00066adbf6bb7535, 0x000ff9d29c4120ba, 0x000f1a65fd1f4042, 0x0009b85519f94391, 0x000007d911a8b098 }, { 0x0008234499aed7de, 0x000763c34bfca38e, 0x000fd1be8863128c, 0x00048439ce0f2755, 0x000013608ea8ba39 }},
        {{ 0x0000cbaad6fa0860, 0x0002b769d4c54230, 0x0000af867172d5b0, 0x0007ea0a19e56eb1, 0x0000613acac087c4 }, { 0x000d6254b0899ec5, 0x00036ded92a5e67d, 0x000b41d22a615f85, 0x00021935bca4979a, 0x00009335529dd4e2 }},
        {{ 0x000b6304a752fd49, 0x0007c13516e19e45, 0x00076a61d0ec2826, 0x000e9d5f656e2e84, 0x00006c970cc1cc09 }, { 0x000935381bb3523e, 0x00069363ab7e433f, 0x0000053ef767b2e6, 0x0002d839f1adea47, 0x0000b39a71be38a8 }},
        {{ 0x00035142c413720c, 0x0009f2d799e9a748, 0x000e25f8b46540fc, 0x0001df95a96239e0, 0x0000745a4815a75d }, { 0x000856e0e8e441ac, 0x000ba7aa2e8807fe, 0x0004324cf2b5bfc3, 0x000bac6f529f2c08, 0x000024c840287b5e }},
        {{ 0x000cb10d572b5962, 0x000ed9d918e49366, 0x000645a02c2b89f6, 0x000f43ab965035ef, 0x00005fc3fbf56b43 }, { 0x0007c7032b9ad449, 0x0000eb7a242da112, 0x000d3f646f3cae05, 0x0002e606b16a4d3d, 0x000066b08ba950ea }},
        {{ 0x000b92420a9f510e, 0x0001d18e4136a147, 0x000cebfbf0b51e6f, 0x0004b2e03bacf969, 0x000059ed4c55993e }, { 0x0008945e1eb1c203, 0x0001c78c64d565c0, 0x0000085e247722f2, 0x0007b021a9f969d0, 0x0000a495ad2457ff }},
        {{ 0x000fa003d16724c1, 0x000aa7d5846426f7, 0x0008a4dd404bee11, 0x0004afe48e7d09e8, 0x00004e388ffd7c0c }, { 0x000d40da8e2b1cbb, 0x000a64f17fd95965, 0x0000d88abe4cfada, 0x00066ec6a49e0e0a, 0x000085fa4175ed03 }},
        {{ 0x0002031717ee94cd, 0x000f705050bad247, 0x000d50a7310c0423, 0x000ee98a78918426, 0x0000a7a77c69c5b4 }, { 0x000fd7cb763dc643, 0x000939960a94120d, 0x000d09c6e3764d3a, 0x0009a67fd478a255, 0x0000291d53e1e931 }},
        {{ 0x00089bc84468e031, 0x0005ab4a595939dd, 0x00084fd839d2cf16, 0x000cd45120355647, 0x0000a31eb877381b }, { 0x00012d8643774d44, 0x000965d85a9184d6, 0x000b3e932a3180a6, 0x0003d448e562563b, 0x0000bf1cf2a46781 }},
        {{ 0x0000885735dbeebd, 0x0001ea7f64085e79, 0x000399222ddb6a70, 0x000b41488b230f30, 0x00007e9e9ee9f0f1 }, { 0x000cdbdf44d9a14c, 0x000b39a652f54f6d, 0x000382d24cab9561, 0x00054a2e365c8229, 0x0000b474dc1a9b89 }},
        {{ 0x000d03d7d3318572, 0x00049a354530bb24, 0x00077e0492176e2e, 0x000bd9f63bde3ef7, 0x0000086e3e2a72d9 }, { 0x000280da9fc53e22, 0x0002d9e43b6a782b, 0x0006ea07cbe66e70, 0x000c1cbd9216db30, 0x0000dfa49ad0403b }},
        {{ 0x000fd508e7dab215, 0x000b68a1635e7417, 0x0009f8e8f9bf57f5, 0x000c8cb30fe14008, 0x0000d09a8cbf4c40 }, { 0x0000c96743cb7c12, 0x0002019d607ad51e, 0x0009ac772f06a74c, 0x0000b7ba20f229c0, 0x0000c906956df251 }},
        {{ 0x00048234a455944a, 0x0003c913e538cd75, 0x000c398678aeeffb, 0x000c8cbffa7001f6, 0x00008153cadc4269 }, { 0x000cbd665cbd5dcd, 0x000d02d3d5b40458, 0x0005bdd2db3441fc, 0x00002b28bb0ffeeb, 0x000086864413478b }},
        {{ 0x000230378fd7d8e6, 0x0004ee0425830687, 0x000b81c6aef3b01f, 0x000e83c4802da2ae, 0x00008bd11eeccd2d }, { 0x00084d402dbd9f8e, 0x000f2b571e8e5cae, 0x000bcb68e4f6d61d, 0x000fcc72e53ab041, 0x000092f0d852e5ae }},
        {{ 0x0007fbf56c3f426c, 0x000c973f37f59371, 0x000db9e90739643c, 0x0004844fceb0f6c7, 0x00001850ac1be86d }, { 0x000ae327a65859cf, 0x0009cd9acf7d720d, 0x0007353f1fbfd5d5, 0x0007b6cef41fc8f8, 0x000076cef0bc0d2d }},
        {{ 0x0001699a1c4e7ed0, 0x000d065c48182672, 0x000b165821a51c51, 0x000ba5a9233aa189, 0x00000691ac4135d8 }, { 0x00081497d823b2a6, 0x000d19cc2d7a7fda, 0x0008f1b3a51aa96d, 0x0000e12c7e318605, 0x000025170752334f }},
        {{ 0x00044990c52f624e, 0x0000e891f5f82dce, 0x00067f1776d921fb, 0x0004105309a7f67d, 0x0000dff3f33f1de2 }, { 0x000830f09a7edeef, 0x000fe95af103e596, 0x0007257ba1c748f6, 0x000b5c9d3ece8aba, 0x00001dd530aeff14 }},
        {{ 0x000520304e191d19, 0x00010b488172d014, 0x000585df6db40bed, 0x00089199a8b20bd6, 0x00006e8ca44ed0cf }, { 0x000244c2cdc93ee2, 0x000f81f2b47f7ef2, 0x000839bb0272e506, 0x000261674e07f528, 0x0000069e938f3b6b }},
        {{ 0x0004b043a81d5e7b, 0x0009fc194d7d9b17, 0x0002537e3b6f2665, 0x0007b94ca5bafdd8, 0x00005587b44c1942 }, { 0x000143bdc27d5d11, 0x0009ea19e71734c5, 0x000e4b01516bbe49, 0x0008531b115d5437, 0x0000847671d73bea }},
        {{ 0x000037543bac95c5, 0x000c653fc5e89202, 0x00087a6599cca795, 0x000b2a908363bf9a, 0x00000d8e95cfd516 }, { 0x000badceac56e4f2, 0x000cacc625640b46, 0x000e20185d8f1a24, 0x0005a34eacbc1051, 0x0000d0e37c834129 }},
        {{ 0x0003505bb870eafa, 0x000d47c6b53f5f95, 0x0004c7c711e131be, 0x000710e1b6424659, 0x00009ecec42b7e48 }, { 0x000febceda1c9c8b, 0x0001361f2ba70b0d, 0x0008ab0f057922d3, 0x0004ebf6ad07e16c, 0x000067dbc750e1cb }},
        {{ 0x0000e4587f5ed835, 0x000a7dc53868c8b3, 0x0002077caf2733fb, 0x000c31964174311a, 0x0000356efdbcfd09 }, { 0x00003d37200e52c0, 0x0002a310811b3b2d, 0x000c3c910cca75c8, 0x00047b8036144d41, 0x00004f8461ce611c }},
        {{ 0x000602500ca3dbe7, 0x000f0f2b1f3390b4, 0x000c5fb3c266229a, 0x0000540a99f7a1b8, 0x0000cbac9856bd1f }, { 0x000289eb1fd8ddb6, 0x000c11c4711230f9, 0x000b6f851ea1a62b, 0x000082d8b058eb37, 0x0000dff1bd082c26 }},
        {{ 0x00009e4ea0d957a8, 0x0009eabd51b9cf59, 0x00020e62d30ee577, 0x000b61a74d2aa9c0, 0x0000dc7f6451320f }, { 0x0007e947df046bb8, 0x0004c966f32dbdb8, 0x00000319c35fe24b, 0x0004a5fe2672188a, 0x00007d6b8255d4cb }},
        {{ 0x000f11c4d76b7c46, 0x000e5391770f5a7b, 0x00004f3326ab38d0, 0x000ed2b2ae4d4d79, 0x0000e4ef511e46d3 }, { 0x000a61e28b58878e, 0x00065e542c4224c6, 0x000bda3bb1915d60, 0x000c95a78d6b4e81, 0x0000df4b98688afb }},
        {{ 0x000f0956569ac4ed, 0x0003394e6f1cd952, 0x000cde0f10e89862, 0x0003727647512f30, 0x0000d19eb33b73c0 }, { 0x0003d7563e475d5b, 0x000eccaa82262004, 0x000cd6e91d2b1111, 0x000b651be02c868a, 0x00008a5278bd077e }},
        {{ 0x000c20578305b14e, 0x0004fc7ab952578a, 0x00017ac7f5884f29, 0x000d5d242b5423df, 0x0000620247456895 }, { 0x000f807ef9d65a21, 0x00075eb21f4c77af, 0x0003ab31f4b5799c, 0x000dc92f11a79c3e, 0x00007300963434bd }},
        {{ 0x00011afff4ca371a, 0x000d03d778f22a00, 0x00085a437474b875, 0x000e1d47dfb10b2e, 0x00002489cd838b5d }, { 0x00011554104be448, 0x0008926ad124418d, 0x00051618d54f49c2, 0x00010937531c081a, 0x000071b18a1c2aa7 }},
        {{ 0x000d41f01735a335, 0x00073f833f6746cb, 0x00054167a0b39250, 0x0003a7ba921e46f6, 0x00006c09c11aae95 }, { 0x000d59e95785f38c, 0x000b834426ffb589, 0x00057781d8acae7b, 0x000a30f7f8055943, 0x0000f1dc76c3a6d4 }},
        {{ 0x00008d0596f30cdc, 0x000bb1c8ad4cf4b4, 0x000fef6e8a21fea3, 0x000272fa40344885, 0x0000efa2f182a572 }, { 0x000acd518b235195, 0x000f9a498c364b09, 0x00046a51392ac475, 0x0000bfb02a4aac4a, 0x0000e0426618d525 }},
        {{ 0x0009dd82ce983ea9, 0x000bf8fd06f56c0a, 0x000d3524f6a44c8b, 0x0005bb79da48af70, 0x0000dc57803535c6 }, { 0x000cdada016942b8, 0x0001263a0ef18d34, 0x0004f091c9bc1fc1, 0x000b1c5039fb7ee3, 0x00008fc41c7e9b42 }},
        {{ 0x0005a0cad09fcdc8, 0x0006f499355a244c, 0x0005898086d8b232, 0x000e727cbe77fd2b, 0x0000f5438a6a25c6 }, { 0x000338ecde1a98b7, 0x000475cef8118650, 0x00089b6cdf5ef5b6, 0x0001cd6bcf52813e, 0x0000ef9cb64dfe26 }},
        {{ 0x000e182e702cbf4d, 0x000093774921592b, 0x000cfba36ec1cce1, 0x000dd0759b4f1261, 0x0000c2c3bb484721 }, { 0x000f11d5488868ed, 0x000c37cdfc50573d, 0x000983acb7fc4b3a, 0x0004278d39ba1afd, 0x00004a4526ea82ca }},
        {{ 0x000db4207965c6e3, 0x00020ee02a2404ab, 0x0003635469f7a3cd, 0x0003dcb71497406f, 0x0000db415a568fea }, { 0x000e3be64146ee7c, 0x000270fd88284423, 0x0007be7a06dae484, 0x000b58415283add2, 0x000031007377bf1f }},
        {{ 0x000b74b0ed03932d, 0x00050c1e038a6759, 0x0002f0e7b233a87f, 0x00079ff53729deff, 0x0000c01ffe0be7a5 }, { 0x000184c1d9aaf43a, 0x0006c23dab407e5c, 0x0004b52e7eecb1c1, 0x00033ff2ccdbd361, 0x0000098af3b980d2 }},
        {{ 0x000904bdbb6109f8, 0x000571981d3a80f5, 0x00016fdb66776216, 0x0003fe8c1fde0c84, 0x00002e8025758320 }, { 0x00011469a106add9, 0x00044d1101b68053, 0x000c189603a82a4e, 0x00069678743c223b, 0x0000a9746fe0a073 }},
        {{ 0x00021ed0b2fc0d23, 0x00093845e43fc616, 0x00057eb4daa4b57c, 0x0000afaa8f328237, 0x00006ae9919db4b5 }, { 0x0003856ec86c297e, 0x0000fe1ad9906763, 0x000b234fd19d2eee, 0x0000f4401d51de67, 0x00002462d036fc09 }},
        {{ 0x00033a00db254686, 0x0007615be6cdcde0, 0x00039bd26eadffdc, 0x000f7aaa61e603f3, 0x0000199ba172e7ea }, { 0x000eef4c32c845b6, 0x000ee09f7bb7fd76, 0x000dd73dccc40b89, 0x0003a2c9570f8781, 0x000026152e934da0 }},
        {{ 0x0000ea7d6756508f, 0x000a13b2f7e85c33, 0x000140ae12343058, 0x000e81db55fce462, 0x00007980f96d8f6a }, { 0x000a553d4d7f0b2d, 0x00088f4e51ab9a22, 0x000a83b13f2b7013, 0x000bf6fa60587c98, 0x0000a508840aeef8 }},
        {{ 0x0000b460d07f52d4, 0x000ae364da29ec28, 0x0003391112464836, 0x000853947dbdd5d1, 0x000061c41be19aad }, { 0x0005f292a360f3f1, 0x0009babf963c31e5, 0x0003765aa05a3770, 0x00003d58767f9f63, 0x00004a15f588fa9d }},
        {{ 0x0003283f79a5eaca, 0x0009eb2cd21ab3de, 0x0001e06b9418e3f9, 0x0001670f9e504f02, 0x0000446de46c5aff }, { 0x000101dc7671519a, 0x0009bf6226bf99fc, 0x00034d2741f38239, 0x000ba7c2de835427, 0x00003411b6f79f40 }},
        {{ 0x000e01db0d46ca69, 0x0004bc2255bd6178, 0x0002de28b6473c59, 0x0009d2a876c96969, 0x0000307faee6531b }, { 0x00082ca080d00c87, 0x000a173893a55595, 0x000aef7740c863b7, 0x000c2aef87630f04, 0x0000b32db7ee850d }},
        {{ 0x00037f9c22a738a4, 0x000b2e10b1894a02, 0x0001b49ccc9d8538, 0x0007871b59203400, 0x00001570db64de21 }, { 0x0009ed518775904d, 0x000a62f4470e2c05, 0x0000fed0fd0d9912, 0x0001e0763b725f7c, 0x00008b2ac0c6733c }},
        {{ 0x000b0b081e23ebf4, 0x000ff16d0115d4d7, 0x000f585e0d7bd2f4, 0x0009316415180b12, 0x0000ddbe5f1eab2e }, { 0x000974e2b1e03fb6, 0x000f48abf4f0dd06, 0x0006e3b043e48d7a, 0x000b8aefe99b289c, 0x00005cf08248acfb }},
        {{ 0x000d737e72363c9d, 0x00021e206ddd6572, 0x000102802afaff9f, 0x0002f4a58e7cbdd1, 0x0000cf910137f5d1 }, { 0x0006f52b1e708142, 0x000696286ec37766, 0x000d2d6a7c82ef02, 0x0006dc73398aa649, 0x0000cffe04031902 }},
        {{ 0x00032721221eeb7a, 0x000defbd803738b6, 0x000781c149828000, 0x000c99a9291aaade, 0x00009626df3ef586 }, { 0x000571b8055894f6, 0x00032f0862c51290, 0x00092dafec9b2579, 0x000acd28f0a8f79a, 0x0000403590545ef7 }},
        {{ 0x000f5f638ebe18a2, 0x0002adec95207114, 0x000ced2c21311398, 0x000e6511455a9e4e, 0x0000ba02dffbaae3 }, { 0x00017596079bc6ae, 0x00070d32bf9341bd, 0x00007bcd3fa21e63, 0x000a719c5d63c132, 0x0000589ce7628195 }},
        {{ 0x000076bc4095656c, 0x000b41ff9e6ed353, 0x000536bfd61a4343, 0x000f2aef7b7b5ee4, 0x0000916f72ab7c03 }, { 0x00070cb65edfe066, 0x000638c02dec312a, 0x00077c8115197531, 0x000bba2c9270de89, 0x0000cc65c87f3d73 }},
        {{ 0x00032e89869168bc, 0x000050ecfe57bbde, 0x00055272392f1fc6, 0x000b3abbd82c7b65, 0x0000528c02084684 }, { 0x000152ce2cb419fd, 0x0009744d11a5529a, 0x000182d52bedc6a0, 0x000877a0ae964ed0, 0x0000f927ef680070 }},
        {{ 0x000871bfa5e6e0c7, 0x0008ca48ec07649b, 0x000f8d2c14ad8a64, 0x00083ea1018b4c28, 0x00004818a1be6e94 }, { 0x000b0cdf9b0c7f86, 0x0008ce59e2e3c9b3, 0x0005099db15ba9f1, 0x000d40a06fbf250e, 0x00000ac8ed941db0 }},
        {{ 0x0006facb0812925d, 0x00042eaf8c511875, 0x000c1d23829fdb64, 0x0007ee64505bb67e, 0x0000b90376a129f5 }, { 0x000841357c4ddd34, 0x000d3e98f6229e49, 0x0003ba2e8493a602, 0x00056697962d103f, 0x00003d04cfbd7b62 }},
        {{ 0x000750cc3c1a7b0a, 0x000b159c34e67058, 0x000af889dc10e012, 0x000957b37c3c7180, 0x0000968c82b0c089 }, { 0x000d86dbd34a175e, 0x000689db7275d7d1, 0x000b10d2d7166c05, 0x000b45693ffa0168, 0x0000abf693ace924 }},
        {{ 0x00071b360f0679b7, 0x0005467e28ef5ba5, 0x000973fdab5aa7a4, 0x000f206202c9a469, 0x0000e0be2b8f2417 }, { 0x0004bed023283151, 0x0009958101c1d454, 0x000dd3c9342fa56e, 0x0005c83466651788, 0x0000f38d09a711f7 }},
        {{ 0x000343b5fb200f96, 0x000952d33da525d4, 0x000dcfd099dce93b, 0x000b1701df193678, 0x0000ef95773ba9d4 }, { 0x000484d4e14aaebc, 0x0006771638127137, 0x000d82dc48b25dc1, 0x0007325d747bf6d2, 0x00005e1c9c82ab72 }},
        {{ 0x000c8d3645c866b0, 0x000beea60dbac1fd, 0x0003fd796cb1baa7, 0x00063a35b23d8c48, 0x0000e075408e6a88 }, { 0x0007c5425467d1b6, 0x00043fab12177b4c, 0x00038b881b47a89d, 0x000d9800bf79bf46, 0x0000a2c54416f53e }},
        {{ 0x000a0d41b10e0cdc, 0x000c447e5911a766, 0x0007fa71dd36a5ba, 0x00049cf4261ffa5c, 0x000079b9ae6dbe81 }, { 0x000e89e390495427, 0x00042f5cc30d3577, 0x0000ece8db061fa0, 0x000e6ad61be9adf8, 0x0000aae7a36a6e46 }},
        {{ 0x000c6cb4fe4e3e7e, 0x000afe5facf69d43, 0x00060b087f4ea2b9, 0x000a4ef04363b7e7, 0x0000a37a66bc1464 }, { 0x00008b945060128f, 0x00034394008e1f2c, 0x000964139e1c25c7, 0x00079cf065e9a85e, 0x0000f97bbcc0ba07 }},
        {{ 0x0003879cbfbabf2e, 0x000922c89f71f0f1, 0x0001ebb607012381, 0x000532fedad8f3e3, 0x000042e5dc9aab87 }, { 0x0007a589356da4cc, 0x00048a56b306a0b0, 0x00065fbee32901a2, 0x000cd9768359c2ea, 0x00005460cecc0d99 }},
        {{ 0x000dc35c554b2226, 0x0003bd168754ab0e, 0x000c817f3e20d9b4, 0x0006ac169801fce1, 0x0000801dc7720a5a }, { 0x00096d0960303067, 0x000c06fa3f660d1d, 0x000ddf8f6807ba00, 0x0005d49aea0184eb, 0x00007165e44fb9ed }},
        {{ 0x0003109d80d2b121, 0x000e1b5f8c2a25b9, 0x0008728167361b42, 0x00003c24c1bb772b, 0x0000da906c6e6147 }, { 0x000cc31b47879cf5, 0x000d7f78ef59fb59, 0x0001eb43ee1bcd5a, 0x00056c59b3b438fe, 0x0000d98902a011aa }},
        {{ 0x00097ba641eb7f97, 0x00074a9b733aa5fc, 0x000b26df4bf350af, 0x0002ceba5dece47c, 0x0000cc1ae7e7d3b7 }, { 0x0007fab43b1d99a8, 0x000ad0409c110608, 0x0004beb49cf2a615, 0x00058d94656ea2c0, 0x0000cf90618174d6 }},
    },
    {
        /* digit=18 (1,2,..,64)*(2^{126})*G */
        {{ 0x000b433b59bb0813, 0x0000341d4c5105f3, 0x000e323c820b4e81, 0x000deaab01ae80f8, 0x0000ba1d2bfa0603 }, { 0x00002fefd81e661f, 0x0004eb693e856387, 0x000273b996572680, 0x000f613ecf7b5925, 0x00002889ae807f47 }},
        {{ 0x000e89f3d33e658c, 0x000bb5d0fefb0c3a, 0x000a984dcf89f8ee, 0x000d20ca4d48fb56, 0x0000b9304c4e5ff7 }, { 0x0003083bb07dcf66, 0x00023c86363d14cb, 0x0000b4a396cd193c, 0x000063a218981752, 0x00009e66befec3c7 }},
        {{ 0x000047f6491b324e, 0x000f3debe949a447, 0x0007e422595c1d64, 0x000d22ddc638e938, 0x000012ead0995384 }, { 0x0007d3336dfbf8ab, 0x000df9041fbab170, 0x00084f98a83458eb, 0x0003252dcda8e0b2, 0x00002ddc2d664ecb }},
        {{ 0x0001d587d483a8ac, 0x0001644bfe209256, 0x00032e076092df00, 0x0008fc46391c19ac, 0x0000ca7c69ca0159 }, { 0x000a11fe134033ed, 0x00086a2a37173eaf, 0x000759658d52a842, 0x00086c73e2384800, 0x0000d0571a330ac5 }},
        {{ 0x000b0b43fae02d19, 0x0005e2e81685d7b3, 0x00018429657b244e, 0x000322fef23f27d8, 0x0000be6797dde7b1 }, { 0x0000b732a6bc6506, 0x000f52663ca72840, 0x00041c968590b9ae, 0x000d64c6b7f5729b, 0x000078435f1bf19b }},
        {{ 0x000b01ea6b5b07b3, 0x00041262048e3968, 0x0005ad3bb37932fb, 0x00016d2ad436b50b, 0x000041c792fd15ec }, { 0x00054b52698a8d32, 0x0007dee97015c07d, 0x000878f05045493a, 0x000466fd246cdf1a, 0x00005d2f84c497f9 }},
        {{ 0x000f1bc8463a493f, 0x00090304e23e9bca, 0x000563526dde04db, 0x000c22e614387f81, 0x0000c1b1d1e0bd00 }, { 0x00058493c7b7d8e9, 0x00018ba527de79c6, 0x0003dcdd27ee6deb, 0x000a262c70e1346d, 0x0000f69deb198370 }},
        {{ 0x000617ff6bb8909f, 0x000aafeb8a24a205, 0x00019f14a5d36b68, 0x0006aa95317ebfed, 0x0000c8fde1bb5304 }, { 0x000967ce0fa78f8e, 0x000a6c5e3c99ebdb, 0x000d5e1c475a7c8b, 0x000cdd9dffc64242, 0x000020302e081fc1 }},
        {{ 0x000d6c2966703b01, 0x000365121c80b5d3, 0x000c65b2a339f5a0, 0x000fd9f7e2e7a563, 0x0000bb183ed3e7e1 }, { 0x000a43951972d8ef, 0x0008e1aaf70b8855, 0x0008d324cf4076c4, 0x000efcb5bafc3bad, 0x0000bcb50e192c99 }},
        {{ 0x00087e191f93cace, 0x00049a85fe4e3142, 0x0000d43a775064ea, 0x0003b4d23e60234d, 0x0000bc6b31f78d5e }, { 0x0002da4b5432efe0, 0x0009ddc09d877d88, 0x000b918d3decee95, 0x000f07ea41c23478, 0x0000526c06550283 }},
        {{ 0x0003639eb906c44f, 0x0001891fadb86087, 0x000d08112f3df08e, 0x000f4bb34049e832, 0x0000c537963213bb }, { 0x00072064455862d7, 0x0005c1fe1e4a71b6, 0x0003195294d3f757, 0x00004b9326ffd6b9, 0x0000135f9599d70d }},
        {{ 0x000418871e270c0b, 0x00029a3b23358342, 0x000a6cd6e14299eb, 0x000d6b00d9526205, 0x000018ce7f6af8dd }, { 0x000fdb3c22d19812, 0x00064814082c1576, 0x000fb9e6c451057f, 0x000efd528cc914ac, 0x000020534397c913 }},
        {{ 0x0009874445537fd6, 0x0003ea6f2f9f8e93, 0x00031665058cdd50, 0x00085d2d9963ece2, 0x0000dfa018b78042 }, { 0x0000899785dc3ccd, 0x000326512bf476a1, 0x0007f945e60b257a, 0x0008189ca598a64d, 0x00009ba39facc3b7 }},
        {{ 0x000b8bc785ae777b, 0x00017fd02eee3ad4, 0x0004e4888aeb0a29, 0x0009b3b75a86b3d3, 0x00009495294c40d6 }, { 0x000cd576bcde4c88, 0x0009754413f9cdef, 0x000e2ed5dfa79ee3, 0x000736423a5e4fb4, 0x0000c8fea236703e }},
        {{ 0x00088fe5f08c69ad, 0x000d03e83731a34b, 0x000e34f3889d77d8, 0x000a91d7c2bb9028, 0x00002a8744970829 }, { 0x00069e9172abed8b, 0x000e0c6edea4eb34, 0x000301f05b05123c, 0x000837be143b9313, 0x0000a5596eeef405 }},
        {{ 0x000e205c1544399e, 0x000710441c23fa36, 0x000053783e19700e, 0x00068fea7b4712eb, 0x0000a112e4fbfa4f }, { 0x000514d89ca1abfd, 0x0007c57b36416860, 0x000d4097b2b1ef2f, 0x000a206500432691, 0x0000b3a43e7e1054 }},
        {{ 0x00090d4325fd9679, 0x0009b7fa8e67fc38, 0x000b22134298196a, 0x00091d5579e2e0b8, 0x00008a49ee4dfeae }, { 0x000a54ce4da649a5, 0x000b96ec55e68a39, 0x000aa8e4fa0ce432, 0x000d91155bf12e90, 0x00004d3d79423852 }},
        {{ 0x0008a43035135501, 0x000e73af1a142952, 0x000c89e1a4c4c49f, 0x000aad9ba916f955, 0x000087d02c81ab86 }, { 0x00053db5fc807a2d, 0x000d2a2b8a9cef88, 0x000e34ae4f1d89bc, 0x000eeb9725517407, 0x0000f797b5d79a69 }},
        {{ 0x00061e9a4969a240, 0x0000ba9e864a51ae, 0x0008886b8141b7a5, 0x0006fca286c26769, 0x0000240005dbdd2a }, { 0x000449c2ec4a9e15, 0x0003b039f84216d6, 0x000748fa0a993339, 0x00074b159ab020d0, 0x0000997a8c005c8f }},
        {{ 0x0008a5b251a46688, 0x000c41c2cf9d7c1c, 0x000a25f0335a2815, 0x000478f0c1320886, 0x0000b3973c66051a }, { 0x0004dada9450a7b7, 0x0005d32c11d23031, 0x00020a289c0afe30, 0x000abe1287da6691, 0x0000967694826933 }},
        {{ 0x000c3bd859c92095, 0x0002e217ef6b0d15, 0x000a205766c8c264, 0x00023768252a9477, 0x0000b7cbdb2f3134 }, { 0x0000ac583b4bc642, 0x0003ef5f22c2d3e6, 0x0000570f148a45c5, 0x000cf5f1c368d504, 0x0000aba17b2c65da }},
        {{ 0x0009f147f44133d1, 0x000ea980eab2437e, 0x000474350cd3f048, 0x000a45f7f8cc08c5, 0x000087556fee2ca5 }, { 0x00043f23a09ea234, 0x0004914d2fd6cf16, 0x0003c910a80fc0fa, 0x000d10df7a3ecd06, 0x0000fce7e70f4fef }},
        {{ 0x00085c4a77db6510, 0x00068abe65ab743f, 0x00003d917e24f825, 0x000a8e568bf7c75b, 0x0000f920e13ab677 }, { 0x000d8f276467f157, 0x000feb2e68acf374, 0x000d5da38820e4a7, 0x0006b9d93a544df3, 0x00004f65f0f3913a }},
        {{ 0x000ce5870a0c41fd, 0x000faaaf0dcbc49e, 0x000883d62d516f72, 0x000eaa5e57de551f, 0x0000f8ef2d69da92 }, { 0x000ff5e2f5425d4e, 0x000c0dd167d79ae1, 0x0002da879bcbf034, 0x000d36191039df8a, 0x00004f16c971cb78 }},
        {{ 0x000fc8cc32aee2fd, 0x000a3396ea51d668, 0x0004f0b4adf36311, 0x000779e24506c144, 0x0000d07ff487d2c6 }, { 0x0003173d2554ae54, 0x000d94d9433d67d1, 0x000f576695712eb5, 0x000cd8a077fa5cb8, 0x00002098fca8b646 }},
        {{ 0x000723c7f687405a, 0x000ff9dd5b340fff, 0x0006b9daaecf62ef, 0x0004f00dcfc6b529, 0x000082ac85fad224 }, { 0x0007febee863592d, 0x00096219773d4025, 0x0002c3531e5edeaa, 0x000d2b53cdf7c6aa, 0x00006f48a45191f1 }},
        {{ 0x0005d48ee54627c0, 0x000fee47bd8e7397, 0x0004388f30907fc4, 0x000d3c92379d4499, 0x000014e6df0129f2 }, { 0x000bb8f6ef5589da, 0x0009daadb5cdf33b, 0x000580b07079ffc5, 0x000738d4e3396e89, 0x0000d3c9aebf037e }},
        {{ 0x000aee99a83de201, 0x0000551cbae2f701, 0x000578bc0f4313d7, 0x0003bf399efc4bc0, 0x00004db1c0ede19f }, { 0x000c5f317b20b8d4, 0x000b1b9ae8274c57, 0x000500d18221751b, 0x000091b816c14d44, 0x0000685f584b909c }},
        {{ 0x00037932990a975f, 0x000b1754ef442271, 0x000ec8a7c1fcf689, 0x00000f4cda17bed6, 0x00008003fe108b77 }, { 0x000ac3793bd9befd, 0x0003378dbbe307b4, 0x000e2c2b1ffa6260, 0x0007dad206dd1c20, 0x000049b2e9cf1b68 }},
        {{ 0x00013eddb15b9981, 0x000b7ebed1f8ca0c, 0x00078d7c47f137aa, 0x000d0086f3e54665, 0x0000ee5e70525d64 }, { 0x000020737111ef2e, 0x000f24fd49eb8f61, 0x000cc72ea2ff6814, 0x0000cd758d0d5bdf, 0x00004f38669d1545 }},
        {{ 0x000ed13c9ad8f9ee, 0x000f8900647a82b8, 0x0004006ec8b4868d, 0x0008b4a03908e0ed, 0x0000ecf41ee539a0 }, { 0x0004380e0ee30563, 0x00093761f460f648, 0x000ab29855275b29, 0x000bf349b53930b9, 0x0000a5933dc89dac }},
        {{ 0x0003bc3770684e62, 0x000ca6bc4fe3c395, 0x000318bfb46f7b63, 0x0001ad259ba4a815, 0x00008206cce146bd }, { 0x000c2731bc0908f4, 0x000d4ab5afc60db8, 0x00049492c0e73d3c, 0x00006d107aa02235, 0x0000a2cf845790c5 }},
        {{ 0x000e83daed4291aa, 0x0005eab18a0339d4, 0x000ce8b1687fb104, 0x00007814fac7a658, 0x0000c1a639fa1a8b }, { 0x000721d740a55e87, 0x00040be2f39ea3e5, 0x000f598395ede4d6, 0x0007ef938ef0c005, 0x0000b5b0afcf29c9 }},
        {{ 0x000530a00b6f6334, 0x000e111fbeeee1be, 0x000309ec6bf9af40, 0x0005d4935d1ef5e7, 0x00009dcd3183c604 }, { 0x000632a3a250e986, 0x00031fef15ebfb03, 0x000c9cefa5780c6c, 0x00071ac37549de03, 0x000059126cd86023 }},
        {{ 0x000f033918a2c159, 0x00061890ef59f78e, 0x0002dae1f2816a7a, 0x000a1bb470dfc644, 0x0000c8d185b680cf }, { 0x0003bfaab3be88e6, 0x0008d5db13839481, 0x0002ad1dd2c94e09, 0x000b5ea68c1fc29a, 0x0000900f4ef1e381 }},
        {{ 0x0007640487bdae6e, 0x00058a90deeaf523, 0x00021500178d835f, 0x0001696c4b003fb0, 0x0000fe99cf01049e }, { 0x0002f3acc9ce3d00, 0x0001bc6222a77da2, 0x00073b543e47375c, 0x000de432c62260a5, 0x0000b372f8b94a23 }},
        {{ 0x000e8055d0741a6d, 0x000dc228e41d155c, 0x00076c40041ede78, 0x0006dd7114d02456, 0x00005f55ba2e84d9 }, { 0x00009bcf6bd46086, 0x0004baff62045490, 0x000ec66a97edf917, 0x000123c29a0a00a3, 0x00001c1ad528832e }},
        {{ 0x00017eca6dedef2e, 0x0000aa822305177a, 0x000e9fe591513c37, 0x000b875c0d51d59d, 0x00005bf481a68fd2 }, { 0x00042c3114902bd5, 0x000b2ed60d9dcec0, 0x00011c80f3407b8c, 0x000452dbad6d28e5, 0x00001d77ed06f416 }},
        {{ 0x0004dacf96735388, 0x00012c58aa68e305, 0x000bbfa881a889b6, 0x000da070c7a6a3d7, 0x0000eb1e4523df46 }, { 0x000d0b1e0fc6e9e4, 0x000856fa4a9af892, 0x000fd9a17add2f5b, 0x000bc96203a41193, 0x000035db9779e65b }},
        {{ 0x000a132f7ea9a6a6, 0x000eb9240be34e8c, 0x0006c77d4fb67e21, 0x000afca2bc7162b3, 0x00001a6fac0542ff }, { 0x000f9b75c44f951a, 0x0002af90e2025b4c, 0x00083cc77b87ef39, 0x0004eb21167aaec6, 0x0000d22d6db853d3 }},
        {{ 0x000c7cce51163e7a, 0x000b1164d348272c, 0x0009ccdd071becbb, 0x00037af16bd99d61, 0x0000d10610c58a02 }, { 0x0007eb5875e27497, 0x00054aad846bd832, 0x000c8134cfd36ae5, 0x000aab5a98775a9d, 0x000086c275896b8b }},
        {{ 0x0005528b5f9ab927, 0x0002651662cc565f, 0x000a1861882d656f, 0x000797e9482353a6, 0x0000a67f2e464b0f }, { 0x000d6414aca8eb30, 0x000d2312f23c5ff3, 0x000a2d9b8ef54a36, 0x00042ada0e3e8147, 0x00008166c958af77 }},
        {{ 0x000188223a7a093e, 0x00044e1a51a168a0, 0x00014726320dde66, 0x0001a00b8b712c67, 0x0000e412af6796fc }, { 0x000487211ce123d4, 0x0008a21f7c18de1f, 0x0000bdf491697418, 0x0002e15297682e45, 0x00006b2b6ce9962d }},
        {{ 0x0009c135ff5a9bcb, 0x00077770c24efc82, 0x0004953601a4b26d, 0x000b112280349fd0, 0x0000209df732d2f6 }, { 0x0007f7525d0fd68b, 0x000ba08f78b0b6f2, 0x00023f94c1dfda97, 0x0000b9365176f5dd, 0x000020eff3bcbb3c }},
        {{ 0x000a7c2c5789636a, 0x000a409748ab1a42, 0x0000348217db9653, 0x00093f0f29cba50a, 0x0000ffd72e054e8c }, { 0x000e9d1cf178ebdd, 0x00089a43f0a449a3, 0x000b6ef3df4bc9f9, 0x00075f35745474c3, 0x000083ae967e4c12 }},
        {{ 0x000fa729a87fe481, 0x0006353f380a6e68, 0x00010b6d0e19961f, 0x00044e00a0b86e43, 0x00009f6a7ecb4a6f }, { 0x0002ed3e3509796d, 0x000026e6c9c217e3, 0x00095ea456ef349f, 0x000c3e2b1be1d307, 0x0000a66932593b0b }},
        {{ 0x00008a89af9a9081, 0x000817f9eb1a8b70, 0x0007c527a6393cf4, 0x000462091bc37eb7, 0x0000c5e0712c411f }, { 0x0003154f5208b269, 0x000bffb7d42c10ab, 0x000e969ea76610fd, 0x000ee44e37fe8c92, 0x0000cdae2d966e9c }},
        {{ 0x000c5def04862376, 0x00079b6d9790ed61, 0x000c8a4c8a36ca43, 0x0001a8e2ea77a0cb, 0x00004f8456e6379c }, { 0x0007052acbcdeaab, 0x000557873b6c1da8, 0x000275d32aae4098, 0x0007648b155cf85d, 0x00002e5661fcaa36 }},
        {{ 0x000e8d5c0dec3769, 0x000c280907eae66d, 0x000f6ee4c07b1ce9, 0x0000e431c3cb068d, 0x0000c8b6234ea0f1 }, { 0x000a212bebfadf0c, 0x00013dce3502e6ed, 0x000073687498df63, 0x000e33ffbc894420, 0x0000da93c59a72f5 }},
        {{ 0x000d7b05dd7857f5, 0x000ad4bf82ce682c, 0x00053a7c1a0771d5, 0x000a2e3de058d381, 0x000087e07d1384e7 }, { 0x0002699a5f9d773f, 0x000b9ed1d9f54ec8, 0x0008306a1b6b36cb, 0x000f3ede28be3d61, 0x00001ffd2fb37142 }},
        {{ 0x000e5782cd47a8b9, 0x0004df571c0c3a75, 0x0009aedc953ef848, 0x0002f1a758f580f5, 0x000042380c4840db }, { 0x000acf2240badb82, 0x000d60c99c82a70a, 0x000e6b1b6a0b6799, 0x00065dc70e8359fb, 0x00005463c0ed77c0 }},
        {{ 0x00044e55ad6bba22, 0x000aaf5e4652f1d2, 0x000ab113020533f0, 0x000a7ca50bd6fdd2, 0x0000ffec60c771f6 }, { 0x000d2dbc0fe69c05, 0x0008e05047d320ba, 0x0004b4536a2fa180, 0x000f3b3ee1ca983d, 0x0000736ebce33440 }},
        {{ 0x000b59f3e30d097b, 0x000bfa29268b3de0, 0x000adb866cb76535, 0x000eac7520d1ca29, 0x0000d07af7edc75f }, { 0x0007f84282de8bcf, 0x00016ff7f4a6c51d, 0x000189e3ecfc03a9, 0x00009091fa8768fd, 0x000063e18fffaf2c }},
        {{ 0x0005978d0a53ff5e, 0x000de0e0546063ea, 0x0001b7ca1ebc5a0b, 0x0009b66007fbadcb, 0x000016843f73bee2 }, { 0x0007fb68ea74f94b, 0x000d61e4f85eaaca, 0x000152cf5d96b415, 0x0008a39cde61e2ad, 0x0000f97b10343ac3 }},
        {{ 0x00048364d3e5d0bd, 0x000fc363daed4c29, 0x000f918392b59488, 0x000d61c0a80a9aa3, 0x000059055976d210 }, { 0x0005f25bd356dc7f, 0x0003d6070985182c, 0x000d6d3eacc1af52, 0x0004d43f5a6db5b0, 0x000050f58b44a28c }},
        {{ 0x000c527f92db6aac, 0x00019242d0a4c230, 0x00045f614cbf9dab, 0x000338092eb5d26e, 0x00008b05b2bd83e5 }, { 0x000c81836dfe11e3, 0x00036382aede0ac3, 0x00083041c5cba047, 0x000916061b292220, 0x000053fe2de8f49b }},
        {{ 0x000a0369abbd0203, 0x000b27fa001ec5a8, 0x000b16d8f40e8efc, 0x000481080e993f5b, 0x0000af7a86da5db5 }, { 0x000551c295b0441c, 0x00071608c927db84, 0x000f4f1394ac39c2, 0x00072abeabf5f37c, 0x000060b2c65c0c54 }},
        {{ 0x000b04f7d8f72e56, 0x0001be7ed77ee8ba, 0x00004804ddee725d, 0x000b92acabddf7bd, 0x0000ccae2b20533d }, { 0x000c088c89918989, 0x000e9944313a8cee, 0x000ba6c1c0748b0a, 0x0001e207ae532e4a, 0x0000e42ef48bf73f }},
        {{ 0x00046b3d3b414b91, 0x0008dcc42ce9e515, 0x000d5df49fef1fdf, 0x0009c67398f9840f, 0x0000bb29bacc84c0 }, { 0x0008c7d4845045fe, 0x0001ba980d309162, 0x00072f844c3fd6d1, 0x0008188084549710, 0x0000d0d70dbdf40b }},
        {{ 0x000ed2436122cc51, 0x000c66788a2ffe41, 0x00050a0f2644da13, 0x0002d783adc506a3, 0x000069129b160058 }, { 0x000218e8eafe18f8, 0x000b292a3d694e78, 0x000d6d37d38998d1, 0x000f855b4c0f43b4, 0x0000bf4a6ab4a165 }},
        {{ 0x0000f3271f894fd3, 0x0008afb185f88448, 0x000a946484dc2331, 0x0007019dfcd7e90c, 0x0000692c2e123626 }, { 0x000981d9ea441cc2, 0x000d8a7d0e62f477, 0x0000cc886efb3ba6, 0x000724e1e88d519d, 0x00000023f082da8e }},
        {{ 0x000afff39d12e58b, 0x000da2c9ac382032, 0x000f2c658c484952, 0x00039c91686eaf87, 0x0000523e755bd5cf }, { 0x0005df02e43ae54f, 0x0006c07256d4eab9, 0x0001eca6a8a4fff6, 0x000375ffe2259869, 0x00006edafe5d45c9 }},
        {{ 0x000d1c4cc4d8b709, 0x0002c6ec7004bf30, 0x0004ce677b1974ac, 0x000529eb7230a08f, 0x0000f295c26eabae }, { 0x000fed0da0e7efbd, 0x000096e7550fee88, 0x00070d0a16dda21c, 0x000c5fae7369cd4c, 0x00001bcd58dde3aa }},
        {{ 0x00044ed736b966ec, 0x000a5c2e48fb8898, 0x0002f73bc5cfb109, 0x0001bb4e226882c1, 0x000063d7619b2c62 }, { 0x0006a413e95e2beb, 0x0005c6fd86e27c75, 0x0009ae0bdbf2ca03, 0x000edf84da04edac, 0x00007492ad45d302 }},
    },
    {
        /* digit=19 (1,2,..,64)*(2^{133})*G */
        {{ 0x00024f4d696da3cb, 0x0003b1d5a74be506, 0x00047c52228c154a, 0x0001bfda2e41781c, 0x000096ec0545b52c }, { 0x00001059ae4af1e1, 0x000735dbdcc9ae4f, 0x0005508ac4dfef91, 0x000d754392573dbb, 0x000045a30ae8165b }},
        {{ 0x000c13673627c363, 0x0003c973112795f7, 0x00071c3b07ab4999, 0x000f308cde824443, 0x000024017b4422c4 }, { 0x0007e6782a430857, 0x000be955dbec38e2, 0x000c10a45929115d, 0x000d83e2f51c0782, 0x0000d2b6c6a41083 }},
        {{ 0x000f9b21b06e4d48, 0x000483d4ed2dbc25, 0x00081ca00bd2f9e2, 0x000314b350eba59c, 0x0000879d4fee9d2e }, { 0x000846aa4551a9b0, 0x0002394af267bae5, 0x000565e76b699192, 0x000e51500aa86cc2, 0x00009d486efc3818 }},
        {{ 0x00023eed8fe44b7b, 0x0001c40e2c2bf152, 0x000587aaa9751a65, 0x000b04a65627a220, 0x000078ae9a0140de }, { 0x0008dfbb97c7d47d, 0x000a883a8363b49a, 0x000920e2f50e570e, 0x000074b5c1a0b6cb, 0x00004368ec853ac9 }},
        {{ 0x0003cdd7ea38297d, 0x000d04c65c7c2aba, 0x00073ef8129a7789, 0x00012cdb51d4610e, 0x0000ae1d9b12315f }, { 0x000a8296356d58cb, 0x0007e83064417ee3, 0x000c72e948f5f2f8, 0x0009e5bc31459b23, 0x00007f5688f80bef }},
        {{ 0x000e7c88b34096a2, 0x000ee1fab192c181, 0x000e51b38d9be8b8, 0x0008dc9a6ec5fcbf, 0x0000c004fd1604a1 }, { 0x000d9b52b3513161, 0x0007c93c92d88c5e, 0x000d236ba8a62196, 0x000c4ae1441c2148, 0x0000e3b813e3424f }},
        {{ 0x0002d6dbe520bd47, 0x00002ebb0a5b358f, 0x0001aba752f14bc6, 0x000e71bdbb154c5c, 0x00004df8bc57eacc }, { 0x0007ad0af3ff3ee3, 0x0000af38665e5b22, 0x000353d24c2f432c, 0x0008cd086488a851, 0x00006ced5e1bbf8c }},
        {{ 0x000c2a360971c31d, 0x000066e846e364f6, 0x000c727ad6974fa9, 0x000639b6d7f33527, 0x000015ca19ee6b97 }, { 0x000b4ef351021aa4, 0x000ba52f024e9245, 0x00044265bc79a45a, 0x000687390e0fa07a, 0x000026bc8f7f2c64 }},
        {{ 0x000eb356d99847ee, 0x000c1c4b6205969f, 0x0001f8142e759402, 0x000504d4842563f0, 0x00002d9aff65fb85 }, { 0x000340185f8291c1, 0x0006b81d1a39bd77, 0x000f7353328a0998, 0x000e743da8f44340, 0x00008523dea1bcc1 }},
        {{ 0x000c87111ada74f5, 0x0000bfb7409ab463, 0x0006445b7897551a, 0x00013f0d1057e78e, 0x0000fca1d2e611ce }, { 0x000f201ffa6c5d1a, 0x000f41c7809460b4, 0x000222a98bccef3d, 0x0006f03c9e751c85, 0x000002a8d1482e5b }},
        {{ 0x000984b27405ee90, 0x000005c3e6721cea, 0x000a272eb4d7fd0e, 0x000927da4d52d70d, 0x00000cba2cc35612 }, { 0x0009d70b01b1eae7, 0x0003e2784de5ca41, 0x0002ae7bd3c2b1ce, 0x000f8f7092f1c987, 0x000030b0c4119210 }},
        {{ 0x0005583b9f7c63fb, 0x000702da6dc1d29b, 0x00094319a5a4c61c, 0x000909776d303000, 0x0000ea690a72942e }, { 0x0000d34175165b72, 0x0004c7895695f204, 0x000148fcd1400755, 0x0009aa443fc84181, 0x00009d399222fcef }},
        {{ 0x000758ee5709e62d, 0x00069f5674198335, 0x00022ee28cb1cdab, 0x00070430c6059e25, 0x00001b376756c012 }, { 0x0008f5c1bf938d37, 0x000401d013b0ea65, 0x000afbd027f9ea15, 0x000a32680fddf524, 0x0000feb88c703890 }},
        {{ 0x000d105f424816f7, 0x00040f17c696042c, 0x000b7d14d99c5d98, 0x0008d2f00d4cba22, 0x0000ca9c23515356 }, { 0x00065cd5704583a6, 0x0002919ddb81e743, 0x00091afae35a6c5e, 0x0001a1285472f2d8, 0x0000a117968098a9 }},
        {{ 0x00061f2dfaaea695, 0x0006dbba2be70539, 0x0005e49a78db79a4, 0x000511030dcbbf7c, 0x0000c25cacbca172 }, { 0x0006bc1247ed44bf, 0x000d53570dc80c4f, 0x0002f9bc2ed8f5b8, 0x00011b222104d6b6, 0x0000be75e1d87966 }},
        {{ 0x0001a05fe05069b5, 0x00097050f15dde91, 0x0007fbbecba1b4b7, 0x0007e7662a47a76a, 0x0000cef9eea25012 }, { 0x000d251439c9ff5b, 0x000996e9d529b36f, 0x0000927929bcb4b0, 0x000321b9855ab130, 0x0000e6913e2a37cf }},
        {{ 0x00002e1795824e90, 0x000a54723d695ead, 0x000d46b77220484d, 0x00039408648ce626, 0x0000dd1d7bdd322f }, { 0x0000941af25d109b, 0x0004e8f7198111d6, 0x00057c65a8da2e48, 0x0002feeb090ca630, 0x0000f1530e7e8096 }},
        {{ 0x000b5c989f6938ee, 0x000d2a222eb51e58, 0x00041f4057fb3762, 0x000dcb106c0bb724, 0x0000ad14845ce016 }, { 0x0007d1e1160ea6c5, 0x0002e16512e1f433, 0x000475371e4c0b2c, 0x0002b464e02eac55, 0x0000f6d7ff9c35f9 }},
        {{ 0x000f15a10f16a85a, 0x000e41aab7b19a8c, 0x000e0fe32c369f9b, 0x000b5a9daf08d067, 0x000024ab40acda51 }, { 0x0009684eaabd75c3, 0x00009e5b738a4250, 0x00058cce645336d6, 0x000db1305ebe131d, 0x0000484813fc96ba }},
        {{ 0x000d8bd56620b954, 0x000129c72ba075b6, 0x00054cbabaaf6b83, 0x000e637a889f78b5, 0x00003d0a3429781d }, { 0x0004afac04fc7b8c, 0x0005a3b805f08d67, 0x000d4f0d15b02979, 0x000aa4c477f48200, 0x0000acd64bf963b3 }},
        {{ 0x00065ecce6ddbcf2, 0x0003a2b7e63d6900, 0x000ffbc500d16630, 0x000c7e1feefb6bbe, 0x00008b274a61c74a }, { 0x000c29f8ff0209d3, 0x000c1ce69f790713, 0x000ec8218d676a18, 0x000acca8f0d9a8e5, 0x0000f615e9cc3915 }},
        {{ 0x000f37825c9ef1ce, 0x000848febae68c4c, 0x0001618e37c4f175, 0x000eb4cec5b38563, 0x0000d423c83cbc80 }, { 0x0009c52aea308776, 0x0008380beb11454b, 0x0000a6fc857c4768, 0x000022bc740a1a99, 0x0000ca5ec3d496c8 }},
        {{ 0x000cd4fde296905a, 0x00062ce45650ef47, 0x00000dcd9ce46f6b, 0x000746a70ae000f1, 0x000007a206cd8fb8 }, { 0x0006cd56404f4acd, 0x000e52ae88408697, 0x0008e8efb046152c, 0x000cc12baa886885, 0x0000b6caba36edc1 }},
        {{ 0x000a22c213a1381c, 0x0005998a9d82abfd, 0x000b0c9eaee2416b, 0x0002e94875f188cc, 0x0000ed487d7becaf }, { 0x000a1bfe43d8e989, 0x000d43afbddb4799, 0x0003d3555d553c34, 0x000ae422e4b606b8, 0x0000a98941400820 }},
        {{ 0x000d93563973829e, 0x0008a092a67d48a1, 0x000de58c53a8a8ae, 0x000c1054acd6a671, 0x00000d14b278f09a }, { 0x000286662fa4b2d7, 0x0004d898fea73ea3, 0x00046daf4982b9ed, 0x000b2de4c8247b36, 0x00007827af7176ad }},
        {{ 0x0004d969cb439a74, 0x000636bb4ec4c20e, 0x00062a667a3273e5, 0x0003e53e8f0a12fa, 0x0000735706c629fe }, { 0x000d4b3454ad59df, 0x00074b743e232a3f, 0x000b3aad70059ea6, 0x0007e3d8fdc7a3fa, 0x000053ec510c7073 }},
        {{ 0x000267aa86866f79, 0x0001686d36e30622, 0x00077edc547bf065, 0x000077b2e626c527, 0x000032b789435829 }, { 0x00088821cbb8b0bc, 0x000a75172e572d59, 0x000721eb0f204226, 0x000dcca57ab861af, 0x00001dc0ca62fb88 }},
        {{ 0x000ce1680d34fa1e, 0x0000722cccc18ad5, 0x000d94c5781a4479, 0x000c8174daadf3f8, 0x000096cf50e285e2 }, { 0x000921dfeb77456e, 0x0004dfcd9ecbee97, 0x000ba38c82c0be59, 0x000d5558a4330689, 0x0000d3b38073c94d }},
        {{ 0x000da886aed8ecd4, 0x000eaf8271d1ca55, 0x00078b0236696085, 0x000b711e33e423bc, 0x000014d2f4ec906b }, { 0x00036e1f5d068ff6, 0x0007034a99643755, 0x0003162eb1a26190, 0x000151fe9356a2fa, 0x0000f7623180c7d5 }},
        {{ 0x000f35afef5374dc, 0x000b53a5c04e4ea9, 0x000e37bb65248044, 0x000884194d549dcf, 0x000056a261a80626 }, { 0x0003eb6a05b04ad5, 0x000b824f1314f53b, 0x000816d16e1d2218, 0x000a9b8b062baf94, 0x0000b751a249239e }},
        {{ 0x0008e248259fd6b1, 0x0005fae1208e16aa, 0x00054f53f5fe83ea, 0x000752f0b1a4d15b, 0x0000322765d11e76 }, { 0x000a06028f32c493, 0x000e0a290b170cfe, 0x000058eb8f6521c2, 0x000831797a1bef33, 0x0000d5b9b2a9415f }},
        {{ 0x000495775835ddea, 0x00038f3dbfb894e2, 0x0006a991d3c0ab53, 0x0007ac0dfbe9b79f, 0x0000a996ed108d94 }, { 0x000a1485e6867237, 0x000b861bb9390993, 0x0009ecbcb0ad8aab, 0x00089ee3b1d6a974, 0x00004490d265e84d }},
        {{ 0x0005d8bafdc81765, 0x00002b4147c15b45, 0x000b25ed58041f7f, 0x000f1de81e34f553, 0x00001e756a2afa4e }, { 0x0008b5d2446a18ad, 0x000cc95f1c0e74e9, 0x000e53045657f07f, 0x00033d5515a9b292, 0x00004e129faf6e81 }},
        {{ 0x00054ce98632c8d9, 0x000e666683425204, 0x0009755b794b887e, 0x0003dd2cd08d6894, 0x000093ff6f3941b6 }, { 0x00011538eb926db1, 0x000ab9eb55b803b2, 0x000a10ce88821a3d, 0x000060e12b03468e, 0x000072cd7c995240 }},
        {{ 0x00032cd90a50d0eb, 0x0005f2f731dea2d9, 0x0001e1c0a21fc8bd, 0x000d579fed5856cf, 0x00009b444a44e0c2 }, { 0x0009e7c0c1c75493, 0x00002d4c0c4ddc0f, 0x00073a980207bdba, 0x000b2d8cbc0f422f, 0x000047b3f99102be }},
        {{ 0x000665427bc303fe, 0x0004fdb0f5f27cac, 0x000ae1a81783a295, 0x0005c1223e85461f, 0x000076fba4039ea3 }, { 0x00022639d9c5cdc0, 0x0005a697587fd52b, 0x0007db23b5f4ea89, 0x000744688935289f, 0x0000887566528724 }},
        {{ 0x000bc9cdaec3de6b, 0x000f720361ecf90e, 0x000f398696b456af, 0x0007dc079001f23d, 0x000063bcc338ae46 }, { 0x0007015cc0864754, 0x00010a1934aa7289, 0x000e179a04332cea, 0x00055a7fd750eb00, 0x0000b649d7589b15 }},
        {{ 0x000c21aa55e58d8c, 0x00038a5bf5151dff, 0x0000b44f6e9cced5, 0x000b2182c21adcc4, 0x000000bf7836051e }, { 0x000405a74bccb408, 0x000bb2f4e51ea80e, 0x000f4195b6421b97, 0x000db135e3dae45e, 0x0000c8a422289ff6 }},
        {{ 0x0005afe3df554744, 0x000a5c69f1c56bdb, 0x00079441b2c565d3, 0x0002dc52b8c176b9, 0x00001263e0533213 }, { 0x00004e3fc1406d5b, 0x000a249145813e50, 0x0009e2650265686d, 0x000129a7f687fc4d, 0x00000832fee46c0c }},
        {{ 0x0001f9addb3fd772, 0x00068b61430c9ab7, 0x0002b03f3bbc1b26, 0x000a8d0ae0bdd41d, 0x0000a6be72385501 }, { 0x00063accbd8fe7a9, 0x0001716c52dd907b, 0x000c73a737c1e0a9, 0x0002868a6d9bdf44, 0x000035c09edac28b }},
        {{ 0x000add2b06f54586, 0x0005c351272a95b5, 0x00074b4a3fc1da82, 0x0009268f70c66771, 0x000044d6759202ed }, { 0x000d2a2bf9fa5f60, 0x000732d14ea5d65e, 0x0005a053a90c3bdb, 0x000554ac6f8e01f0, 0x00005fd71f56197b }},
        {{ 0x000d3d1774087028, 0x00051b5e9d3a7c31, 0x0006647adbd73b2d, 0x0005492758e0ee5a, 0x00003bba70bd8115 }, { 0x0007246f125d77c6, 0x000ab127747ae836, 0x000fb258fc2e4b6e, 0x0001e4e0031f4315, 0x000079fb7523dc70 }},
        {{ 0x00022cef7e044521, 0x000ab5aef0c31335, 0x000d2be979bd6bea, 0x000e49649247cc45, 0x00008daf607fcbc2 }, { 0x0004be69d5e2999c, 0x000580559ab93b3c, 0x0000da59a69c02d5, 0x0009972d0225fba1, 0x0000cc519897bf32 }},
        {{ 0x00000ed311acec40, 0x00036680448087c4, 0x000f31345a9c7960, 0x0000d7f47ac30903, 0x000014a52f1f5454 }, { 0x0006f2e733d21f38, 0x00041fef8fd24247, 0x000a837ac257323f, 0x000ff2f3afdfd974, 0x00009ba0f6470d39 }},
        {{ 0x00050c547e9953dd, 0x0000eb52a6017539, 0x000c306fd8ca8187, 0x000b82024b286514, 0x0000a0dcbe1ef184 }, { 0x000f64a7dd340b95, 0x0008469e5d32bfe8, 0x0001e5a8c4619dc8, 0x000d6ea3a30bc147, 0x0000e933dd6721ac }},
        {{ 0x0007ccc59280d519, 0x0008037633dfb1fc, 0x0004cfb25fb4af10, 0x000498d46ea82c39, 0x0000e384e83287e5 }, { 0x000e6e0a4144dfdd, 0x0009c6281761e131, 0x000f73287e1b3d37, 0x0001fe070da57596, 0x0000b0e289bb07e4 }},
        {{ 0x00066b4e44ff11ff, 0x000882c4152fcf57, 0x0000280c990f629c, 0x0003c0009441c87b, 0x0000b659204e90a4 }, { 0x000df4368ea4d935, 0x000701211f1c7924, 0x0002fdef25b0372d, 0x00033119eea02ae9, 0x0000eb3caf4bdf44 }},
        {{ 0x0002ad31f429e92f, 0x0007dab4baef62e3, 0x000275ae5ff44e2e, 0x000fc47109f5a220, 0x0000f78da2e8f242 }, { 0x00089cf1ff54aea4, 0x0008ca425bcf4d6c, 0x000288b364e70dd4, 0x000847d56ea95059, 0x0000d5e592ce8a8c }},
        {{ 0x000bb30b4c5c4041, 0x000f81304e60cc0f, 0x000c4a72c8291975, 0x000e6a462ce811e8, 0x0000eb639b79afd0 }, { 0x000e60a95e94bf15, 0x000e32c485281182, 0x00059ec7afb2ce90, 0x000767a287c6fe08, 0x000067e5869c6204 }},
        {{ 0x000b6d146193e6d2, 0x000ab3d9b785dbf2, 0x000b7beca8857459, 0x0000f4039530889a, 0x00009e5aa29e84bf }, { 0x0004f877f4c051e2, 0x0004a0451a7bbf7f, 0x000f81f9d270263e, 0x0006a8ba030ad99e, 0x0000b432329b85d8 }},
        {{ 0x000cd2b7ced02b6a, 0x0008c2d4c44f1190, 0x0006dce4bc47e034, 0x000836c3b4555f53, 0x0000788b19828abc }, { 0x0004d4598c6d92d3, 0x000557b46945fa35, 0x000081a82e04d1a0, 0x000a4ea08f8785b2, 0x000063de08f3d528 }},
        {{ 0x000d3e738b6fd0d3, 0x00087cbc46d7ce17, 0x00070dc9e4ec18cd, 0x000a97d509a515bb, 0x0000c0ed53ffb19b }, { 0x0009fd2c8134d2bd, 0x00005ce2a6cb0ebd, 0x0004b57daff8514a, 0x0002ea6766f2e295, 0x000010bc9a64cffe }},
        {{ 0x000465c200623f5c, 0x000db38b2b836a16, 0x000017a7a4228368, 0x000823ef1855b41a, 0x0000cb16497d171a }, { 0x00094b4447686be0, 0x000e23f6974b99e0, 0x000b99263876de79, 0x000c0065875a7cf8, 0x0000dfac4f57f79d }},
        {{ 0x000a8e6f779b6900, 0x00085edb940cb5af, 0x00018239bf98af89, 0x0009c26fa6706d79, 0x0000b769a307f467 }, { 0x0007c08b475deaab, 0x000f234591ee4f0d, 0x00066866b61f19ff, 0x0007c69575f46e33, 0x000028a209b8748c }},
        {{ 0x000b82c392e80537, 0x000f93c1c08ad635, 0x000ff1019ae85ec6, 0x000ee34f79954343, 0x000039a812312260 }, { 0x000b0e72020ed2dd, 0x00035c4928435d5c, 0x0000ae24917dfa24, 0x00007284a071cf0f, 0x00005e0b1098229a }},
        {{ 0x000e70062ee0a4a2, 0x000917ac0b7eff35, 0x000e6774850d2a32, 0x0004c985f8552225, 0x0000ed692c26dd93 }, { 0x000068c99ecd36b0, 0x000a048e7073969a, 0x000922d3ea339e56, 0x00093f20f392d2a2, 0x0000a61b1d58dd81 }},
        {{ 0x000a4141443fe874, 0x000f11fb87a29137, 0x0004c28111acbcec, 0x000845fe4ea3e3c1, 0x00002b3d24c59afb }, { 0x00010dc2aa355a06, 0x0008f1aa71ff4939, 0x00022ac5ba4c733b, 0x00075e8fffe980c9, 0x000092ea09a1baa9 }},
        {{ 0x000bd61a53336f7c, 0x000c326cca9f54d3, 0x000b572ddc8897d0, 0x0007f6a7c928bbdf, 0x00001633d0f6cb76 }, { 0x000e19742c3c3494, 0x000a4e6395c9a791, 0x000d43d1c100cd6d, 0x00052c2316940ca2, 0x0000fd9b397c3285 }},
        {{ 0x000baa11cf6966bf, 0x0001a0823fc2e6e0, 0x0008fceb7f79dccd, 0x00086bea5a76e29a, 0x000020bec34acd1b }, { 0x000a26322d5b0637, 0x000b564961849da5, 0x000f551759fb7518, 0x0009cb74493b75f1, 0x000085589c5471c1 }},
        {{ 0x000097196463dab3, 0x000235861a023efa, 0x0004194e0b09b4ae, 0x0007b8bc01d72aab, 0x0000a76ce5fce481 }, { 0x00008748d29fbf98, 0x0003d4badb9b5b76, 0x000163d30978b6be, 0x000794e0873fac1a, 0x00001acf21ce255e }},
        {{ 0x0009ab9f40344f97, 0x0008094c486094f9, 0x0002f24809967c07, 0x000bb7725869254a, 0x000017af2e8590fc }, { 0x0003a576905d94dc, 0x000ea8a6706c02e5, 0x000c15d397dbab92, 0x000dc5b6c22b5e76, 0x0000e77621b7d00e }},
        {{ 0x000d38e9c47b4340, 0x000b852edffcf3a5, 0x0003ded185911220, 0x0000e5b9eb4d2ed0, 0x00005ef944ff1997 }, { 0x000cbf19807bc19d, 0x0008bbf303747e2b, 0x000c38f6a80d0b53, 0x000aa5c23a208e77, 0x00001f5c1520084e }},
        {{ 0x0009789f1f62f6fe, 0x000f230273231941, 0x000b1a24067e38a4, 0x0006f97203b055a8, 0x000043fde981f2ac }, { 0x000ee11ececcedef, 0x000a568aaf018013, 0x00004c88b6cdcf5b, 0x00006d17972ee119, 0x00001719e065c6e0 }},
        {{ 0x0006e4617f3e69bc, 0x000caec1ed66f181, 0x000bbf3b6cc4ad74, 0x000551b4d225d906, 0x00003684306257aa }, { 0x000516497c9675a0, 0x0000addb2fbc3dd3, 0x000f55d795decef3, 0x000598fadedb5484, 0x000015e8ee776bd7 }},
    },
    {
        /* digit=20 (1,2,..,64)*(2^{140})*G */
        {{ 0x00020f2beb76812c, 0x00013ecfa181e695, 0x0000dec76ead7889, 0x000698d5d9ea02f8, 0x00000cd75bcfa2f0 }, { 0x00017a069c2d2cc5, 0x0008f0a1df843618, 0x0000f08066c134ef, 0x000c931ff1203772, 0x00008f19ef39b4e2 }},
        {{ 0x0000aecc1c7112bc, 0x00073b6b294cda6a, 0x00051854d3181244, 0x00049ff216ce9bf7, 0x0000f5f14402d398 }, { 0x00056e372344bdbd, 0x0005f3af02909e50, 0x000f436065b91304, 0x0002579f8c7d59ec, 0x00001766823e2146 }},
        {{ 0x00027d200e7695ba, 0x000fb3189e800ca4, 0x0006d1d4e8ef137a, 0x0007f003750fe0c1, 0x0000dad25c5ac540 }, { 0x000807804fa82f49, 0x000994fe616e2c00, 0x0008e610d4715daf, 0x00004e1739c25f4c, 0x0000a1ed59eb55e7 }},
        {{ 0x000c966787f80791, 0x000d43a4f0d56f34, 0x00077d92507dca1a, 0x0006bb24b961e404, 0x0000a0d775222852 }, { 0x000bb6d594089b2b, 0x0004142864fee422, 0x000a2f57f8c8c37e, 0x000969659c1be93d, 0x0000e98561f48eb8 }},
        {{ 0x0003d36eea5a411b, 0x00083c9b809b7ceb, 0x000b2ef3bd41c883, 0x0005fa4368a41486, 0x00005327a94036a4 }, { 0x0009d81a294550be, 0x000028a328cc987f, 0x000b405a4a382a8d, 0x000c01dba0a3bcd2, 0x0000ecbc7c687492 }},
        {{ 0x000ee9ea4399d83c, 0x00029ade4d559419, 0x000d914e5643a410, 0x000194f9bdedafa5, 0x0000ab6a2f9c77b5 }, { 0x00023fc56d6b71dc, 0x000ce1637a55a4a5, 0x0003af1fce4bba9d, 0x0002a5998eb19a51, 0x0000fb6b0a026533 }},
        {{ 0x0005384859f3a770, 0x000970cb98fe684f, 0x00091d11cbfe5c75, 0x00014fb3fd60fbe9, 0x000024a2c6896e9f }, { 0x000a7731d1c175f6, 0x0007bf7b59e763bd, 0x000880e35c8c898c, 0x0002e8c923d4606a, 0x00001705d921c944 }},
        {{ 0x000bd17983bec34f, 0x000e79390d458714, 0x000fba44f409e51f, 0x000e5b503e976403, 0x0000bb28b50bfeca }, { 0x000460fc77585d24, 0x000439ed5d2c53a7, 0x0008896e3f104db9, 0x0000d454e5ec4bcd, 0x00005c92699d9c5e }},
        {{ 0x000cb1236b66c01c, 0x0007f8295caabee0, 0x00057be3de780986, 0x000aeb665de024ca, 0x00001dc5e9d49aa0 }, { 0x0001359afecfc3d8, 0x00005bf0972ebf9d, 0x00045177093370a2, 0x000872776f1dd387, 0x000022a719d4d0a4 }},
        {{ 0x000802eabd5b22c0, 0x000fa2b91502cf37, 0x0007904acc9ff780, 0x0000df1c92c3832a, 0x0000c8e7c1032083 }, { 0x0006aeb0c9d5f503, 0x0007027a37fddab7, 0x000b8400f8a95d7b, 0x000f1f61a65d6f2a, 0x0000d4946c17ee3b }},
        {{ 0x000af61472ec944c, 0x00013f62f93a6750, 0x0003fafe7206fbed, 0x000f69c84a6ec968, 0x0000e64f69ece348 }, { 0x00036a8a8778d2a5, 0x0009e72a5f696be1, 0x000a5bb870e8f960, 0x0009a4590c65e57e, 0x0000b18bf5bd851b }},
        {{ 0x000b4b36c8c2e8ff, 0x00082dc0f9f44d43, 0x000ff0fb2191ea75, 0x0005ed2b1f09a695, 0x00007a5902599c01 }, { 0x000a904b3c1f4b94, 0x000837b3b6b234cb, 0x00045ca08e940188, 0x00034bf1d096a250, 0x0000986acd027cd0 }},
        {{ 0x0009958546371d81, 0x00058d493a3147a2, 0x000c70c0cffeda75, 0x000d2f1129f30a5d, 0x0000599e39f58537 }, { 0x0008d658f17504df, 0x0005fffc17db9ba2, 0x000ca23febb75e2d, 0x0004520bccb18548, 0x00007564d18ce2da }},
        {{ 0x000e430a33a255cb, 0x00095cda99f8c71a, 0x000a9db438cba264, 0x000f5749967b7abd, 0x0000945a43d068ee }, { 0x000cb8f38cf63ea6, 0x000d0d388ace52f4, 0x0004061e12f02828, 0x000c9271b4bc0fa2, 0x00004dad48391d98 }},
        {{ 0x000a1c5643447901, 0x000bd808c3bff364, 0x000208478f323f53, 0x00022dbd1005a0bd, 0x0000ae406a2896e1 }, { 0x000137007e0dbeb6, 0x00025b99fe4fc88b, 0x00081b60bb371dbd, 0x000646bae09418e2, 0x0000cc34af053693 }},
        {{ 0x0000693a18674889, 0x00096e1c63c4962c, 0x0008d7b6a10d190a, 0x00003a26b50541e8, 0x0000e282f1e08996 }, { 0x0001ed4e5703fca0, 0x000c04e0117f203b, 0x000258ac6a79aec7, 0x000248c44245f196, 0x0000ff00eb7253c2 }},
        {{ 0x0002a6cd6b58c40f, 0x000e675f8547be3b, 0x000cf73d922da584, 0x00000f2000a7033c, 0x000044f163153dc0 }, { 0x000273a484b87506, 0x000aef557933c6b8, 0x000ca899db77a846, 0x0006120b6a7538eb, 0x0000ded4a50d08d4 }},
        {{ 0x0003995b1a10381b, 0x0007d450168e508a, 0x0001f60fefa3784d, 0x000788b718cbc9bb, 0x000015ddd98778ea }, { 0x000b59fe5b0f3810, 0x000568e1239e1525, 0x0008605ee199d7d9, 0x000b969635689255, 0x0000ae3fda8f50cc }},
        {{ 0x0000f551a177b5e0, 0x00040eb34a9f7aec, 0x0002165c9e43067d, 0x000084f37e5e8cf7, 0x0000983560921b62 }, { 0x000e549e04b370fb, 0x0006768038f67254, 0x00051f3e35b1a2ef, 0x0005adb861dd38e3, 0x0000ac6cfc2b7810 }},
        {{ 0x000ddba4b3d8e704, 0x000f53b08b193400, 0x0004bceb33e96771, 0x0002e98c7c2fae6f, 0x0000d016b8e9d8b6 }, { 0x0009d9e008a1dcc9, 0x000feb51dd95f8c7, 0x000754cbe24d3872, 0x0006dc34c1d163cd, 0x0000d6587365fd85 }},
        {{ 0x000846f5f2308016, 0x00088bfe6ddd5053, 0x00047cc4f727d222, 0x000b2675737064e7, 0x0000ae37a4ce7376 }, { 0x00090c603077e9a7, 0x000cd9160df1b9b3, 0x0004f411426f0fd9, 0x000e7cd3c4cfb289, 0x0000f64f81bf99d6 }},
        {{ 0x000762921d03717d, 0x000d6128c0a78de0, 0x000a17234d538bcc, 0x000a9e882ccd02bd, 0x0000eed2d646e6a2 }, { 0x000287c7f8080a37, 0x000d123d85010c02, 0x000addbd0727cb42, 0x000abb08ad73aaac, 0x000037e0c3c8c39b }},
        {{ 0x0009ea4996a3c06e, 0x0008942580b1a014, 0x000d1657f1e2a72a, 0x0001e3344f080415, 0x0000093afc473704 }, { 0x00090e3df700aea3, 0x000eeef571f3913f, 0x0003fca41c93b37a, 0x00008fb8f0610f21, 0x0000984f4e2a8e30 }},
        {{ 0x000c69c29d48bff8, 0x00065eb43e2e0349, 0x0003a7d6620d144a, 0x0004ca77f53fb5a3, 0x0000f354b8c1cb22 }, { 0x0006812c52e71fa4, 0x0007bebbace47b69, 0x0009082476997303, 0x00043e13ebdce028, 0x0000fc56789759ad }},
        {{ 0x000a5a5512b3d42d, 0x0002a7157151692d, 0x0007ff33c7322027, 0x000656ee1eb2721f, 0x0000ca99ca38747e }, { 0x0001ef52167f204a, 0x0008b5dc6f567809, 0x00037bd855ddf45e, 0x00073d003e20d300, 0x00004fefff6aeb76 }},
        {{ 0x000c08620a2c393a, 0x00045c14092d85f7, 0x0005e041820ffc39, 0x00031138172d223c, 0x000087580008988e }, { 0x000769b5baf39b4b, 0x000017dfc588b090, 0x0009510b05f510e9, 0x000b0b995728cd49, 0x00004ab8441733f4 }},
        {{ 0x00082ded8c15f913, 0x0000f8bf99c24714, 0x000ad0f0c3845063, 0x000745d24f2d8a11, 0x00002d13601a08bd }, { 0x000fc739d67f549d, 0x0000cd8dadd48854, 0x000ea20b7b714a04, 0x000b658507004477, 0x00002739d35a18aa }},
        {{ 0x00010188c19608f8, 0x000b727786f08a98, 0x0005c1c0eabddfc4, 0x000452e708700bb2, 0x0000b09a61e07f86 }, { 0x0009d4f07bc35bad, 0x00023c9abcd3297f, 0x0006faac186ffef6, 0x000734030ae4c896, 0x000068869b2a6b55 }},
        {{ 0x000a11de9f5e8da7, 0x000e7d583fa1e08b, 0x000e0499f35bfc4b, 0x000ad6a967780d33, 0x00006387ac199d31 }, { 0x00028fa43af1c617, 0x0001dda02c2d064f, 0x000cb1fe3e00767d, 0x000ca72d87ecf360, 0x00008656d8273918 }},
        {{ 0x0003edd0e98b762e, 0x00054cbea4570b35, 0x000b51fdfe00672e, 0x000f31ed5ee65d68, 0x00001bd0155d4d2d }, { 0x000500888c6c33f6, 0x0003d07beb9bc6d2, 0x00077a5bdedc774a, 0x000be200fe9abb90, 0x0000328aca3a48e5 }},
        {{ 0x000d700c7f811577, 0x00027d9926fce430, 0x0006f76a1cb0b72a, 0x000373b27809dd1c, 0x0000b7a49c601709 }, { 0x000574395a4d1adf, 0x0000b33ab64a0652, 0x0004edc4a625cb0b, 0x00049167faab9b73, 0x00008e3cbcb5b34e }},
        {{ 0x000d0301187bc101, 0x0000eb6700a1acd0, 0x0003f09695693995, 0x000f3e0bae823fd7, 0x0000bc494f5d06f0 }, { 0x00015737c0a7b0f1, 0x000bfc989fca8cdb, 0x000b4882d64acc9e, 0x00080e66f970d01d, 0x00002a01327ea3d1 }},
        {{ 0x000860368ee95e14, 0x0001fe8a8b484e77, 0x000901c6a838f33b, 0x000c0ed36b2fdbdf, 0x0000699447c715b5 }, { 0x000e12a5ec3c2973, 0x000b91cf9ad3e183, 0x00072a8ca88d9d13, 0x000267eb66d6c5f1, 0x000078786b1aa576 }},
        {{ 0x000c54b9cb562934, 0x00040e9a56b872f8, 0x000c70a1d4d6fe0c, 0x0002cdaf7a1ad550, 0x0000563272b206e9 }, { 0x000acf260393473d, 0x000cd4a088ac342d, 0x000b3aa7a0ccb38c, 0x0004f4ce9ec450c7, 0x00001b6e01fa188c }},
        {{ 0x000cba2eeb210411, 0x00026c6c8cb6b42a, 0x00030cb4c496878e, 0x000ec67251326fc9, 0x000017ca1f4e8294 }, { 0x0007a18a6d684fc4, 0x000c47a633c30000, 0x0002d02a45040c90, 0x0001d2dbecb6cd17, 0x000008914cbb715b }},
        {{ 0x000b3294258afecb, 0x0005b799edce57e9, 0x000443cd891c286d, 0x000a327c652b892f, 0x00002fc850e5378b }, { 0x000c76a331bc95f9, 0x000e1f2bd2e21df5, 0x000ca5d06767748a, 0x000c7ba2e55298d2, 0x0000676b98053ae3 }},
        {{ 0x000d2c801bc4bf64, 0x000fdfff8a4f29c3, 0x000099630d2de628, 0x000f1ce1c70dc924, 0x0000ea1da14883bc }, { 0x0001e8e630540696, 0x0006172425218760, 0x000c3f839381a073, 0x00037c5c90bed93b, 0x000079f89d6bd4d2 }},
        {{ 0x000f0e75ce977889, 0x000de38c6f3431e1, 0x0002c417ff72df79, 0x00013fe7ce609cb0, 0x00009a2977125eea }, { 0x000b453193606cdc, 0x000ee641158544d7, 0x0008f171ed3d9490, 0x0003d6b555d777e8, 0x0000d3214ae1a449 }},
        {{ 0x000a448a130c115a, 0x0009c8839f05ffea, 0x000fc33dcd3bcf48, 0x00017d31b8f4f4bd, 0x0000992eb014f9e2 }, { 0x000af5cebde70108, 0x00089b3f7914967f, 0x000a1e12d72fc4cd, 0x0000f3ae74537f09, 0x0000f1941d7b610e }},
        {{ 0x00011f21588397cb, 0x00032874ca6b39ac, 0x000c41d0b1e9ef78, 0x000c4c3db70fb72e, 0x0000a9adbfdfa153 }, { 0x000e9af3e27ac409, 0x000ed35bf3bfebf3, 0x00042d3e6c3d4995, 0x0005cc2733a1ff0b, 0x00008605fa3d4149 }},
        {{ 0x00028d0a1e02ab66, 0x000c672b41c34403, 0x0005b0769041de1c, 0x000dc2fa3175239e, 0x00007eb126a76c52 }, { 0x00054fa107e49200, 0x0005f2ed3788a049, 0x0005d171799725b7, 0x0004492d625d8ff1, 0x0000ec4457bdfdba }},
        {{ 0x0004ee0734a79d5e, 0x0004df2e98aaa995, 0x000041c2f15629cd, 0x000b6a46a90cd8ba, 0x000006e82e9e4e19 }, { 0x0002e25421fbf8ee, 0x000b05445b4147a9, 0x000107702d4ac0ff, 0x000bd40773d102f2, 0x0000f9d11eeb5c3d }},
        {{ 0x000a2e45bf1c1c78, 0x0006071bb608558c, 0x000e2ca6b9dcbb05, 0x000d5fe3e60e4eba, 0x0000055d8f3a5a25 }, { 0x000d5d7316bcf10b, 0x00067ec2bfd7a5f1, 0x000f57733e16a9e0, 0x000cd65dc528a8f2, 0x0000538112b6221d }},
        {{ 0x0004396860fdc301, 0x000174adf423e315, 0x00015aa1b1af53ef, 0x00098c9b9cb41729, 0x000050c9de633264 }, { 0x00088ae797a2c6e3, 0x00013c4a54ae57ef, 0x000ff5d9a705ab8d, 0x0003413bc0ffeb58, 0x000021a5f9a644c8 }},
        {{ 0x0001abd8610e73d8, 0x0004544d2359ed9e, 0x000311a87481122b, 0x000de0d69ac68dd0, 0x0000f82040d9e6ee }, { 0x00048969704b7037, 0x000e1d9343b6e3a4, 0x00086f80b23dd859, 0x000cf0767480797e, 0x00006cd8d794a707 }},
        {{ 0x000fc09d2dcb11e8, 0x000285552a82e8de, 0x000488856faec190, 0x000f18978b2caf78, 0x000085276cc6351d }, { 0x0006b1386691ac89, 0x00070ee9113be5c0, 0x000fad47eeb5cd81, 0x000d773542fbda78, 0x0000a86b95d248fd }},
        {{ 0x0006d449d8e3dcd4, 0x000740f43e3865d4, 0x0007d70afe005519, 0x000438bf772dd77a, 0x0000b18fb1739868 }, { 0x00018bef015b1939, 0x000e0ea185d706fc, 0x000b06519061374c, 0x000a2e731e47e02e, 0x0000ca817e6aa631 }},
        {{ 0x0002887ba15c20fd, 0x0007d9c6bdf22da8, 0x000f0e54073dbcd2, 0x0004f3951efbc432, 0x0000469ec570ed01 }, { 0x000deaa2e18dbfd8, 0x000c5dab920ec1de, 0x0002fe53ea59ef2c, 0x000e8af47d0d7e8c, 0x000015e430f8b3be }},
        {{ 0x000111a7619e2658, 0x000c5936cc874756, 0x00020c11dc5ba7f4, 0x00069dd46d92a52e, 0x00007437d17bf839 }, { 0x00049cab0510e4cd, 0x000b7feb3d5f6126, 0x000cfc4b7de6fa00, 0x00023a3305ea69b8, 0x000003184fab8043 }},
        {{ 0x0000efb2747d10ac, 0x00041cf8bdce214d, 0x0001b5bfc0eca8dc, 0x000280de2b5622f7, 0x0000eaba332135f3 }, { 0x00040a52c6704d02, 0x000f68b79df975b7, 0x0005a58efd781132, 0x000ebea46856bf28, 0x000034b117e458df }},
        {{ 0x00080784d755ed6d, 0x0004e152cbb1a3f8, 0x000d8fa0af51d84b, 0x000794e74c3eb99f, 0x0000d39835b41a87 }, { 0x00041e50a81e7f57, 0x000f51cb4e7a0be1, 0x000da8f75687bf48, 0x0006cd9e18780510, 0x0000e64f1df80763 }},
        {{ 0x00043afaf8568dee, 0x000691847be87ae0, 0x0007f32f4879b707, 0x00079d5d368a6141, 0x00002b47d1718f6f }, { 0x0004001ae4f33965, 0x0007c211552a4d28, 0x0002374d2fcfd8db, 0x000feb1a4b6dee69, 0x00000e25e4bee3ac }},
        {{ 0x0003473caf8cdb36, 0x0004d44d795fb455, 0x000d645818f5cf12, 0x000f1a9fc57326e7, 0x0000b402e73630bd }, { 0x0005614508afbc82, 0x0006993a13df4c82, 0x0006726a9f66c94a, 0x000284af8a73e51f, 0x0000c55c0677d85b }},
        {{ 0x0000b9c41619d1d1, 0x000286211a71b27b, 0x000abafe3623481e, 0x0000622efdf71412, 0x00003f1c33b3a212 }, { 0x000e3bcc6e9333c9, 0x0007c52014cdde5e, 0x000a343f7f4d5b8d, 0x000b5bff3702c624, 0x000071d00de55440 }},
        {{ 0x00046ed52661de90, 0x000f22a42ac6c853, 0x0005e98b1d913b21, 0x000ff8e582db0cea, 0x000088053a8d84e0 }, { 0x0005662908845062, 0x000c7951a2faf097, 0x000006bf38262f45, 0x000a502902fa8a58, 0x0000693f983dd1f8 }},
        {{ 0x00020c5984ecc7c0, 0x00035c53d080847c, 0x000890357b714942, 0x000ce5a63cb081d3, 0x000080dfb69e8b8a }, { 0x000fbd9c6a0c8dae, 0x000fc7cf902f245a, 0x00059e95ceb430b2, 0x00025654f9cb1273, 0x000018bbaa36708d }},
        {{ 0x000322f9d772ff68, 0x0007b0ac7088c7d2, 0x00000ee43ec82717, 0x0004112535ab6571, 0x0000826ae5c2a874 }, { 0x0003b5b6363c24f8, 0x000fa2d7430de4de, 0x000602d6f37384cd, 0x0006960681938269, 0x000039706131bb55 }},
        {{ 0x000a9ac3ed8e4ce8, 0x00011803fbc43419, 0x000fc0c429608622, 0x00035fc1d359f2c7, 0x0000af92e79860e6 }, { 0x000cd6f4e3313ee1, 0x000c5fe91fc50ae7, 0x000b493d05996022, 0x000f4105a89ccc66, 0x0000e63d8e773195 }},
        {{ 0x0000f3ffc1435cfd, 0x0009b69ce56b40e9, 0x000c2f39ded7b89a, 0x000912fd7f93e094, 0x000003a249c0e7aa }, { 0x0003ffdeb1ddd9da, 0x000235608241aedd, 0x00072cb1d136a3ce, 0x00057335295ab7ad, 0x0000f449ec6ba460 }},
        {{ 0x000389cfbf4c9e0c, 0x0005091d11a8594e, 0x00074e459df97c5a, 0x00017d3d02e74d25, 0x000044ebb0b8e3b6 }, { 0x000ea3c28478a740, 0x0002405941768d8b, 0x0004451bbfd95774, 0x00025d7e24510399, 0x0000fd86525ee1de }},
        {{ 0x000301e47f2cd335, 0x0008b6ed9a475207, 0x000a3c833f852147, 0x000b42f8afbfe18a, 0x0000580fe7496353 }, { 0x000afffe040e720e, 0x000ebddb299b9777, 0x000b2ab785106b87, 0x0000a82e776697a7, 0x0000b8a44884646f }},
        {{ 0x000cfb5fa19ed6c2, 0x0007c4b28c22cee9, 0x00076d01f60cbc18, 0x000539b6aefb0ecd, 0x0000f84ec5d3906e }, { 0x0001527a37e73f9e, 0x0006d57f36e1ac55, 0x000ce5788145521e, 0x000c926db008fa9a, 0x0000af255a03386b }},
        {{ 0x000db8eaa84aa125, 0x000c9305bd213110, 0x000282e3a62f6f6c, 0x000e571d36ed41b2, 0x00002902def1b031 }, { 0x000bc226cb0016cb, 0x0003337b24b4b6f6, 0x0007e66af842d281, 0x0008210d9fe58afe, 0x0000f22b8a2b6b8c }},
        {{ 0x00062dbbba9d84ad, 0x0008b4ece53c2d04, 0x000d18184b2003a6, 0x0006765b0779d897, 0x000067fc9538f5d6 }, { 0x000035f7ff931704, 0x00005f2cafe37b68, 0x000f6c983617d6e7, 0x000df03fd273d1eb, 0x0000249da2e138a9 }},
    },
    {
        /* digit=21 (1,2,..,64)*(2^{147})*G */
        {{ 0x0009d1febf2de9a9, 0x0008bc77e6c55202, 0x000fb2e7eb995763, 0x0009cbd6dc27df9e, 0x0000444476cdaaa8 }, { 0x000ba7ced0785f36, 0x0004b9e93470afed, 0x000906fab0ce1fe7, 0x000bf2f043e6a966, 0x0000d9c1876fd26a }},
        {{ 0x0008d94f6be7acf2, 0x0000341338d6e434, 0x0002fc6886610503, 0x0000f96ca56f7dd7, 0x00002f54a7c972af }, { 0x0003e19d89ed269e, 0x00005bb2ef8279cf, 0x000a1736ca68762e, 0x000fbb351d575465, 0x000061e8deb175a7 }},
        {{ 0x000f96f33f9e0680, 0x000b268e32dd620a, 0x000eb2e15662f4a1, 0x000310882913ec99, 0x00003ddc488aba8e }, { 0x0003f9b5a7ed9adf, 0x00058e5dc3fa8a51, 0x00019cb68475d2cc, 0x0005558b0236bb53, 0x0000fb0d86c8819c }},
        {{ 0x0002aae2b99711c5, 0x00031ce51efb3108, 0x000412e3130475fc, 0x00003c959cb5b2eb, 0x0000efeaac806f9f }, { 0x0004837f98bf8cb0, 0x000da39411aa636f, 0x0004e03d299b3bd8, 0x0006376b77152ecc, 0x0000b7bc15a5f18a }},
        {{ 0x0001162ade988968, 0x0009ac4c753c45fb, 0x000648b216204571, 0x0006f3be618ca81b, 0x00009a6be30dbc18 }, { 0x00042cd4528f9326, 0x0008720236e3ab64, 0x00056e788ed3e0dd, 0x000e9967af88cdaf, 0x0000c45162d8c709 }},
        {{ 0x00083eca38629302, 0x000f164b71698f58, 0x0002173c96261574, 0x000a82abb6ba418d, 0x0000a7d774e7d73d }, { 0x000c1dd5576a4dbc, 0x000283ff3437f24a, 0x0001f86ddde594be, 0x000503b65e910b43, 0x00004e079387d0c8 }},
        {{ 0x0004a222c42e90db, 0x00035f924a0a3fcd, 0x000413dfa13526e5, 0x0007cd19938e17bf, 0x0000dbc59707ebbe }, { 0x00038d8fd101963f, 0x000debe13f542b66, 0x000d80984486c605, 0x0005578791fc65e0, 0x0000318e04247e1c }},
        {{ 0x00039d3ee2963f2f, 0x000233a626332d5d, 0x000281c47c620310, 0x000452fa27bc3883, 0x00006bc300c4cda7 }, { 0x0004c341a09ee9c3, 0x0003ad3e7676b0dc, 0x000f36ad1e76c678, 0x0007b36a8620e35f, 0x00009710f4c3af53 }},
        {{ 0x00073207beab3a63, 0x0009a2f69beb6e85, 0x0003163e05455030, 0x000493e4f3a50b99, 0x0000ae1c490dee61 }, { 0x000c1b26b4aa0c62, 0x00020e309fe7e5b9, 0x00039f25812cb0ec, 0x00023f4e2957678a, 0x0000be251e8728aa }},
        {{ 0x0000e7c2becbd6ac, 0x000da1cf405ff065, 0x000cf786bfcc5059, 0x0009287fcebac86b, 0x000006628092b297 }, { 0x0003a21f33a3f231, 0x0006f95fa90d7679, 0x000b52ce8c481fe2, 0x00088d46dae60eb7, 0x0000dd27b34c3095 }},
        {{ 0x00081eea1daaae26, 0x000110babb886433, 0x0006eac32b7bbb8e, 0x000c7f07191df36d, 0x00000e8acae417e0 }, { 0x0009aba1faa58bab, 0x00046ba39966e6ca, 0x00083d7db2981427, 0x00026c07f7c464a2, 0x0000d90bb6000f64 }},
        {{ 0x0007112386fc20ce, 0x0005122369b87ada, 0x00088beb81cf895b, 0x0008f96663c00e5d, 0x0000786fe6f72494 }, { 0x0003cd4c08b1b97b, 0x00066986d9bd5f51, 0x000845f5fa36c27c, 0x000c259b22dcc7e3, 0x00003a7a6a264018 }},
        {{ 0x00066ec8de9b7d5f, 0x000fef8e32377106, 0x000e65642ed171bd, 0x0002ca92b0dadb26, 0x00004cd467fccd49 }, { 0x000b7d34d2311a46, 0x000d8d5ee5242161, 0x000b265d7be3eae2, 0x0006c90bacd93c59, 0x0000677b279563a0 }},
        {{ 0x00014104124194f3, 0x000f16022caf46b4, 0x000f5955fa3d17a1, 0x000fa544e6a45dd8, 0x000027f7e6432277 }, { 0x0001bcd329f0aee1, 0x000ac1241de97bff, 0x0000cbc3ebca8120, 0x0002c1f37b8547b6, 0x000028b36f4746d4 }},
        {{ 0x0003e297caf789c4, 0x0006a0910caad676, 0x000ff14f0a163a1f, 0x000c604401f59195, 0x000036cf5f7f1567 }, { 0x000cd19846f4da2f, 0x000dc2e49abad67c, 0x000c206c18924750, 0x00014ee406ec405f, 0x000043041d3be98b }},
        {{ 0x0004a07f82ebea7c, 0x000b5769bb816483, 0x00053090a14d150d, 0x000e04062d69eb48, 0x00003488c7727160 }, { 0x00062b085ef31ed0, 0x0002787bbfb55541, 0x00004cb77b99391c, 0x0007343a0b06ed4d, 0x00001334165040ae }},
        {{ 0x000ae1cac5e3f9c0, 0x0000b31e0dd3eca2, 0x000cba4d48dfd8b5, 0x000c10297c6237fb, 0x0000108543603d43 }, { 0x0006f85dc5404165, 0x000a2d47845ae4f6, 0x00026761151b0706, 0x00037b0bbfec7dd9, 0x00008f9d062c7bdd }},
        {{ 0x000c6ff6e5be4b0a, 0x000ae88bdbcf0ca3, 0x0006a2b7a7ba1b36, 0x00009df34d75f5da, 0x0000309466d23eb2 }, { 0x000f30b5ece886f8, 0x00084f4047d26e4d, 0x00077119364a90d4, 0x000524daff370f7b, 0x0000a8cfc76f60e6 }},
        {{ 0x000e32f8475f689e, 0x000ae624896246e5, 0x0006905dc38a3b9a, 0x000963d735652c01, 0x00005aec6e092828 }, { 0x00069e5622c8ef2d, 0x0008612cf933c835, 0x000dc3e0f1c33e14, 0x000acf715d05bb76, 0x00001af4af25e9cb }},
        {{ 0x000f6d12f8d61cbc, 0x0002f3a6568013e8, 0x000e93853103e5cd, 0x00025b74a8dbd203, 0x00007ce46d1f9bdd }, { 0x0000b6a7c6e4974b, 0x000c5a2c9eade291, 0x000337266751d4ee, 0x00084c246bab4f32, 0x0000f12b5a16b482 }},
        {{ 0x0004d506bf73235e, 0x0007ff33035b41cc, 0x000ed4ef51d687f6, 0x000a4f9e5d336343, 0x000094104ca078e1 }, { 0x000895041f284f32, 0x00096cbbbc33f758, 0x000a09cdc35e1489, 0x000caf6237281f08, 0x000051a7f97c87c8 }},
        {{ 0x00013888f004cd15, 0x000d5499c438edfc, 0x0002d0b53154f812, 0x000ed3bf07678dcc, 0x00000b340fc797dd }, { 0x000ce0637b7a5424, 0x00036d41140af420, 0x0003858e6e2c47bf, 0x000a2455f8e5104f, 0x00009c5fb7f1500e }},
        {{ 0x00000c81b681942b, 0x000d6e6fe631e8f0, 0x0007d7b150953146, 0x00006a94f1c5563d, 0x0000b00f364d03b5 }, { 0x0006d17634ee0c1b, 0x0008e51aae1fd9a7, 0x000716897755713c, 0x0008cab22e35626c, 0x00003897cfcc2e16 }},
        {{ 0x000127ee79d5b629, 0x000e4daeb300f7aa, 0x000c4c8df9876122, 0x0001efb4c9dcf7df, 0x00001b98eaecee9f }, { 0x000be67d99c079ff, 0x000cd53575f1dbba, 0x0004d4ff98848810, 0x000ac51d095037e8, 0x0000af6c7913e968 }},
        {{ 0x000227b7d5fa2779, 0x0001b4b3f8e10652, 0x000610e6d14e4258, 0x0002b8836ed908cb, 0x00003ee0ee50ff75 }, { 0x000381ae0cc87753, 0x0007593b5cd6dd54, 0x000162a5e8db86de, 0x000a18e2c82225e1, 0x000026e1628fc502 }},
        {{ 0x000ea9dd3d739ece, 0x00045d37fafa4c8b, 0x000145e7cd82c0ba, 0x000ca584c3d91808, 0x00000a483c77e1ca }, { 0x000b89ee5c5235ad, 0x00023673eadd3e7c, 0x000739e9ce7c6b70, 0x0006c9b6c0033c1b, 0x000042ac04521d0b }},
        {{ 0x000ec25669d8ed84, 0x00024f322532e5d5, 0x000013bba5e2b6f3, 0x00079a2dbb6410ff, 0x000030dbc6caee49 }, { 0x0005a7ea04cc0c9d, 0x00091064c3beb017, 0x00000526bb51e340, 0x00076ab7a852123d, 0x0000ef64f923f62b }},
        {{ 0x0004049e0b602df5, 0x0009a356798e4ba6, 0x000563cc4285eaa9, 0x000fb0f039214e6e, 0x000054c04a0b7d50 }, { 0x0008dabf1a099126, 0x00034f13ae9841ff, 0x0008b7dba38bfa5e, 0x000b40e58dbe4ca1, 0x00003b573ac345f0 }},
        {{ 0x0000c60f36acc716, 0x000daaacce2793bb, 0x000e1526388b0989, 0x0005c39a971db851, 0x0000043ab896078e }, { 0x000ff069e32c40e4, 0x0008d21774eadbd4, 0x000f7e41ce7fd901, 0x000c7857625367fa, 0x0000b451922fe6a0 }},
        {{ 0x0000e7363ef92c3e, 0x00014568776e6676, 0x000c515535767ae8, 0x0000e4c876e76142, 0x00007574380ed322 }, { 0x0004548dca9348c5, 0x000800eb7366d800, 0x000a43e34e14c3f8, 0x0008b6c6fe4ee14c, 0x0000f0365a22d0d6 }},
        {{ 0x000b8dd3d3c731b7, 0x000974c5ae888eb9, 0x0000bb5bd8c5d56d, 0x000390929733abb5, 0x00006206511f4cf3 }, { 0x0006c1ff0991bcef, 0x0005d920bc5d11a2, 0x0007759d899a17fe, 0x000604db8f20127a, 0x0000afba4da06484 }},
        {{ 0x00027d09f86b87a6, 0x000672f7189e71f4, 0x00081c5287eec6df, 0x000edd421c643874, 0x0000dd3e802a5f6f }, { 0x0005cba1123d99ff, 0x000318f38384a7a9, 0x00041aa78a8746d0, 0x000ea5919aac3acb, 0x00001546626df6d9 }},
        {{ 0x000d7ba8b08355a7, 0x000a87fb00009c49, 0x0006f276e2aeebe7, 0x00041f05db8396c2, 0x0000859783ac9299 }, { 0x000c3a6250aec83b, 0x000b59f34f485b64, 0x000c17d7c2c584c4, 0x0000c36febc0ddac, 0x00001a06deb2c157 }},
        {{ 0x000630745c2443f6, 0x000db03d1295837c, 0x000f71d776d34583, 0x000701e91a2893e4, 0x00009134f5638299 }, { 0x000d8586465dff85, 0x0008b9e48661ae16, 0x000b7ca5a02e6a92, 0x0002951b58d17e7e, 0x0000e3f1660e4165 }},
        {{ 0x0009be4d8ef6f6d7, 0x0004d3e19115ead5, 0x000df1ee024075e8, 0x000331b5b1ce1393, 0x000093c9ce9841a5 }, { 0x00058519cc609023, 0x000bd6cd0def72b5, 0x000d8aaf0d7bacb6, 0x000dd273db923db5, 0x00006b02deb40132 }},
        {{ 0x000c548b69ea5df5, 0x000f6f8827097ef5, 0x000b740069f3544a, 0x000d0db89259353d, 0x0000e612d2045282 }, { 0x0008b57bb32e6372, 0x0008416994b995dc, 0x0005dcceda4ea724, 0x000bc67f04d1531f, 0x00007cded8a96b16 }},
        {{ 0x000c6b9000bbb209, 0x00006cb1cd5ad73c, 0x000594d94a4eac92, 0x000b0b5b8b9c860e, 0x00007f19f8ffa9c8 }, { 0x000d03e590e97791, 0x0007ca420c8a471b, 0x0000bafb9a1ec30a, 0x0003e497769ae05a, 0x00000ae589eaa8d6 }},
        {{ 0x00098231b2fdae8d, 0x0009e5eb2940d9e2, 0x00062218c3b180d7, 0x00034763d3c663d8, 0x000079ac5ceb7ec0 }, { 0x0008b408269ba673, 0x0009a2c552ad093f, 0x000c06d13e6ea662, 0x000789a00a4799d1, 0x00006bad7b20b037 }},
        {{ 0x000f3764f7b38e54, 0x0005dedf1bd72186, 0x000adbed6377389b, 0x000079665d491c53, 0x00000d1755479780 }, { 0x00048c7a526618d7, 0x0003a6dcdc348009, 0x00062d56bf03752a, 0x00092516bcdbcd45, 0x0000cec167b1ee0e }},
        {{ 0x000439be9201d0e9, 0x000c1377378058e6, 0x0000a00752c2e9b0, 0x000b1a1c741c746a, 0x0000bb3ea39e2af9 }, { 0x000f7de5688e9cd2, 0x000e2b043a2fbad0, 0x000f733802d7bd56, 0x000faeb4c39e6dad, 0x000033ce90509fd4 }},
        {{ 0x000b539dc2f62b1e, 0x000a7a1d2ffbfc15, 0x0009f177c68e623a, 0x000880758f74211f, 0x0000f1010e0ac5ca }, { 0x000a989eca797cca, 0x000b9d02f3af6f6c, 0x0007cefab003de17, 0x0004d1f835213207, 0x000053ff9d904563 }},
        {{ 0x000c547b711b42fe, 0x000ca29d6f62f995, 0x0008efd6009b4f01, 0x0005dcd0a8f06a30, 0x0000d50d91362f7d }, { 0x000916e5205bf484, 0x000bce1682afe112, 0x000a522adb87ea28, 0x00018d9369e0fc77, 0x000083acc24ddd86 }},
        {{ 0x000fc96822eaed69, 0x000e9e83f883faf0, 0x00091bbd25130ab0, 0x000b42f81396f963, 0x0000c9a426aabc61 }, { 0x000ba8446d25cd38, 0x0003d8766f1c68d2, 0x0003cf9f27a01b56, 0x00052259010fc6de, 0x0000b7f2dba90c5e }},
        {{ 0x0002ac509835efa4, 0x0001520ef4a9975b, 0x00037083f2c76c48, 0x000c843b560bb600, 0x0000295a37fec299 }, { 0x000d83ca9a68bcb1, 0x0003d2fb0d1ea67c, 0x0009a7419796877a, 0x000c6d9e6ae779a6, 0x000089f4f8df2e42 }},
        {{ 0x000ecbc81ce29d74, 0x0004af5e33deed3a, 0x000e514cca8bd88a, 0x00042493ee6356f6, 0x0000b9f30d92af98 }, { 0x000e29242f1443f0, 0x000c909e6e861dcc, 0x000385ccd3d2d0ba, 0x000f07e5790a247a, 0x00004947dfd69c12 }},
        {{ 0x0001d1645c39f241, 0x000e04a2f998a912, 0x0001b2e98e1ecaea, 0x000d02622271ee9b, 0x00004196510475ae }, { 0x0005bbdedb159a7a, 0x000aa2c7501779c0, 0x0007f452a12df2d4, 0x0007df729f006140, 0x0000ac021b7f9c4a }},
        {{ 0x000c314056ab1eb9, 0x0008f3225d984730, 0x0005503443d0bd10, 0x0004846a1f207fe8, 0x0000f455eb506396 }, { 0x000e1c005b309371, 0x000938f4107e28f1, 0x0001230567e4dd24, 0x000104f6e74ed387, 0x00008f86c0103426 }},
        {{ 0x000a793b2657b1ce, 0x0005c73f37ec3c39, 0x000d42e0b07a8381, 0x0009b6457c65259b, 0x0000b5dfca037683 }, { 0x000a1d4307f2b6b5, 0x0002e4be43756f0e, 0x000db0de13c200e9, 0x00069d6272f76bdc, 0x0000caa7d0f8726e }},
        {{ 0x000ad40c5a7ab2a8, 0x000ab308bdf623a6, 0x000a196def2311a6, 0x0002b55955f5accd, 0x0000420ad92fbdb8 }, { 0x000ee1c791d19a99, 0x0002189aeca8709a, 0x00007d42c30b7260, 0x000c8e176c64a7cf, 0x00004c391ad3c0a0 }},
        {{ 0x00097b04cf4dcd27, 0x00071fd21a843b21, 0x000d738557e2a32a, 0x0004063be56e4ed4, 0x0000f569b199b0d1 }, { 0x00027ecdb28f119d, 0x000161ffd42c13f0, 0x0002b190c58ab2e8, 0x0004f1dd45ef7feb, 0x00003bb3c53e8fce }},
        {{ 0x000985b93206998e, 0x00056d14e3352feb, 0x000eb9d3625dfe11, 0x00040ef6008414d2, 0x00001dabe9e71cb1 }, { 0x00010ec86988d8f2, 0x000e1ab96bf36f90, 0x0005bad089761cb9, 0x00093257d012dbed, 0x0000392d438d5756 }},
        {{ 0x0006074b7b7f3ff9, 0x00007f6bf926dffa, 0x00014eb820473ad7, 0x0009a18f57c9fdbd, 0x0000f0b5518a13e5 }, { 0x000bb55bbfa1756b, 0x0009566e84f44a48, 0x000846c6398e72b1, 0x0007e239a99998d3, 0x0000d5d77e99f313 }},
        {{ 0x0002f49e3766fab5, 0x000bdb40ea2ab303, 0x0000c448caa69039, 0x000c82cebd365a34, 0x000050f4208830b2 }, { 0x000004213cabc507, 0x00050f101d8b4d66, 0x000bc6b055c40b77, 0x000137b6a03c8423, 0x0000747dddf18bf9 }},
        {{ 0x0008f4ffcfd9c86f, 0x000f5a1c1c06681f, 0x000ef60c8e811dc3, 0x0007a43b316e0a16, 0x000027129f4ec201 }, { 0x000aba00062f76e3, 0x0008ecd4cc5e3e06, 0x000a366fe3da9e55, 0x0003be8f1d55c7cf, 0x0000d191984e99e0 }},
        {{ 0x000c5d7ec808f4e1, 0x000fa30d232cfc4d, 0x000df92cbe274727, 0x000ebb0499ddcf12, 0x00001e2f6a31d47e }, { 0x000abfb950d7fce1, 0x000b76e0b50d7dd6, 0x000715bf57d98c4a, 0x0000e08f688fdbaf, 0x000030c606a9d9d8 }},
        {{ 0x0002335df2a1fb08, 0x0003c878a429f4f8, 0x000b591e08172eac, 0x0006eaddf0649712, 0x0000e4d7ee646545 }, { 0x00038956fb888a73, 0x0007d675c93d9950, 0x000a64de24ea9aff, 0x0004f221b59ac367, 0x0000ca2c7aa20399 }},
        {{ 0x000ac7d898fb100b, 0x0003dcbbb734e37e, 0x0001411dd9d6657e, 0x000a195d9c08c0bf, 0x0000b128040fa2fe }, { 0x0004303f220f9f80, 0x000cbafbc1a6dea1, 0x000f02fa70b9b9bc, 0x00096441afdd0e2b, 0x0000f73ba67d9a72 }},
        {{ 0x000a9157dd651c99, 0x0009d210408f237b, 0x000a32417a1054e9, 0x0008f1da2cad3b09, 0x000057f0f2c67ced }, { 0x00053cba344ed082, 0x000b3074f013efbe, 0x000ac4fc0a6eafae, 0x000d5c1451b112c4, 0x00003f0ad4322c4c }},
        {{ 0x00074ea0b52dc146, 0x0001a63ded40d232, 0x000d2d41ed701785, 0x0007af6313d5f1f4, 0x0000d9a531a3ad12 }, { 0x000744ef50a57f11, 0x0007c99451d1b71a, 0x0002e834e0295a02, 0x0004f26f580e8264, 0x00004d9fa000c2ce }},
        {{ 0x000fe5e5592df016, 0x000d1f5cf551396c, 0x000db6a682124601, 0x000541636c616898, 0x0000d5c526596e5e }, { 0x000fa2cd6fae11ac, 0x0002c187051f476d, 0x0005354d66d26431, 0x0003324f656fad2a, 0x0000bcf48f474c5c }},
        {{ 0x0002136d7196d889, 0x000f83ad15d15174, 0x0001f569d783786a, 0x000612a5f61214cb, 0x00008b285ec5b82f }, { 0x000ec5c4199a8b19, 0x000581b084172219, 0x0002b38e963dfce9, 0x000c53f2d0a5546b, 0x0000d57c27a12776 }},
        {{ 0x000378c25b25433e, 0x0009ead771849f1d, 0x00042e5efd140744, 0x000ce30b2b0c932c, 0x00002dbc7048e363 }, { 0x000fbfe01a53e807, 0x00099da9e6e3e319, 0x000d16caf5b6722a, 0x000dee6b80417a54, 0x0000e7b46d452fce }},
        {{ 0x000e093761507e7d, 0x000a805ba9384a2d, 0x000d86a4325fdf2a, 0x000bcb8a621b8596, 0x0000d301c511e847 }, { 0x0004f1d7d5d3607e, 0x0004ec8f7bffe6d9, 0x000ec8a24e675bb4, 0x000936e3027566bf, 0x00008ff4238ff05a }},
        {{ 0x000842e7e3646f13, 0x000721aaaca5e9b7, 0x0000de2e53b43348, 0x0009b4ef6518aa52, 0x00004c1f8413e3b0 }, { 0x00027091000d9504, 0x000ab656868a5489, 0x00024df86b81806a, 0x000fc7333d963bd8, 0x000027dffecf8b57 }},
    },
    {
        /* digit=22 (1,2,..,64)*(2^{154})*G */
        {{ 0x000de8cdd0926a17, 0x000b2a7d298c241d, 0x00097897af2c8ee1, 0x000fad3dbe3b697b, 0x000095a39dc131b2 }, { 0x00088519ea3f0b69, 0x00087a75a4d2604d, 0x000e93901ec55560, 0x0004fb6cc27154d7, 0x00006bfc52e1469f }},
        {{ 0x0002571fa1c005d3, 0x0000a1a7cb1282c7, 0x00064bef7798f823, 0x0000303b2a08d762, 0x000094b95e409f27 }, { 0x000cc832f936b83a, 0x000392ff22dfd98c, 0x000633bedd944ef9, 0x000cfe67a87ab01a, 0x00004e149b05af22 }},
        {{ 0x000f839761bec299, 0x000fc0e36ff3bf0c, 0x000f212faed6d78a, 0x00014bf5f8ad20f6, 0x0000003cd07951f7 }, { 0x000f568724f96e64, 0x00061b38288c545c, 0x000a43bae4ac5ec1, 0x000b2bcef2f8e4e9, 0x0000482335f80a39 }},
        {{ 0x0005d70b7657b7a1, 0x0007085ca8d9d1ae, 0x000cf3b0d35bf0d7, 0x0000cc024adf2c77, 0x000069d110cf7a09 }, { 0x0003a7564e157769, 0x0001d506260e70ff, 0x000703a97ab76d5e, 0x000435a6439d75ea, 0x000076aec3e36360 }},
        {{ 0x000b491d7bce26e6, 0x0001ecd2f52d2a7a, 0x000a379638de4a08, 0x00078ccf74fde149, 0x000012daea2770d9 }, { 0x00082aa266a6dcf5, 0x0007718c6312c7fe, 0x000c827977f0f2fa, 0x000766ea0ca0d5f3, 0x0000282fd2c5bf2a }},
        {{ 0x000b4394784f1e40, 0x0000700ec3128c2f, 0x000192f4aad1458d, 0x00011e4eb6c76d86, 0x0000e130115cd871 }, { 0x00052289429acd15, 0x000614a8dc796230, 0x000daf1e7270619b, 0x000f8d90d5b86995, 0x0000166c13057d31 }},
        {{ 0x0006fdbec46ef830, 0x00010a51e6a75c1d, 0x000537aa19534622, 0x0007f159724327c7, 0x0000279e5d72bd98 }, { 0x000d87b2825485ae, 0x000effb13c90f48c, 0x00029ae73fe0cc43, 0x00070a0ebd6ac527, 0x0000b44da3075aaa }},
        {{ 0x0002f99d592d4607, 0x0009d3e3fcd3efca, 0x0004150ef1418504, 0x000f73feadd26c03, 0x0000c8303bb7708f }, { 0x00028bb4209cedef, 0x000e4552f1da46cd, 0x000d486f19b140bd, 0x000e6167872c3f8c, 0x0000b4a36e89cdf5 }},
        {{ 0x000c77a0a3360371, 0x0009611745edc116, 0x000cd150d4b51af8, 0x00099ad649dd9ad7, 0x00009c97f502587e }, { 0x0009ae48860379bb, 0x000eb296baf7cb10, 0x0001c348e49c5f3f, 0x000e9d571cf065f9, 0x0000b56bcf3f6375 }},
        {{ 0x0003b56be8e6e5b5, 0x00086ecee8314f30, 0x000dd834bda67796, 0x0009af7ec1c068ae, 0x000043ec03cb0895 }, { 0x000635be527dea03, 0x000b5f29222882d6, 0x00065e79cda53115, 0x000db7a8c95ff38f, 0x000042689771e2a1 }},
        {{ 0x0006fa038810126b, 0x000e776a5177900a, 0x00074f8b83c9f4be, 0x000030e6c4e1dbb4, 0x000046c45fbd9ab6 }, { 0x000f312eb8992e97, 0x000c6562844c6d0a, 0x000a547118c55450, 0x000088ad6c633cf9, 0x00009ee0bbc48794 }},
        {{ 0x0007709b7b97f8c5, 0x000cdd2e7417ce17, 0x00011f9e4c798c2e, 0x0006edf1eff42bc7, 0x0000d3407d05f1c4 }, { 0x000b2930ca06e395, 0x000de1dbe217f2cb, 0x000ca08df0ad2e70, 0x0003a3b592af2a8c, 0x000040a4a94b9f52 }},
        {{ 0x000519834b80ba88, 0x00058477d930cfc2, 0x000114564ea90584, 0x000e06fe14c262ad, 0x0000425017607c82 }, { 0x0004968949e7bfda, 0x000414cd105f961c, 0x000511a5410a9516, 0x000e2177ca5415da, 0x0000759aabd156a0 }},
        {{ 0x0008eb87587c0983, 0x0000e26cecdaa7af, 0x000d12279d229b41, 0x0007b3ec4fc8c7e0, 0x000092acdf0c79b8 }, { 0x0009f00adefde818, 0x000ea12c00674527, 0x000e01ab878d0446, 0x0005ac2f43f6f69e, 0x0000c28cfb3ac1f1 }},
        {{ 0x000aae38b1dc4f8d, 0x0004e7181236c972, 0x000b14323f6d85c1, 0x00094867274f7685, 0x0000efff992ec9c7 }, { 0x0002818fa5d10725, 0x000bd22f6dd9d1bb, 0x0007a909d4e14581, 0x00027b9aa5769544, 0x00000f6612b6817a }},
        {{ 0x000cf60f96d10e48, 0x00095daf176f2c08, 0x0004556116c14d5c, 0x00025e7fb01ca460, 0x00006c4e588656b4 }, { 0x0007fb754d3440e4, 0x000851cc9071c4aa, 0x00038c48b2b6677b, 0x0005b2981fd58874, 0x0000cc23558b384c }},
        {{ 0x000a3165c2d1e96e, 0x000ce5cd51805e1f, 0x0008b86876e5f775, 0x000018700ab4ccd3, 0x00005515bbda4876 }, { 0x00052c258fdcb035, 0x0005654e160e552a, 0x0009eaedab8f23ae, 0x000ea157adc4972b, 0x000066592aa69e2f }},
        {{ 0x000ffedd92412715, 0x0008af4c1e99d816, 0x000633c73a2eb72e, 0x0006556a708e1a7d, 0x00002684c134b0e3 }, { 0x0009bf250d515cb9, 0x000a0d2d629d2080, 0x00052ad12e01390d, 0x000f3a089a9d141d, 0x0000fb8a94900ae6 }},
        {{ 0x0005d66a6d649943, 0x00090b1b38227904, 0x00004de22caebda1, 0x0005eb7626c5926c, 0x0000f67f61d51111 }, { 0x0009445c606af316, 0x000c063987d2792d, 0x000c656ec0cad92f, 0x00039a3e64a73caa, 0x0000e4821ae46138 }},
        {{ 0x000981afc00085d1, 0x000d0d5af71013f7, 0x0004926c19866a71, 0x000ed9fa0e68216e, 0x00008c7bcfc44bd1 }, { 0x0005694612f9623f, 0x0002b679449f0e1f, 0x000d14dea79b08e7, 0x0009c8b2286cfc4b, 0x00000d346cc223b8 }},
        {{ 0x00096cf17200d729, 0x0008e23901515ea5, 0x00051b2ffec6cf8c, 0x000c727594c6c77a, 0x0000320acc1aeeaa }, { 0x000269e868e20629, 0x0002afa6530ae7aa, 0x000d6528bf3d9e09, 0x000d558c90f5e631, 0x0000229c5edcbc0d }},
        {{ 0x0004cc22b4444cf0, 0x00009e12cb7191e8, 0x000860e46bf2a7c8, 0x000e63bb978ea1bd, 0x0000d8fbe677e701 }, { 0x00039a0269232967, 0x00082e9d49d5bf18, 0x0005956e7b3add6f, 0x000cd7c57cd4ec1c, 0x000020dc2c6c5c8b }},
        {{ 0x00021474c07616b6, 0x0003562c6d8607f9, 0x0004d89521244fd3, 0x0009dcd17759689f, 0x00003d544887a945 }, { 0x000f77453607db96, 0x0007d5b2a354c8c4, 0x000231b85f7fc2e2, 0x0005589e8d51e52d, 0x000081a4d861604b }},
        {{ 0x0001287ee355832e, 0x000a8715bde48f8d, 0x000250613d9a672a, 0x00086c08ac970387, 0x0000dae19fd8bb71 }, { 0x0005fdcbcf36d30f, 0x0007974db1dbf1fd, 0x000ac30ebd07464c, 0x000637413ea46588, 0x00007cc18fa7cb4d }},
        {{ 0x0000a61162698c8d, 0x000e043810d5ab00, 0x00090431d46a2727, 0x000a6ef336739d8f, 0x0000a26b0295f5a3 }, { 0x0005c0032a22784c, 0x000f53b379f0a1ff, 0x0004f422a86b7376, 0x000e6cd9cfe12660, 0x00003c940fd45b0c }},
        {{ 0x00089fa086f4a3d4, 0x000903037577dd85, 0x0003d1265e0202f1, 0x00019ed3283b82af, 0x00002538201d9fd9 }, { 0x00032bd123d0e234, 0x0006138b13799dff, 0x0000df69ff10325c, 0x000bba8ca85fa8b2, 0x000057bcce5a1bec }},
        {{ 0x000774df733efabd, 0x000cba7858b044cc, 0x00019335042416f4, 0x000dd98591f0d69c, 0x000024cd837a5001 }, { 0x000d42a35cffec65, 0x00084f6d3bd2e1c7, 0x00002b1b5b735de0, 0x00000a4c4eb30da2, 0x000019e181b75ed4 }},
        {{ 0x0008561eaab750ec, 0x000fd959890272db, 0x0009861b1b6617ec, 0x000313bb875f3432, 0x0000725d0e14f52b }, { 0x000a16eb56377ba8, 0x000efa08332aafd7, 0x0004f969b00c6f26, 0x000e75b6f9b0d8bb, 0x0000fe905ed9a6d4 }},
        {{ 0x0005c59502191a75, 0x00080f678e62ddc9, 0x00037237e0292642, 0x0002906be2267c45, 0x000076e18afb6945 }, { 0x000b67703ed66163, 0x000fc85f2b254d9a, 0x000c814dbb101610, 0x000e0fcb5844b617, 0x000051187f47b6f8 }},
        {{ 0x000df6ef65292f82, 0x000ccc13cf482834, 0x00054e9380f92393, 0x0004dcd84b09daa2, 0x000002ad22636b32 }, { 0x0004d671be580fab, 0x0005c63aa21ba469, 0x00080dbd873d895b, 0x00010ffbfc6b1d17, 0x000037d32eef447d }},
        {{ 0x0008511a63429641, 0x00057cf4f5792fac, 0x000648d037fe39e7, 0x0002f65d47c5e0d6, 0x0000b0155eccce65 }, { 0x0003bd08cf6b4388, 0x000da1ca5a60d998, 0x00084ba13ffba0d2, 0x0009a6608f7fbab8, 0x00000b92a4ebd38c }},
        {{ 0x0009badc329d1f15, 0x000798a2ac13e274, 0x000f04f6cdc35ac5, 0x000fff5624494f6d, 0x0000234123bdb8fa }, { 0x0003d01cf2e4388b, 0x0001a41004f7571b, 0x00002b4c77fb6c86, 0x000ef3131bac67d1, 0x00001a55a5e1aed2 }},
        {{ 0x000dc88b3456a9ee, 0x0008d1b55fd0b8ba, 0x00035123a63efb0d, 0x0006484b7a099d18, 0x000092c90c828310 }, { 0x0005f1462aa51ab5, 0x0006c6af989d9051, 0x000013c446d68fcd, 0x000caa593c2e68d2, 0x00006aa1218b67ca }},
        {{ 0x0000bd29c9340da1, 0x0003275ea3d62f2e, 0x0002210b6db3e508, 0x0006d85f6a17765a, 0x00006dc8a9e841b0 }, { 0x000bb9f1ed8d6637, 0x000606d08d84858f, 0x00046e14e3d3d3f9, 0x000cec356370dfe8, 0x0000614d73954f19 }},
        {{ 0x000e324aa43d7d34, 0x000b853060dd7ef1, 0x000d50332ed5585a, 0x000234e35eef2dac, 0x0000e1739c044fa5 }, { 0x000d402c7e387317, 0x0006771e6e676710, 0x0000b212ce791c9b, 0x000c9af2abccd519, 0x0000dc23b3bd97e9 }},
        {{ 0x000480be272bb944, 0x0007c39306a5a3b6, 0x000d31f77bf47852, 0x0009a89d44e0a41f, 0x00005f69215972d6 }, { 0x000a8f43e1bf1716, 0x000481d2285f3821, 0x000fd17c18552f1f, 0x000f83782188d8d8, 0x00007bd976ec665f }},
        {{ 0x00075ffd7e18ac01, 0x000216d77aefb3a4, 0x000dd712c4d568bc, 0x000ad4cf1f69a751, 0x0000e55f50391cd0 }, { 0x000dd2e229415715, 0x000adb13844e95c1, 0x0007bf400d6b744b, 0x000af280085caf64, 0x00003a42515d76af }},
        {{ 0x0000db2ffa6b60c9, 0x000179d73d172440, 0x00068eeb23f6cd90, 0x000c0191c3ede774, 0x0000e518c9f2fb70 }, { 0x00079a440a790688, 0x000a43930cfd9057, 0x00042e26ce9b9861, 0x0008576954934d07, 0x000024bf5ca7badd }},
        {{ 0x000b28c70e222015, 0x000dd1897d6fbdcf, 0x0006e54ea39dea03, 0x000f2d3dbd3a5cd8, 0x00009af5dacd64b4 }, { 0x00051b3088c8cb5e, 0x0006928b61301a0c, 0x00080e5712cc3715, 0x000c4a0dca628216, 0x00007183a4f877ad }},
        {{ 0x00033f48d51a3fa9, 0x0006b7c3902dda59, 0x00066b2a2adf8b68, 0x000179c7e35d2f70, 0x0000efe1c34ffaa8 }, { 0x00032c6af415357b, 0x00034884912beb26, 0x0005b30274a0ae30, 0x000fd784b7f5a4de, 0x000068784fca87f6 }},
        {{ 0x000f4a94d8f05720, 0x000d2e3753e43a97, 0x0006a58a4d66687d, 0x00096c6a670e1d21, 0x00003923588250bf }, { 0x000b64615e973715, 0x000f5968d4faa6a0, 0x0002a502553213a1, 0x0007fc6fdd9ed0b3, 0x00008fdecf871ce1 }},
        {{ 0x000e553fc34528cf, 0x0005148cfb9fad53, 0x000ce5b636f9dd65, 0x000b208bcbeaa386, 0x000021343a11054b }, { 0x00047831ec12559a, 0x000419b9d5f97505, 0x000a7e142922e027, 0x000f2299b3e85c52, 0x000013d6e4fdc33f }},
        {{ 0x000101b9ab1418b8, 0x000ea274a82f424f, 0x00078d463ef2df0e, 0x000ae812c36919c7, 0x0000a9078f376e2f }, { 0x000e6b4528b15e10, 0x0008e9c1c5009d2d, 0x0002be663e1d0b7e, 0x0007067c7fb55ea1, 0x000065c8a8ad08f2 }},
        {{ 0x0005e18cc10dacdb, 0x000c8339bb26f5f6, 0x000d9e6bf8f50349, 0x0006ca3541e85db8, 0x0000696b34c84786 }, { 0x0008da3ee5c6c70a, 0x0004550a471fe6ea, 0x0009863904b6dcc5, 0x00040278bf031747, 0x00002271b171d4f9 }},
        {{ 0x0000398ec951344f, 0x000641c558216b17, 0x000bbf4057f8b421, 0x0005a9e3c90c7810, 0x0000ef7c615f84ad }, { 0x000d37bb5223d947, 0x000fd9ef701b2350, 0x0003de5d38438590, 0x000f22d310ceecdb, 0x0000b63f014f16ac }},
        {{ 0x0007f5e17c602df8, 0x0001538b880d2dd4, 0x000b1d9d2bdaa4e3, 0x000343df5a60f177, 0x00003595db60eb93 }, { 0x00042647efbb60cc, 0x0007f57db394af4c, 0x000cac10af9a0c05, 0x000c6e87239ba215, 0x0000a63e0d4e1e73 }},
        {{ 0x00049c0c51c25079, 0x000fc4564488f1d6, 0x000890337983826b, 0x000946663aa92270, 0x00003ecfce6e8391 }, { 0x0006ae337281abf8, 0x000de4e3cffcb6c1, 0x000e3a48ea23ea44, 0x000159ca0e3bf93f, 0x0000c6e00209e63b }},
        {{ 0x000bda54e9b5f5de, 0x0003cdf5ca48f3fb, 0x00062650469b09ca, 0x00013c04a64352f0, 0x0000a9c900d6b80b }, { 0x00034c2438695df8, 0x00045f56f6e55fbe, 0x00033ccb4b4fad0c, 0x00059bdff1ca44a2, 0x0000d8197cd15076 }},
        {{ 0x0006b3a610c31e52, 0x0006975861421033, 0x000ae881563cbbe1, 0x000a35f226704ba1, 0x0000fa0894f27e4d }, { 0x0009ee25bb322589, 0x00037a775a0d0508, 0x0006b2f5187779d7, 0x000d880be066f919, 0x00000c76c7c238f7 }},
        {{ 0x0003422f090ad74e, 0x0009022fd29dc766, 0x0008d3af43571f67, 0x00073f3267fd32ed, 0x0000d0b442ee5f1d }, { 0x000b361400cf8a31, 0x000f3f40807e199e, 0x00003777308f86a0, 0x0002880b4fe9ce5d, 0x00000da749380f3e }},
        {{ 0x0003eea570082bb1, 0x00099052680813be, 0x000c722c6366b500, 0x000f4a458527aa55, 0x00005bce35af9e9e }, { 0x0003afeb82f8170f, 0x0002917a0ff58b3a, 0x000476b9faa4836e, 0x00052a6d6b67d3f6, 0x00006f5fdddf36b0 }},
        {{ 0x0009432bf6f80693, 0x000ae6d3e5548920, 0x000d39cafe4e671d, 0x0002c0fa08413b53, 0x0000a88888d1f798 }, { 0x000b84f7978aae7d, 0x000d3c6daf0ecc11, 0x0005404428b0ca79, 0x000d2960749ebce5, 0x00002f4edff10ec9 }},
        {{ 0x000b1771c2c13be7, 0x0001d818d1d7a0dc, 0x00026f583393b81d, 0x000627635d9fabe0, 0x0000399a341196f9 }, { 0x00046124aa70c355, 0x000f2cae9a6312bd, 0x000f80ccb41ae366, 0x000b5f216a34b355, 0x00003e9a5b7e359c }},
        {{ 0x000bc3dd47503d26, 0x0001d5d68afe8b4d, 0x000f9049b15a067a, 0x000db429827f2053, 0x0000558730c6cd22 }, { 0x0007e4665dbf09b3, 0x000e542c03b0c057, 0x0003013d5638d610, 0x000167b0a8b1b3bb, 0x0000b3ec98ffe86c }},
        {{ 0x000ac7168a05d60b, 0x00002fcd23e39845, 0x000c910a2bac02e1, 0x0003b40a7ab86e8b, 0x00007c554cfc6e51 }, { 0x0003931577b8f581, 0x00022840139d9af1, 0x000862d13578d007, 0x000e1ce4a72733f8, 0x0000f2968562b2ac }},
        {{ 0x0001f568c4e93444, 0x00095eb70f8d5a4d, 0x000705c64963ec81, 0x000e7c732375adde, 0x00006e492766c122 }, { 0x0003224c6ff30b48, 0x000b4835459df02e, 0x000a2cab4e0e2287, 0x0003bf30a2beac0f, 0x0000b7c08ff94828 }},
        {{ 0x0002d191d34652ff, 0x000d10ee87edf5b7, 0x0001ae15eb2221b8, 0x000efd2b230385f0, 0x00004abd2d05ed06 }, { 0x000564f3ba13cfbd, 0x0001dca0917d9e55, 0x0008ca2af9c50195, 0x000eb7b96394eab5, 0x000048607473d36b }},
        {{ 0x0003da3500cac108, 0x0001505d73654ef8, 0x00038ffd58742011, 0x000886ba068d1a9c, 0x00002aa8c6ea07a2 }, { 0x0008af07c42cfaab, 0x00051a468ae01189, 0x0009377e98a53d43, 0x000b0d802a1b9a4c, 0x0000f99f7b9d03be }},
        {{ 0x000dd2d01e319688, 0x000eb26995f0f9ff, 0x000f73e2bba85402, 0x000cbd22b467fadd, 0x0000ea7ba8013c8b }, { 0x0005ee2fa76c4cbc, 0x000f228c693f9fa1, 0x00091ea149f1bfeb, 0x0001539834862232, 0x000070ee1dffe6b5 }},
        {{ 0x00052e2214ff073f, 0x00089455777e1891, 0x000030410614a29e, 0x0006eecd5cc10bee, 0x0000bd74b27941db }, { 0x000b6dff74a64cd2, 0x0003c727cb23c74a, 0x000ffd9e676af10e, 0x000132a53fd5d766, 0x000022b0948b387c }},
        {{ 0x00015648065531e4, 0x00083816f4cbe9d9, 0x0008d90103d196a3, 0x000769dfa0ed8b24, 0x000076059de2f82c }, { 0x000b69dcba54041f, 0x0000c085b8628093, 0x00011a1bb0c183bd, 0x00005ca2a1facd1f, 0x0000198f2db6c25b }},
        {{ 0x0002cf1f8a6fd4d2, 0x000bc9f967365efa, 0x00024532d63e9c0d, 0x000ae632f57300eb, 0x00004160be5ee065 }, { 0x00025c377f9c4ca1, 0x000fa744dcc40cc3, 0x00057b48ec7bdfbf, 0x0004d2b4dfee38c4, 0x0000461e8ab76c74 }},
        {{ 0x000377f264a991d9, 0x0007fce476cc9ff0, 0x000cc35cd8ff3792, 0x000e04f9b580e0b6, 0x00003f7b7baf52e6 }, { 0x0008e50f30877c3e, 0x000156288e591a5c, 0x000cac3ebafa9eec, 0x000630733a453a7b, 0x0000a9492c65a23a }},
        {{ 0x00010ee253308b7a, 0x0006b0f6d3549cf8, 0x000f79be840ba3a8, 0x0008682b46f33696, 0x0000b318d895599d }, { 0x000485717b66f888, 0x00087e17159bb2cd, 0x000da105c3fffe4c, 0x000a76272f7dfbeb, 0x0000bfbd7894f96f }},
    },
    {
        /* digit=23 (1,2,..,64)*(2^{161})*G */
        {{ 0x0005fd5fa205d0f4, 0x0004ee8ee36860ce, 0x000c5b16628b839b, 0x0003f4e13daf04df, 0x00008b3aaf4c153e }, { 0x0004e879df3f3f34, 0x00029941a4e0551e, 0x000d33e8877228e4, 0x0005911236772cfb, 0x00007681b72c03ca }},
        {{ 0x000e349a4b31aebd, 0x000f2b0285b94916, 0x0007b017d6137900, 0x0002b0dab01a0be8, 0x00008ac2977211f9 }, { 0x0006974d05362415, 0x000fd464a1c6a163, 0x0005f2a0e55b98f7, 0x000493f71d99e6b6, 0x0000a0e9ca0b6129 }},
        {{ 0x000b8f7229523466, 0x000e3a1cfe89d80f, 0x000cdf0d11487037, 0x0009c00d2b42c026, 0x0000172110f51188 }, { 0x0009ff21f71bf171, 0x00008ecd850935b3, 0x000c9c32bfbbecc4, 0x00028143434c1a74, 0x00001cad52349f90 }},
        {{ 0x000d9cce837c6d61, 0x0003d148f9290579, 0x0006232b855bab85, 0x0007c64ce7a64ae0, 0x000028043d63ffdc }, { 0x000f181a75f69e00, 0x00059796e93b7c7b, 0x000172a383b1d5d0, 0x0000029a0e1e4709, 0x0000b9ee91160db4 }},
        {{ 0x000b670a385747ca, 0x00062658ead09f79, 0x0004fc19c7159df7, 0x000d39cbc1335e1d, 0x00000025ace0a875 }, { 0x000ab38fcace8fb2, 0x0002a8128efbeedb, 0x000bc873171affc4, 0x000fee8630c74e77, 0x00001249e10307e9 }},
        {{ 0x000ba8e478b7779e, 0x0004ac85c0a3f1ee, 0x000190dedf06d75f, 0x0006c198dbcad249, 0x0000bec9ca1c42e7 }, { 0x000b123f368231d5, 0x000b7b0eeaede4b8, 0x0007ee649f7fb5a1, 0x0002f72fb0bc498e, 0x00003fa9b3e1cde9 }},
        {{ 0x0005c8295bc7ccf1, 0x000ebc7de79dc241, 0x00023fc74071d988, 0x0004b5fdff1168a3, 0x0000a46cc6dd3945 }, { 0x0003a26e3c4f8d24, 0x00086367d8184ffe, 0x000d4ca20d9fa6d4, 0x00057819ec396228, 0x0000217db807076e }},
        {{ 0x00026fc9c4cd4abf, 0x000e23f1402b9d0c, 0x00026a0bfe9f0668, 0x000417f6e573441c, 0x000042560b13ff8a }, { 0x0003ef07f65b14ba, 0x00061fd7493cea35, 0x000ec7090c603bd2, 0x00077a68fd05d4b3, 0x00006ce1efdc940f }},
        {{ 0x000613ddd45cbe56, 0x000a5d27824fc56a, 0x00076d14b2a2aae7, 0x000062d493467521, 0x00004912a1180184 }, { 0x00016c4168b43a86, 0x0003b9c5450d8660, 0x000c8e186318cb3e, 0x000c25c3946f4409, 0x00002251eb5dbfdc }},
        {{ 0x00035d95c1f7dfd4, 0x0004ff0c74359787, 0x0007ad857857c300, 0x0001a9bafb152c88, 0x00002220a65483ec }, { 0x0002c5cc53ddbb41, 0x0003473039553173, 0x0009bd757d41255e, 0x0006c05000b4b02a, 0x00005bb2616a0592 }},
        {{ 0x0006f267f3450bcf, 0x0009e1f606241297, 0x0004236bc36ba3a3, 0x000fc5e72b7bc5d6, 0x00006d062e175519 }, { 0x000d792d60b7e583, 0x000e00fb36eed930, 0x00072e7c3aa2aa86, 0x00041b3a055230cd, 0x00005222efcee916 }},
        {{ 0x00025b58889424a0, 0x0003bfdd5c7c5d2f, 0x000614cfdba46ef4, 0x00069438ab38f8da, 0x0000c9d7330553e7 }, { 0x00099f94c0e0b82c, 0x00086842946ad608, 0x000d6031dbbdd21d, 0x000b21909854f29a, 0x0000f80169f11b4b }},
        {{ 0x000d22a1149324b5, 0x0002751f4a55b03c, 0x000d68c111c13c46, 0x0005dc651261762d, 0x00005612c643fd71 }, { 0x00026455eddae243, 0x000931a247c9569c, 0x00030eee68dcf34d, 0x000c4d3d459a5097, 0x0000bc8501472388 }},
        {{ 0x000cd3142c078a4f, 0x000e7e049ce89e7e, 0x0008dfb5f889206a, 0x000b1dba11890853, 0x0000fe4b40447fd0 }, { 0x000af34eb1a2b760, 0x0005f59b3f5ab84e, 0x000807cb5614c7ce, 0x000ecf41f3c320a6, 0x0000376551d686ca }},
        {{ 0x000d8eec2b43abc0, 0x0004129977c97c4f, 0x000ac7efcdced8cc, 0x0007704785ff9bee, 0x00003fe869d8f984 }, { 0x0005c75015e7e220, 0x000f94ef027458c3, 0x0008e754910013fb, 0x00084b6f4b4ff531, 0x0000df161344c360 }},
        {{ 0x000f63cfc83f45b8, 0x00011d646ac49d20, 0x000b73afaae16770, 0x000dca23842c77c0, 0x0000cf54b1e93428 }, { 0x0004dcfda1adde56, 0x0005a1bc6441f959, 0x000957b146ed74f3, 0x000a15bba7d38f71, 0x000080b43552bdca }},
        {{ 0x0004512fad93b7bf, 0x000f4395f6d4a09f, 0x000bfa01887930c4, 0x00063d40a6b4f355, 0x000015dc769eb061 }, { 0x0001dbaca543e3fd, 0x00084d5206c4c74f, 0x0003d97181a31083, 0x000245f2dd021d4d, 0x00004c9b52a51d86 }},
        {{ 0x000f5829708831a1, 0x000f226490a1aca3, 0x000c01fa4191db95, 0x00065f158cb64dd9, 0x0000b704509d211a }, { 0x000d54e389cc560a, 0x00066d17c213ba70, 0x0005db4beb4b0a15, 0x000ad2ec1c28febb, 0x00007d2a923ea7f4 }},
        {{ 0x00069bcfe21bda90, 0x00044604af206ee4, 0x0006314dacd7546e, 0x00089f02e786c88f, 0x0000fea9761aff90 }, { 0x0009a4ef238392e0, 0x000540d470622302, 0x00079c0fb4897908, 0x0005c65ace743bb5, 0x0000ac57eea586d6 }},
        {{ 0x0009fcce8362831a, 0x00014feb483689bc, 0x0005873df3dedda1, 0x00023c1722575d3f, 0x0000658b868344aa }, { 0x000525a7345027f1, 0x00021ff0149362d4, 0x0001d725423d3479, 0x00035076c95e5758, 0x0000632aa6fb7444 }},
        {{ 0x000caab33910184d, 0x0003e628f3383609, 0x00028c433ff2d754, 0x00072cdec483ff59, 0x000059db5682d119 }, { 0x0008c75049787047, 0x0008bfd668927090, 0x000cac92ccf7e399, 0x000d9fc942496b4d, 0x00000b039ef1d169 }},
        {{ 0x000389359d183e8e, 0x00043c2c905d85f7, 0x000f93aa78bab2ac, 0x0003e4f94e9d7f78, 0x0000cdc97efebfd8 }, { 0x0007303b35c1dcbb, 0x0004eb7367e45f70, 0x0007c44e3fa733aa, 0x000da6d181f20306, 0x0000693cf0cbca05 }},
        {{ 0x00016554135f365c, 0x00088e4d13b7ea1a, 0x00043f81ed6cbee8, 0x0003a9370e6489b2, 0x0000a763c3cdfd1f }, { 0x0005fb3cd26c7a98, 0x00026fb7af4194e1, 0x000c788e80c9e8e4, 0x0000ec056533c1c3, 0x0000e7c9daf6851d }},
        {{ 0x00014e77a0de8421, 0x000cc45526f09fd2, 0x000c5fd9ac6926fb, 0x000a7dc8de8a4f10, 0x0000d25068992420 }, { 0x000b6cda791fe0f2, 0x0007b7314faa40ec, 0x000b3679170d12a8, 0x000c08f3e767867e, 0x00000e1e221077f7 }},
        {{ 0x00083667c4ab99b7, 0x000646d349d51aa0, 0x000edb9151250bb8, 0x00006dfdff56b5a9, 0x0000e96a55350487 }, { 0x00075a324beb86c4, 0x000dd3518087f2fd, 0x000bf9dcb2114ad7, 0x000280589f1b8eaa, 0x0000a4dccd763888 }},
        {{ 0x000cc4569fc6534b, 0x000850101dae185d, 0x000abeb63e1cf161, 0x00085800b45434e0, 0x00003dded22e1035 }, { 0x000e50258feb27e0, 0x000518a09a512993, 0x0006746000488c61, 0x000c0779281b4d20, 0x000087a3225e890f }},
        {{ 0x0001a7ec6bfaf389, 0x000c3db2996864b3, 0x0008eda07338e1f4, 0x00058e15aa2e6708, 0x0000a57bd6c84a58 }, { 0x000d24ee7c6e4db1, 0x000c666d6bc20da4, 0x000c39512313aed3, 0x000225f634acdb5e, 0x000017899b25ac05 }},
        {{ 0x0000ef95acc5b65a, 0x00091d3a19fd8680, 0x000cda9e478efabc, 0x000ae7e52612e481, 0x00005879a89ba754 }, { 0x000ff9d98e9d1fb0, 0x00022edaa0a6c469, 0x0005a37d6e017817, 0x000202a106129408, 0x00008051d4fc5997 }},
        {{ 0x000a9868340ef12b, 0x000847c468b8835a, 0x000ac8766672f855, 0x000de1f81977a31c, 0x0000ae09622e4cb3 }, { 0x0004b60159822469, 0x00032f3a6bd9d340, 0x000b2f0fd6fae8ce, 0x000eed25e6a62fe9, 0x0000a1ce552aab34 }},
        {{ 0x0005f9ee8b1f3b1b, 0x0007bdf5579bea49, 0x0007c6f4ec662d97, 0x00081aab110e35ac, 0x0000838e05f272c4 }, { 0x0008c93722dc320b, 0x0004eee82b6eb0fd, 0x0004bcf973ba5062, 0x000f5f1fe2e84576, 0x0000987ab3588bf6 }},
        {{ 0x0007afbfe9bfa97c, 0x000db48f83e8a3b0, 0x00068574d6094e13, 0x0009ff29ba579aa9, 0x0000ac0b5352b6ee }, { 0x0008e78e9f640109, 0x000d36d5b7bd2931, 0x000f573aa39dd6c0, 0x0000935763b592cf, 0x00006b052765f938 }},
        {{ 0x0000cd9c328d84f2, 0x000ec10296c36eff, 0x000191f73e449397, 0x000f344da7ee8967, 0x00003d52cf283e17 }, { 0x000adab3ad961303, 0x000c8f7e455fe908, 0x000af39881456bea, 0x0006ae1ec3fb53cb, 0x00007d83567df6b8 }},
        {{ 0x000d359f6fd4efcb, 0x000232f53d648293, 0x0007652ba9bb3daf, 0x000db7f2c7a471e1, 0x00009d7cb3d94a0f }, { 0x0006782372900602, 0x00065860aacd6975, 0x00031f4ad535a466, 0x00057f24e92b444c, 0x0000a4012d7fa548 }},
        {{ 0x00018c74a1ce7f88, 0x00008f2bbbab3b85, 0x000e39780cc99267, 0x000135cd24851f8a, 0x00000d44135a09ce }, { 0x0003ccc547f7ecdc, 0x000d90af9877d4ca, 0x0000054d05740761, 0x00071d6b614578d9, 0x00002bd6390f7cb4 }},
        {{ 0x0007167876861ca8, 0x0004c8e327f83495, 0x00069d263dca7c3f, 0x000f26885f3e9734, 0x0000cc0b31df68c9 }, { 0x00011b5e5c6b6cf5, 0x0001fb9411925a6b, 0x0007eff6df88f06c, 0x000be215d7c078d4, 0x000083a8bd8ed8ac }},
        {{ 0x000abc1295834d88, 0x0001f6c1eea7b85f, 0x00009818e5c95ff1, 0x000938dbb2352b46, 0x0000387993489f2c }, { 0x000c5e12df15edd0, 0x000fdc05233f9251, 0x000bcb21d976697d, 0x0007be659e0a802d, 0x000053b74438966a }},
        {{ 0x000b70d5488e28d0, 0x00088c64b8457ee6, 0x0008e40dd4aef873, 0x00037e38fa5360f6, 0x00007f082ebb5766 }, { 0x0006f388f3012c36, 0x00001f13bfae3154, 0x000284ec6a763ea6, 0x00088e0f93379785, 0x0000205217d30224 }},
        {{ 0x00034f30d0a7f21e, 0x000285889d8ddbbc, 0x000dada45e17bd7a, 0x0001ff4b08830b26, 0x00008227c3ea8c36 }, { 0x0009e76d098d7344, 0x000eda356ab64b81, 0x000cbfb4f01d1e3d, 0x00024d9947f77f0c, 0x000077da11412e32 }},
        {{ 0x000ecd83bcdac8cc, 0x000bec396238f398, 0x000e19632faa0ca3, 0x00085e899c0a482b, 0x00002203b4334230 }, { 0x000b32d5095835f0, 0x0003cb7a2f3776f8, 0x00046985b5d63ee5, 0x000e9b446b42176a, 0x0000db866498ec65 }},
        {{ 0x00097128795b26d4, 0x000007d53b618c0e, 0x000b80b150a9c145, 0x0006c0564c424f46, 0x00006ae4b9ab6582 }, { 0x0000d841a1380e4e, 0x0005832f815561dd, 0x000502e81430573a, 0x000171a4f85f48ff, 0x000063896020863d }},
        {{ 0x00042ad522e18a65, 0x00044260b9267c6a, 0x00012add234060bb, 0x000833924c78913f, 0x000039e7e3dc6b6b }, { 0x000136fd8d64e5e2, 0x0003549a2c989d55, 0x000f90beea015851, 0x00038c4da928292d, 0x0000b65cea1cb0a5 }},
        {{ 0x0005f43682904822, 0x00022b1f2c34a996, 0x000459641f29f5c4, 0x0006715be3b00aca, 0x00008ad64a961e6a }, { 0x0000473f5f609763, 0x000f501a11b12e13, 0x000cb1c79515bd02, 0x000aa7dd578a7ab2, 0x0000493e312a1f39 }},
        {{ 0x000c13966b2406fb, 0x000e59d169c7d97d, 0x0003f39b6362b933, 0x0009d4d035cd5baa, 0x000048a4caa8a041 }, { 0x000310fa4ebd46a4, 0x00087e82b8357c1f, 0x0004f6c481db4a34, 0x000b8923e0cb1b4c, 0x00003d448294e348 }},
        {{ 0x000fb357b1ee936f, 0x0004452f97f68c6f, 0x000f0b4f77a6ed24, 0x0008c27d90c773b4, 0x0000c6d90e5f6f4d }, { 0x000ae575c3e40e20, 0x00004cde0f28039c, 0x00097bbe8620d5bb, 0x00038bb6c0b36c95, 0x0000c238d358e07a }},
        {{ 0x00073da0ba4fff56, 0x000bd02a70dab865, 0x00091a87ec422f67, 0x000e17592fcbd12a, 0x00003dc4b150156e }, { 0x00022cb8d1d292ad, 0x0004ffe0d41d5505, 0x0002cf29172c74c0, 0x000c83548e7eec0e, 0x00001f902993f981 }},
        {{ 0x0009a38d4ab89a62, 0x000342a70cbb96cb, 0x000a032d975e6da2, 0x00090838de034362, 0x000051f017afdd83 }, { 0x000aebcb837a69e7, 0x00043821ff140fd2, 0x000ff5c31c5d3793, 0x000bb3249bd162f2, 0x0000c09d95c15150 }},
        {{ 0x000f921c720023d1, 0x000e15f937b05271, 0x0007cbeb7d25606a, 0x000816929f42e1e4, 0x0000a8cc384e52b8 }, { 0x0006ea82172f000f, 0x0008f7b5b26bbd51, 0x000d247724699de2, 0x0001f0d7e7a8ef5e, 0x0000f1fe050d2c8b }},
        {{ 0x000f7199dc46d818, 0x000722b599ff0f94, 0x00073193628eba9f, 0x0009137f368a923d, 0x0000ee5360c30393 }, { 0x0004f710bd7921fc, 0x00093f6e46f2a79c, 0x000d25010260474b, 0x00092d27c08b5dea, 0x0000fab67c859c32 }},
        {{ 0x00064519deca0cdb, 0x000d55778bde41ac, 0x000fedd3517b736a, 0x000b97c416474bf7, 0x00003208530088be }, { 0x000025813a05a641, 0x000f7460edfcc4f9, 0x0006b8cc2931d960, 0x000ea5e77319aa73, 0x000029012049c8c4 }},
        {{ 0x000de5048b827884, 0x0004fcac0c227196, 0x00045be510935647, 0x00029229cc60209e, 0x00001a8f07ca2a57 }, { 0x000130388eeb5d29, 0x0000fd8e115ac04c, 0x000f5b3ade432bbc, 0x000316be77df7093, 0x000022d0dba6d494 }},
        {{ 0x00039aaf90e9d532, 0x0008c8477f54e6e0, 0x0002623b60542e7b, 0x000bc14d69f258a7, 0x00009ce5ba732f82 }, { 0x0000931ba5ed41d1, 0x000a9739460342f0, 0x000c3ee643e44bb0, 0x0003fd87e4703148, 0x00003a0863fbd5a9 }},
        {{ 0x00004525097c5017, 0x0005a9d786f337c0, 0x0002e6bf14745166, 0x0000c18efd9c01cd, 0x00002c8b553116d9 }, { 0x0000641053c931ba, 0x000c64784db8e765, 0x000a436499b28b73, 0x0005fda223bacf1a, 0x0000938a983f292e }},
        {{ 0x0007bde5834630c1, 0x00077749f7144aea, 0x000e9c16c77a33ff, 0x000ad12ef170c963, 0x00002afc875cc612 }, { 0x0007882dab2f3b77, 0x000638b6fa687508, 0x0001cfa5d49dd82e, 0x000de3631fa64e53, 0x000078a45e28f1f6 }},
        {{ 0x0009c5714d71771f, 0x00098316c67a7751, 0x000ad6005f642df7, 0x00040e7b2cb10471, 0x0000cc0be522bf70 }, { 0x000bc82ce45c7f2a, 0x000617bed03fd3f8, 0x000ea5f3ec6a645d, 0x000e3f6b5dd03344, 0x00007777234fc32f }},
        {{ 0x00081d837a740b62, 0x000d301b35f1341b, 0x000c6d86bac884d5, 0x000a3565ce53156d, 0x00002f8f9bd103d9 }, { 0x000c596b31a8eb5d, 0x000f76e2e4358674, 0x00049f4c722e721f, 0x000256e9d9020eb9, 0x000000946b032fd9 }},
        {{ 0x0004828817b2b97d, 0x0002c0f0e0b040d2, 0x0002217d0167ff6f, 0x00040e422baf02d8, 0x00006eb8e36ece4e }, { 0x0001f86203c5e993, 0x0008a31113ec3567, 0x000da3f78ff4f368, 0x000ea84159e48861, 0x0000bb7e93050f1a }},
        {{ 0x000560b0e9aafe66, 0x00042da3fd19408b, 0x000b722cecb04b0f, 0x000c6b6904aa716d, 0x0000052ee2e70dfd }, { 0x00015ca19443e93c, 0x0008d92fa34bdad6, 0x00043b7857295ea6, 0x000c094d3f024fa1, 0x0000ea3f81cb62df }},
        {{ 0x000d3a74d028189e, 0x000aa3d6f73c51e3, 0x0002acd34e0d4511, 0x000a079ffa4cb241, 0x00006e3b446570f6 }, { 0x0008027c148bf575, 0x000240f4a7a92024, 0x00095faee0289dbe, 0x000ea1d08a9bccf5, 0x00003daa96c5c819 }},
        {{ 0x00079738d8eab83c, 0x000fb7c40b557f08, 0x000645d10d87e8cc, 0x0005ffe27e001c36, 0x0000017e39f092f7 }, { 0x000943d1b495d0bb, 0x0000c4bbda948826, 0x00083d40e0e510b7, 0x0006a7c24d1bbcc4, 0x0000acdbfbfa5df0 }},
        {{ 0x000bf3dd51c8d19c, 0x00007bb098cafb4c, 0x000f94e0af5e1773, 0x000601c4d27ed230, 0x0000285109152371 }, { 0x00024c7557bb1196, 0x000891641b1d830b, 0x000d18bfa8cf69f1, 0x0009cdfdba7ec851, 0x000077b4e1b088fa }},
        {{ 0x00040f8ac3df8d0b, 0x00006dc1d4636c38, 0x0005e40b0e75786b, 0x000f4a2b3f843652, 0x0000977ae94715bc }, { 0x000123d8f150f27a, 0x0001e11f4387f1af, 0x0002c3f18f8f2228, 0x000f40e677c501cd, 0x0000110598eb168e }},
        {{ 0x000796ff08cfa1b0, 0x0005e71d607039eb, 0x000515a3463d5019, 0x0009090912b70e21, 0x0000a34adda11483 }, { 0x00038e8cdf8fccad, 0x000cd863c57c4aa3, 0x000f84360d47a30b, 0x0009660982aed9ca, 0x000082fbf17ecca1 }},
        {{ 0x0005213908309c96, 0x000cb8266cde8dbf, 0x000f3851c42e5054, 0x000d1ba724e15997, 0x0000c913344e336a }, { 0x000a1c0c559056b7, 0x000681d8ce4396ea, 0x000ffdd114ecbcec, 0x000c3f86b37a1a67, 0x0000bba57201413f }},
        {{ 0x000e3d7a82a07312, 0x000248b4ed80940f, 0x00057c32545a8fa1, 0x00024459f67e6d05, 0x0000781e5623c72a }, { 0x0009422f1dd9d9ed, 0x00045027e096ae27, 0x0006ab7164488446, 0x00002f2fcb1f3e1e, 0x0000a08771e4d556 }},
    },
    {
        /* digit=24 (1,2,..,64)*(2^{168})*G */
        {{ 0x000b1adfbccd5f72, 0x000bafb9db3b3818, 0x000e49c42a8e58da, 0x000a5741f9c3a2de, 0x0000e1b4d1992caf }, { 0x000d2ae779d25bd8, 0x00001397e053a1bd, 0x000689b00f8d9c66, 0x000aeefabee2be5c, 0x0000ed75eb0e9aae }},
        {{ 0x00070ef12df3aecb, 0x0000e7a205b9d8b0, 0x0003fe5865a61087, 0x00049560e6eb8f06, 0x000018c288645dc3 }, { 0x000c1f205200dec5, 0x000d0053bcc876ae, 0x0007bb212c914ca5, 0x000c3165e12a7533, 0x0000fee6eaee8fb7 }},
        {{ 0x000b625175d3e131, 0x000ba79b6828f364, 0x0007b65b0a28d9d9, 0x000a31a0c9d7a025, 0x00003f761efd974a }, { 0x000cea06f50c8e7a, 0x00025dd9669b6210, 0x0006ea0e74a30782, 0x0007f7cbc88a2ca5, 0x0000eefdd32a930a }},
        {{ 0x0006927bdc72fcef, 0x00092b5c4e83d33c, 0x0008986accaed0f0, 0x000ee5e0fd9f3587, 0x00006fc2b4d5332a }, { 0x000bd4c284a559fb, 0x00092f79f9e0f036, 0x000e91031f24a068, 0x000494df12868661, 0x000064b67a214c5a }},
        {{ 0x00038062d4c1e75b, 0x0004591289a8619a, 0x000fc2f14e9e6431, 0x000a96b32ef796e5, 0x0000cf84b53f10ce }, { 0x000e2d93f2a93799, 0x000b1200573274eb, 0x0003eaf97fa1c33c, 0x000a47520d07b67e, 0x000099241c28bfc2 }},
        {{ 0x000e16a8fa9459c5, 0x00069533f36d1411, 0x00042fe5fb485de4, 0x000223d3ae84bb3d, 0x0000362e47c092d7 }, { 0x00051ac53cf453e0, 0x00072adddd472e03, 0x0006d8041bea2700, 0x0004e95997d405ee, 0x000072103589e10b }},
        {{ 0x000c45b260b78e4a, 0x000cab84444896b8, 0x000f6cfa759ac76b, 0x000f5fe7d64974d2, 0x0000fc1b25688826 }, { 0x00018f2e67924f42, 0x00079c2d84634875, 0x0006ee2d190516ad, 0x000a501c0d1b2b3f, 0x0000036290195036 }},
        {{ 0x000578dc4ce14fbe, 0x000b08c06d75fc1f, 0x00063cd0cc5274b3, 0x000f629dd2dcf7bb, 0x0000f36db3fef100 }, { 0x000304c0d907ee38, 0x0005103df2ce7a06, 0x00083934eed34414, 0x00075ccabaee3628, 0x0000f1816df3580c }},
        {{ 0x00082f4b52e4cec2, 0x0001aa2a91f6791e, 0x000fb37b53a9c983, 0x000cdb8c12abe418, 0x00008cd67259a170 }, { 0x0009b47fddf3ac31, 0x000ff274073e81c2, 0x000cd201171eb3c4, 0x000d8c60cc0276fa, 0x0000bde77950ebb2 }},
        {{ 0x000db1d445e12d1b, 0x000c603459b19402, 0x00092d4f6a4d5460, 0x000260fd3ec881c5, 0x00002404d2934d68 }, { 0x0000c0e5d7b17ee8, 0x0007387df239fd84, 0x0004d45c3f1714c5, 0x000bd7a59c718af7, 0x0000cf1f5fc6cd42 }},
        {{ 0x0007e104334dd14b, 0x00045f4721b33f2e, 0x000b16fe00b71c74, 0x00074a4e72124f1f, 0x00000c5ed583ab49 }, { 0x000b6a0a0c1b3b39, 0x000698ee4d0c7e6c, 0x0009bbbb310fa8c1, 0x00008ad43847e339, 0x0000d0eb2a823c1c }},
        {{ 0x000c83d2104da3af, 0x00087c327d02dccc, 0x0001ade4eab2adc4, 0x000bae22d5ad0098, 0x00000d9c44913bb2 }, { 0x0005934268273a75, 0x00009c1666fba0ca, 0x00066ee203cd0f18, 0x0003334c26994819, 0x00001a7e83e4402e }},
        {{ 0x0001240501d5f60c, 0x00008eda714181da, 0x000abf43ae40161c, 0x0004fd31f609ac13, 0x00006d341d5395cf }, { 0x0002c2adc36656d7, 0x00035231409ca976, 0x0007bda0de8008a3, 0x000a46fcd254cc1a, 0x0000ead8778cec76 }},
        {{ 0x00082b49f9e98b30, 0x0006df056e464833, 0x00002b48195cdcf0, 0x000c68c52a55abb2, 0x0000455cf3a69edb }, { 0x0004c589d9fde490, 0x00091ad4dff8d47e, 0x00045eb86afca0dd, 0x00060d50979caf61, 0x0000ed7832e28abd }},
        {{ 0x000716dd735d4299, 0x0000fab3fd40e095, 0x000e201a6135ca74, 0x00012ff8be455842, 0x00006f917d8d6ec2 }, { 0x0007e76496178dbd, 0x00054e54e8bf3de2, 0x0005f39e59a54bae, 0x000840fa69a0e77b, 0x0000a545f75f92f1 }},
        {{ 0x00065ca49bf70465, 0x0002633070d3aab6, 0x000b33d03149eda8, 0x000f82c732643672, 0x0000dae397b7ff25 }, { 0x0005986e7c2b0613, 0x000c759b3efb9983, 0x000ccf96a4c52f87, 0x000f392308a5b922, 0x000053a40c602f11 }},
        {{ 0x00060575d954191d, 0x0004e5cff3513cc7, 0x0009bb938203e64a, 0x000d85286bb0cf8b, 0x00009896e7ac48ed }, { 0x00058c95fcf57592, 0x000bc169f7aa1811, 0x000fdf7d571f4181, 0x000cb1291163a3ec, 0x0000a74ad3d22246 }},
        {{ 0x000d52e6bccfbd9d, 0x0007e6e13cda46f1, 0x000516dcc813d3db, 0x000adbc707948241, 0x00009aa625122196 }, { 0x000fc7bf7e178d47, 0x0005ae78608dd3d1, 0x000aca3fa6085efe, 0x0004ed3fa4930db2, 0x000095d11cde96d0 }},
        {{ 0x000d1764efa19423, 0x00089cdd9eee96cd, 0x000b08758747c9b3, 0x0006ea12a0ca277f, 0x00003fd5bca22445 }, { 0x000cff95c47d1a9d, 0x000dfde7b3ed7397, 0x0008dfea4bfb8703, 0x0000b151250cb745, 0x0000e035718c4eac }},
        {{ 0x0003e982abb759a4, 0x000a8bd8088b454e, 0x0006b6b489deac94, 0x000caa3cbdb7f32f, 0x00004686e7f56cb3 }, { 0x000bbf1d3400c329, 0x00098e4c2a2a5938, 0x00047bb4d51b0609, 0x000c26b2372c3686, 0x0000c3163f597394 }},
        {{ 0x0009e8e77eb6f0ef, 0x0001a73da50c991b, 0x000ac8448763bb7e, 0x0000c887148afcd5, 0x00001fe304790c6a }, { 0x0005b82406b0e023, 0x000ec4499a7703de, 0x000d7bd612ecce7d, 0x000a058be6972930, 0x000055f6a476b172 }},
        {{ 0x000a9e252eded6d3, 0x000b8e3bd3790620, 0x000d23302563dbb4, 0x000be5b37d64b7a1, 0x0000f8d8432196d3 }, { 0x00008e4790ff828a, 0x0000b6bc39d3bcbd, 0x000fe29ce5d2ddc1, 0x000bef0181a31c9f, 0x000059056576275c }},
        {{ 0x000e5d55febb0e31, 0x000e9e9255c09801, 0x00010ced1da0ddbc, 0x0003c14a14bd8638, 0x00002df70d0ce263 }, { 0x000e5bf8ecb0386b, 0x000b0068ff5cc292, 0x000ee05cac07c42d, 0x0004c2124026b389, 0x00009cd54793b7a8 }},
        {{ 0x000c6360bfa4642b, 0x0005cd0ad886aac2, 0x000a8b0e55855f5d, 0x000dc1aab5c16878, 0x0000ad287ec6b022 }, { 0x000e29d50b1fec3f, 0x0001cb5b9c972a3d, 0x000d9e7098af043d, 0x000bb5ba6c9e6f88, 0x0000ccd2c2291972 }},
        {{ 0x000d8f2c9dc84b95, 0x0005f83d309cbe6a, 0x000a332dc0b1d6bf, 0x000348ee35b020d8, 0x0000d07b2b00cfeb }, { 0x000c5bec9951a99c, 0x000921de7400e938, 0x0008c66271b3c77f, 0x0005ce4bffe3e18a, 0x0000fcae372683dc }},
        {{ 0x0009779f4565f3cd, 0x000992a74a55d1f0, 0x000efaf7c792c505, 0x000e87a94e3fbf28, 0x00001e0023586f6e }, { 0x000da67cfc184107, 0x000b6afc3d19d8d3, 0x00041a3029e3c052, 0x000ed5b55d5a7509, 0x0000fb679388fd7c }},
        {{ 0x0007c48aae979814, 0x000e22a2c160dcd0, 0x000387cc6040b48b, 0x0000e495a34e6c5e, 0x0000ebc559de6b12 }, { 0x0000871f6c9b95e8, 0x000c6da50068527a, 0x00027c0973fe1c51, 0x000e34a4c2c09ef2, 0x0000a42cfbc74a4c }},
        {{ 0x0006897ae28b8aa6, 0x0005568d84d835d2, 0x00095f9447042666, 0x0000a52d7c90caf3, 0x00007477e6db63ea }, { 0x0002c7980e3a62d5, 0x000b508a4d6755dd, 0x000b36ca63cc8293, 0x0005c42e8403ee41, 0x000072dad9713001 }},
        {{ 0x000f22110209bd6a, 0x000f832d2e11305f, 0x0001f49e696d2947, 0x00057e69d966be49, 0x0000a57523d1fdb1 }, { 0x000e0267ea28eb43, 0x000134aabe30129b, 0x000f756bfce543b4, 0x000213d22c6f2c93, 0x0000707ad02d7862 }},
        {{ 0x000a76e9e4386d11, 0x00081f193bc042b7, 0x000437c0da73ea74, 0x000bf3068f085b53, 0x0000015ec7b9d94a }, { 0x0001889c8d4274a0, 0x000e2fab88911b55, 0x0004635272281033, 0x0003323ffc85345e, 0x0000f694aa06193f }},
        {{ 0x000043f2c0486dd3, 0x0004069a5e829e5c, 0x000b55c3b7815495, 0x000afb1c1cbb4c6f, 0x0000adfdd639e5db }, { 0x000d515f74e0a7f0, 0x0009f758ea5e1853, 0x00099b2b5e2aa1fb, 0x000e7d6018bda40b, 0x00002841bc77e94c }},
        {{ 0x00003172599604ef, 0x000e04fda79e5acb, 0x0009d5feaf05bd45, 0x00080866e68b83b3, 0x00000b424807d53a }, { 0x0005296e9538c34c, 0x000381ac5ccc2c46, 0x000ad873e1d42e72, 0x0005408bd7d7dc96, 0x00006a74e1c17bc4 }},
        {{ 0x0001fdfedce79aef, 0x000c3d0bc8fa5bc5, 0x0006fe0e289f0f1a, 0x000059b5d8a5ead6, 0x000023b6e31609f8 }, { 0x000d66af2147f6ba, 0x000890b289fca32e, 0x000eb0352d1e23d2, 0x000f354dee36ff0c, 0x00005f8a7192aa10 }},
        {{ 0x00010989c077ae7b, 0x0001b8b5fa9f4d67, 0x000dc699d9268932, 0x000596c722d4066c, 0x000090b03f888845 }, { 0x0004d874f8f53831, 0x000ae35ce40cbd89, 0x000f826ab444e0a6, 0x0006a1d2cab55c5e, 0x0000acd6cecd4b95 }},
        {{ 0x0004c0e1b8f9d216, 0x000dae8a518001b4, 0x000658440d56a993, 0x000bb3725ee605eb, 0x00002b0f4a4a14c0 }, { 0x000bf395cc3b7ed0, 0x00054e88b6740dc8, 0x00036229b9409031, 0x00046137289087a5, 0x0000be581ada8fe4 }},
        {{ 0x000d51afe901e219, 0x000b0a1deb8568b6, 0x000013a98d491c8f, 0x00016e281a35daea, 0x0000537d475b7d2b }, { 0x000c6865cafe90b4, 0x000f86e51803a198, 0x000ef9b92ab5832d, 0x000de923ce3d24b7, 0x0000b2e4a54bd2e0 }},
        {{ 0x000c70ca45125561, 0x0002a7db743f4ef2, 0x000bbb6053892073, 0x0009d8ae83793909, 0x0000dbb97d8fdd1d }, { 0x0000384cb6274347, 0x0005229cb1131b10, 0x0008255041c3b80a, 0x000952d0a7826f29, 0x00004174450b76ca }},
        {{ 0x0009570c16f8f650, 0x000c25b3ab480953, 0x0006297944efd26d, 0x0009f08453451c25, 0x00004a96dc9b66ab }, { 0x000ad4b67319678a, 0x00020d3adecf263d, 0x000101a22a8fbe78, 0x0005f1e36f1c3d8d, 0x0000ed9d0a1de53b }},
        {{ 0x000d2db93bcd3273, 0x000260ff0d796a03, 0x0008a1e0d1c6f14c, 0x000d98dad5fc6247, 0x00004b1ab345b835 }, { 0x000ea6e007bc5637, 0x0005379d85d5ddc0, 0x00075d099f2d2836, 0x0004877d5e907caa, 0x00002c068a8b32e7 }},
        {{ 0x000651f5d8d828b1, 0x0004069d23233f3a, 0x0005d1796da707e5, 0x000781e6c3e32279, 0x0000b55160373545 }, { 0x0008bdd37cafc9ac, 0x00003191a230c767, 0x000254ec8ace2d98, 0x000e2217ebd52727, 0x00005901810eba20 }},
        {{ 0x000491ccaacfb314, 0x00002c217cd93fe1, 0x000dc64b1a37f286, 0x0009b2ef865b97a4, 0x0000b59b701c4047 }, { 0x000da2b7ef8c43e7, 0x0009b80231963324, 0x00070ad88faa8816, 0x00098e9ed3c3862d, 0x00000eddebb7b375 }},
        {{ 0x000ee5cf3c1ad26f, 0x0003f33b126e48e4, 0x0002a5af12fbde15, 0x000885bdb404a0b3, 0x00001567d76438aa }, { 0x00055c5bfa632ed7, 0x0005cb97df3c8c32, 0x0003920bad02d03c, 0x00016ea6abd905fe, 0x00005ccd15e48747 }},
        {{ 0x0004610714435ee0, 0x0007cbfc3dd9b4d7, 0x000f87a583732e37, 0x000fc229bce23edb, 0x0000d1868b201130 }, { 0x000f1bc7d1d5a193, 0x000645f1266bf52e, 0x000129c8d3e49b75, 0x000cd86e30204672, 0x000000c5c9aacd67 }},
        {{ 0x000c3fefca460bc8, 0x0005fc17078c4322, 0x000c8b3ffded87cf, 0x0001d3d70e1961b9, 0x00005038a807927c }, { 0x000eb77640143e4f, 0x000bb36c2af389a8, 0x000d2eefa6aabb49, 0x0006c356396c610f, 0x00004a75c050691e }},
        {{ 0x0004778ffba62fa6, 0x0007e542a296e775, 0x0002ee0bea399904, 0x000da2131c871868, 0x0000d6cf7f2694c3 }, { 0x000c13e651883a51, 0x0006eecaf06defd3, 0x000f497ed0b9e2f9, 0x000a01014fe3c105, 0x000029455e6852d9 }},
        {{ 0x000dffbb5f63e5d5, 0x000d329855938e83, 0x000e5fe590137f7d, 0x000e725b92eeb5d9, 0x000084e49f54aebc }, { 0x0005d787c5f79f6e, 0x000f0431acc093d5, 0x000ce38d469b87c5, 0x0004a46600cb6cc3, 0x0000f30c7a69dbee }},
        {{ 0x000a17ef8e44cede, 0x0001f47deadbf02f, 0x0008b22da0da0860, 0x0002433f011e8021, 0x0000e0cbc364982d }, { 0x000764196f27ef16, 0x000e72593e43bff7, 0x000e947fecc38758, 0x000f707a35cb5ff3, 0x00007e04f36d9766 }},
        {{ 0x000cfc53b4244564, 0x0004fedf0290a8fd, 0x0005ee4b6fd35ed5, 0x00009974fcae8196, 0x0000c1f220ef19f4 }, { 0x000bedf5f9d4ef28, 0x000ca3c4cb632f9e, 0x0005cd318a6d91a9, 0x00094f00ac42a1ad, 0x0000689a17da238d }},
        {{ 0x000ea61e2a63d186, 0x000c69f97e3e0442, 0x000fe32a819f8e55, 0x000bce8234201851, 0x0000668bd1c6ed6d }, { 0x000868c740040cde, 0x000b3bc575eaa945, 0x000160968aed23c5, 0x000e36f6f9e7bc72, 0x0000880b167f7fa3 }},
        {{ 0x000c19e283331498, 0x0002adb7be0eead0, 0x0002f357aeb26c71, 0x000d5f8d7511e784, 0x0000b48101676692 }, { 0x000621b037cb2a49, 0x000a7cfe9206c550, 0x00042f321032bfce, 0x00035370489389cb, 0x000054fee486602b }},
        {{ 0x00049dfb6757e9ab, 0x0003b3596e78cc4c, 0x00014768b940c469, 0x000fed2035385c8b, 0x0000146b998f7458 }, { 0x000d882de450b14e, 0x000937059f707727, 0x0003bc9587871fc0, 0x000d9a3903a073a8, 0x0000d3d672691eea }},
        {{ 0x000882984a6412d2, 0x0006d0ab2cf89408, 0x0007e711f7072d50, 0x0001381c872b0b17, 0x00004f08b915e614 }, { 0x00012be0ff5c5e45, 0x000180c6a19d2bbd, 0x000bb5cba1cc5bab, 0x000322b644b9e7f0, 0x00003b06299d88d1 }},
        {{ 0x000246e8af14e4f9, 0x000d93d0e165dc33, 0x00021bdc901f20a9, 0x000d3a7873d47ce6, 0x00006860c9c10ae9 }, { 0x000c598bba00d6aa, 0x000a4c703ce2b4df, 0x000abf66ec1544c8, 0x000c296c1e9b4ca1, 0x00006a2a90d3380d }},
        {{ 0x000a1105556dd7cb, 0x000fb2b45036099d, 0x00013c6b6083296c, 0x000e64e43a9dee74, 0x000078a017dad06c }, { 0x0003baaf9a80f826, 0x0007f2c651a3be27, 0x000d14899a0c2d08, 0x0003b98d1cda5111, 0x00004e13797d5a7e }},
        {{ 0x00000cab02e61927, 0x000e59d7fc63668b, 0x000c5bf6f1db6ce1, 0x00098fc516886c9d, 0x000015963851966f }, { 0x000d956d9d123a74, 0x000a2bc81b69442a, 0x000007c6cf517a7b, 0x000612e784148670, 0x0000acd5ae38e4e2 }},
        {{ 0x000d774e4ee4ff1f, 0x000d38b638a7e819, 0x000ebc098f74b062, 0x00070057a54155cd, 0x00004efa631b47ed }, { 0x0006843cb3f5f8f6, 0x000782e078637d27, 0x000ce786a3b624a5, 0x000d6ad541c363e2, 0x000040f408018af3 }},
        {{ 0x0004197ed9721891, 0x000d491d4b66947f, 0x0007102c8bb348b1, 0x0005dfe55b452ce9, 0x0000b7e62fe46cd1 }, { 0x000ada31f6ad76d5, 0x000df72400dd0f70, 0x000f40443e9cb7aa, 0x0008c63ae59f5ca8, 0x0000bd128c94245a }},
        {{ 0x000c2a3869098900, 0x0004eb694a2c2152, 0x000ad97b424b1af0, 0x000e689fed83a43a, 0x00002b03a91bd120 }, { 0x0001ce6e973b5ec6, 0x00073408439d9b28, 0x000188feb07a97d8, 0x000880619715ea67, 0x0000f2fc52299eac }},
        {{ 0x000af556ee5923e0, 0x000ffe0d809c7bd0, 0x000c841af9bcabc1, 0x00080376fd7b2580, 0x00006abbc0368a7f }, { 0x000e61844f2bc503, 0x0008873921de0acb, 0x0001c045ffb0ea65, 0x000091e7452ac9c2, 0x0000c85b8dcd9bc4 }},
        {{ 0x000b192421f0e56b, 0x00086bc789a283b1, 0x000d6cef12d213ac, 0x000ffd52d72d3ac3, 0x00005f7fe8a58197 }, { 0x000a3bc86647bf41, 0x00058989dc84cacd, 0x000cc232ad52144c, 0x00059dbe2b0a8482, 0x000095d11cda69f2 }},
        {{ 0x0007b4d51ba3adb8, 0x0004e53056e6f8fb, 0x0000ad1c5e8c26eb, 0x000bd064084861c4, 0x00001e7739155ad0 }, { 0x0000518677787161, 0x000d111220a255d5, 0x000bb1f017f68d42, 0x0009aceeac997152, 0x0000559f1f223eea }},
        {{ 0x00043d74bc914746, 0x00072bacf902b0bb, 0x0004839dbb7ea13c, 0x00008cbbc8d4b4eb, 0x000057a993c570e4 }, { 0x000004512228d85d, 0x0000ec98adba3503, 0x000c8517c8053c9c, 0x000d515ec364eaf8, 0x0000420c9b7173e7 }},
        {{ 0x0008a61cf4dc1641, 0x00058b7ea7b979b7, 0x00046e551ed89117, 0x000b9bafecd78cd7, 0x0000ea75547ea9aa }, { 0x00033effd03f2de0, 0x000e8723d502295f, 0x0003524db2b8913d, 0x00001a586f137685, 0x00006e05ca06d0f5 }},
        {{ 0x000fd495f78c275c, 0x00068cdb30cfb3a6, 0x0000fcc91ed14bb9, 0x000a17ddf6d09b8c, 0x0000645d0ce04a7b }, { 0x000dc229b0415b16, 0x000009f275264daf, 0x0005e7bb59b2b9b3, 0x000525c2280c2b74, 0x00002b3172744708 }},
    },
    {
        /* digit=25 (1,2,..,64)*(2^{175})*G */
        {{ 0x000fc20755e3176b, 0x000828452666a58c, 0x000016e7bc6ccdba, 0x00078f9084bcb6e0, 0x0000554faca4c643 }, { 0x00094e142cf0b0d6, 0x00046505294dba30, 0x0006822fbef1afda, 0x0006df474a30ba28, 0x0000e6be6e6ae1a8 }},
        {{ 0x000b296904664fc3, 0x000b53e979f39254, 0x000a642320a351cd, 0x000cc34fa1efd130, 0x0000a26d827b4096 }, { 0x000df088ada01cc1, 0x000d534b9db65b69, 0x0001656914dd4d71, 0x00018a2f335c82e3, 0x000058dcd3dda1a0 }},
        {{ 0x00036f4fb314c50d, 0x000a366bb3fd30a1, 0x000adc80e59bd10a, 0x0005fd66c429169d, 0x0000d081f2b59f20 }, { 0x0002bf6af341698f, 0x00020011fbba712e, 0x00036dac36ae06bd, 0x00045183e6ef0f0b, 0x000041383caf3e36 }},
        {{ 0x00008fc59cde3468, 0x000e7124237b64b8, 0x0009635d376aaed3, 0x0004ff5d8688ebe9, 0x0000c1b55d497018 }, { 0x000b143a98f532c8, 0x000632ba3585862b, 0x000532de58edb3d0, 0x000ebfa9bb66825d, 0x000060efc436424d }},
        {{ 0x000ae5a380deb2a8, 0x000ecce12ae381b9, 0x000bebe7370f573b, 0x0005442ca3a7f176, 0x0000600a0769fd67 }, { 0x000d65f035454aa2, 0x000eb15ed251b464, 0x00007d646e2f56a1, 0x000a76d071f5d6d3, 0x0000bb2c5fed005a }},
        {{ 0x000f8f66b8a7cdf7, 0x000c324f111661eb, 0x0003f176844bf6c5, 0x00056be78edced48, 0x0000217f1b2be94c }, { 0x000ec85fbc8973e6, 0x00067f4d7ed8216f, 0x00068f645f12fdb9, 0x000735154bf07f37, 0x0000c3cd2d5edd0f }},
        {{ 0x0002b052011404ae, 0x0005feea5ba5b0b7, 0x000dd02000988637, 0x000ab7f42e6a116c, 0x0000b74febd1fdd5 }, { 0x000b000c06b573c2, 0x000ad7d31a3b4ea6, 0x000820bd8f2c06dc, 0x0003d6897e589307, 0x000078133015312f }},
        {{ 0x000f53916fc6beae, 0x0001122765fa7d04, 0x000fce16c2004cf7, 0x000c22d9859805be, 0x000052df10ff2d7b }, { 0x00099f450e1f9830, 0x000c61f33ddf6269, 0x000e06e68bf551b7, 0x000e86ee34206238, 0x00002aa249bfa9c5 }},
        {{ 0x0004c88f374f6f88, 0x000d851e4c79e9bf, 0x0002d1a210b3161a, 0x0004ef3231394cb7, 0x000034829a179e77 }, { 0x00060e7f8aef7dc8, 0x000aa1f4aaa499e0, 0x0005ddecaae02032, 0x00014bd1d68b3a7d, 0x00009e90f751f24b }},
        {{ 0x00047aa4c7ac2b3c, 0x00082731a3a93bc2, 0x00057da03752d6c3, 0x000d7d2fd42bbf46, 0x00008348bc06dd42 }, { 0x000825653cf027b6, 0x0001032f60c77da5, 0x000a69fded019b66, 0x0001dc3cf6ffbc26, 0x0000333cf94fd444 }},
        {{ 0x000648487f9f745e, 0x000ab7ebc1aa2b92, 0x00031157eabb1368, 0x0002c7d380788939, 0x000092962173f198 }, { 0x00006a7a345f44e5, 0x0002e363fcef2614, 0x000cce4176f57b8b, 0x000e286e26239c81, 0x000007adc1ccaa3d }},
        {{ 0x000ddab96e59ccf0, 0x00077ba4a493b31c, 0x0001ce1d36b7f226, 0x000c8a14b54f20a5, 0x00001a428916f43d }, { 0x00047a670dede59a, 0x0002d0aed25ea138, 0x000afd5154d9620d, 0x0009faac5fa1d09e, 0x000024f267bf7958 }},
        {{ 0x00006f4098b6440f, 0x000331e64bcb626e, 0x0007466fa01d3aef, 0x000cc256caea638c, 0x00006bacbc3672fc }, { 0x000063d8a3c3a04d, 0x000c7527512a30b9, 0x000d5525ccf645f5, 0x000307480f3d867f, 0x00004f61af99ab8f }},
        {{ 0x00017e76ea8b4aec, 0x000d995d314e7bc2, 0x000b24fa2303625b, 0x000d3e9d32ee7464, 0x0000fd86bbb7517e }, { 0x000a5817e3ed6c66, 0x000681fe7cda917f, 0x000a903127323a60, 0x0005ae96b12d8016, 0x000044afeb2ca0b2 }},
        {{ 0x00098ad9f490f5c1, 0x00080640821ee4c9, 0x0003723b3f9d8e3f, 0x0007cc69f1583eab, 0x0000a66bfbb2018c }, { 0x000ee4ed6e631317, 0x0000250a41157f70, 0x0006d9f54d0fbb05, 0x00013a25fa9d7e1e, 0x00007eea65ae7d21 }},
        {{ 0x000b69cbf65aaebc, 0x00090c5ac751e7b7, 0x000281d845c75cee, 0x0002f93c693f9647, 0x00005a6bb7c3dc6e }, { 0x000be7048732b0cf, 0x0003da2f4bf94e19, 0x00085541905b0af8, 0x00067070b1a5325f, 0x0000e3a10d49e546 }},
        {{ 0x000c686e55c3531d, 0x00063bfe1dc5c055, 0x000a92567d01f185, 0x000bcc9a3d738add, 0x0000afd5ebc87e1b }, { 0x0008a947fad2853c, 0x000cf50dfd0b7ec5, 0x000b11a03426ca2b, 0x000cf3096bdd0264, 0x00002575e52919b9 }},
        {{ 0x000fab1ac5f1c3a5, 0x000e84e49ccd6d74, 0x000c39123abacef0, 0x00010ceb2ff8fb02, 0x0000b9696ac02316 }, { 0x000d8260aa363274, 0x0002fbc97576ec06, 0x00079dfc1c84ccdd, 0x000fde200e8c946e, 0x00005e0955dd2239 }},
        {{ 0x000fba35ff441c20, 0x000104f70bba85cb, 0x0000e4e69d11bf4c, 0x000f9cec1ad090da, 0x000045f5d2d270df }, { 0x0002160efb05643b, 0x000c69ac1e919380, 0x000318e632645d14, 0x00055d30fa47638f, 0x0000433a1488c0b7 }},
        {{ 0x000cd99e61699f04, 0x0000c73615cd9e45, 0x0007944a9da57d26, 0x000dc398c498ec04, 0x0000524b1feda847 }, { 0x000ec9746fc9036f, 0x000a3df7851dafeb, 0x0009e8c6ed72b8df, 0x00008ea0ef156f5b, 0x0000108b102601bf }},
        {{ 0x00012050d9a73af9, 0x000b0c1c8ed1f7a4, 0x000566fd0f18765d, 0x00065804a09e43bb, 0x0000103c0ab78e66 }, { 0x0003868fbbf33bcb, 0x000b14e90747e405, 0x000b62a3a6cab563, 0x0005268536dc05eb, 0x0000757f03238b3d }},
        {{ 0x00080f9fb1b49db5, 0x0001dc3cfdcf7dd0, 0x00036a302b02766c, 0x00057c7a341a9fce, 0x00000720671b4553 }, { 0x000320fd327ff00b, 0x00083849ba7a5317, 0x00067a79174cfc18, 0x00058b83607d3558, 0x0000ab6cf754792f }},
        {{ 0x000b96d16da0f72c, 0x000db98b18bd1ff8, 0x0000244657980980, 0x0004b3bcd52c2e30, 0x0000cd07defe68a2 }, { 0x00019448d72e9986, 0x0008a494b45706bd, 0x000da2a8dd214d19, 0x000597a76e7f58b8, 0x00003146bb218459 }},
        {{ 0x000769788c7ccd96, 0x000f6b9d3210a041, 0x0003c391d9dca1fe, 0x000f5b29f6bc7d46, 0x0000a7a3bdce7dea }, { 0x000971da0e302072, 0x000462dbd6cb838c, 0x000559fb85145a33, 0x000e7e16e7066d6e, 0x0000166a58b45338 }},
        {{ 0x000941b2a22935bd, 0x000eb6bc49a2fac6, 0x000fc75134d7d87d, 0x00039f8563b8420b, 0x000028c1f97d5e54 }, { 0x00075bcd770bf80b, 0x00058080658ff2ac, 0x00026cc8ddbf0813, 0x00059d7998780436, 0x0000b046effada4b }},
        {{ 0x0002a6b802d70649, 0x000f7907f8f875d6, 0x000df41bb3bcbbe4, 0x0000a72e79c9d754, 0x0000ec89ec7b5eaa }, { 0x00019fe96bba031c, 0x00025aac6e7394b3, 0x00037bb805a4ed99, 0x000a3d80c445b6a8, 0x0000d094b5aa7fb4 }},
        {{ 0x0000920d0920b3ba, 0x0000c1f385f63cb6, 0x0003c22bc41dea5c, 0x0003588c241250c8, 0x00006627f300e7e9 }, { 0x000a9bfe7766cbd7, 0x0008046724d81a52, 0x0002311e3421bbd3, 0x000755932b7ffb89, 0x0000c38dbc2a2a77 }},
        {{ 0x000c94a8b3fecb0b, 0x000d7393191f4fd2, 0x0006a74204edcd02, 0x000ab8b0643093e1, 0x00000fea7bdc3f59 }, { 0x000ac3f5f4222e4b, 0x0006ec05c437d8e4, 0x00038e15fdd446f4, 0x0004245a23d4ae42, 0x0000b1ce9686cd0c }},
        {{ 0x00048ef841ec905b, 0x000adfc6b6af9917, 0x000fa6cdf37b4794, 0x000b3b2d70d5270f, 0x00003f3048a38ef0 }, { 0x0002c56f83b0eb6a, 0x000940edfc6fafb1, 0x0005e46a9fa3f7f0, 0x000cee4c9eadfdf0, 0x00001b1b2c4fdc4d }},
        {{ 0x00000f6ee71a6cd6, 0x0006c03d39b8c34b, 0x000581167ef45d8b, 0x00076c7c5ed8b1be, 0x000070c61c8b5adc }, { 0x000c324341fd93fc, 0x00072cf57d0ea992, 0x0007ebae2901bcb1, 0x000c74faadcae0f5, 0x0000ffd2c5f6a211 }},
        {{ 0x000d8607e66825bc, 0x000c355e126d4682, 0x000f48ee862317f1, 0x0008401c2ed325f1, 0x00004e9bb5b46467 }, { 0x0006b0cdc1954e42, 0x000fc276009d660b, 0x000dd65ff7df7908, 0x00041c7a03a525d2, 0x000009b10a289aa6 }},
        {{ 0x0002dfa557bfa495, 0x0000476e1337c262, 0x000db45dc38dd4f8, 0x00012d3d96faadeb, 0x0000c13ada75bd32 }, { 0x000c6ca5801dad5e, 0x0007c17be93c6d61, 0x000188985039af29, 0x000bbefa124866c6, 0x0000462261edbc6f }},
        {{ 0x000e8d3ac716b9cf, 0x000de8ce964021e7, 0x0009849e47ecab8b, 0x00035c998b83bffb, 0x00005192ceb4ff31 }, { 0x000b2b0564f91674, 0x000ac91bc2d124af, 0x000c0a865a8f2693, 0x00065fbb7bd54c3d, 0x0000ca36576246cf }},
        {{ 0x000570dfff87d7fe, 0x000c7575b45afb42, 0x0009ca47b12dc69f, 0x0001ee757e9ded64, 0x0000ffe130d9f5bb }, { 0x00098ca255a80b64, 0x000977888e084bd4, 0x0008a9f3435e8330, 0x0005936eea024b23, 0x00002da3669daeed }},
        {{ 0x0005af3c6e99cba1, 0x000c5eaaaa07e869, 0x000b5db14917a1fb, 0x000f6b8c80db6fac, 0x00000dbef989db23 }, { 0x0006542e486c2d07, 0x0008b8c02e30afbc, 0x000c3a365a70473d, 0x000f0979d9b5356a, 0x000015f1407ea962 }},
        {{ 0x000afe3247ac4682, 0x0009de87adfb6cc6, 0x00032d694ef9c6bf, 0x00039cf5acc826fd, 0x0000ca422fa86990 }, { 0x000a6778b884ff75, 0x00093b9b6ae8a751, 0x000d3067d9288acb, 0x0006244d6b8cf388, 0x0000c49ee251a7e4 }},
        {{ 0x0004f1a6a51a864f, 0x000aec1ecc4c7e38, 0x000c91e9c227bdba, 0x0005bc7aade9fe47, 0x0000637ef4323989 }, { 0x0003365b1415cdea, 0x00093725a7b162b9, 0x00091a91bc8f680e, 0x000e7a346c5cfda1, 0x00002d033d623c62 }},
        {{ 0x000f201022c136d0, 0x000022526a1fc900, 0x00023e50603a6d0e, 0x000d575f40d6d10d, 0x00009c8b622aee15 }, { 0x000394ff6b33c8ff, 0x00067c5658fde413, 0x000d4a9260b29b00, 0x0005a0ce7a8d2bc7, 0x000020c000e376cb }},
        {{ 0x000550bd2e68ff28, 0x0009d098132c9af4, 0x000af0cdbb3a6f91, 0x0002dedaaaac4b0e, 0x0000ab754349a57f }, { 0x000fb1b899561971, 0x000c08eea5399d41, 0x000a71b1d2e6e9c0, 0x00031302dd989bff, 0x0000dd9508dd374b }},
        {{ 0x000f01732de8a9ab, 0x000ce6944c44d6a8, 0x0009c8e6ee09ad42, 0x00004014491df144, 0x00002eeb080df28b }, { 0x0009476f970fa389, 0x00014253aa0a6332, 0x000ca08f6605dcb4, 0x0004bf71006de5e7, 0x000099a78f12c1a2 }},
        {{ 0x000aff2f226112af, 0x000c8c0d34c35659, 0x0001809fef7bd257, 0x000cdd102ef1105b, 0x0000c029fa506998 }, { 0x000b332e089dab46, 0x00009853a80e75d8, 0x000de08a787917ca, 0x000ecb77ade71853, 0x000002996859d6e3 }},
        {{ 0x0002ca07544326df, 0x000bce8b9c20cda9, 0x000543030543c2b2, 0x00092628389f7aad, 0x0000462f560eaa18 }, { 0x0004fdfd9c7882f6, 0x000e51174a1e0359, 0x00077eececf624e2, 0x0006205f202628cc, 0x000053d65edaa5d1 }},
        {{ 0x00013045bf671e99, 0x000c75463428c6ba, 0x0001ef83c3e5fed6, 0x0002715112f7205b, 0x000071bd670eb998 }, { 0x00099f3ec14c07f5, 0x0001fff03ef77fee, 0x000f16504d2e9402, 0x00092e922689cd59, 0x00006897e1a72bd2 }},
        {{ 0x000d2466e3ef3d22, 0x0005b91ea1b3c3ec, 0x0002eb5dd366b2c8, 0x0008d231e2d1a705, 0x000040cdd3ee863d }, { 0x00007c4b82673e6c, 0x000882fc8cb762d2, 0x000e5613d59cd383, 0x0009208507e312aa, 0x0000c2327bddf0de }},
        {{ 0x0005670fbf65ef80, 0x000a5c1375913eca, 0x0005c229726133e4, 0x000785bf694e45e9, 0x0000158117a0c867 }, { 0x00077fefaeecfab4, 0x000024a66f5b8fca, 0x0009d0effc797a1d, 0x000049552d5a56ae, 0x000049ee3eb9b0d4 }},
        {{ 0x00004a61164aac1e, 0x0000f7c98379d445, 0x000b00547b73dedc, 0x000c3c664d000bdf, 0x0000f6c15b16639b }, { 0x00067ecf4a4fa56c, 0x0004c71effea31ae, 0x000d0a434b0a070e, 0x000a1e7bf2d3de26, 0x0000cf8b029be733 }},
        {{ 0x000004ceed9ae923, 0x0004652151fe690f, 0x00038f260c71df2e, 0x000cd078d0351580, 0x0000d3995d9f2836 }, { 0x000008e04b1b6128, 0x000c27cf0386fac2, 0x0002b1f3c705ed36, 0x00018dcec3b62ff1, 0x0000371099c99400 }},
        {{ 0x0002a2c2241af6c3, 0x000518e41dbd7725, 0x0002cff16430b650, 0x000ce81f72b419f7, 0x00006d2ef331afce }, { 0x0006510652363465, 0x0008914634dfb02e, 0x00000e2763a12775, 0x0005ed5cc5f074ff, 0x000022c4d1e63b6d }},
        {{ 0x000fde089e1449b9, 0x000db82f1012e7a7, 0x000028c75e109fa1, 0x00076543b51013cc, 0x00008371ca42a533 }, { 0x000f1f5764038bdc, 0x00078fec731b4b3b, 0x0003dade48fdff7b, 0x0008e59bc12a136a, 0x000052c07ce95456 }},
        {{ 0x0000061987d3a8bf, 0x000e66da16d4b346, 0x00097c9c02c81980, 0x000e525867fba9d0, 0x000068da7a517899 }, { 0x000d7e28755b891e, 0x000f1869da28ff0c, 0x0007764631272fb6, 0x0009fad0d3133f70, 0x000092271cbf4392 }},
        {{ 0x0008dc98e16ac555, 0x0009bf272ed91fcd, 0x0000c3f80a91efc0, 0x0005fd03c85214f3, 0x0000571a08a1dcae }, { 0x000cbff4f341eeda, 0x000d5fe585cc3a25, 0x000d169c04872f31, 0x000c46b81724952e, 0x0000703ab0cce8fc }},
        {{ 0x000126332eeca533, 0x000d70e87d8cc7b9, 0x000360ac4da1a08c, 0x0004994049beaa7f, 0x000085d58a9542af }, { 0x0006092eed4373ee, 0x000d3b48ed3bd6db, 0x0002233cbc36be87, 0x00016f45535bb2c1, 0x00009ff16f286157 }},
        {{ 0x00011e4bcd460f2c, 0x000e83fcc058865e, 0x00018d064d550766, 0x00041a7537b0a5af, 0x000046efe0747324 }, { 0x0002573bfa73d9cf, 0x000ca3aa14a90fa5, 0x000c67b515d9d098, 0x00058982d7706e20, 0x00007d1585bc34a1 }},
        {{ 0x000717596c322d2f, 0x00087a93b5bdb831, 0x00008d8fc9ac384e, 0x000bdd75f08da65c, 0x0000f1e0a2f83e31 }, { 0x0002f3b50d9aca4d, 0x000a3d0b266d88b4, 0x0007164ddf4c7663, 0x0005edfacf688ae9, 0x000069a04a1ed5e1 }},
        {{ 0x0001bc81d1567fa5, 0x00054986a90fc34e, 0x0002840e08bc1725, 0x000d25262c8f257d, 0x00007d37c1cef912 }, { 0x0000aa9ebd5aee4b, 0x0004467251a2a12f, 0x0006cddfd21ec0d8, 0x0000a022e10d8658, 0x00004f408c5bc296 }},
        {{ 0x000652b5c82520ed, 0x000d104275968936, 0x0005df6c8f6344c2, 0x000a5abc8b6bb02a, 0x00005bc743f852cf }, { 0x000c773c3afe64cb, 0x0008afd4f5c04999, 0x000818dbeefd7cd2, 0x00084a3f76ea1655, 0x00008333b4274ea4 }},
        {{ 0x000f9a78b85b575e, 0x000dc409b323abc8, 0x000161475ed8d00d, 0x0002fb77ae26605e, 0x0000ce1190f5abef }, { 0x000c94bddefb0e25, 0x0001faac407eddd6, 0x000836c6d870d410, 0x000fa149981ddd1d, 0x0000e0d5b79ac001 }},
        {{ 0x0003624f3e74bdef, 0x0005bda8545d185d, 0x0009a029698ce572, 0x000c9f20e82ae44e, 0x00002103e5d5a56a }, { 0x0001f363f57898e9, 0x000379b14be42542, 0x000c9bc8cab053ab, 0x0002a92be7d0efcb, 0x0000217605b86c8c }},
        {{ 0x000f7246a4a10356, 0x000e71ca85015501, 0x0001b938ca17004c, 0x0005e6f1c2251e0e, 0x00000d5fa3abf1c8 }, { 0x000120a64c2cae9e, 0x000a6ee5f2591f26, 0x000efd41cb0b0208, 0x000451f683c47d33, 0x0000a218fb5a7cf6 }},
        {{ 0x0001c18807f2e6b7, 0x0000fc07b5d07df4, 0x00083d320f7b47b4, 0x00053e44fa92b955, 0x00000d924dd02cb8 }, { 0x00001b784f6b6a60, 0x000708980ba9b1c1, 0x000eb517318e9114, 0x000ba0809ac8dc6d, 0x00005f1ad1849fe9 }},
        {{ 0x000fd8ae60415444, 0x00001bbb6f116037, 0x000dc1f9d1be7f45, 0x00020f0a857b09db, 0x000074bd20b5cf68 }, { 0x0007958e09141208, 0x0009e6678308435e, 0x000bebf0fe4490f9, 0x000c5a2b09572f77, 0x000007396bf5f419 }},
        {{ 0x000fd7296995f44b, 0x000e342253466896, 0x000040a5469aeb31, 0x000a0c3e3716da29, 0x00001e5c16338121 }, { 0x000ef6d5416eee55, 0x000430308b640d30, 0x000e79883d5fe34a, 0x000403e115b0026e, 0x00003880d9961f05 }},
        {{ 0x0001e0f84f93b4ba, 0x0006df0f9c41135f, 0x000876c5514402dc, 0x0008ccd5cf21d60c, 0x00008207421ebd51 }, { 0x000f04b824048527, 0x0007c5906002d609, 0x000b67145d2e9747, 0x00092f7c3be2da60, 0x00003cc3292a8a5c }},
        {{ 0x0002edf52c078d85, 0x000c57dd4b79bb87, 0x000adfda839ce9d2, 0x000ba33e8aee806f, 0x00007fc8b3dee585 }, { 0x0003b2818874b38e, 0x0001a4e2127015a3, 0x000069054d6b7749, 0x000ba60c89051d0c, 0x0000fcc7ea0acfb4 }},
    },
    {
        /* digit=26 (1,2,..,64)*(2^{182})*G */
        {{ 0x00063f4153af6dcd, 0x000c683171b445fe, 0x00029af26bf85199, 0x000cbbae89bcf21e, 0x0000e2560e6db219 }, { 0x000e1d75ae224a6c, 0x000f6a930add43df, 0x0002cb637aac5049, 0x000bc7d6862ebb14, 0x00005e73664b3718 }},
        {{ 0x00080034798ca4d4, 0x000f2c7af8685c8b, 0x000d778ecd30d2f0, 0x000cf952ad4c1c8c, 0x0000e198f05db898 }, { 0x00063d51806a84e5, 0x000e1e52eca3cdb7, 0x000b75c78219120a, 0x000a6e0cfea1db16, 0x000052a4a35bae24 }},
        {{ 0x0006c8bebe50eb45, 0x0007cb808bd98d05, 0x00040ead3381f366, 0x0008a8d70644b1cd, 0x00007b632a717522 }, { 0x0008239899b73b3c, 0x00096acc4632b46f, 0x0009b7b269a90fd8, 0x00021394617da427, 0x000076f048091e12 }},
        {{ 0x0008e13092cf573a, 0x0005c3f4c8303203, 0x00005750d3512460, 0x000db40fb842c7aa, 0x0000494598d37817 }, { 0x00077989cb6b3d8b, 0x0007062757558f1a, 0x000150731ef6087b, 0x000f58b273cc3e83, 0x0000ffa7db45a98a }},
        {{ 0x000e632a990cd2b6, 0x00061a144a081e00, 0x000da8aafed1d92a, 0x0004cbb9cd797fb7, 0x0000eb29b6b6973b }, { 0x00010f9eff77898f, 0x0004216c5f0c2848, 0x000eac88dd631243, 0x000a457cff51088a, 0x0000f820a5bcdad4 }},
        {{ 0x0002830fdbcd7a9a, 0x000fab97ffe4b3ed, 0x000113774d45e378, 0x0008122750b691d2, 0x0000dbabb5016bd0 }, { 0x0004aef03d79fd03, 0x000219c52434f573, 0x0005635a7d6e7d8f, 0x000fbbdb34b1f754, 0x000089b90d825715 }},
        {{ 0x0005157b6a28f2a3, 0x000aef4b7ac7ec57, 0x000a6ff82c6bcfde, 0x000476bb8d6ceb95, 0x0000cd5591ba46b3 }, { 0x000117f5ae655dc1, 0x000064ec5413d62f, 0x0006cb47cf197d67, 0x0009cdcdabe329ff, 0x000034f79ff63bfb }},
        {{ 0x000d84601a040653, 0x000c4f4b0166f7a3, 0x0007916814ccaa7c, 0x00080f8fdfbd3e3f, 0x0000bdc4370992dc }, { 0x000a6279f7786e45, 0x000a95c7c1620b02, 0x000d01014b0992a4, 0x000fa801eba68b4d, 0x000052fa0f983aed }},
        {{ 0x0001c6a659019d30, 0x00039b06ae85c104, 0x0008dc372d4d1620, 0x0006a5bec7294697, 0x0000ab9976bc0f2a }, { 0x000333af21cebd7b, 0x00095240e0239001, 0x00019dd9652bff2c, 0x000cd3ddcbaeec12, 0x0000ff15e9da5236 }},
        {{ 0x0009088b4d137854, 0x00096e03aaba1f19, 0x000da268b982cbad, 0x0003344bc2cada16, 0x0000badfeeab305e }, { 0x0002071cf6a8f781, 0x0003fa03549dbd42, 0x000fcf51a6b55cde, 0x000f6558542cbdf8, 0x0000c6c6a717b9de }},
        {{ 0x000cdbcd7623a9b8, 0x000ba1fff2811e50, 0x0003f1a3e116af2c, 0x000483fbbdb81b70, 0x0000d353c95f79b0 }, { 0x0004c3074db8bf13, 0x000512a566a808ce, 0x00034f3d5d88967a, 0x00069e76f02b7445, 0x000003ceb7b97c4c }},
        {{ 0x000a03aee51cbd7a, 0x0003eb0b796d2197, 0x000eba706ef4966f, 0x000307cf5c3e95f4, 0x00007329daec404c }, { 0x000aacb62b09eeb6, 0x0000d76f53a89aa8, 0x000109dd72b102e5, 0x0003bfa4f0d8af9b, 0x0000584ec8a3c986 }},
        {{ 0x0002c97a20562f30, 0x00055ebf62ffd521, 0x000e44ce44efba3a, 0x000b3daefe670b54, 0x0000098ea607e96f }, { 0x0001985e228d3952, 0x000f4ce1fe32093e, 0x0005576f4aa6830c, 0x00035206b13cb4ff, 0x0000bb374024af47 }},
        {{ 0x000f278cb2b3179b, 0x00043a910a230cd3, 0x00038e096c95125c, 0x000ca660a24f46a9, 0x00000266524e1cf8 }, { 0x000207cf075b1512, 0x000b47d50b920eb4, 0x000d4c6e366a9240, 0x00080a509f1cb9a2, 0x000070955e1401ad }},
        {{ 0x000747d648fddd2d, 0x0005181c4127fe1a, 0x000c93a1ef1d3d25, 0x000e28ea5a9f8b9a, 0x00003db1465f08ae }, { 0x000a7234813df238, 0x000f35c2ba97bf75, 0x000d6287fea38e24, 0x00093ef9d15e0f77, 0x00005486c5bb1638 }},
        {{ 0x000a4c97a4abc3b1, 0x000629f8f5fcda83, 0x000483a1c45951ca, 0x0007bee77f3e558c, 0x00000e48e3d45037 }, { 0x000f9b7025a36d96, 0x000524bf6dce6a73, 0x0005d44b52ac36ba, 0x000ec10e9ff373d9, 0x00005254e73f1733 }},
        {{ 0x000917ab9ff1c65a, 0x000f031317cd0627, 0x000da087d792e7af, 0x000eac10b30779d8, 0x00007850b5ef6624 }, { 0x0004d90fa2ac1df6, 0x000a6be28fb1d8e4, 0x0002ce95ca910a5c, 0x00053b26c2e31356, 0x000059685056c6f7 }},
        {{ 0x000012062fd2dce6, 0x0002267d6b3e243a, 0x000cb8f28618953c, 0x000962f389fe6cd9, 0x00007eebcb8ff8c2 }, { 0x0002260beaa35c3a, 0x0008ebc9399d0822, 0x000a1fb906f7ddab, 0x00093057347e8c54, 0x0000f595f07f2d13 }},
        {{ 0x0007401e4d1294c0, 0x000661e1a18cc20d, 0x000e4be1b7e97e55, 0x0007cf6a02192999, 0x0000422ab974c7d0 }, { 0x000b5cbc27a1ccf6, 0x000a3836fbca2bae, 0x0002a46340237734, 0x000b9c94046ad329, 0x00002357b30dce95 }},
        {{ 0x000f6e149c30d036, 0x000786c252bd4796, 0x0009a3476ff36be0, 0x000f9406305e0f88, 0x0000f83a58f41674 }, { 0x000e1a0a8596c4ab, 0x000785cde601a897, 0x00087559b39a4f3d, 0x000826d27d17bbe9, 0x00009be39f3ad611 }},
        {{ 0x0007ac8746207af9, 0x0003896270580c34, 0x0009aa1d02ce4c08, 0x000b9e7a51e23383, 0x0000cf4081696e8d }, { 0x000d9da6c4963c2a, 0x0004d9a100dd467c, 0x0006814766449538, 0x0002461c9274a433, 0x00004c28dc524eed }},
        {{ 0x000774457fbdd20a, 0x0000f9f355116ac9, 0x00091f5d496d688e, 0x00079b671cc38bb5, 0x0000fd56d3b79f71 }, { 0x0000f140b199642b, 0x0007ec8c6db88bb2, 0x000f2dd84d526d42, 0x000887f8b6429d07, 0x00002d0578cfc96c }},
        {{ 0x000e6fa37e3db750, 0x000bf09ca4946937, 0x000073e3458ba1d4, 0x000f9fb1248d3a02, 0x00009a0ba3d63119 }, { 0x000d35c7755bbb5b, 0x0005a0649a0c938c, 0x00079eb02caa0b2c, 0x0007a3912b72c0d4, 0x0000f4f8e7d64356 }},
        {{ 0x000e3c3c1d7cda3d, 0x00032ea87baa6279, 0x000158ee6a629579, 0x0004420c958c1fec, 0x0000d64ac7b719e3 }, { 0x000792f805f03b27, 0x000eb7b2bcfe0b0f, 0x0001327ffd07ff8c, 0x0003e8c973510710, 0x00002665759bab58 }},
        {{ 0x0006229c472f32af, 0x0009904a9b46402f, 0x000ccaec6c09216a, 0x0002f59aaedddf85, 0x000022cb9423951c }, { 0x0004f9e47fc8c2c3, 0x000011e7727b9b64, 0x000b654a63c2fce5, 0x000e684ea289e4e4, 0x00003b8864805a20 }},
        {{ 0x000e57c4ea5be12e, 0x00073fe611f8b8ef, 0x0009f6c12f3d9e96, 0x0006d8cc3fd2e416, 0x00006523896385b5 }, { 0x000a38efd1795651, 0x00002acb1aaf6131, 0x000fd01f6a74847a, 0x00041ff3431df210, 0x00000190aa223cab }},
        {{ 0x000936a01ee6f056, 0x0009c370b1e9e212, 0x000cf487e88bd896, 0x000d164f9b258514, 0x0000b0dbf25b69a9 }, { 0x000717e61d824da3, 0x0003b986dd5b20c8, 0x0000e9e1a8b25329, 0x000df8cd7b586bf7, 0x000098c804cbf541 }},
        {{ 0x0000291d43fc236f, 0x000e82235e6fc06c, 0x000e4b14f6918efa, 0x0006704723477728, 0x0000cd51067b09ae }, { 0x0009f8b71be82198, 0x0002467cec5a196b, 0x0003a235d360dea7, 0x0004c8451deeb31e, 0x000071e8bcc32913 }},
        {{ 0x0006bbee27162ea6, 0x000fc87968a2ab91, 0x00092fffdcf8b836, 0x0007f9930e3c5ce6, 0x000052f83aed92d1 }, { 0x000aee94a5c4b253, 0x000cd17fd42cd227, 0x00007dc33dfaa1ab, 0x0006df6321e13997, 0x00008a5c209883b4 }},
        {{ 0x00046af04a0b75ce, 0x000336089821a663, 0x00085b2d261f5d51, 0x000566845f7c3767, 0x000046b01307b23d }, { 0x000e266e7ea58076, 0x0003c32370bf29c0, 0x000e34bfe26f7ad7, 0x000a63590f90c73c, 0x0000fa778d8e6bfb }},
        {{ 0x0008c7ca3988b700, 0x0008eb90fca65d93, 0x0009dde77b7ef7a0, 0x000105a4e7e2989e, 0x0000f1c42756f442 }, { 0x0008f99897246c90, 0x0003494d402676af, 0x0005fe32e3b161e4, 0x000cdb64e26ab5a6, 0x0000d779dfab8fa9 }},
        {{ 0x00005ab8b7207ede, 0x000a6911a335cc88, 0x0009e9fe5b2d6bb2, 0x000344b19563459b, 0x000063bbb632aaa7 }, { 0x000b647545ffb149, 0x000c9da8d006086c, 0x0009f1914bb72b13, 0x0006fefd76ca4846, 0x000088e4372592ac }},
        {{ 0x000ffea6429715c7, 0x000278fbd7d90402, 0x000e519cf6536d14, 0x000921044ee66a2d, 0x0000fdbc110ce347 }, { 0x0005fc22ff710224, 0x000b7d1114344eab, 0x00064711998bde17, 0x000ea5b910430453, 0x00004bbc5151aab8 }},
        {{ 0x000e608d3b84a63c, 0x00027a91ee682e0d, 0x0000202539ebdc33, 0x000ca402ddf48abc, 0x000039b2f368d7a9 }, { 0x000a5e736f80cbe2, 0x000e08bedafd8830, 0x000eb3366aaf8525, 0x0008d5c413c362ed, 0x00001f043dd12d92 }},
        {{ 0x00051c374c6eccaa, 0x0007911f68cf9d16, 0x00075c5d3e0a50dc, 0x000351809a4f9ef8, 0x0000ad1a87ccd2e7 }, { 0x000f8b5ca64e7fbd, 0x0004e5888b5cead1, 0x0008501e7f03fb15, 0x00096363a0ef0005, 0x0000e724123c7fff }},
        {{ 0x00090f925a13bf98, 0x00028a3ee9523f07, 0x00021e4f171e207b, 0x000b3eff4bb75eab, 0x000087b67cf8d7fc }, { 0x000a80ca34fe2f62, 0x00001ad1e4f6a5f6, 0x0003f7d0d01bf374, 0x000368fee06e08b8, 0x000092f43fe95274 }},
        {{ 0x000932ad964afa6f, 0x0002868aaba4864c, 0x0001c41ffa4036b8, 0x000b294f1b13cabf, 0x000056378b24b243 }, { 0x0006cecce0d322ce, 0x000a28ecae1e27be, 0x0008343f2967812f, 0x000aaeabc9dac426, 0x0000540f92402c0c }},
        {{ 0x000d95bc6cfae79d, 0x000e75e95081181f, 0x000f6bfd9968c225, 0x000cbd8a8cc8a791, 0x0000480e6df000c8 }, { 0x000bdbfb94f56234, 0x00036266888523a0, 0x0006bf25a4d854ae, 0x000577f974d142bd, 0x0000a963460af4b8 }},
        {{ 0x00001be6298ca10a, 0x00098291660f6a6a, 0x0005b5001e039d37, 0x00022f5e87f6abcd, 0x000023ff88c1c17e }, { 0x000064091feffe54, 0x000ed030ca1c3287, 0x000d489f896e7ea4, 0x00092f0b0c74114b, 0x0000370a3f4a2eae }},
        {{ 0x0004f21fd5b99340, 0x000b521e447f2c48, 0x0009db50721aea61, 0x000c5e6de81b8da7, 0x0000050b21dc70c3 }, { 0x0009f72f7ea729c2, 0x0006f814e9df3d85, 0x0009dfb45db575f4, 0x000efe84a1ca4861, 0x0000e4cdd0716db3 }},
        {{ 0x00033c65bf36ee47, 0x000f841386d5087c, 0x000c1a2b70906c0a, 0x00081da5840bf739, 0x00003727fb23462f }, { 0x0008de9cfbb71ad6, 0x0008f7c38ca61533, 0x000aa39a7c7b16b2, 0x00053493997f4519, 0x00002e4ab8b82e16 }},
        {{ 0x0009ed75a8075803, 0x000e8a37b793f853, 0x0009b67a736cdcf0, 0x000bcab87e9f8dbf, 0x0000d28da38fbee9 }, { 0x000320fecff5c637, 0x00011487ffba390a, 0x000ac7d8bd39b7eb, 0x000ae3fdb67f8cb8, 0x0000a584d17078a6 }},
        {{ 0x000c04d32f4cd7b7, 0x0007902b2aced2bb, 0x0008a56701019371, 0x0005a0770dfba180, 0x00007ee3ef2a1576 }, { 0x000b1949fd771f7e, 0x00022d10f9f142d6, 0x0004dcd51025a0e6, 0x0006a36e32c44a0c, 0x00007e3d8d8e45c2 }},
        {{ 0x000b87bcd0217256, 0x0004808be0f44920, 0x0003de4e28470c89, 0x0002210a83ff0da0, 0x00000219c682d1d6 }, { 0x00045c0f29f6bf67, 0x0009583ab1f58ab6, 0x000a1d22cd60f4a5, 0x0007c00e56d9598d, 0x00007cfc00573c26 }},
        {{ 0x000e4c720d49953f, 0x0009e82eabaf21cb, 0x000963f4f390dba7, 0x0000a319182b9602, 0x0000a7b7f6d4dee4 }, { 0x000b26d3ec37e7f8, 0x000c4036f08d6c97, 0x000733e9eb5434fd, 0x00029071236e8a99, 0x0000804003513506 }},
        {{ 0x0008486600572aec, 0x000e6d0c9469ad61, 0x000d18673a5b5b15, 0x00040b686c4a11b4, 0x00004c8a42d0ce86 }, { 0x0001f6653acf2e26, 0x000b0ba8d9ce5be6, 0x00059d5595d2893b, 0x000d2a9730d8f68c, 0x00007023d6e1d087 }},
        {{ 0x000ac7e2a0e832ad, 0x000962b5aff5a48b, 0x0008b7f257c9170a, 0x000e5ecef0d81c4b, 0x00009e1bf2654911 }, { 0x000e23feb138dff4, 0x0004b6c1c664399f, 0x00005da714ac9df9, 0x000cb88bf4c288de, 0x0000a0bdbf12fa64 }},
        {{ 0x000d63ff677c67f5, 0x0002f66cdedca850, 0x00000ca5542522b3, 0x000a91c5d140bb71, 0x0000061b8edde9e1 }, { 0x000dd6d4fcf285d8, 0x0007d517d234afbb, 0x000dc049bac482d8, 0x00004735d61c2db4, 0x00000bdf475cd200 }},
        {{ 0x000379bc0760a44f, 0x000536d3be01cb1e, 0x000b40c0e4757120, 0x000abe6029ba14ff, 0x0000b2768766a251 }, { 0x000499f0b7403ff6, 0x00022e9d4ca8d68a, 0x000bdc215072febc, 0x00005b368083558e, 0x00000674a9be8956 }},
        {{ 0x000524194f020327, 0x000303cb8349abe3, 0x0004fb90e2cd1e88, 0x0004f22903baab3e, 0x0000cf54c8866bc9 }, { 0x00078e9bd6163cb1, 0x00066257d4bcc42b, 0x000b01726de44c34, 0x000e730e0e90a3d9, 0x00006e99c9f280fc }},
        {{ 0x0001b393ef075bff, 0x000b8469e03d2781, 0x0005c55acea80038, 0x0006d79558017860, 0x00004b0cbffe0992 }, { 0x000975eb80eb4487, 0x000de82d8884f278, 0x000a35a3cede5daa, 0x000dbdbc6bd44737, 0x0000d4e86779cbb8 }},
        {{ 0x000a12aeaeda2167, 0x000f0c7d7fe71f3b, 0x000002851f89faec, 0x000cf14631534219, 0x00002b71cc5b2f73 }, { 0x000a778c8bea0c7d, 0x000b8d2a9b4f8f67, 0x0005805f4d10c1d4, 0x00088167c04619d6, 0x000025b9fb741b8f }},
        {{ 0x000fbfee7fb522a9, 0x000dd2320a71b899, 0x000c7c62a2e5d72d, 0x000d59199241a2ae, 0x0000474a8ea015f5 }, { 0x00083dd738132807, 0x00051b95c4017e32, 0x000fecd4344ed7ce, 0x000db466f1dc7f9d, 0x000070c33943a538 }},
        {{ 0x000e55d0c2e5f7e9, 0x0003d09cd85d6a2d, 0x00051c9030b43387, 0x0001b15de6ebbc34, 0x00002dbfb11b5868 }, { 0x0000178c71b5b3ad, 0x000367f24dcdd5a7, 0x00050eba05896a00, 0x0004e910d8d602da, 0x000031562bd0e4c0 }},
        {{ 0x000d3b63b60559ae, 0x000f5863fe54cf05, 0x000ea58050bff57b, 0x00043e6d7a4a3ec4, 0x000068c2c4366a8c }, { 0x000e0ffffce55a24, 0x0000fd85ad3d2aff, 0x000cd39c7a3a9a4d, 0x0001d7214fbc586d, 0x0000260f7349e15b }},
        {{ 0x000c326d123c14a3, 0x000437f906151e50, 0x000f49ae99370325, 0x0006854b7cea27f5, 0x0000400c840acf85 }, { 0x000dea4b70fa8920, 0x0008c337dd35df39, 0x000925ac906e0b6e, 0x0002ebb1c0c3b763, 0x00001965b1ca3b09 }},
        {{ 0x000ebf9747546784, 0x000ea53aae3f639a, 0x000c4d67f464f2f6, 0x00028b35ffaba0e7, 0x000050bca1ceda40 }, { 0x0004311819988796, 0x000193ff41ad5d66, 0x000c4eb87a0aad72, 0x00047c0aa03c4623, 0x0000ecb71c39431b }},
        {{ 0x000e526f3fe97340, 0x000b2118be489ac8, 0x000d2b4ea10c02e4, 0x000f1a0072821895, 0x00003dbc2e993576 }, { 0x000c5a3e30980bb1, 0x00019775b645aee2, 0x000f53b3d2ced666, 0x0009145de3d1dcb9, 0x000083a266843180 }},
        {{ 0x000c4e57b437a58e, 0x0003f3e2a6c6fbf0, 0x000a3fab215289e1, 0x000f7db01a74516f, 0x0000f786a2c596ef }, { 0x000965f3c5b529a6, 0x0000c29b6e386b6e, 0x000cb078b861c0d4, 0x000478029b46793f, 0x0000a2baaffa251a }},
        {{ 0x000f1ef3355edcbb, 0x0004083425c65590, 0x0005fbd17ca52ef5, 0x000fc9bdf7fc00ee, 0x0000d8394a59efd4 }, { 0x0004329d7084b6fe, 0x0005345b79b8ea2c, 0x0004f98d5517a604, 0x000615fb740fd294, 0x00000a2b05ab5452 }},
        {{ 0x0006c177171f353e, 0x0003d52ed1be45d6, 0x000d7e0bb55daa85, 0x0000bfbb1891dd20, 0x00002712c1116863 }, { 0x00091b3fa0fa9f46, 0x0007b412843c4e0c, 0x0006f063761e63bc, 0x0006d4fc7d5ac481, 0x000065f27cb5b34a }},
        {{ 0x000c1fcd2acc0b57, 0x000dd0aab9d97f89, 0x0006f48308e66a00, 0x0009ba0e5fb57bd0, 0x00003a73253811da }, { 0x000f78f55b916830, 0x0009ca9c7ea74322, 0x000c0107819b05c3, 0x000f31ca51b0b924, 0x000084e2a1bbee98 }},
        {{ 0x0006f2e3dafbe296, 0x000de4ac18a414fe, 0x000da94a26abfc55, 0x00062a955c611eaa, 0x000053a9ca6e3b2e }, { 0x00087883af2cb656, 0x0000a179320dc3cc, 0x000ec57e31f52ec4, 0x0008aa490692e382, 0x0000c6a39be83100 }},
        {{ 0x000cfd57c06983bb, 0x00098e453544774f, 0x000b3dde4afbe78c, 0x0005fb6f5834d0ec, 0x00007bae9c3b3751 }, { 0x000ee4d65bdcce8d, 0x000ab3dd6901b1d3, 0x00047673a7680203, 0x00093d2623b49fbb, 0x0000be9b2e57312a }},
    },
    {
        /* digit=27 (1,2,..,64)*(2^{189})*G */
        {{ 0x0009ae30f9bf470b, 0x000087d0d63d1fae, 0x000118fba7a5a59e, 0x000fc5809658fc05, 0x00005a1e22227a90 }, { 0x00083acb655ee723, 0x000edd1818baf918, 0x0006032f40bacfef, 0x000334844e27e9e0, 0x000040f4d53495bd }},
        {{ 0x000ee84e265cd5da, 0x000988dc2c6ff707, 0x000f46f3d40f2a5f, 0x0000c9df979d6122, 0x0000ed01e35bdcbd }, { 0x0002d0793015eb1a, 0x00095b063cd880a6, 0x0005d2a436307bbd, 0x000f365423f3ea7c, 0x000081c0177d53ec }},
        {{ 0x00065f36c733ecb7, 0x00097438a5807cef, 0x0002edb163b75a2e, 0x000e8e6daad28389, 0x00005a009a4a047b }, { 0x00026231865878db, 0x0006b9ea60d32b2d, 0x000a6c1f604286b8, 0x000e303b5ee93df4, 0x0000ae2226c13edc }},
        {{ 0x000968734e4dd7f6, 0x0008df9dec31a21e, 0x0002817162256254, 0x000a3ed65988c8b2, 0x00009bb859066057 }, { 0x000f94739f3adb65, 0x000858f5da6309bd, 0x0003e25d5a77565c, 0x00068a42110f3a62, 0x000053d37190038f }},
        {{ 0x0004cf63c0f10c8f, 0x0005106cce85c9b6, 0x00063ff85f12ae8f, 0x000ed8f60360e70d, 0x00002db912ea29bd }, { 0x000f8b36c18dac42, 0x0003da3e1cf6f904, 0x000686550e256d64, 0x00088999fb7e937c, 0x000091b309d5b740 }},
        {{ 0x0002ceef4d55dd08, 0x0005da434a35ea84, 0x00070a7523a8c814, 0x000ab2ec33418e0f, 0x0000b26e001161fb }, { 0x000772f4c008038e, 0x0003c0b543a1957f, 0x000490f9bc634d5d, 0x000955f30c33761e, 0x00003c76f2e45ff6 }},
        {{ 0x0009ff51a206fb0a, 0x00036bb4885d4104, 0x000051a6a735edea, 0x0009dd5b17287f81, 0x000044dfcc90c3d7 }, { 0x0009e739a001c317, 0x000e3536bc3012e7, 0x00019d261a7db4f3, 0x0004c32663d31fd7, 0x0000a707f44faf49 }},
        {{ 0x00003f3e034d81e9, 0x000580ef4f8b16a8, 0x000e443be8670976, 0x0001a9ec197241af, 0x00001dd783168b20 }, { 0x000097440f10fef1, 0x00018a804dbec69c, 0x00067b238c506e04, 0x0008f287fa83b85f, 0x0000d016711f649c }},
        {{ 0x0000c59ad98e2c8c, 0x00023b63524ff164, 0x000e417a2bf4e4e2, 0x00011bec0d7f9b51, 0x0000f07523fe58b6 }, { 0x00038fdb3274f644, 0x000cf6f17c387a48, 0x000567f6a51af73c, 0x000642fc95043b8d, 0x0000e865fd6bdff2 }},
        {{ 0x0007b53ed41360f7, 0x000bfdd418e7ddd1, 0x00032f79db02b1a6, 0x000c6f44d39888d9, 0x0000587a59dee9f8 }, { 0x000b2fea9bbd72cd, 0x000351b2b01ae4fe, 0x0006e4189be73eb4, 0x000eb608788462e8, 0x0000369123795063 }},
        {{ 0x0003fd41b6333ef3, 0x000685fb821fc28e, 0x0006f820e91c2a59, 0x000f1c76289d2e19, 0x00007ba4f718025c }, { 0x000b586236bb3b57, 0x000552eb9a96d612, 0x000a0b3be1bdfbfd, 0x000c84d407f8567c, 0x000013167b56f73c }},
        {{ 0x0003132b63c9bbbe, 0x00042c756b95bce4, 0x00061e5a7b1cf210, 0x0006584525ec0390, 0x000035ffbec182a8 }, { 0x00063e8924f392fe, 0x00041f22f3263bf6, 0x000c3692181d584a, 0x000ae5ecbd5b3733, 0x00000ee6831f5d50 }},
        {{ 0x0000089cda3b5352, 0x000505e5236344e0, 0x000dbd9ed20c4505, 0x00078eb2282e8df9, 0x0000041841a4fdd6 }, { 0x000a6297cdfe9726, 0x0007a71bd38a3db5, 0x000095102157933c, 0x000a7b9754ea97fc, 0x0000a0b0d44f8deb }},
        {{ 0x000a11dae24e01a0, 0x0000b35ce7c42d4d, 0x000ea8ecd02dbf37, 0x0000f7b459fd493e, 0x0000a4b9f5308b34 }, { 0x00014369a85af1f5, 0x0000d785545a7fb2, 0x000c1f919e3e8182, 0x000750558bdb2601, 0x0000d24bcc8a4315 }},
        {{ 0x00019d679200106c, 0x0009dcb06e0956ce, 0x000e1f6f6bb91946, 0x0006139324c9d254, 0x0000df1391186504 }, { 0x000a6bd1e8bf9fff, 0x000a728db5c7be33, 0x000877e20c1d3762, 0x000fb2f2ec714121, 0x0000a3614c505432 }},
        {{ 0x000a1a96613ebed5, 0x000aaa0b311898c7, 0x000452f4160400b5, 0x00055fbbce2a272e, 0x000044105e96f0ba }, { 0x0002280dcf97d62e, 0x00039cea1c8cde9a, 0x0008d0499e144dcd, 0x0003fe7cd9d958ba, 0x00007fe9ad9ee6a5 }},
        {{ 0x000c0f4672046418, 0x0008dc1cf6401471, 0x000d42748aa65277, 0x000c0ad11ea5a2e3, 0x000085ed223cfbd5 }, { 0x000a05f641ba27d2, 0x000e5a9695653812, 0x0000b4c92919d77e, 0x000cdfe09dc96624, 0x000016ab44addb42 }},
        {{ 0x000306333a079f4d, 0x0008866d4312483b, 0x000cb137ee35846f, 0x00064f7935179a95, 0x00005b20b179b20f }, { 0x000dc92787c3fd1b, 0x000d1db0081a726e, 0x000561df74341404, 0x000949bb90149b0c, 0x00002a861cf9d27f }},
        {{ 0x000e08e3479258df, 0x000e34aae6a8e13d, 0x000dc12fb46451af, 0x000c1281585833fd, 0x0000a5f7d00a3745 }, { 0x000c0dcfd1df99fd, 0x0001b4ccdeff2e08, 0x000a8a50deff90fa, 0x000e2498a49f9387, 0x0000a9b6e98e5797 }},
        {{ 0x00094d3f1ba6043c, 0x0009de37a6a308bd, 0x000f72da28b830d5, 0x0008882b66237582, 0x0000b9540188ab4d }, { 0x0007f04703513909, 0x0006a5bba21f248e, 0x000484e1a8ad4170, 0x000772984272ad0e, 0x000027dd807898b4 }},
        {{ 0x00077c871440e1a8, 0x000184acba2ececf, 0x000fec2ab72d4975, 0x000078f250c18126, 0x0000bcb6186f88c3 }, { 0x000109fe96d12373, 0x0001e83fa7f9f909, 0x000bfb8d0916c333, 0x0009e56a3f2e7aa4, 0x00009e0eec3e2355 }},
        {{ 0x0003f901e0ee3da8, 0x000c0a9e9adfe1c3, 0x000cfd5537daacaa, 0x000c08f8f3984403, 0x0000096fcbfd4004 }, { 0x000dbd3ddcb09925, 0x00000fe3652438c6, 0x00063014b42f3978, 0x000265abdae9ec4e, 0x000078614ff4c20e }},
        {{ 0x000bd4d8f33bce8b, 0x0009fecedd36776e, 0x0007fdb0b7462639, 0x00060e565f612c47, 0x000003f48d640dc7 }, { 0x0006125c1b173232, 0x0006eaaaac4cb325, 0x0006bfdae3ea27f1, 0x000704c63ea3aadb, 0x0000876e11548998 }},
        {{ 0x000320c5cb34fa83, 0x000c1a43794f8dcb, 0x0009951b966af168, 0x000a28da0dcca923, 0x0000bdaf4528f40a }, { 0x00083be9a0dbe4b2, 0x00076ff47b50a951, 0x0001b6b449d08629, 0x0002b4f53933e3b0, 0x000077b42a87e4bd }},
        {{ 0x0008df26cd861bb0, 0x0002c5ad6bbe24db, 0x000f9aa510c7c822, 0x000d07d4cc6c7462, 0x00007f4555c709df }, { 0x000cd8924978fe2f, 0x000b4995a483f8df, 0x000680fa9ef11fdd, 0x0000da47572d8a95, 0x00005ee12e0f4b32 }},
        {{ 0x00058280d4ddee6f, 0x000d4d77613aa81f, 0x000f907be4afaca5, 0x000c29388801007f, 0x0000bc59683be7a1 }, { 0x00091554cb1a8b9b, 0x0003d2f7c978c385, 0x0000b4cef8affa72, 0x000d48a2fdf1204c, 0x000070620062b926 }},
        {{ 0x0000f9e4bb715381, 0x000b6d6a05bd5254, 0x000ac69b9680aff6, 0x000bd1ac948d5f09, 0x0000a232a16a1065 }, { 0x0008d30a61a7b345, 0x000b503f46ad2ec6, 0x000f5b10ef8d1b72, 0x00087007700f766c, 0x0000f4ea35a1072c }},
        {{ 0x000b867e6badbcba, 0x000c7ea77b46a081, 0x0007abc1882dd01f, 0x000773c83cf5a647, 0x0000cbd5df06c8b8 }, { 0x000a7c66d6817ff2, 0x0001f141f60558ee, 0x000244c4de37c212, 0x000e31c8bcdbcdd6, 0x00001916c3d12b4a }},
        {{ 0x000dfcf4d119ad85, 0x00033a504e99fd8c, 0x000179885fc958ac, 0x00057cde198a8dd1, 0x00006e59f62aab6a }, { 0x000f97af8853b43a, 0x00033b253758da79, 0x00034b31e858e0f4, 0x00058217ee643bb8, 0x0000da4a5c964eff }},
        {{ 0x0001f37864c9d751, 0x0003720fbeee3f97, 0x000fd715a2986b25, 0x0005aa25c93e5348, 0x000088ce2f7480d0 }, { 0x000c9682db64cb9b, 0x00099e0943aa75e5, 0x00084ef974089e2f, 0x000037219a18c9c5, 0x000082104449d5cf }},
        {{ 0x0007678cab68e47e, 0x000cea6b709c3de9, 0x00019e0e2c451e58, 0x0001cd6bb64a8426, 0x00005633f2cb7399 }, { 0x000c5ab10a85e889, 0x000f05f951a95c33, 0x000314a21afe0190, 0x000f06731dd35b16, 0x0000667bfe6e1cad }},
        {{ 0x0000a02b0eaceafd, 0x00088c341dca5663, 0x000aa7c1ff9509d6, 0x0003070fb7de7fed, 0x00009716331f83e3 }, { 0x0005f449ec69c78b, 0x00092aa2fd3cf36d, 0x000709959bb35e00, 0x00081dc33d131954, 0x0000174691c29863 }},
        {{ 0x00091481c1059a6a, 0x000b11b25fe531df, 0x00017108d4857909, 0x00043e61af979743, 0x00005c928c67b126 }, { 0x000f9f87695e4b85, 0x0005f65f8b05bc60, 0x000ef348b5716c43, 0x000905e3b1844e1c, 0x00004c192e0ee156 }},
        {{ 0x000c992a8a0219ee, 0x000c3c038b27fe29, 0x000b0f0ad855471c, 0x00051edab63b6359, 0x000025a641ada563 }, { 0x00048d3f95b0f902, 0x0008d152a60aed16, 0x00047c320a49af15, 0x00049d79aeb089ca, 0x00008fe88effe77c }},
        {{ 0x000de501a3cb5123, 0x00060a8e1add3f34, 0x000a017fa1a51172, 0x000dfb8dbf42c0c6, 0x000001a82938cb34 }, { 0x000025a2cad7b920, 0x0005c1f4b9a64bc7, 0x0002b4712a8e7257, 0x000cfbffba77f0ad, 0x000059630c29979e }},
        {{ 0x000c477cc78dbf35, 0x0007dd1d1a8f4e7a, 0x0006c70d15b57029, 0x000612235f5b3da4, 0x0000dec5fdbedc4b }, { 0x0009b795f963ab82, 0x000f1d2a883239a8, 0x000642d3ed5e9773, 0x00073cd86e12572f, 0x00006777294aa397 }},
        {{ 0x000ef0e314d85ff1, 0x00065f1b54398dc7, 0x0006c0fd0bc04f5e, 0x0000d10401890efd, 0x0000f8bc668d0ac4 }, { 0x0002c591e55172f8, 0x0006e31cc61dab32, 0x000dd1c0dc69ff57, 0x000c343394a39801, 0x00008f9576364d5c }},
        {{ 0x000e02fc57543709, 0x0009a08de02a53e8, 0x000f4586f7004114, 0x000bd02662a6c060, 0x000038ad03da91a7 }, { 0x0003660ea088e386, 0x000f19573adb4ba8, 0x000c244a29655bb1, 0x000aae9a67b86122, 0x0000a46975ac7183 }},
        {{ 0x00097b8de565ca26, 0x0006b0521ba473cd, 0x000e0559aab8550d, 0x000b275c0b6c50be, 0x00008c7a7b476844 }, { 0x000df7725c5e3459, 0x000e570ab3bbddbe, 0x000e37a00efe4bc1, 0x000e15ad93898aab, 0x0000dfe696962d4d }},
        {{ 0x0006166b44b3b527, 0x000e8abe085116b3, 0x00055111e7cb7191, 0x0000aae2f04a4337, 0x0000b8cbd7d43a2f }, { 0x000af71e91ff6ddc, 0x000673cfd4b322cf, 0x0006bf828cbf933f, 0x0006d39eb726aa81, 0x0000f4fde56a46c2 }},
        {{ 0x0002cadc98401f3b, 0x0006b40641fad989, 0x0007969748c19d4a, 0x000225edcb799591, 0x00004bb11b390d9e }, { 0x00095f7b90617d6a, 0x000b98b256534e4c, 0x0005bc35c509aeaa, 0x000e05b9b73e80ce, 0x000010012b3ad66e }},
        {{ 0x0009cf84aa0fcd26, 0x0004a14d856ac253, 0x000d7282c920e466, 0x0003773f5441bd9d, 0x00002c3a08dedb4f }, { 0x00017edeb92c2909, 0x000a755b2aa95a02, 0x0009ec6039a6d8be, 0x00016624329e33f0, 0x0000c812d89efc16 }},
        {{ 0x0005b9b9cd84c5c7, 0x000783583cbea55b, 0x000d065ea3a2d3f2, 0x00099a292c485ee4, 0x0000a2524bfd6d99 }, { 0x0008ca325a560d39, 0x000b58e4e9e8a506, 0x000af99fa8a3269d, 0x000da58c6462e907, 0x0000ea5c18d58fa5 }},
        {{ 0x00060fff17c23588, 0x00035f213b5ba885, 0x0009dfdf0bac693d, 0x00084130860294c8, 0x0000c20857b9d557 }, { 0x000c2f31600cb009, 0x0007b87d67703743, 0x000d5e27c525487b, 0x000416cd9a373202, 0x000070a468d0a739 }},
        {{ 0x000f684368d3db41, 0x000a8dd7b9520e80, 0x0001fb87f44c3d55, 0x000c0cf4cc078f9e, 0x0000fafd402c5e4a }, { 0x0009cd516968dccf, 0x000e6ed6247e7ef6, 0x0008cfa0acb86b14, 0x000bdd3c2b32d33a, 0x0000aa46051a8ae1 }},
        {{ 0x000f0837e6eccb46, 0x000940f4b3234da4, 0x0005654f66d7528e, 0x0005930c7cf0b023, 0x0000ef0adfc93e80 }, { 0x00089a4060c4f953, 0x00022bb906f6e98b, 0x00040ec7a770d547, 0x0007a5c10253bb43, 0x00007cea8d76fea9 }},
        {{ 0x000f5d57fd84b794, 0x000acc5b8cc037fb, 0x000ab4a8634a4c70, 0x000abb4b83d093a5, 0x0000a4275cfe559b }, { 0x000ef22458521286, 0x000430c8c3bef483, 0x00031b1871b237e4, 0x00018bcab23ac762, 0x00004ef2dd9f1303 }},
        {{ 0x00081dcee597d414, 0x00087dac97e65165, 0x00006a1325b93f7a, 0x000a0d6498877936, 0x000087e6a262fd86 }, { 0x0009a34fc66e7cce, 0x00055918f483f148, 0x000c479481754c72, 0x000f76c426d2dddf, 0x00002d39a7634863 }},
        {{ 0x0008653ffddba12a, 0x000cbb312e08ce5b, 0x0007ef8b72cc3850, 0x000f6fe05a80540b, 0x000001a887b97925 }, { 0x00059b30e1948bdc, 0x000ab8702b0b4c9b, 0x000ee52fe628f9ef, 0x0008038e52675261, 0x0000cae50b48aefe }},
        {{ 0x000aee5ef0da9c40, 0x0005ea60200ec7db, 0x000850dfb43a4325, 0x000a1e7e0b6f4a3f, 0x00005fd883bfa016 }, { 0x000d4f59c8134ed7, 0x00093aae4c6cf389, 0x000633e73e4d5bf1, 0x0008679c6ab4cfe0, 0x00000083cf7bbfdc }},
        {{ 0x00032363a7734ed2, 0x00081b151deb6adc, 0x0003ee6413a23cd6, 0x0001134e1812399a, 0x00001e7ebc657857 }, { 0x000b6f7d030cfaa8, 0x000a7bedc36c64d8, 0x00011c72315df52d, 0x0000f314a86c81e3, 0x0000b2a701f2e2ac }},
        {{ 0x000d56d36a2b2bd9, 0x000140756faa38ab, 0x000bd5aee7eac178, 0x0009b5761a3790e6, 0x00007e4559f47050 }, { 0x0008c39768edb7d6, 0x0008349229a8aa9c, 0x00067cd254bfe89f, 0x000f379e6dcca8f4, 0x00007ea637445818 }},
        {{ 0x0009ebbecd961518, 0x0004736523f2b367, 0x00031821d4ee4c8f, 0x000dc4b2b995c649, 0x00002214d933e9fa }, { 0x0007d928a6a9b4ac, 0x00091db79592f48f, 0x000a67afc0f21e06, 0x0004c6d3e67216a7, 0x0000a28afa8ce512 }},
        {{ 0x000011b6acb767b5, 0x000cf17e991f600b, 0x000d796ff2879e7b, 0x0004986de2a91113, 0x0000f0d31fd49b4f }, { 0x00076430677f20c6, 0x000ef18045ec6135, 0x00016b9162a2f02b, 0x0000d9fde522d2d3, 0x0000d682877747b9 }},
        {{ 0x00075b4a21968cc3, 0x000ad5d1c7193f0d, 0x0006648d3b6cc785, 0x0003a1cc7f0f32f4, 0x00007e378db65c00 }, { 0x0007387e1e912b73, 0x0007e7b342659d41, 0x00042db7b9710081, 0x000269f9de596ea3, 0x0000a3bf0865713d }},
        {{ 0x000cca31c7a57625, 0x000ade949dee1686, 0x0002d160c071cd59, 0x0001de16a1ae0522, 0x0000da394a68f678 }, { 0x0007da93fd13da3f, 0x000971d9bc499082, 0x000cce082821a131, 0x0009544ac4a9b4a8, 0x000025d76c70fa80 }},
        {{ 0x0005f1bfec0f543d, 0x0000f964821c8c54, 0x000d624a5608dadd, 0x0009ef3048de49de, 0x0000fc43c74c346b }, { 0x000087cdaa1179d9, 0x0002d2654b9df609, 0x000c18d43d897611, 0x00089848e5e4fdc3, 0x000036663c4320b0 }},
        {{ 0x0009a7b853608cac, 0x00070238319ade44, 0x000a94e545a4d669, 0x000e28b05766f287, 0x0000f55f670cf4d3 }, { 0x0002971e277b213e, 0x000baa75cedadfd1, 0x00057b1b89ee3e92, 0x000d551958f56281, 0x0000b62941f52c86 }},
        {{ 0x0002ad741cbec2f6, 0x000c096d0ff34fce, 0x00012f74fd700b35, 0x000d2324d20824de, 0x00005c4e2951824d }, { 0x00078a169ffa86e9, 0x000bddf54043ccfa, 0x000b30b9fc13145c, 0x00042a1f016ff479, 0x00009b49c42140e5 }},
        {{ 0x000e5064503a3e24, 0x000634699c930e9d, 0x00076b90b3559e70, 0x000e6519f1d33e85, 0x00006cd16fbc642b }, { 0x000249ef84233f4a, 0x0001300caf3155ed, 0x000bea29f7db9bad, 0x000cbb7db672d66a, 0x0000fcd7af334203 }},
        {{ 0x0004b29cdde6a208, 0x000ced473978fe39, 0x0001e09863c37a8f, 0x0003dc7e9cc4e454, 0x0000cd9ff54a2771 }, { 0x000d2de14cb33c91, 0x0007bff9bc7edf24, 0x000d86ac84a73116, 0x000ae7a488ac236f, 0x00008575be2d1315 }},
        {{ 0x0004d1f3d3a5e8cd, 0x000eea760430e540, 0x0006d56095d81a1d, 0x000aecf5910a2d0d, 0x0000183d1a538a31 }, { 0x000c80b505a3cfa0, 0x00084738a0983cd2, 0x00089c0315b5fc22, 0x000d5e99528e25b3, 0x0000e560b7c69fa3 }},
        {{ 0x000d37c5b1792718, 0x00023714a953beb9, 0x000e790b2245754f, 0x000aecf22502e223, 0x0000c74bd4088fe8 }, { 0x000016a8607ff0a0, 0x0004d70f8330bc73, 0x000645907f288125, 0x000de2cc2139850e, 0x00005677b1f28c7f }},
        {{ 0x000e658aedcda8bb, 0x00031644cd8f64c3, 0x000fe937f063f6ea, 0x0001e975658a2d78, 0x0000049c304fd752 }, { 0x000473ffef5b241b, 0x000792e05da59eb6, 0x0002a9683732b03d, 0x0000a6487ba3d60a, 0x00005d72e137e219 }},
    },
    {
        /* digit=28 (1,2,..,64)*(2^{196})*G */
        {{ 0x00099ee566eb922a, 0x000364f692ac542f, 0x000028dcac6c9771, 0x00018cfbdf079ffe, 0x0000610be982fc2a }, { 0x00013889ee80e815, 0x000511fd62191997, 0x000663545a2e0935, 0x0008cc4048c883bc, 0x00000920149ca0bb }},
        {{ 0x000eee0c0ea6e1a4, 0x0009b8f81a1b3bed, 0x0004ef395891f049, 0x000e351f74f3ce14, 0x000069aac489bd81 }, { 0x00027729a694a1ff, 0x000aec0d39e25c35, 0x000071f076d9e952, 0x0002d5b584061982, 0x0000a6863991eccd }},
        {{ 0x0001ad919aac7231, 0x000d837b1b76ba6f, 0x00045b7295421df9, 0x000677d0a1a2eab8, 0x0000c06c35fd7a6d }, { 0x000314874f18d1ae, 0x00031b807ea7b0ef, 0x000e4bc5643a1ce2, 0x0009b5aabedf5496, 0x00007faef00cb014 }},
        {{ 0x000747ea0bede08a, 0x000e0753f73f449b, 0x0007d5fd126e01d1, 0x000e0524d1d1c94f, 0x0000db80899bb94d }, { 0x000c10697e01e74f, 0x00074fe228d291e6, 0x0004d7b0c0585031, 0x000c5761635b804b, 0x0000e6909f3f3acf }},
        {{ 0x000a8f2439972c6b, 0x000d88f0876fd4e6, 0x00071943cf4a2c61, 0x000dcc9cb45f0c30, 0x0000274cda09b319 }, { 0x0004426efde75793, 0x000b01b65ccde7b6, 0x0007c3a4fb720a13, 0x000dcf1d741e2cd1, 0x0000483cddf39166 }},
        {{ 0x000d32fba5127c0b, 0x0008344fc94dea3f, 0x000edda3a50b098e, 0x0009531ef34c8cdb, 0x00009ea6e5479af1 }, { 0x0003173f9743bb5a, 0x00036f6795dcfb75, 0x000330414c181516, 0x0006467c0fef01b0, 0x0000608f1137cc86 }},
        {{ 0x000731021c804f19, 0x000272aba16f73bc, 0x0003c5c8b7dfaace, 0x0001fdebf2fb3101, 0x000060e66ffcf1c4 }, { 0x00051f55a60c6b7d, 0x000ca244fee99d47, 0x000685a66c4490cf, 0x0007f1df74bca48b, 0x0000e008421c6eaf }},
        {{ 0x0008949a50ad4589, 0x0009c1acb452fdbb, 0x0004985e6bffc0ea, 0x000efbfd931ee696, 0x0000aba564d3b7e5 }, { 0x0007a12f75fecfe0, 0x000a3263c88f3bd1, 0x0000c37c6a1321bb, 0x00004ff03ab147c3, 0x00003bd493c68746 }},
        {{ 0x000b70130c3e980c, 0x0006d6d36cf69db9, 0x0008baf6f90bfcc1, 0x0006f018e1219eee, 0x000018a15b3aeb3b }, { 0x0007bfbcdfd435df, 0x00045620ae18e0e8, 0x0002bf00e958626d, 0x000f8d1d42021a62, 0x00007271917a37f1 }},
        {{ 0x000c923aa6870f8d, 0x0002805cdd61d649, 0x000c2c2f4db1707b, 0x00004e8158c7a53f, 0x00007b90d120c88f }, { 0x000555917e796337, 0x000d2cb5c18871f5, 0x00081e5e3af648e3, 0x0006e9dc47d53bec, 0x0000007d33f08184 }},
        {{ 0x0009ea61f1bde96d, 0x0004eda548f5a0e4, 0x0009f1d559f00cff, 0x0008e42c31910eab, 0x00004e2256a193b8 }, { 0x0004b491b0ca83fb, 0x00059784f7c39221, 0x000d0c11fd3eade1, 0x000f39fb995d337c, 0x0000591ec14b3514 }},
        {{ 0x000f450138039993, 0x00028e028e3fe894, 0x00047665bfb376d3, 0x000e4be40e7e1fee, 0x000057339c7f6d65 }, { 0x000251437e2a615a, 0x000058495cd81854, 0x00018b61a205b101, 0x000fd13e474be0ba, 0x0000b796f513feb2 }},
        {{ 0x0004e39da416895e, 0x00014a0ac6203668, 0x000c31f1f5a051e4, 0x0006217b94be3fb9, 0x0000b0003105438b }, { 0x0005627d144873ff, 0x0001590b38f04092, 0x000306c8616def58, 0x0001ca3d80295194, 0x0000cebe7a096ec9 }},
        {{ 0x000f637e4fd3515f, 0x000f1aa658a6d877, 0x00075038f4d59b63, 0x00035e9c91405aaa, 0x0000302534f4de89 }, { 0x000f15b373895ff9, 0x0002e7614ffad2c7, 0x000d3be5b41e6f6a, 0x0002f9afce58fb1b, 0x0000c8215fd1b036 }},
        {{ 0x0008d69d5541959a, 0x00013d2d842ca726, 0x0000d602c5aff83f, 0x0006b2414a71e439, 0x00004b346f996925 }, { 0x0007ac7bdebe7e5f, 0x000f46737183bcfe, 0x000a4bc46418e5e0, 0x00093f35b7152b7b, 0x0000aba65a5ecc00 }},
        {{ 0x00032811a7abd874, 0x000db7881fdad909, 0x00057a04e50379b8, 0x0002cd9e5cf638f5, 0x0000f7d1bc229820 }, { 0x000dd099becd5a2a, 0x00089309bbd11ff0, 0x000fa633afb561a6, 0x000f7a86676108a6, 0x0000803fa54a21da }},
        {{ 0x000c23f829f27693, 0x00017aaabe329866, 0x0009e4b1a3fd664f, 0x000b9e0050fe9dd2, 0x0000a20ae7baca9a }, { 0x0009a25a18798042, 0x000e0180b136a798, 0x000dedffaf7dd73a, 0x0003b0ba67e2174d, 0x0000f25ede8ca147 }},
        {{ 0x000ca5bdc1609c2b, 0x000777ec11ee01d3, 0x000aa94b503a968c, 0x000505e3cf900553, 0x0000383c90f106a2 }, { 0x000b715a6d47b75e, 0x000b73797eada9f1, 0x0002f4b407c350c0, 0x000850bb359c50ab, 0x0000b660a6466bea }},
        {{ 0x000adfe0af71e0c3, 0x00011766ae06e0be, 0x000db26301b0323d, 0x000d19720cdd7888, 0x00002c963a41c4e4 }, { 0x0007e4b0dd7b445f, 0x000f3a9699500937, 0x000aba9adc633558, 0x0009b296a3525d9c, 0x00009fbb8308b6ec }},
        {{ 0x0001610588bce8e4, 0x000ce0a517d7061b, 0x0002baccf0fa5475, 0x000de186777fe343, 0x00000b73f8bc3f10 }, { 0x0006327dec759e0e, 0x00076cad9c4ddbe2, 0x000c1742cdf5be97, 0x00048ba94b3164f1, 0x000038569d7d4e4c }},
        {{ 0x000aeebb5d055643, 0x00083de97b5cd328, 0x000e2afe337097f3, 0x0008894fe40835da, 0x00006140cad890d4 }, { 0x00048990ad4be3f7, 0x00038b75b8ee9ecc, 0x000cc9b3106b05f0, 0x00010490ed7a17bd, 0x0000a0c831b76ce3 }},
        {{ 0x000b19131435469c, 0x000e12a2199e347e, 0x00053f20cd3d5899, 0x00027b5dcbee7555, 0x0000ffa0bf10725c }, { 0x000a33a79b0335bc, 0x000a50d37d341eda, 0x0000aca174906343, 0x0001371ba434fbb4, 0x00002e3e4d934cd4 }},
        {{ 0x00031af02f504322, 0x0002e54e4bd6d3d2, 0x0004379a61a0f9b9, 0x000ffb1a609540f0, 0x00006ab7f4c255f8 }, { 0x000c8be793b491d3, 0x000b9ac0992c163c, 0x0000bc025420deb3, 0x000fa538c1fef8e7, 0x0000930f5d791209 }},
        {{ 0x000a5e00b2f8398a, 0x000f1aec61c3855c, 0x000c8e63fc8830ec, 0x000d209b8ebff429, 0x00000550b4100d93 }, { 0x000174306911bdd3, 0x0002e17c3fed8b79, 0x0009f027d18dbd75, 0x00051ea54d49ac6f, 0x000037ee2449542a }},
        {{ 0x000db067bf6ab9fa, 0x0009e1eaff2ef812, 0x0004fa6846c16bc5, 0x00070ae92c3654d3, 0x0000e7bed9e284b0 }, { 0x0007c60de3cf3529, 0x0007586703dcc974, 0x0007996dd4321892, 0x00065515ce66f9b3, 0x000045556034c314 }},
        {{ 0x000ee790cfd235d5, 0x0008439d860f1b25, 0x000c9705bc7a9c0a, 0x00080e7bd3e9a1db, 0x000066b271a5cef7 }, { 0x0005613245b1f9b0, 0x00084b15bdaf9bbe, 0x0006674163ab3eae, 0x00094878c02cedda, 0x0000e912b227e619 }},
        {{ 0x0009176f1bf4b9dc, 0x00046cc76b23a502, 0x000ce154b2b21706, 0x00010e7f54a2793a, 0x00007203d4f46495 }, { 0x000438b216a5a2bb, 0x000a6874f6f59eb5, 0x000937a3b3776086, 0x00040490286e9bca, 0x0000d17e13cdbcb2 }},
        {{ 0x0005ab800dbb99e9, 0x0001d3c7672765ab, 0x00034f9c39b14ed0, 0x00017bd3337a3ce6, 0x00006e48af02badd }, { 0x000ed04363ca7756, 0x000e95911a5acd6a, 0x0006974fcd00271e, 0x0001ecb0abaee615, 0x0000ef5280b8e5db }},
        {{ 0x0002ace7a063fccf, 0x000ae1ca4a82b765, 0x000ea0d384496b84, 0x000df9cb55586960, 0x00008b8c0663e8a5 }, { 0x000afe4322af8875, 0x0000118cbbc9b701, 0x000a454e62bf1e10, 0x000356c334f06fd5, 0x000087253946a7be }},
        {{ 0x0009b42e68963555, 0x000d7a6c44b2c6c4, 0x0008c5444f9610d7, 0x000340e77ab72679, 0x0000087d908b7398 }, { 0x000d2878a8e58f15, 0x0002fc3cea0b1760, 0x0007bdfe18d92915, 0x0002b88cb53a8ddf, 0x0000dccc4d1142d6 }},
        {{ 0x000803fc21b1eff9, 0x0003cd5d019b6a39, 0x000da86b222452c9, 0x00067c811bb5cf88, 0x000099bb7d29f4a9 }, { 0x000469591564f145, 0x000d79ec07764a90, 0x000eb2f2773fea31, 0x000f584ded93286e, 0x0000a5e431316829 }},
        {{ 0x0002ede0e2398f58, 0x000858a58d6cd461, 0x00057b7853efdb37, 0x0008c6b289cb633b, 0x0000c7e7d1c765be }, { 0x0009e66e59813ff9, 0x000d162065a6f4f9, 0x000ccdd7da21b1ec, 0x00007f41cc3a47ec, 0x00006d5b783c2d8e }},
        {{ 0x000ff6ea2132c1aa, 0x0009aca63d0ff123, 0x00047619b63e3c43, 0x000ebd147e5efc78, 0x0000e1e9f3d79049 }, { 0x000b88bb8cfbac0c, 0x0006d1bf21db8b7e, 0x0000ec5079d8d077, 0x0009fcead7dac70b, 0x0000f978502a906d }},
        {{ 0x000d9e8a3ff2c9a2, 0x00092b4dfdbeb1be, 0x0001c74fc3aad1fc, 0x00088a62122f5ca7, 0x0000f11570bb28f3 }, { 0x00061bd711a3d136, 0x000f6b1b961f49a5, 0x000124e4bf5ef7ad, 0x0007f446d7fee2ec, 0x00002e82c2e6427e }},
        {{ 0x0000ba9d0379c1f5, 0x000a7cc320bbcaff, 0x000eee634788d256, 0x000e6fc28568434c, 0x00003ef3f9080a95 }, { 0x0008117623d9511e, 0x000220432ff9f60d, 0x0005ea4c590f9a09, 0x000762eef3dd8e4b, 0x0000d0f362b6f269 }},
        {{ 0x0007df2ca0800f08, 0x0002076f51fe76eb, 0x000b59f6f8e8b673, 0x000134f188a6811c, 0x00006d8c2c59c4c6 }, { 0x00047ed447b1ad6a, 0x000e452dd7b205bb, 0x000793240fcc896f, 0x000fba748b695947, 0x0000bc03136d4c3d }},
        {{ 0x000c521308e5a687, 0x00011b904c9d9bfb, 0x00005f271d3f34ca, 0x000785d61661e288, 0x000077daf5dd5dca }, { 0x000d465b18f0dbea, 0x000b6fc2477de506, 0x000c1076f6d6d82d, 0x0004d5f05d80bb41, 0x0000b02bf176d362 }},
        {{ 0x000288573948f643, 0x000b030fb6840fd2, 0x000411798f8ece00, 0x000bf71f8faaeb21, 0x000084d5f1bed683 }, { 0x000b394dafddac9a, 0x0009c7c7bc467dfe, 0x0002ed0984ca7407, 0x0003d7aa6926562e, 0x000088d23eb26432 }},
        {{ 0x0004b74cc6bc1151, 0x00094d211c0be981, 0x00016c97349e82d9, 0x0001a7d1cb1e6ed1, 0x0000a0c06a8af629 }, { 0x000cbac448c5b98a, 0x00087868c060985e, 0x000d4102bb339e15, 0x0004fb8ca071047d, 0x0000652742014308 }},
        {{ 0x0003ca782157f884, 0x0004ce2ebf3232f7, 0x000e515fb6bc3716, 0x000bcdbdbfff80f9, 0x00003291c8c0121d }, { 0x000f24a939bc6d67, 0x00079a0e6251b462, 0x000c1ce8e9f3b13f, 0x000f03a9a89bc6b5, 0x000034c1818b3d4c }},
        {{ 0x00027c1c2e5d8f7c, 0x0000246766561b79, 0x00057c5415fc3c25, 0x000fe9c65736600e, 0x00002dd1b7dc8305 }, { 0x000cdd998788c6f5, 0x0002e370cc5df696, 0x00019cf8b021a08d, 0x00028e62799b37c0, 0x0000aee1d95fb4e2 }},
        {{ 0x0004274408c7cb95, 0x0007431390420d28, 0x000b855cab9ba1b9, 0x0006e19ad299c40a, 0x00009b7daa386fe7 }, { 0x00051fddc7f82d4e, 0x000c1b991a711a0c, 0x0007da32cd4f739f, 0x0000f20d6461592c, 0x00002ee7a072ece2 }},
        {{ 0x000bcde201ec9e56, 0x00068b6da50d5311, 0x000d8ad9d7d8a81c, 0x000ad8689521b840, 0x00004b23d5891c53 }, { 0x000e61d34d4d78fc, 0x000ab8b3b76a59a8, 0x000d791746cea6ae, 0x00018f11cf84841c, 0x000090aaea3d7172 }},
        {{ 0x000e7ba65e352870, 0x00054515541b8923, 0x0001be21f7f5e835, 0x00096f10db186ee4, 0x0000fd944bfe00a0 }, { 0x0006f4ce31269aef, 0x000897aa83694867, 0x000717c0ec22773f, 0x000ad2cc309f9dab, 0x0000350aa14102af }},
        {{ 0x000ff862f1ec19ab, 0x00055cff4f066b52, 0x000316254cd0bce0, 0x0001eee9889ab514, 0x0000f50054e6e5b7 }, { 0x0001743d0a1819e9, 0x0001085b05556e3e, 0x000d52f2cb14146f, 0x000ac4eeacd96058, 0x0000e5b40fed6cd5 }},
        {{ 0x000d1e0148639a3f, 0x00033e27eb37726e, 0x000d27639eef0363, 0x0005a6b79f7fa264, 0x00006a16397995e2 }, { 0x0000ad13cee1fbfd, 0x000fc7982e7abe2a, 0x0001a54920299425, 0x0003bc23a19eedc7, 0x00001d27e9fe8f7f }},
        {{ 0x00045446dee19896, 0x0003f241c0e651a2, 0x0009de79db9fb24b, 0x00019e8caab1a6e2, 0x000018a4f366b869 }, { 0x00075841e5ee98b0, 0x0007a2156c07613a, 0x000e4ccf12b26b49, 0x00064594643deafc, 0x0000c457c3bae579 }},
        {{ 0x000e4bf888b90c57, 0x0004f777b0ac93d1, 0x000d5aa9e84f32ca, 0x000e8f22ba6e37b0, 0x0000c2bfd71f0dbd }, { 0x00029438c4942ce8, 0x00044e04df0ba2b4, 0x000d0dcdfb573987, 0x0007fc420935d5cf, 0x0000d2dbc8e26eee }},
        {{ 0x0002e6bead7a7f69, 0x000d2c9049d33fa6, 0x000a3188a951a288, 0x000226da358b82dd, 0x000075c3d6d80aed }, { 0x0006a933d8e7223f, 0x000590fde46c58c0, 0x0009631fa3e61563, 0x00058f6eca7f6937, 0x0000e503ce844e9c }},
        {{ 0x000ca3f44a175048, 0x000c69b077ff55dc, 0x0007ee82f2366b2b, 0x0004c6d4f5397779, 0x000038a3ff5c2a06 }, { 0x00093eed830e7f4f, 0x0000d87a07eda0fc, 0x000f34251345035d, 0x000668b72c138bcc, 0x00000099d6e25005 }},
        {{ 0x0009915109fb7643, 0x0006e574d0b75c00, 0x000259f4b65201c7, 0x00004e9169716eb4, 0x00000af81e525286 }, { 0x0003c7b8c6ede208, 0x000a1285cca6c3bc, 0x0009713e103947e7, 0x0002278c4b25f7d1, 0x0000e315b9cc5975 }},
        {{ 0x0008f732f82265f4, 0x000243c5a1f8e24e, 0x0009b4db22f81d33, 0x000196ea6fdb118f, 0x000088f3204c98a0 }, { 0x00032a76c6083b31, 0x000fc910be786ab0, 0x000f079b2a2f4a0e, 0x000eac5945d32ade, 0x00004fa0648f108e }},
        {{ 0x000ceb151b607fe0, 0x000b1a3ebeb15447, 0x000bc1ec25a82383, 0x000ecd8b344b7b88, 0x00002db5e0795ec4 }, { 0x000e9e1a3e4dbbe7, 0x0007fd05bc1e6a8e, 0x000a1a70ed816faf, 0x000d78e89ec70cec, 0x00003b862411af2f }},
        {{ 0x000a22d40d38f45d, 0x000ad84745b98d25, 0x0000cb34193377df, 0x000202fb9da429a2, 0x000043d54875e04c }, { 0x0009c0bb328b4510, 0x000a9e570e8a4cea, 0x000af4ed4846b07f, 0x000c6c47ebfd1445, 0x000063d3c380d98e }},
        {{ 0x0008653ca70dc05b, 0x000f2600a4c3f6b0, 0x000923b0b29d901b, 0x00087efccbdfaad7, 0x0000bcd3b0c08f71 }, { 0x000be1fdf72ebf6f, 0x0001824538873a39, 0x000d10ffc9f0003b, 0x00023c3ba09299e5, 0x0000ad9a8d1c6f7f }},
        {{ 0x0004957d29e4f3e6, 0x00077595758b11cd, 0x0002f3cfd216bfa7, 0x00043ae470b85289, 0x000035c69d5fca6d }, { 0x000e1c2610168c00, 0x0004363646559b01, 0x000c2b5c647e66bc, 0x0004254a9f4a8273, 0x0000abe6b046a81e }},
        {{ 0x0006133b9250e9e5, 0x000fd28cf6a4a81c, 0x000255354be09066, 0x000bfdc5b1fa3b6a, 0x0000d8a9469a7e74 }, { 0x000ceecdb6a205f7, 0x000de5c9dffa978f, 0x000c624a983be8bf, 0x0002cbe2fba1d1c1, 0x0000bd9e811e29d5 }},
        {{ 0x0009e1cce6add388, 0x000825f10f2b8f5d, 0x000e826a627d023e, 0x000263a13c83a6cd, 0x0000bb21395fab4b }, { 0x000121696926c8fb, 0x000c17caba7e3721, 0x0008ceb03971e6c8, 0x0007d1de5fcdc74c, 0x0000c71b97399710 }},
        {{ 0x0001acfdfce2e995, 0x0000323f6021c5dd, 0x00041e444626f1df, 0x000e37e96880d5e8, 0x0000877660535d8a }, { 0x0005ab89623c630f, 0x00055e8bbb66840f, 0x000d7d09faf7745e, 0x000d5d9ecf7f4301, 0x0000fb09525a4469 }},
        {{ 0x0005b6c3c5a8fbdb, 0x0009e740e9547505, 0x00080f430f00f4c1, 0x00089482e0121de8, 0x00003c819e927d30 }, { 0x000f3e226365392e, 0x000b747519cb9fb2, 0x000f54ea3daa561a, 0x000484b1af134019, 0x00006b90423d137b }},
        {{ 0x00062246dec7aa4b, 0x000e51773d273b06, 0x000610ffa8bbeae1, 0x000a69139ccd2107, 0x00003c66d4c6b3ee }, { 0x000b65168903af15, 0x0003dfa6b83c0d12, 0x00076c84b6bd8c9c, 0x000e810e4a72df26, 0x0000ab60b5424485 }},
        {{ 0x00088043d808618d, 0x0000e7902f345331, 0x0003f3f77d7057e6, 0x000f1ef785e354c1, 0x000037a49cbea99f }, { 0x00040c9ad6afca4a, 0x000e5428c1d50261, 0x00067fc80fd4ebd6, 0x0008d7a502e0c8a2, 0x0000c659782ab4fa }},
        {{ 0x000ce3f19260a150, 0x0005a4225ac0f182, 0x000ad7ee0a60bfc6, 0x0000ef824f7b1295, 0x000023483fef7f44 }, { 0x0008e821b9993c9c, 0x00081558f2e1b72b, 0x00057742808a033d, 0x00081f954fe93228, 0x00002729d121fb3b }},
        {{ 0x000aebfa4044443e, 0x000fabe67e573e06, 0x0008d598ca0bb6e3, 0x0007d84c505891db, 0x00008b6a89573352 }, { 0x000fd835db9f7005, 0x000d7e9e56e55c19, 0x00028cd210f50d2c, 0x000afe3eb7148ced, 0x0000282da8971416 }},
    },
    {
        /* digit=29 (1,2,..,64)*(2^{203})*G */
        {{ 0x0009d7b51caacc1c, 0x0008509081e9387c, 0x0006c2bba30bf8a1, 0x000749415fb9780d, 0x000016b6886a163a }, { 0x0009d9971a12e8dd, 0x000f041d6d9f6711, 0x000897c914fde7c4, 0x00037415fba6c1e2, 0x000044ff79be19da }},
        {{ 0x000a38783edbf56a, 0x0008a62060b5f619, 0x0003e197df84183c, 0x000ec5565a56f46b, 0x00003d764abc2d6b }, { 0x000a0edc3fc096b3, 0x00080a9da710718f, 0x000633fc0eb6b9c6, 0x000931e875a77998, 0x000072910f080d6c }},
        {{ 0x000f379672eac03b, 0x000fb216a31e09bd, 0x000ce1fd77c2b332, 0x000e09452de89e44, 0x0000aa34a9ecfe61 }, { 0x000697c8bf759643, 0x0000e0a2591d61b1, 0x000741c37bbc5394, 0x000923c7779aa87d, 0x000040557c30456f }},
        {{ 0x000c351b873ccd50, 0x000f849cb198ac73, 0x000cd2f12adddfbb, 0x00052b678a884a93, 0x00006d2e4199685e }, { 0x000ba6f9e2ce488e, 0x000619fe2c4b44b8, 0x0006a77f7f29e16a, 0x000b9984a580f6c4, 0x0000c4fae9993e3c }},
        {{ 0x0003c99a73f9f984, 0x000135f8fb547385, 0x0007d3fa5060028e, 0x00050845a3516078, 0x00000a2971005ebd }, { 0x0009a4598291f660, 0x000fbe05b81c51b3, 0x00088463f934f561, 0x00056b9a7925bafa, 0x000056cef7017b16 }},
        {{ 0x00043775095179f6, 0x00074b65eb03c0ee, 0x00038e84cad4f821, 0x000c08ad2f19b795, 0x0000a815addc931c }, { 0x000a6a2475d15354, 0x000e250bb3ee8a3e, 0x000bc6c7e2e9f012, 0x00084e0f675eb14f, 0x0000728fb5f890a9 }},
        {{ 0x00058b5fe31ed9d4, 0x00029950f8d67583, 0x00010b00a7fa7e99, 0x000085e21cb84e10, 0x00000fa25fc01a87 }, { 0x0003e86e791375d2, 0x000a5ec95f40e756, 0x0000d98b855d088f, 0x000a5c966f015545, 0x0000404ce330eea9 }},
        {{ 0x00079fb23be5ff7d, 0x000b7abd3c095f18, 0x000e5d17bb3553d9, 0x0003eec8404b261b, 0x00004c7b8e343501 }, { 0x000f0ac52ff88cf7, 0x0007572dfd754907, 0x0004e2b22c9118c3, 0x000d179073a97d08, 0x0000d6bca24f52c6 }},
        {{ 0x0003c66b47990520, 0x000abc46eb564d4c, 0x000bda90a2e18e10, 0x000f0dd5d8d6c752, 0x00005bcd40fc8105 }, { 0x0008ac0b779ee6a7, 0x0000d010f87de563, 0x000f8e19f83df780, 0x000692242ce62c06, 0x00009114b5e7e8df }},
        {{ 0x00062d8c5230f8f7, 0x00029ee4b049136e, 0x00014f3cb9c19a54, 0x000e78a288b63bf1, 0x0000675ced19a43f }, { 0x0004ac0245017d70, 0x000da67379e7896c, 0x000206517d607078, 0x000c44a6ab25237a, 0x0000c32d492b5336 }},
        {{ 0x000eff66bf542bac, 0x000fc6921fea4ca8, 0x000764e07402005f, 0x0003cb3afd36d081, 0x000058cf051c60a0 }, { 0x00069fb9dd0bb00e, 0x000a0e70b57235e6, 0x00009a66b74e0c7f, 0x0004cceb0ee3c39b, 0x000096f6178742b5 }},
        {{ 0x000e008c5bcdd3fe, 0x00003fb319d76820, 0x0008fc97a392e47d, 0x0008db544b029312, 0x00005611953b5d34 }, { 0x000d3a1aadc08c32, 0x000ab1c0278ca331, 0x0006c870390417da, 0x0008770cf666f76a, 0x0000e48921cecb9f }},
        {{ 0x000a418ea93670f8, 0x0002d3173faa4851, 0x000fa665d9cf986c, 0x000a06e7dc8ee6c4, 0x000003f9ec391ff3 }, { 0x0001e3de1002180f, 0x000ae9251f73971d, 0x0002084afa8fd8f9, 0x00025f4193e4b3c8, 0x0000beca7924e4e5 }},
        {{ 0x0000ffc1739db82e, 0x000d6ff50f75f9cd, 0x0002ad7569ae9e9d, 0x000e1e3181d8eddf, 0x0000eddba8e1699e }, { 0x00002aac66c37326, 0x000c6e3037d90f29, 0x0005d02ad905e85e, 0x000d947afe3f307d, 0x00000675780cc1ba }},
        {{ 0x00081be1ba9dae65, 0x00082bf94cdfadef, 0x000c87aaf47a18f3, 0x0004cfef86c97e1e, 0x0000bf0bbd559305 }, { 0x00049a602bc81752, 0x00013e4d14b4206e, 0x000dab067b39bec0, 0x00068756cbecc2e0, 0x00007986052ab4e9 }},
        {{ 0x000709def8c8c5c7, 0x0004c1a567193ec5, 0x000a8eed0812adee, 0x0005924ddaf3c305, 0x00002a0743a5403a }, { 0x000f431d23ed5fa0, 0x000f9830eb2b6692, 0x000b5818530569a5, 0x000050c164d80ce1, 0x0000cf41a7008416 }},
        {{ 0x000bead645c0c39c, 0x00076a6f9294dd32, 0x000f6f60373224a4, 0x00024945cb448d01, 0x0000848887261b44 }, { 0x000041af6242087e, 0x000c54d84fd8b072, 0x000fd7682f82c87c, 0x0004d3f85537b983, 0x0000ee0f932db6ad }},
        {{ 0x0003d2646d5d67ea, 0x0009d4cb940c71e3, 0x0009a78ce6cd30ea, 0x000b333c3211935a, 0x00008dbdca8a24d2 }, { 0x0003929f2c8f6353, 0x000834bef9114044, 0x000a56a9bfc7874d, 0x000d1d9789d1f25e, 0x0000546cc2b4beba }},
        {{ 0x0001b0310c8f78fa, 0x0006c67c7196e295, 0x000041e18d3c15ec, 0x00067d36e7992c2f, 0x0000cbe87245b10f }, { 0x000d6e5a5a506ca1, 0x000e9cfdd9f22cb0, 0x0001a8177b29d312, 0x000d4c288bb1d81a, 0x0000a9d05bcc9b3b }},
        {{ 0x000d04e18964b4be, 0x000696638235d4e4, 0x0006fc69510f82cc, 0x000629dc81c62bd6, 0x0000d69b284282da }, { 0x000f0e6f4ef7aeaf, 0x000670185a8068c5, 0x000133a42b363af9, 0x0005edb3bdb58e4e, 0x00008a81469b37cf }},
        {{ 0x000d487b80c0f188, 0x0000e8469ed23707, 0x000c8b2972a54721, 0x000aad30534dc30b, 0x0000077e05f5c7bb }, { 0x0008c4e13b06dfc6, 0x000f437dd0e71799, 0x00011b3f36d639a9, 0x00094b5a545f8019, 0x0000cc8a58f51d91 }},
        {{ 0x0003ff120d6b43b6, 0x00080e6e82c9f9e4, 0x000c7f3afe366902, 0x000473605711db87, 0x0000b79687b6563f }, { 0x000c63a68310f640, 0x000492fae77779bc, 0x0008c078f8ffc2c7, 0x000cf71bcf0fe0cc, 0x0000f50ef21963e0 }},
        {{ 0x0007fcd27a7b8803, 0x000b70c09dbe19a1, 0x000d535fa1484c62, 0x0005d09302f3585a, 0x000034ee43c2d08f }, { 0x000ec3df2b85cbb5, 0x000983410b3d5a5a, 0x00017bd6edc62975, 0x0009ead4209c107d, 0x00006cece29a740e }},
        {{ 0x000eb585fc03a598, 0x000fab73f894ae03, 0x000578a992d2287e, 0x000ae32d0f59df3e, 0x00009041a752d68b }, { 0x000f18757657a10e, 0x000fe8632c7f717e, 0x000ac07bc9d5c038, 0x000db43d2eb8c795, 0x0000beb76ad83f9e }},
        {{ 0x000f678f4d369698, 0x00025b20389168b6, 0x00087aefedb6701e, 0x000dd9882c4ecd25, 0x00001a019280de1e }, { 0x0006b6c517470dd7, 0x0003acf65e58dd88, 0x000e4d4bfbb60e27, 0x00029ea4cd53b4e3, 0x0000c0f835501259 }},
        {{ 0x000e54523f9dd595, 0x000453eb38dff6f0, 0x0002dc0f19a88a5f, 0x00007a2f8be012c5, 0x00001ab8ae664cc4 }, { 0x0008e19b0ec8804e, 0x0006636036a05d8c, 0x000e55b225f11a9e, 0x00001cfdf83e3cdf, 0x0000493f71b10088 }},
        {{ 0x000a39687fe60d79, 0x0001b3fb569a5fec, 0x000409fac4f5e2bf, 0x000df2700ad0090d, 0x00006a9ddd1bcbe4 }, { 0x0005efb775a4c718, 0x000f5f765959d77e, 0x0007b6a6922442f3, 0x0007a3c5139d7821, 0x00007d9756c3730d }},
        {{ 0x00063dec3e3e2558, 0x000b9a4e822f0d08, 0x0000d44918ba007a, 0x000a12a0fbc647ad, 0x00002a7318c6f712 }, { 0x00063dde65759c1b, 0x000423af017bfc7f, 0x0008a12da3c38c63, 0x00020f23b2005443, 0x00007385de06d196 }},
        {{ 0x000223802a112a4d, 0x000a2b32a4c94e9b, 0x000de4d4a76011b1, 0x000d92c3dafbff0f, 0x000052dfb56d8723 }, { 0x0002e8a4369887b3, 0x0008b578ec3fecad, 0x00036655cc1ccce8, 0x000df81c406c3831, 0x000054665660b4e3 }},
        {{ 0x000668270095818c, 0x00004525e9ca895c, 0x000dd0b84bde7efc, 0x00044e95c2f4dd31, 0x00005870893f8e51 }, { 0x00050bf75458d456, 0x00020a30cf43895a, 0x000fdeb71226c646, 0x00064b80ea9d604d, 0x0000539558e1b50d }},
        {{ 0x00090b3defe902aa, 0x00013def6a702510, 0x000b335f737a1d94, 0x000cddd62548b801, 0x00003e17f46314bb }, { 0x0009b0f6c59961be, 0x0000d8969cb91584, 0x00044fadd77dee87, 0x000783df544a9074, 0x0000f348045d2e71 }},
        {{ 0x00037dccb38225c9, 0x000f315adf7cccff, 0x0005ec2414fcf3b3, 0x00075f87e81a3e5d, 0x0000f61ae1e090a2 }, { 0x00009fa4f71b333a, 0x0000e73907fba12d, 0x00097997840dbf32, 0x000506cc535daa6d, 0x0000b54ff864cf47 }},
        {{ 0x000f979bec99a237, 0x00067c9aeb19a362, 0x000e982823c7ceec, 0x00020c3a6d913f1c, 0x0000fee386354fc8 }, { 0x000f2edcdad2a280, 0x0001d0f359679ec3, 0x000cc86c2785e2f4, 0x000583870ebcf523, 0x00007790dea32d3e }},
        {{ 0x0001f3e3242b67c2, 0x00037a76f568431c, 0x000b4104a667e1fd, 0x000a399d5f848c27, 0x00000c5d8c8cbcfa }, { 0x000519b637651d15, 0x0001c9c26c185ddb, 0x00079f55548ce1cc, 0x0006288576d85b7d, 0x00007ef27cc4babf }},
        {{ 0x000f6557c2e6a3cf, 0x00094a9ab87d59c5, 0x0002da65524e84bd, 0x000d3697bff9f58c, 0x0000bbd202141b4a }, { 0x000ea096448fc959, 0x000e70636a830417, 0x0009f74e8327d786, 0x000c320409b1c96c, 0x00005b046b469a20 }},
        {{ 0x000e20be26cf50fa, 0x000b9b56aa39dff0, 0x0002916d8f65e5bf, 0x000fb519f59b43da, 0x00002869fe6dd6ce }, { 0x00029808bacacede, 0x0009382e8ee47c94, 0x000b3b2178464d49, 0x0001271f6d9687cd, 0x00002d0288e392d7 }},
        {{ 0x000aa219cfc47b4c, 0x0005a0d4944b6639, 0x000909570115df01, 0x000bb97b1f901f7a, 0x0000200ebd199ed9 }, { 0x000598dee385254a, 0x0009875056a2512b, 0x000928ccfa78af5a, 0x0007a79eda4d643d, 0x0000a165f63eb2e4 }},
        {{ 0x000ef8428d09f84c, 0x00075e5dda9e5c65, 0x00026bb1ba085e32, 0x000339e444e0d3c7, 0x000054c2fab4516d }, { 0x000e342afaa45999, 0x0004b7a460ff0166, 0x000f041e56d127f3, 0x000ca11d2da6d12b, 0x0000227aebe64d5d }},
        {{ 0x000d33b92ef918a8, 0x000130b93a37c047, 0x000b5fdd04671ae8, 0x000939eb762b1bc9, 0x00007fee369cbc14 }, { 0x000ef80e1f1184ae, 0x000fb4be61fa29a5, 0x000ea74a1414172d, 0x00088df374b2be13, 0x000023e963704321 }},
        {{ 0x000d96dd292ac476, 0x000d0996d16768e9, 0x0009c4f514edec88, 0x00089d30d9fc4ff6, 0x0000a3049df36aea }, { 0x000386b3c21696a6, 0x00033d8ce5d72c41, 0x0004ad95bd8d614c, 0x0004dfa3e9d06c5a, 0x00004d05ba437463 }},
        {{ 0x0007d17dc0d747a5, 0x00043a6fb283f61f, 0x0004fbda463b273b, 0x0000591227df203e, 0x000021ce3dea9063 }, { 0x00056d9dd063bbcd, 0x000a144522055f0a, 0x000aea39ab0b5fd9, 0x00061ff14de3012d, 0x00000a88edfa935b }},
        {{ 0x000ff42bc607a539, 0x0004db68bcec9c2f, 0x000868177340b0bc, 0x0002d59817cf24df, 0x0000d7cc1c5324b9 }, { 0x0004c5f529e4d5e0, 0x0005c404d289427b, 0x0008ffbe999a2d50, 0x0006aa00c3d5f189, 0x00004942095ae6eb }},
        {{ 0x0009c75b689ed4a6, 0x000998fde94fdcb0, 0x0009bdfb94f62d7f, 0x00051ece1e96af73, 0x0000523f6b4acde9 }, { 0x00018c7501c3e7db, 0x00062f2b5932a7a7, 0x000e6c7d04ab9f0e, 0x000febb312658252, 0x00006c7c749a3df1 }},
        {{ 0x0006ba6d4797b4e0, 0x0004d69ea40dc3a4, 0x0002681194478eef, 0x000bc104adc84ad2, 0x00003476bddf339a }, { 0x000c417d369d684d, 0x00084c593e58a8e2, 0x0000e5ca8b55cecf, 0x0000639daa239473, 0x00006f34e5122d2e }},
        {{ 0x00056f1bda88811f, 0x000dac58b6f8efae, 0x00095202695bd2ef, 0x000b8aa2b671a2fa, 0x00009c76e0d6ec2c }, { 0x000d267e667d7ee6, 0x00088c49ff3c8eb5, 0x0001e6229e840d8a, 0x0001551942deae43, 0x0000a944be2e849a }},
        {{ 0x0000009e494c202c, 0x000e56db07159d03, 0x000f9188c3c7511c, 0x00075e979ae07a67, 0x00001fabe7db3e6f }, { 0x0007b796c365a4fe, 0x000b122e31328195, 0x000c9a831e3c24e7, 0x000b8d887cffa197, 0x00006565fda180ba }},
        {{ 0x000195969620544b, 0x000ef6ca1779ad79, 0x000da3e6732eeb47, 0x000605a7ea33cfe2, 0x0000c3bd61477f1d }, { 0x0004f291ca3b4ea1, 0x00018a6e546c8792, 0x00043da3370bd63f, 0x0009ddf6b8e996df, 0x0000fff3f513ba1a }},
        {{ 0x000136d7f23d9040, 0x000e3bb440e22296, 0x000212b875128f14, 0x00081ac39324673a, 0x0000c58b94474200 }, { 0x00061d7940cb50ff, 0x0002999b053a9270, 0x0001a7e9058caf2a, 0x000269dd93eaa266, 0x00002b4a7d3f8737 }},
        {{ 0x0003f67568134616, 0x0009283f06d18bd9, 0x00031e7992db2ac5, 0x000d4884d4ba6de5, 0x00005538b6570045 }, { 0x0002d4d4f88e4a39, 0x000381b4347237bd, 0x000f14bf8b95cbb8, 0x000ad048cbc041e2, 0x0000ef112474fce2 }},
        {{ 0x000533837f402aea, 0x0009089b3e93ce7b, 0x0004a806a89c97a7, 0x00030378af7b5a87, 0x0000d33d95cf20e5 }, { 0x000972a7c51e92cf, 0x0008b73579fab2e2, 0x0001c3649bc0b31d, 0x000cbc215b41055a, 0x000013a756b9af25 }},
        {{ 0x000fd1823a519a79, 0x000ba4ef678e68a2, 0x000069ded5b31a6d, 0x000323d9a4e4160f, 0x0000ac5ec7087e01 }, { 0x000c3930bb0e4209, 0x000fe2f3298f0557, 0x0009828e19b3d13c, 0x00072c7418c0566f, 0x00000b7752b85e4d }},
        {{ 0x00058d3c48fce8ff, 0x00042fcaaa2902bc, 0x00091f695e8ae388, 0x00056a0fd539ad79, 0x00003fdda1024b84 }, { 0x000ceb97675efd50, 0x0001fad5dc64e964, 0x000965554bed5435, 0x000a497a93aafcaa, 0x0000072f2cc19c4d }},
        {{ 0x000fde3ee29e89ec, 0x0005249e32a858e7, 0x000e2eb23ccb4e1e, 0x0002fc8944c32892, 0x0000fdf45fd7d390 }, { 0x0000793aacf7c0a2, 0x0000909caf549579, 0x000e557c9b281bf5, 0x0004340c0bfb028b, 0x0000eff9d16da328 }},
        {{ 0x0009ee5f4330c617, 0x0001779c81710a00, 0x00061dde4e912dce, 0x00009560c557e4a3, 0x00004f997a74131b }, { 0x000606bdf720e7be, 0x0001de9d4292f19b, 0x0005adeaad214a89, 0x00013f62a56f74c2, 0x0000372d1eb10015 }},
        {{ 0x000b1ef85cb235c6, 0x000974d044573ac0, 0x00064a4c13476c7b, 0x000d830f77dc3cf8, 0x0000e93ec667d713 }, { 0x000aaf7c181ff6b2, 0x0004684c96958bee, 0x00034ab98d4b092c, 0x000a94e617c32fa9, 0x00007362c69c64ca }},
        {{ 0x0005faa9c8bd37eb, 0x000f34687032d585, 0x00030831d017617c, 0x000cd0423c54f3d8, 0x00007c6ebecf2b9f }, { 0x000b53e0cd41d986, 0x000c96f89b95355d, 0x0009dbed7bb78cfc, 0x0003534d34860d2e, 0x00008bc6614104c3 }},
        {{ 0x0004603695bcdea6, 0x0001a319feb65939, 0x000fd66cc881fb52, 0x00076b610dd87f30, 0x0000fa4b825b15e2 }, { 0x0002aac5a362f271, 0x000a0c32e83358c6, 0x00048c503bd59590, 0x000c81552508dd9b, 0x0000887ce9e262ca }},
        {{ 0x00078ebe5bca260d, 0x000c5a08b1ea7b31, 0x000396fc5629d03b, 0x000df2299d495ab6, 0x000045eb0b5e28bc }, { 0x0004348796ae1fc6, 0x000f6ca5b36d17fa, 0x000306c3e75354eb, 0x000d923e0b5e4cce, 0x00000bbf054089ba }},
        {{ 0x00055e14b2edfcf1, 0x0003209eef1a7522, 0x000749cc8d3ad05c, 0x000038cd25d7162d, 0x0000d5775e923a21 }, { 0x000c80242e03d505, 0x000db720303b6972, 0x0005c3d6fcd1b6b0, 0x00053e6b6c5d17e2, 0x000040da0a6e404d }},
        {{ 0x00094cc7f33d38db, 0x000ef5fabfae1ca1, 0x000c00d47da85466, 0x0002dc9d2937afaa, 0x000024c1f8bc28e2 }, { 0x000c53b0093312f3, 0x000213a8aa116906, 0x000c9d61edff7a12, 0x0003d930abe40ad8, 0x000054fe955092b4 }},
        {{ 0x000b70ce22f66999, 0x000a85d1356eb809, 0x0002fa0810b5bf08, 0x000261a8bb8bf9da, 0x0000447d5e591368 }, { 0x0008f8655fa050ca, 0x000378a829a81803, 0x00037f266d35dbaf, 0x000ec208384329f9, 0x000095e38ca342be }},
        {{ 0x0009bb9dc9c786dd, 0x000c3d01d3ec517a, 0x0001a76783e4deda, 0x0002248989874465, 0x0000b42617e4c825 }, { 0x00071bb70b07ff38, 0x0000b5578b4c2405, 0x000b09f0f8d10cc6, 0x000394c6b5ed62a3, 0x00006bb834b02c13 }},
        {{ 0x000e530573a2840b, 0x000ae5b2da705d20, 0x0006a34d8a2512d2, 0x00068f0e53ef8ada, 0x000041196132343b }, { 0x0007f458eb02c886, 0x000c75521a9255d9, 0x00015aff273f5d8c, 0x0007ef27563a32fa, 0x00008ba09cf2c5ca }},
        {{ 0x0008797e91aea7cc, 0x000a480eb6b242d6, 0x0002d2f863b4d485, 0x00089479dd30bd02, 0x0000e4b655e68a0f }, { 0x000d53005ec1aeb9, 0x000290a6b4e185ab, 0x000d8586b6a88091, 0x00048c81b82f2c67, 0x0000bb2a23d0098b }},
    },
    {
        /* digit=30 (1,2,..,64)*(2^{210})*G */
        {{ 0x0002f4209bf9bcf0, 0x00059be9d8e68fd9, 0x000288102284ec39, 0x000064398db0f053, 0x00004a5dc6b917e7 }, { 0x000420b560d4b030, 0x0007fcc1a739d4b6, 0x000f3e6a037e1521, 0x00016c1d009aea75, 0x00000584c6da5516 }},
        {{ 0x00082b921a859ae8, 0x000619c6254dc41f, 0x00038257ec641913, 0x0006a3ca77e29392, 0x00000c021e167183 }, { 0x000a3d8c1544aca0, 0x000f62e1402cec5e, 0x00070ff1a4d2048f, 0x0000bd9cb9f17ca1, 0x000074174697d3a5 }},
        {{ 0x0006d4822ab85c35, 0x000e0e75e0dee6e5, 0x0002858e08782681, 0x00005869374f2487, 0x00000b357177d3c0 }, { 0x0000031f8adc8033, 0x000e4cfc59826f90, 0x00092404f713d9e3, 0x0001ec585ca26096, 0x00001aca73b07beb }},
        {{ 0x000e0ace9790f135, 0x00011e60ebcf7262, 0x0001342678767f4b, 0x0008aec094482b83, 0x000086d2546132b7 }, { 0x000cf8daedabfb38, 0x00071d31f8d1f420, 0x00014a5069e4864b, 0x00083fb4b1b1e83f, 0x00006eb2e2034cf9 }},
        {{ 0x0003ad49be80c260, 0x0002ea800ee11c1c, 0x0003f23693b6b015, 0x000c9e3c5eb053cd, 0x000021587ac78054 }, { 0x000143e74d589844, 0x0004a41b64279905, 0x000d1f3ac6d14165, 0x00013731e12cf15b, 0x00005b3101abcdc2 }},
        {{ 0x00088690b3b5dbba, 0x000a936c88bdee1c, 0x00058fefe5c87a4a, 0x000975f9625f2930, 0x000004895caebe10 }, { 0x00007be238c00228, 0x000042d852705b44, 0x00040d8d72f4674d, 0x000634dd1b0bf8d4, 0x000089f5af04b2d8 }},
        {{ 0x000bcaced2f7f0f6, 0x0003a49a639fc6b8, 0x0008fa356752166d, 0x000db2046e75c35d, 0x00000c7919830565 }, { 0x000757ac98dd7581, 0x0004e1db55367425, 0x0006c34659f304b9, 0x0000388c5013dd89, 0x00009e9e11e632ad }},
        {{ 0x0004d5ef9bcae914, 0x0000b0d1e935abb5, 0x0003c077f3debaf5, 0x000b3d7b5defd10b, 0x00005a2eed07f66b }, { 0x000687601569519a, 0x000a7ba74f17e266, 0x0008123a7dcee411, 0x000877efdc0a0e3c, 0x000028a9a1af6c1e }},
        {{ 0x0008f0539945768c, 0x000b29cf36658f0c, 0x000ba907b1e581ba, 0x000f3f75ac433ef1, 0x0000e501b89aacd5 }, { 0x000e4ebf5dc2f5a3, 0x000e9dfe983c4a11, 0x0001e568f9d43c43, 0x000c3615bbf5e816, 0x0000c937c01fe8f4 }},
        {{ 0x000792a7ad2ea66a, 0x000c06043a24fe7c, 0x000e3820a17ecaed, 0x00089740529c1191, 0x0000da0679e69c61 }, { 0x000df141eef101e6, 0x000a4d8b737982c6, 0x0009342d1ef43ed6, 0x00081731a7e3a031, 0x0000448cc1825582 }},
        {{ 0x00001f483720d8ad, 0x000f339e6d6f4714, 0x000ed17988f78de8, 0x0001638ac14b2ba0, 0x000005cc7b06291a }, { 0x000e8e9670cc34e3, 0x00054964fd03f663, 0x00099f00a8a162af, 0x000032c183fdb4e4, 0x000086fc98b4855d }},
        {{ 0x0009432af25a44dc, 0x0003c0c207674f17, 0x0006ba6b64e961d4, 0x00069db1e112e09a, 0x0000b97a68266210 }, { 0x00016fbbff8bf955, 0x000008fde2ddec64, 0x000f11e7b94049f2, 0x000e7c08a81392d1, 0x000090f2310497ae }},
        {{ 0x00073c690595d548, 0x000b5fd0d4a9dcf2, 0x00086d11760d0890, 0x000b020667bc29e4, 0x0000e82ab371d3a6 }, { 0x000ed3d505fc5b0f, 0x000d518cdfaa6b88, 0x0002c18df1996da1, 0x0009c4f27606bd82, 0x000024b05bcf9a6e }},
        {{ 0x00060e86489642ef, 0x00039416ab25b6a0, 0x0007250bbedb1912, 0x00086af03c6c0ffc, 0x0000643b4ceb328b }, { 0x0004f435d3812e8a, 0x0006f0f93c7f3988, 0x000afe049b37d0d9, 0x000a26fe39f4a96c, 0x0000c8d9ac21a5b9 }},
        {{ 0x0008aca1dc8cc929, 0x0009c5b8e33787f1, 0x00058d8ba496be3a, 0x0008890f0ef42f97, 0x00009f738f8c1ac3 }, { 0x000f6c9c74fac735, 0x000eba72d4bfe80d, 0x000c25d1d6b75c29, 0x000709516fd823b3, 0x0000df2d17686f7a }},
        {{ 0x000a995e5dfbc6e3, 0x0004fc7904fc3fa4, 0x00057d4a050cf2ac, 0x000caf6aa14a23f4, 0x00009e74b3de65f4 }, { 0x00016c563b4a41b8, 0x00083ecf3a5598a7, 0x0007407e5359d663, 0x000c3283c56534b1, 0x0000a377a93e6974 }},
        {{ 0x000bf3352222107c, 0x000fba41d6178e77, 0x000fecaeca6c05a9, 0x000037a3de4c80c3, 0x0000151b1e5776e8 }, { 0x000829e76053679c, 0x000858e10b365bbb, 0x000357ae0314b797, 0x0004c0d258102180, 0x000045a0e1b22a51 }},
        {{ 0x000867e8fbae98cc, 0x000a9a54d53e5fb8, 0x00024cbf2ee7bd43, 0x0000b62c204a5dc0, 0x00007dace033343b }, { 0x0007f6fd799a066e, 0x000ee6ebec9889bd, 0x0001f350cc14257d, 0x00005bc1254f491f, 0x00000465189040df }},
        {{ 0x000fa37d174d524d, 0x0005f30acde5e173, 0x000028af37c05817, 0x000b76154d0f996d, 0x000078f2cc1b9df8 }, { 0x000a99951b3d4f35, 0x000518434b8fb41f, 0x000ca461cfd62559, 0x000f9acf299a2acb, 0x000083fa11ba4326 }},
        {{ 0x00064110e3d7be93, 0x000db892f3b26e7d, 0x00000446416395d6, 0x000c44aa3f470986, 0x0000d33c4ed3954c }, { 0x0002e3c7706265bd, 0x000d02cfad414e41, 0x00081bec63b3b22c, 0x0001c5ef12b155f5, 0x00004a979c09ab69 }},
        {{ 0x000f6db89c6efc0b, 0x00052e59fff83814, 0x00020c510ff875a0, 0x000517dc41105323, 0x0000b7c05ea42fd6 }, { 0x0007d34d61e350d7, 0x00068d6ca563a441, 0x000541073d5751ad, 0x0002419a17f5c221, 0x00006c4fc28b0bb1 }},
        {{ 0x0004b95e3dbdc078, 0x000c1101c474019d, 0x0009007390bdd0dc, 0x000a1b42bcaff262, 0x000042e2d657f503 }, { 0x00021d322b01c90c, 0x0009a1f1594438c2, 0x0005e28dacb4f9f3, 0x000183d401eb6e6a, 0x0000c764dc01c967 }},
        {{ 0x000629a6c01aa769, 0x00040f6b7cb179a6, 0x000d5169a70cd2e7, 0x00061af0ddc7b764, 0x00009db12efb8043 }, { 0x0007d3a142e389a6, 0x000b84e9f5a915e1, 0x00074a30a2dbdc3b, 0x00082b953b1db9d3, 0x0000f4c3e36d8caf }},
        {{ 0x000e709afd4ebe8c, 0x000b8ea308780f21, 0x000119d55ae78e9a, 0x000664ba3dc8de76, 0x00001b9c38ab8027 }, { 0x000526074ae5ec7c, 0x000c776021704560, 0x000ac3d34419af49, 0x000944e68bf204b3, 0x0000bf2f1a243170 }},
        {{ 0x000f57691dba40a3, 0x0002bf8ecd92af61, 0x000a4d7a0b3cdb98, 0x000e520e4e6b9fa7, 0x0000cbd83da71fe8 }, { 0x000c2134f4315826, 0x000cdfe9d2e1a33d, 0x000ecfe50841b739, 0x000bb7d27d430093, 0x00008342dc8c1e23 }},
        {{ 0x000a626f3a19601b, 0x000e32c5c2102869, 0x000d8248f65290f1, 0x000d557923f68aae, 0x000026d9214bd1df }, { 0x00023bbc1636c361, 0x000daf3199d78e38, 0x0006ef80072fc767, 0x000961f2cb7a2af7, 0x000031f1a12763e7 }},
        {{ 0x00049f321e4bf811, 0x00028f299d0a4221, 0x0005dfb794dd3969, 0x000277ad69ac3df6, 0x0000a5e53844a824 }, { 0x000d0299708cd13b, 0x000bc14721b7ebaf, 0x000734a097ba251b, 0x000f39a81e17df09, 0x0000236afe787c48 }},
        {{ 0x000a45755beb5e5a, 0x0004c3e7882f14f1, 0x0004ed815580f741, 0x0002464d0c9f6dc3, 0x000088180b1213cd }, { 0x000ca2ca9f747a28, 0x00054277c4608cf7, 0x0001d9dd48232ab2, 0x000f8730e7ccbdf3, 0x00005ef4880c3dec }},
        {{ 0x000a91f32af5551d, 0x000cfb16f640f620, 0x000b2f63261c003a, 0x000339244e274b92, 0x0000106a1125d96d }, { 0x000ec1cc3386a669, 0x000be4bd6aaba315, 0x000fb0a6b8c5bd06, 0x00052b520719a163, 0x0000859041c6c354 }},
        {{ 0x0008724aa3301862, 0x0007c963b54a0599, 0x0005574ab31008dc, 0x000a32ba60e0015e, 0x00004faf233b361d }, { 0x000a0642cb6dca07, 0x000815bfdb6558ef, 0x0009cd8357cac326, 0x00063ba12f9f7a28, 0x00002446724efd19 }},
        {{ 0x0007e5eeb46274cc, 0x0009d81e7dfc8d6c, 0x000f57cfca95b2b4, 0x00069313767cd444, 0x00007aa955fa3cac }, { 0x0002cf71b93c6df1, 0x0009d3d81d950286, 0x00020de171a3c63e, 0x0001a2ee30088f17, 0x000063c5a29ef2a5 }},
        {{ 0x0006539c7de5db9b, 0x00043a7e0f222c2d, 0x000b9c92b4c46a68, 0x0003b3230309d42a, 0x000069a4869c5fac }, { 0x000331fb46a1f47e, 0x000277c432ac7d72, 0x00000f4ddec961d1, 0x000121ade72692cf, 0x0000f86aeb4b1886 }},
        {{ 0x0005cc064fc464c6, 0x00034322867e6332, 0x0001e4d0c1c60ee7, 0x000f969a578709a8, 0x00000556972914c2 }, { 0x0008ecec23a18976, 0x0009f1c493a65c5a, 0x0000aa2b3840fd26, 0x000bca2e25fb5d94, 0x0000d962837b0d4e }},
        {{ 0x0003895ce010bd56, 0x000728bb96209bd8, 0x000b30586d00bcad, 0x000d038ac65fa8cd, 0x00004331b11c1539 }, { 0x000e7f5c64f975b7, 0x0001359b56cd3d1b, 0x000b1c7425cdaf38, 0x00092539f10c8350, 0x00009e660c7a4eab }},
        {{ 0x000780dd3a8a4d1b, 0x000253997403a113, 0x0006da9ab974a228, 0x0004b23c294626cf, 0x00003195ea2589e0 }, { 0x0004c61ed6713b28, 0x000e6462baf8e3b1, 0x000d3d0c7ad8714e, 0x0003efc08abe68aa, 0x000057bcb3c222f3 }},
        {{ 0x000a3f84425ebffe, 0x0004614c53352e78, 0x000b88f8502c29f3, 0x0009f2dc85a34889, 0x00007c35ac8f1e9a }, { 0x000c80f9252c564d, 0x00051975224da68d, 0x0007e76a07c43bb7, 0x00015767560cd6e8, 0x0000e6884afd696e }},
        {{ 0x000dbc7e7af4947a, 0x000ba156ac774075, 0x000ec785880677b6, 0x000732a58a1e5ea8, 0x000073eaf3f59951 }, { 0x0004bc8be8ae5699, 0x000ec1b31590a478, 0x00099131e20fb613, 0x0005da6c0039e85b, 0x0000ecfb39af4929 }},
        {{ 0x00021c3831e88d1f, 0x0000a36f20df03ab, 0x0001d3ec9c404310, 0x000bc012a2405438, 0x0000e8ad27a0d2a8 }, { 0x0002abde10a71547, 0x000e959d6fdee104, 0x0003359cb06136c3, 0x000e7b4259da47ad, 0x000064fe3586ecf9 }},
        {{ 0x0008660001e36d74, 0x000a1fc17eab9fe8, 0x00044b810970315e, 0x000c49d2cd6e863e, 0x0000d984aeea5a80 }, { 0x000086431bc9586b, 0x000ac5a2c088eaef, 0x000d91aa75b86dd0, 0x00006d87913afbca, 0x00002d1d48a935c8 }},
        {{ 0x00092167c2a88bb1, 0x0006855c298b9743, 0x0000bf4fb6418eba, 0x0000aec18905fb5f, 0x000010e7e5087856 }, { 0x0000016589c91554, 0x000b92d28c511da9, 0x000c2668c0a43364, 0x0004ea9f0f127c7d, 0x0000d032d8aa0c34 }},
        {{ 0x0008dd39e5a927e3, 0x000c43c11aa600d6, 0x000d36d8716ccbaa, 0x0008f1d682f5379b, 0x00008d24e8fcbed6 }, { 0x0005ff84acc088eb, 0x000e4572ecaa9c3c, 0x000b2224fdbef927, 0x000a8b8c4ba0e103, 0x00004e5b4b7fde85 }},
        {{ 0x000b9042d1372832, 0x0008c22ac95aff87, 0x000b3a142a37f646, 0x000c61d8d1c9bb6d, 0x0000bffd2852091c }, { 0x0000111675a26c12, 0x000de80830cf6bdb, 0x0009beb790506d1b, 0x000c2efaa4861d19, 0x00004ac5fbd62b7d }},
        {{ 0x000f2bed06be1b4e, 0x00088cf1afb20d91, 0x0000213b2ebeaee5, 0x0006a9c9b1369500, 0x0000b3302a11d566 }, { 0x00088a52e02d7437, 0x0004b176d799e206, 0x0006c9e76ffe5798, 0x000db221093415f3, 0x0000011fe4eb093e }},
        {{ 0x000ea4725092fb05, 0x00056a3e2a9a6db4, 0x000d0a54b43a9d4e, 0x000b5eb4f4601303, 0x0000be0291a05ceb }, { 0x000e240a57294dcf, 0x000b898d33fd3e7b, 0x00092d1f2cb16361, 0x000af81ad1ccd8a8, 0x0000360a29a8915e }},
        {{ 0x000acada29bfbe26, 0x000d798ca0e7ebd0, 0x000adc7d87452c91, 0x000bd8499f586783, 0x0000c65aa3596607 }, { 0x0004fd5e3650c743, 0x000937d0fe723ac6, 0x0004d4c60de7e613, 0x00084a7411d8e53a, 0x000016d0a718f6ce }},
        {{ 0x000d2fe3065bcddd, 0x000429057908805e, 0x000c71a49dd73b22, 0x0002f50d0b5b9ea3, 0x00002ca50a57bbe3 }, { 0x00041b9241ca1156, 0x00092926e9197e27, 0x00093734981b7b64, 0x0000dcb3f11719c0, 0x0000987c423fab1e }},
        {{ 0x0001f06bcf00e99c, 0x000df6849a5f4f41, 0x00028dd3efae77c2, 0x0000c5755f91d70a, 0x0000a828a98575b0 }, { 0x000341205886affa, 0x000d85d038d05e1c, 0x0004f78afcf954d4, 0x000dee60b04838ee, 0x0000776112a82e17 }},
        {{ 0x000a0f5ab82fff45, 0x000741364d9c2f41, 0x000386014e477e81, 0x0003658417468bac, 0x000084f5c1ea7f07 }, { 0x000549d8e1b91689, 0x000242f659d0ce0a, 0x000ac23bf70fc30f, 0x0000b09c74011bcb, 0x0000c926584930c5 }},
        {{ 0x00079ff428e80f59, 0x0002fd8c85438b17, 0x000d4485d6fe9846, 0x0003ca4f45680332, 0x00008f4d41216d32 }, { 0x0001d426d9f3da01, 0x000e898f059987ad, 0x0001fa94470d97e4, 0x00062fcb2f6ce8cf, 0x0000101ab31670a1 }},
        {{ 0x000bd0c9b88b7971, 0x000803fc781a2419, 0x000e3b8298f02847, 0x0002fa0843e89360, 0x0000acbc78ffab3d }, { 0x0000be4b4b26462d, 0x00020538a912a4b1, 0x0008ac36cdb271a8, 0x00021b4e0de5e15f, 0x0000a17f8d3dad9e }},
        {{ 0x000be980de325ae2, 0x00032f2668621c9c, 0x0000dfb311e3a6b1, 0x00079993cd5518f5, 0x000052f4db6222a4 }, { 0x000f159768ae1fb0, 0x0008f10ddd83683c, 0x000ccab52fd3a1ed, 0x0006425503b6cb35, 0x0000ef327d2522b2 }},
        {{ 0x0006283e8437a770, 0x0005f5343ad73b3b, 0x000f98db1c104d18, 0x000473c691b528c1, 0x0000c3c4141f72fc }, { 0x0003347161a895a5, 0x00024590a5f25b1e, 0x000475107cc11b68, 0x000483fdeefaf8aa, 0x000020a387cf4433 }},
        {{ 0x000de20ec9734397, 0x000ab198d9ea9f87, 0x000b761a0bec4dd4, 0x000ae95da090de5d, 0x00005d35adbfe798 }, { 0x000182de8a62cc1b, 0x0003d8eecdaecabd, 0x000cb6b148b91280, 0x000dd1dec6ca4b0e, 0x0000550839c85391 }},
        {{ 0x000cd9daab10e14f, 0x000763ec8a4186a5, 0x000b3793b388c571, 0x00056173cb3e878d, 0x000035bc8bc8154e }, { 0x0009c5d26f779d48, 0x000d17f586d5e720, 0x0008ba7df7ea2a57, 0x000c7dc2447500be, 0x0000ece6c6ab436a }},
        {{ 0x000636393a1d534b, 0x0005c9aa74d3f7b3, 0x000d99286b1f5422, 0x0007e296d0428fd8, 0x0000b23839cb4288 }, { 0x0008021fc06fa7bf, 0x0003a5510350bf49, 0x0008c8a8c7c25f86, 0x000c0c909b381743, 0x0000c67eee6023ce }},
        {{ 0x000caffa79164dbb, 0x000ec0c0e367a98a, 0x000462b2bd497a28, 0x000b04843bea1bc1, 0x0000c78fe367131a }, { 0x000b512e742ec763, 0x0003664457bf9c43, 0x000ad1dd7f0fc57f, 0x000ff90eb0db6407, 0x0000e05c4ba13983 }},
        {{ 0x0007f983c7b84b15, 0x000f91ffce104080, 0x00035264a1b62bef, 0x00017d0b389ddc00, 0x00002960ff9e4a4b }, { 0x000967d1accb38ae, 0x0009f155d5231e7b, 0x000c51e7c366f7b0, 0x0009802b5cff22ae, 0x000081c1f665309f }},
        {{ 0x0007012c7d47081d, 0x0004ec6cf777b6c0, 0x000d456becdc46a4, 0x000edaf850062bdb, 0x0000cc93e16237c7 }, { 0x00095cb26b4e0be4, 0x0001562ac47800d1, 0x000ba7a63b09f80c, 0x000ace5d353365f2, 0x000034e8d2215ffc }},
        {{ 0x000f81609ee40822, 0x0004c2905f2c4ce9, 0x000af8d535cd568a, 0x0004a4e15bd2bdec, 0x0000cfa19d6c9ba1 }, { 0x000241b6f9fad805, 0x000662cda350002f, 0x000007d9c63cdde1, 0x0004574bcbd05469, 0x000047e52739fbec }},
        {{ 0x0005ce69cde21889, 0x000fd432fe3cd4c0, 0x000baeedf96e9a6a, 0x000fb161fe0f33ac, 0x00001790f40c1e5a }, { 0x000b8e29796a6e6a, 0x000b116f5f98aaf9, 0x0005d395b776372e, 0x0005f93d345bb71d, 0x0000430258e053dc }},
        {{ 0x000fb4879b3fe9bf, 0x000f346ae0a6828c, 0x0003a0c44e0f5c7f, 0x000cb3cdbba533d2, 0x0000f6ea637a253b }, { 0x000dbf6a4042ef68, 0x0006b0b3281705b5, 0x000be2907641494e, 0x00070e6df7e361c6, 0x0000876051625bfc }},
        {{ 0x0000e78507af85ef, 0x0008d234b54d9bf9, 0x0000a42a3ed21495, 0x0002cc8581374e68, 0x00009d3702290f12 }, { 0x0000946dbd4e2626, 0x000d38d5d86f1749, 0x0002f55502275d25, 0x0009fad44d74e0bd, 0x000004488f9be27c }},
        {{ 0x000f710571dd1d02, 0x0002158fe1a838c8, 0x0004d299c4671897, 0x000b1580b05aae6c, 0x0000e3dbc53866c4 }, { 0x00091241f3b00662, 0x0007b48182caaac9, 0x00096af503dbdd89, 0x000b2dc6d8c4aeef, 0x0000048abae31743 }},
        {{ 0x000ffdfc7e7c2a09, 0x00085eae457a4773, 0x000bae6fdf8723eb, 0x0009a0d71d19857d, 0x0000d6ef525ea59c }, { 0x000d15002a515a26, 0x0002c5426ee7cda3, 0x0009a1edc37de8e0, 0x000e199c8341b086, 0x0000bbb468e51820 }},
    },
    {
        /* digit=31 (1,2,..,64)*(2^{217})*G */
        {{ 0x000f6927b0d79bd1, 0x000614c85edfa308, 0x0003f86bcc29875b, 0x000193b862597655, 0x00001c7d62051005 }, { 0x000d0721ecd294e3, 0x000feba55f2f94c5, 0x0008d744240dfdf9, 0x00015625d6a996f9, 0x0000e456c18ca0d6 }},
        {{ 0x0004cae01e64c20e, 0x0009b2079e5fb67e, 0x000a8bdf924006e5, 0x00033ca37e1331fe, 0x00003ed077fad719 }, { 0x000c957822ca746f, 0x000cfe60412a77db, 0x000c22af18030eaf, 0x0007aeb3106ff7ca, 0x0000cacc54eeaa59 }},
        {{ 0x0007e0aa7adc7a48, 0x000db0cb0ce1c552, 0x0004e3df0b8cb07c, 0x0009bee3b5b534d0, 0x0000fda2b88e9831 }, { 0x000f7d9eeff2ac2e, 0x0007e2d79362c410, 0x0000823dcdc0db71, 0x00077b12467920c9, 0x00001801931c732f }},
        {{ 0x000f5fa5904e2bd3, 0x000209afa8338349, 0x000206d2dfd21b18, 0x0004529102295172, 0x0000fd30a26b44f3 }, { 0x000137b286ed0846, 0x0004125e77d9a3b0, 0x000a624d3959c964, 0x0003d9c4a11235ce, 0x000037f27954916b }},
        {{ 0x0000d262f88de12c, 0x000b5ce44ace1509, 0x0008b2af868437a8, 0x000e478b8e0f8d3d, 0x0000f156f86418d4 }, { 0x000dbdf5922834ef, 0x00065c9283e13c98, 0x0007211e7af3d393, 0x000a7cfcb2d04fde, 0x0000f09ff470d31f }},
        {{ 0x00073c8530e34004, 0x00002950de1e5786, 0x000782baa695ecd5, 0x000f9e8a52e1463e, 0x0000607baa031037 }, { 0x000c1d160b022745, 0x0006100747f91da5, 0x00016dae62deceaf, 0x0008e89f5b08d43e, 0x0000e0b5caf53925 }},
        {{ 0x0004669f46350981, 0x0002d5328c8e13bb, 0x0003fe873d8bc483, 0x000c485ad375f10c, 0x000035558e6d3927 }, { 0x0009293790f894ed, 0x0000cd05bbd8699b, 0x00092e6e3582dcff, 0x000e93bd35528a4c, 0x000007788f960423 }},
        {{ 0x00061bbb05f9956d, 0x000106ac42bd6d29, 0x00082503dba8e1c1, 0x000ae458e6df8646, 0x00004572780d144e }, { 0x00004d881133b185, 0x00070f070a6a26b8, 0x000cb240e6288319, 0x000ae43d370686c0, 0x0000da343e03be55 }},
        {{ 0x0008584ef74bf3d5, 0x000b27b207f16700, 0x00084609039a722e, 0x000afedbf9b0dd18, 0x0000dfa5f66ee54b }, { 0x000c334771dc5db5, 0x000dba5dcd898ffc, 0x000f5425a6cc0213, 0x00029fcc126286c3, 0x0000264ddb1fa583 }},
        {{ 0x000725af16318964, 0x00085542f770fb1d, 0x000f75b1a448a814, 0x000806afc97c1c61, 0x00000e2e1a79320d }, { 0x000e079d3c015b41, 0x000914fc01e20ecf, 0x00039cdf74e30b3a, 0x0001c76ecdfd3749, 0x0000e980c17f36eb }},
        {{ 0x0000d6a5c42da56f, 0x000780a2e6ac8d23, 0x0002df97b64102c1, 0x000402c8c9c6b5a4, 0x0000724fa29031c0 }, { 0x000d131ab892404e, 0x0002e224fc0a9ef0, 0x000388ae52355bac, 0x0009dc1de190ff08, 0x0000ec43401fd702 }},
        {{ 0x000eec726a98db71, 0x0006f7e857012919, 0x00087a53ef3da9d1, 0x00019338594793ed, 0x0000b72ad73d6c3f }, { 0x00032b2c8ec764fc, 0x000be590c9d8005a, 0x00077b4e8d42c947, 0x000eb6e1e5feca45, 0x000035b3f8239855 }},
        {{ 0x0006c04225a0b97d, 0x000447373dd0b4ea, 0x00039de012a075b8, 0x000d1d6f137a2821, 0x0000c37b6f44f5ad }, { 0x000f54feb8e2353a, 0x0001134cf671e3ce, 0x0001de54badb3643, 0x00075fe4881bbecd, 0x0000e52b6354c0d0 }},
        {{ 0x0003469898a0538f, 0x000eda7fb8088fa5, 0x0006fa95fe2d2ec5, 0x00023e2760142724, 0x0000b5efdf4b7190 }, { 0x000e706aca43c3d6, 0x00061b83a2aaac68, 0x0002cc8322d12925, 0x000fb5299e9f91dc, 0x0000dfdaec18b984 }},
        {{ 0x00093ea6c3ee3c56, 0x0001e95f688dc983, 0x0006d90da43852a0, 0x000dc215c26490cd, 0x00000e9b4f9bc030 }, { 0x00059cacd2538ba3, 0x00090437f338e0d8, 0x000adab73ae2ca93, 0x0001271d539fb832, 0x0000b214d5f9080e }},
        {{ 0x0001567f755e516f, 0x0002de74462007dd, 0x00047b5f76420568, 0x0000d6ee7b8ab48a, 0x00004f2bc1635d97 }, { 0x000931de26c2af42, 0x000d96b0887bcec5, 0x000e8847159b8388, 0x000b324cb694497c, 0x000039c7e289bc5b }},
        {{ 0x000bd0a121216b32, 0x000ad7c0e58d4939, 0x000fbce69df1668d, 0x00035914db1ae5ec, 0x0000c149038863bd }, { 0x0007f13e98b5b5ed, 0x0009bb7cc863c683, 0x000e052e0eccfa57, 0x000f268933a1cc11, 0x00008167fd8f4e54 }},
        {{ 0x000ee1b6170663fd, 0x0008967bc947badd, 0x000d4b5538170a6f, 0x00013ffac738e07c, 0x000023f0b6b7b6bf }, { 0x0007cdc47eb25821, 0x0003d672e793d12f, 0x000de923c25b844a, 0x0008859f7d6aa6ca, 0x00005201d76c5ace }},
        {{ 0x000236ac0915006e, 0x000ff9dab15247f0, 0x00004703a77613ae, 0x000a2c6e77c789c1, 0x0000808d9373070e }, { 0x00042da81b066709, 0x000b4251d7a51446, 0x000a8d111cd1a17e, 0x00089a2ca098b9c2, 0x000013d8733bafcd }},
        {{ 0x0005441757848811, 0x0004b054f7269b13, 0x0007ad43ac2df8a6, 0x0001c5e21fcf3077, 0x00008d6031c1f86e }, { 0x000728edac03ee8f, 0x000e999fd703431a, 0x00056fb46aceed43, 0x000d9a62da438d7a, 0x0000f9c288edabe5 }},
        {{ 0x000356c23a712a04, 0x00086bc42531689f, 0x00070db2a1b53dad, 0x000edd5b30a51a0e, 0x0000c4d54a3bd065 }, { 0x00050964756fe418, 0x00027ff8d3a3dace, 0x0009588ae4a9b9ac, 0x000663ea4409597f, 0x0000d6d7b2598a4f }},
        {{ 0x0001c02c179a9537, 0x0001a265c3427b0b, 0x000c20c29da60ee6, 0x0009725456401405, 0x0000041eff03bf87 }, { 0x000a43463e984012, 0x000db75d9635bc92, 0x000743e10ef12662, 0x000b1be66226790c, 0x00006caa30be64cd }},
        {{ 0x0009f2f368831642, 0x000033adb4837611, 0x00054be273c309ad, 0x00028ba87e7cc608, 0x00001790a225635b }, { 0x000590dd601ac1f6, 0x0004d175c0b800bd, 0x0008b5dcc9827fed, 0x0008cd15e1f1e7c8, 0x000021b22c5fb59e }},
        {{ 0x000b15d6243f1091, 0x0000ca11f17a34c7, 0x0008a8443e31d5d4, 0x0003fa53b5420ab3, 0x0000927b5e2d1cf0 }, { 0x000424051138a243, 0x000fdb1e274e49c4, 0x0004528d80f9684c, 0x0000da2a45cf5074, 0x0000abcad67dae2e }},
        {{ 0x000248c0344ff4d0, 0x000734311b2d595d, 0x00091c801f234930, 0x0007f0d05f136749, 0x00005dba5d3c2cee }, { 0x000e8c07a9b72e10, 0x0003f13a77739f5e, 0x000a3546a79917e7, 0x000b691cfbf4288c, 0x00009c7306d20c7b }},
        {{ 0x00089e1462801321, 0x00028996fe8393f2, 0x000fc08125d060fa, 0x0001c42c80f809a3, 0x00009fb87e79ba00 }, { 0x0001826047e09dee, 0x000287e9889feea9, 0x00024f16d8c20c68, 0x000b29cc6d5c9c0e, 0x0000ca96182e9075 }},
        {{ 0x000a074b4cbdc635, 0x000527a9a4a27515, 0x000270dfb3609166, 0x000e3e44171bd007, 0x0000f49c9e777f48 }, { 0x000fb0efe038c226, 0x0003334c1cec8edc, 0x000ddb0b13dd9e14, 0x000c85e15a6ecb49, 0x0000c013efc7e2a5 }},
        {{ 0x0003ceb3f5d13058, 0x000edee23ab3495c, 0x000747a89247407e, 0x0002caef4a132df8, 0x00002b96856566b0 }, { 0x000c0899c9fa2622, 0x000071de88fb1da6, 0x0003b19887afb52c, 0x000d8d520ea05797, 0x00000f3ef4a9c2d8 }},
        {{ 0x000f5d86d67ee298, 0x000dd04752ddad71, 0x000f824903c11932, 0x000766399d60bd74, 0x0000bacfccb4d377 }, { 0x000571960bc9fc41, 0x0007b424b5a6d88f, 0x0002254d6471eea8, 0x000cd841069eb2c3, 0x00004e6a49e4dc49 }},
        {{ 0x0009b988353bd1dd, 0x0001ecf704e23c66, 0x0001bef11e2642bb, 0x0008c5f83c71b7d2, 0x000034725a0787e8 }, { 0x000a08f0037397b3, 0x00074a3963bfeec0, 0x000ad474b7c8163a, 0x000d0f6b199947aa, 0x00006b872dacd87a }},
        {{ 0x000ed489e3948b33, 0x0000d9478ef24870, 0x0004f3fb6243555f, 0x00051c028c8d2d41, 0x00001b4ae6438b23 }, { 0x000b9784bc1dfe6e, 0x0002a6bbef3366e9, 0x0000524658fc7d96, 0x000df6de20de59c7, 0x0000fc82425dd0cc }},
        {{ 0x00084e81ad96c3d5, 0x00080021a93507a4, 0x000744ed85217d67, 0x000286a40b4cd118, 0x0000702de63abcfb }, { 0x00077e27e30a727b, 0x0000cb5272e9d6ec, 0x0004ff812967789d, 0x000a6af8eea1c93e, 0x00003caac07df9b9 }},
        {{ 0x000fcc837acc3f62, 0x0004936d2b713c5b, 0x00000fae8958d09c, 0x00070206b86274ea, 0x00001d84b95f783d }, { 0x000f1a38bcdc1a88, 0x000769dd6529b73d, 0x0007848e4745cca3, 0x00002bdb9708aa22, 0x000012697ae997ed }},
        {{ 0x0003317d153dbf41, 0x0005926b8bad853a, 0x000293f26dd1f0cd, 0x0000ae7302d52cea, 0x0000259fdb316c19 }, { 0x0006bc5834b6d375, 0x000edfefd4ddedab, 0x0009fea120f6e9b2, 0x000f37444ed2a153, 0x000027704cde3608 }},
        {{ 0x0004269d64c13d9e, 0x000b4b383d102b6b, 0x000652201ae42133, 0x00030cf37fe695a4, 0x0000c2b2d20a73d0 }, { 0x000282e5be5c5683, 0x000d3a10a1d22850, 0x0001444f87b4312a, 0x000fdb1a7feef6c5, 0x0000eff77ff509ef }},
        {{ 0x0005aaf78c7288d5, 0x0001c81721c486d5, 0x000066ad8ac27b77, 0x000ce2b31301baf1, 0x0000f49e14841d8d }, { 0x000dd8c0882b51a7, 0x000ec7159c639522, 0x00045de99b26fb30, 0x0009386c33a3bd27, 0x000050ea586ccf8c }},
        {{ 0x0001dbe32c1fbaaa, 0x0008874f5531461d, 0x00008bb9666f009a, 0x000c64fd28b4a128, 0x00008430a73d0dfa }, { 0x000431531ac44d42, 0x000df0d30c34169a, 0x0002d4895218eac4, 0x000ebb054ba74a95, 0x0000755d2f2b53df }},
        {{ 0x00006ae448efaa11, 0x0004d22f669d7ec1, 0x00042514d0cc257f, 0x000f9ecb3ca374f6, 0x0000f169c333b6fa }, { 0x00010a7d3f6b9eaa, 0x000609033c8e9a1f, 0x000180a2db4623fc, 0x0006c4314234645e, 0x000084b79c5ee4b0 }},
        {{ 0x000d8727967cf447, 0x00097ee40d5c9be3, 0x000f7cfdc601ff1d, 0x000e146ddb2bae7f, 0x000039d35690a32d }, { 0x000213071385c0a3, 0x00027818569ff132, 0x000b784d085ec27a, 0x000ae88868b89a5f, 0x00001eb069c3b03a }},
        {{ 0x000098baf9cb4cd3, 0x000b81e48ac28403, 0x0004bc21d97de9e8, 0x0004798431831129, 0x000013750d1196db }, { 0x000ae9e34b83b95b, 0x00066584198da522, 0x000be98219cfe30b, 0x0007d4e08ab4fc17, 0x00008387d9c3f13b }},
        {{ 0x00010ea88587f6e9, 0x000699e3fa11b1f7, 0x0007263c4a2f6c1a, 0x00039903b21e4367, 0x0000789308d6093a }, { 0x000722a5d5edffef, 0x00032fcc5dc67cf9, 0x0006f11db1430e0b, 0x000e4afb708d5164, 0x0000f530330b49e1 }},
        {{ 0x0006370339f57e7c, 0x000d4e2c861baa5c, 0x000ccecea904d470, 0x000fd8a3fc0c21a5, 0x0000fadeb312f995 }, { 0x000ef031de2f466c, 0x0002dc7638167c3f, 0x00057a6d812df399, 0x0003edd89fe6ffe7, 0x0000825bd8675575 }},
        {{ 0x0002fcb419b70685, 0x0001a82d57c667f8, 0x000e490ec2d75faa, 0x0000b05ecc70312c, 0x00004cf30e7acd12 }, { 0x000d8db5533f5c10, 0x000a385ab45d44d1, 0x000243edce4fdb2e, 0x0008c0b52fba0722, 0x000052d22dab8a19 }},
        {{ 0x000dc5e37371a728, 0x00090d4e344a8732, 0x000dd070f1cabef0, 0x000cebf507d8d116, 0x00005db19373e506 }, { 0x00072b8ab82adaeb, 0x00047e6e799eb95b, 0x000eb16e5c349584, 0x000d038d5e9e5f4d, 0x00009e31731daff4 }},
        {{ 0x00089197287c3914, 0x00066e02cecc84ab, 0x0007c349eff41179, 0x0008b5b58600016d, 0x000061064bc7eb5b }, { 0x000c91e7660a8876, 0x00045bddafd60884, 0x00061f4025d3c53e, 0x0003b9a721a74014, 0x0000074a6b09578e }},
        {{ 0x000f2874056008c5, 0x00020ca08819a786, 0x00095a30051c1c8d, 0x000b3077dcf48b72, 0x00004df9fccd5a7e }, { 0x0007e550b59c1a48, 0x00012c226cd44ecc, 0x00060804d4595c21, 0x000eca3c588ae32a, 0x0000363ef9fbc480 }},
        {{ 0x00061b1ca7d7673a, 0x000e2f25c727bd2e, 0x000036ddf9088a43, 0x00035ff202e4badb, 0x0000a50c99cb451f }, { 0x000f2e941699137f, 0x000faccbd6bd51f4, 0x000131175819f7a0, 0x000db8f628024e40, 0x0000d71497878c8c }},
        {{ 0x0009f80d55b0c703, 0x0003dec0ccced589, 0x000b73ac42429524, 0x000510fc625cd4b9, 0x0000a65aaf5a02d6 }, { 0x000f34bb38b3eac6, 0x0007ac9ce6dc1532, 0x0007a93199e8a328, 0x0001c3b4d138d511, 0x0000ca319150839a }},
        {{ 0x000b70ab66820487, 0x000c8d22ae8921e6, 0x000e1dbaf67c7245, 0x000bcf9b2b039288, 0x0000edb438dd0e2f }, { 0x000b5629d34a9d60, 0x0004bc9e70574810, 0x000f912ab26e33fe, 0x0008be6824581dde, 0x0000b84bd043d606 }},
        {{ 0x000fbfb416c23b23, 0x000a49eb06f5b41e, 0x000d5e6480526105, 0x0001f4ea9a42e84c, 0x0000cccc1d701e42 }, { 0x000897f9fd8a78ab, 0x0003f239d33cd6f3, 0x000871d410fa9b73, 0x00055c4969caedb8, 0x000057a5bd2f9d4f }},
        {{ 0x000f306e8d3ef41b, 0x00096f88e5d788e9, 0x000842a42261b628, 0x00064b03c1aec7ba, 0x0000bdc61d3e538d }, { 0x0002865801e16113, 0x000ed6e5040a0acd, 0x000328c1172c7e8b, 0x000896723ca9d504, 0x000064b076c1e6dd }},
        {{ 0x000671c6f1b51569, 0x000c03f6967469a8, 0x00090718c7e7267c, 0x000b208222894d83, 0x00005bb66d861506 }, { 0x000fc913dfbae46d, 0x0008bd6d7de9853b, 0x0005f229a8e82949, 0x000b8e3cf2e3ce34, 0x00006ed57b771cce }},
        {{ 0x000dbe9e4865fdfb, 0x000959fe61f7908e, 0x000eadfddce11bc6, 0x00006b838192e304, 0x0000a93c2ce3f05b }, { 0x000ebd91e9d1f474, 0x00075be2db30660f, 0x000ec22deba43a12, 0x00039956eda3e613, 0x000030ac3120ad60 }},
        {{ 0x0003a058ce6c6132, 0x0009ac4df655a499, 0x000bcb11a94222b2, 0x000f584eaa9e00df, 0x0000155d23f64159 }, { 0x0003455bbbdf94ba, 0x000d64a67bd3448e, 0x000a104fa1a088b1, 0x0008da5524a3d4e1, 0x000055bd521fe9fb }},
        {{ 0x000381dd5350039c, 0x000fc5b4b9ad1ce1, 0x000648361a401b32, 0x000ae0816a981853, 0x00003aa8a7a07cf8 }, { 0x00041e541e27a4b9, 0x00054617a4a76209, 0x0005a59a402eaafe, 0x000f30a3ac841292, 0x000004cfa2d4a052 }},
        {{ 0x00061e15b0193f73, 0x00026e84a34f239b, 0x000c02e5ed252835, 0x000fb353215fdb83, 0x0000611a80a40f29 }, { 0x00053277336c58d7, 0x00082d4a127f89a5, 0x000bb2b59fa73e65, 0x000bfdf925d541e0, 0x000059c1e0662639 }},
        {{ 0x0000309532509fa6, 0x000fafc95f57552b, 0x000fbc7c1b975eb3, 0x0008218013294764, 0x000077c5af065b77 }, { 0x000e03f65c2dc5db, 0x000e2f505c2e54da, 0x000c78e4e6c62bf5, 0x000b84353158f27a, 0x0000694c1d649f0b }},
        {{ 0x0007332f3b061b87, 0x000c7bd9c95f0f98, 0x00019d30d836a35c, 0x00039dac45ed9deb, 0x000018b00ca2bbfd }, { 0x0005cb917ba93167, 0x000079e2b9e0d888, 0x0003d0d4cecd60b0, 0x00027c84852d910f, 0x0000f373e4a9fc00 }},
        {{ 0x000137a6d994065e, 0x000be934c8998d19, 0x0000e1d6feb6644a, 0x000fc3fb02186a3f, 0x00003352e74d4d1d }, { 0x000ca1b89b828df6, 0x0008937748bc82c0, 0x000173a4e6f0ad53, 0x000fb24a287c2e93, 0x000028fafbaf323c }},
        {{ 0x000737d6fb05a03f, 0x00004aa5e1cd3cd7, 0x00082350a52d4bcc, 0x00049e09f8939780, 0x0000f18ec22e9ed8 }, { 0x000429759f59ae9f, 0x000d56a60f1524b3, 0x000585dbb50813e4, 0x000e39fa536985e5, 0x0000b06736fc8393 }},
        {{ 0x000ac8aaf9b029b8, 0x000b6c3092e9f414, 0x0006175cc0c40ea0, 0x00083907d3a238c6, 0x00008d3d8a65f0c0 }, { 0x00090005fc438ce7, 0x0001ab15e4ccab05, 0x000244e145fefc14, 0x000a4a29d9bf430f, 0x0000798b9c3dca21 }},
        {{ 0x0000075aacd2e5f0, 0x000a12f24544cb60, 0x000ad7936836f151, 0x00072b2129987b71, 0x0000b2e6c26f30ed }, { 0x0007813de7e02a59, 0x000f78b6cac83696, 0x00078dce2ea1a080, 0x0009ad9a487842be, 0x0000b0205681558b }},
        {{ 0x00040896be942344, 0x0001dcec3427239e, 0x000fe1609e1ac9a8, 0x000f5d1915f3c7ce, 0x00006570948e3d4e }, { 0x0003d4708d9dd274, 0x000f3910492d25f2, 0x0009c36c11e8ef73, 0x000b536cc8aeb30c, 0x00006263417807f4 }},
        {{ 0x000f9b71b8c0f425, 0x00077f2d0c051995, 0x00029b4c129bcbe6, 0x0007e7c82502fbc2, 0x00002b22165d2734 }, { 0x000b6e3e8373047f, 0x000bb592b82dd077, 0x000519b4c4ab6dee, 0x0004bdb327630273, 0x0000227777e79e51 }},
    },
    {
        /* digit=32 (1,2,..,64)*(2^{224})*G */
        {{ 0x000c41114d0d0f0a, 0x00071c035d0b34a2, 0x000b56e6af5ad632, 0x000f458d1440b386, 0x00009070851ee09e }, { 0x0000477abf63470d, 0x000c1f1ad95a0b12, 0x0008478dc8a2c85e, 0x0009d79c9c09b37a, 0x00005669d660129e }},
        {{ 0x000a68df70882277, 0x00065b3292a92874, 0x00062d47b35717e2, 0x000498f05ddc15cf, 0x00002045f41bf3f8 }, { 0x000a8b9343580755, 0x000f94505bf7dd0a, 0x0008e243ec49440b, 0x000aea3afa4e63d3, 0x00000f5462133be9 }},
        {{ 0x0006113c503cd9d4, 0x00061b51e706ad97, 0x00044d98af8ed595, 0x00086b990b99cebb, 0x0000a86e1c215f82 }, { 0x0003cbb144e6b9aa, 0x000e4b097e2b5aa3, 0x000c2fed61bf9a24, 0x000125c6c7e1022c, 0x000044eec8aec086 }},
        {{ 0x0004e7b4f75c69c7, 0x0009d717af715d2e, 0x0003eb2b959f67ad, 0x000b50256e2f7f59, 0x0000faa39a85f847 }, { 0x000657624d4d6888, 0x000779788d5374a6, 0x00031a2adb0e9860, 0x0008607e22b915e6, 0x00009ed17ced0865 }},
        {{ 0x0007e49f538144e9, 0x0000801dace5aca9, 0x000179c203139436, 0x000579d09c4fdbcf, 0x0000b8c43e3ced43 }, { 0x000f036040802177, 0x0007090937e2ad5c, 0x0007636ab7b11bc7, 0x0009a49dc846e250, 0x000015f05617398b }},
        {{ 0x000858e1e42fa26a, 0x00096e07442f1d58, 0x0000de8801c7a755, 0x00023d647475607d, 0x00002129ca073288 }, { 0x0007cdea3e2c120d, 0x000050b8231ee10a, 0x000abbbc34902c47, 0x0009866a41b80e7e, 0x0000ea4fb6751c9a }},
        {{ 0x0005b9d57b4ca325, 0x0009b07f8ae7c38d, 0x000b67ba2c17122d, 0x0004048b36db07a5, 0x0000c13547ebaf13 }, { 0x000fe5a101822457, 0x000dbe78eba20f2f, 0x000e71d40250d287, 0x000744f58ef11ca8, 0x000067b29ced3d7d }},
        {{ 0x000af127793627f3, 0x0008811e51732d26, 0x0007ba47d495211f, 0x00011fbc5dfd6eb2, 0x0000d0277a7c5305 }, { 0x00059aba7caa2e14, 0x000712d06c425aef, 0x000b6687611ec682, 0x00054599c6df92f8, 0x00003f2120a92224 }},
        {{ 0x000eee4194a7b79a, 0x0003ed5da64309d0, 0x000682c4c56fb4d7, 0x000e3be0861a6de5, 0x00003c6f7e87d587 }, { 0x0000847a2186a0fc, 0x0009eab771e4caa3, 0x000d4837a79d473c, 0x0001408f583965df, 0x0000b258cc7ec22d }},
        {{ 0x000c06085b99b726, 0x00006f152bfda056, 0x0008f591b1313934, 0x000ea4c6fe0e4c4e, 0x0000984ae625115e }, { 0x000eea86f0bd9963, 0x00061d74463ba14f, 0x000201459c9b5827, 0x0008df06d9d292f8, 0x00000dd1c0ca9a79 }},
        {{ 0x000c48ec0b853213, 0x00079d366131e2e7, 0x000f04b53a77349f, 0x000e3ef89f31d974, 0x0000a4bfcdc5045e }, { 0x00080a6ebc348716, 0x000fbd8d1989aa7f, 0x0009f3eeb83d4cf9, 0x000be314ab34fa0a, 0x00001217dc8a396d }},
        {{ 0x000f4dca515118cd, 0x0009354afa05dd8e, 0x00010d61474d56e0, 0x0004bee5926dfcf2, 0x00000943d9f591b4 }, { 0x0009b587257c123f, 0x000448a65d306723, 0x0004c3aff7615195, 0x000c356c88b08d53, 0x00005b8eb7fb6fca }},
        {{ 0x0008c318b548154d, 0x000c3593de9abe37, 0x00012418eb094690, 0x000d8818c4043488, 0x0000566a78469087 }, { 0x00030ce97683fa8d, 0x0004216fde24c99d, 0x0004cfb615e6cd11, 0x0009b82b42a77cb7, 0x0000a14d76ffb4a3 }},
        {{ 0x000ac78848467605, 0x0004e931a62cc262, 0x000625f9580ab732, 0x000502ef323c69b9, 0x000090bdd65bcd79 }, { 0x0008b8974d570b21, 0x00057f9187d073d8, 0x00090dc6e92ff16d, 0x000634c8f6c2be91, 0x00009044aaeb8adc }},
        {{ 0x000c764803a21e34, 0x000dd77ff57d0517, 0x00077ef3d15d6733, 0x000b936d42ff3803, 0x00001b21b2a67101 }, { 0x000a30e0b2c6a5f0, 0x000e75f48879bc89, 0x000fbe9d8b0eae8c, 0x000e37c16ea7eacb, 0x00009ff4968601c7 }},
        {{ 0x0006605ecd65db55, 0x000d72e421d3aa42, 0x000cc1ef49735da2, 0x000798f1cf926407, 0x0000115826b66fae }, { 0x000b337ce7ef919a, 0x000a7a6d6a5eabf5, 0x0003c637e9a63491, 0x0006f67021edb84f, 0x0000746c950ad014 }},
        {{ 0x00034288799ea710, 0x0004c77c055dc14a, 0x0004de9614663360, 0x00056bbaf2a0ffa2, 0x0000e0884c9d70d1 }, { 0x000fd61e66f6e09b, 0x000fecf884655fb3, 0x0005f6771fda3095, 0x0003d81b7fa5b5bd, 0x0000cdb900fd5f4d }},
        {{ 0x00059455d0433f8f, 0x00013dccc78cf662, 0x00044a47a9a7232c, 0x0004db768b301817, 0x0000faaaf90a0ad3 }, { 0x0005deb6ca3cd246, 0x0001ec5cde3acec5, 0x0004bdcefbacb8b3, 0x0005df9657714192, 0x000045397101934d }},
        {{ 0x00068f895e9004fd, 0x000b957b1fa70fbf, 0x000ee177b0a26d56, 0x0009b2800fa07f50, 0x00006b90a3ae492c }, { 0x00066663c21efba8, 0x00078605cae2a6aa, 0x00075c20e4801115, 0x000e8d3d301bd721, 0x0000b7b5d77ccf5d }},
        {{ 0x00030d7189e49226, 0x000dfd73272d8383, 0x0009c755243ba1ba, 0x0009db432e22924f, 0x0000ce330e0db587 }, { 0x0000a894fbc9777d, 0x000292cbde1f7140, 0x0004e5228567b078, 0x000a5063235771c9, 0x00004ff8170ebc9a }},
        {{ 0x0002b9e66ddcbd1a, 0x000899b36194d408, 0x0004f4a73f93d7fc, 0x000121558e857a7c, 0x0000f54e23d16472 }, { 0x000c57761e509ec5, 0x000a434e4e8b9dd4, 0x000d57cc01b97144, 0x0009c1f3bd2372c9, 0x00008d5623e3a8bd }},
        {{ 0x000653a576829f13, 0x000ba0a1058a3184, 0x000531b195245a3b, 0x000277ae2369cf3f, 0x000016030f4bd9af }, { 0x0007f95f135e7fcf, 0x0000e5aaa4e5c975, 0x00015fb1dd60b809, 0x000e5f3835be80aa, 0x0000a477556e695d }},
        {{ 0x000d4690afc07e00, 0x00014e24415503b6, 0x0005df748739e313, 0x00060c966c08ff7c, 0x0000127236d69e17 }, { 0x000d769c04659257, 0x0008c2eea74237b0, 0x000ebfe0b48a99dd, 0x00039e9fec436d17, 0x0000486dc5e3dd88 }},
        {{ 0x0001787c9e286fcf, 0x000e7de34eaacda9, 0x00079a28754099d0, 0x000d3bf55d9e116e, 0x00007562dbb12950 }, { 0x00014f95d0e6cf70, 0x000e32bf129ac2fd, 0x000788e74c5ba2af, 0x000d048f3d307b7c, 0x0000f8a713e0a48d }},
        {{ 0x000d063b675b0d68, 0x00068296ae550430, 0x000f37a2427aa344, 0x000d796e28931e98, 0x0000fa3386495eae }, { 0x00040f65cbe226f3, 0x000a5b9ed0e4316a, 0x000a8c5b647a8e82, 0x000b3a4c7e74a736, 0x00005f2b0889266d }},
        {{ 0x000497e5c3ee0d06, 0x000300416df4285e, 0x000aec2d851e2818, 0x000b8c9c3b0c9bab, 0x0000197c784b55e1 }, { 0x000c8f0069e898e6, 0x00085b4164bda960, 0x000994284d372970, 0x0009f40ba1ee4241, 0x0000ca6bcd47d957 }},
        {{ 0x000394f375958bf5, 0x000f481c27253c98, 0x000b0f1e76cff0db, 0x0002bc64605a6aef, 0x0000f0b8234cc107 }, { 0x000753d452dd6652, 0x000759c5f109d0fe, 0x00057438f1b79715, 0x0008a4c68cad88a7, 0x0000fe76d824187b }},
        {{ 0x000c362a70e2529b, 0x000d9a15b0ec6f5e, 0x000366538847e1f8, 0x00084617347989fe, 0x00001b6a7bf51376 }, { 0x000a56149e2522b2, 0x000e6a7b3a8ad418, 0x000ff10dc6b2e1e9, 0x0002983f473a20fb, 0x00006887f8bed5cf }},
        {{ 0x0001f4d76b4d26fa, 0x0007e819ac6e0f4f, 0x000eb21ede824bca, 0x000d9b6fd360fdea, 0x00003d6157db8955 }, { 0x0005570e9aed5fe8, 0x0006909587f76b89, 0x000c8dc0ef1ab00f, 0x00008c8958b4ee08, 0x000053e011751c9f }},
        {{ 0x00043ef52d2d581c, 0x000a76f35d33ae72, 0x00032c7c6567763d, 0x00057cc4b200ea12, 0x00009729e0b08fde }, { 0x0008914cb19c5b91, 0x0003477e23ae233d, 0x000ecbcf853a9cd9, 0x0009ee5a4b93807a, 0x0000427af498005c }},
        {{ 0x0008f30be32aa9eb, 0x00099943fb42622f, 0x000b5715fbf20368, 0x000ec01633b2700a, 0x0000a02527da25ef }, { 0x0004d487b147ea98, 0x00060d269e55f156, 0x0001673e9c38aac8, 0x000fb56f61fead85, 0x00002fdd4c705f91 }},
        {{ 0x000add3db7ec68b8, 0x000b386d23ddc820, 0x000c732406385f9e, 0x000123ac503fd344, 0x0000078adedd4745 }, { 0x000755e7ed4c6729, 0x000153f8260e01bc, 0x00048d4be4a45000, 0x0008bbb33fbeb49e, 0x0000d816465d0546 }},
        {{ 0x00045bbb58e226bb, 0x0009639296077453, 0x00006e136fcfe782, 0x000a03edd0ca0742, 0x00001978f2908e69 }, { 0x0003f5cdb0aae739, 0x0006a8cbe0f27a29, 0x000bd029fda61b77, 0x000fb8e5767c24d7, 0x0000f943175bf6fe }},
        {{ 0x000c4647529cc278, 0x00031d7c5ee30a1e, 0x0004ccc36fd68fe3, 0x0001c8748d159363, 0x00001530c91aa13e }, { 0x0001789e131a19dd, 0x00082f39e49af6a5, 0x000a1dc463a96fbc, 0x000b4fb43f2d6238, 0x0000ead936950af1 }},
        {{ 0x000ce1ad11ac24e3, 0x0001629763231dff, 0x000ceaea30de2a0f, 0x000017aa846df9f7, 0x0000e2b10bbfaf1f }, { 0x0004deb125091869, 0x000a4f6fc6a0a7e8, 0x0007e8c574b783f8, 0x0001c41bedd431b9, 0x00002151fe66860f }},
        {{ 0x000e9dd70df9b33d, 0x000b4693c2a59987, 0x0003faa941ff8de3, 0x000a633149b9247b, 0x0000fbd0cf3f9717 }, { 0x0006a6e285611fca, 0x000aef7cbcd5e64a, 0x0006f52dddad338d, 0x000a258ae0359ac1, 0x000096f58aa22a1b }},
        {{ 0x00073f78f0342793, 0x000723f24cea1a04, 0x0009ec229ba14f6a, 0x000120be9c97bce4, 0x0000c16b99f54401 }, { 0x0001a21357d12b82, 0x000ade0f63d3d0d4, 0x00006e094fa470ab, 0x000ea0183e620b88, 0x0000c42192da71c1 }},
        {{ 0x000a9e41a1d48ae1, 0x00001fac7dcfcabd, 0x000dbaa89e6546d4, 0x000254d07650acfd, 0x0000deba6e56a7dd }, { 0x000c591b9fe8b554, 0x0005327175a54ab2, 0x0000be0471701886, 0x0004842a9ca0cd19, 0x0000e042468b20ad }},
        {{ 0x000a78a81b42cff9, 0x000dbe42bad4d5f9, 0x000064775d802ce4, 0x000422994fa4c359, 0x0000ec1267726fb2 }, { 0x000cc8b1f090da45, 0x000c079a98e84e7f, 0x000f4ccf33c47417, 0x000479ce4fe1242b, 0x00008d7b8bba4b39 }},
        {{ 0x000700ef747f8ab2, 0x000f10292d0487a1, 0x000c124aa28a6283, 0x000e05c8f1442dbe, 0x00008c88040d8a3b }, { 0x0000b08ab557d66b, 0x00094c370f1fc62b, 0x0005b91712159de5, 0x0001fc90d3b31d4b, 0x0000bf8fe20af650 }},
        {{ 0x000b9ed428d2bc49, 0x0000e01ae3400501, 0x00060151f1407234, 0x000255151ba4b301, 0x00005711c0145b29 }, { 0x000bf2831a40ba61, 0x000844eeb4819f5f, 0x000ca2263440d5ac, 0x000281a0b616cdfe, 0x00006655f6b3a3fe }},
        {{ 0x0003e7363ac7c9d0, 0x0001c85e0e60d843, 0x000fe9d0b83e0ebc, 0x000727d6e68037e2, 0x0000a4ef48579238 }, { 0x0009b9aa509df363, 0x000d76108d0756c9, 0x000b95f568fafb67, 0x000401523d727f85, 0x000004ff00a6a67f }},
        {{ 0x0000cb3198286b38, 0x0007904e2b4e075d, 0x000472b1178e49c3, 0x0003e802d4998729, 0x00008d0984c74c1e }, { 0x000afaaaa6c45a17, 0x000f87dd7808c569, 0x000eb6fdeccc2f83, 0x000631aaab5c54d8, 0x00005179697da3e9 }},
        {{ 0x00019ed11ae91fb6, 0x000a82104cf667af, 0x000aab9a64bcab50, 0x000e9707cfc2a811, 0x0000554d8be76583 }, { 0x000d6d34f69204ef, 0x0008d09067b6269c, 0x00028f41c0a35e1d, 0x000eaffbc967b853, 0x0000dbd9f34c4cd6 }},
        {{ 0x0000c8e44f888e23, 0x000b2d36cceb3707, 0x000936c497a9954a, 0x0002857622aeac57, 0x0000b6752a402de8 }, { 0x000c4acd0404bf66, 0x000a35086a34d092, 0x0006bac87c03cd3c, 0x0003e6d15e0fca25, 0x0000b1bac43b987c }},
        {{ 0x0004309c3d5be708, 0x0006d173b5a9ba6e, 0x00052eb47e145ec8, 0x0002e4097b9cbe1f, 0x0000dd504546a602 }, { 0x000270b2a44f2d47, 0x0001105f67f37466, 0x000a40f34f5ba628, 0x0001c535385fc37b, 0x0000ffcc4377268a }},
        {{ 0x000b87878a9289fb, 0x00069f5974ad33b0, 0x0007f2f3a6671d24, 0x00026447304d8597, 0x000094b1402774b8 }, { 0x000a86f6939e30ae, 0x00091015a0617323, 0x000e68a3226dcef5, 0x000e4a218c25430c, 0x0000225b9562aa3c }},
        {{ 0x00074a9e86f273ac, 0x0003768da9f3804c, 0x000dda1996154227, 0x000ef0ea5470f07f, 0x00007a00585a4292 }, { 0x000a3d7f108e6847, 0x000633543471a24c, 0x000deddbc6014539, 0x000748d4d239446b, 0x00003d82eda4eff3 }},
        {{ 0x000052b13e4c4fc6, 0x000ea85fdaf45207, 0x000fd4f729b3d2ed, 0x0005d9642bb6d3e7, 0x00004fd193b24bcd }, { 0x00032412778a49c4, 0x000491b682b90604, 0x0003691032b4873f, 0x000ee41b4a71d388, 0x0000b5eb5db6e3ed }},
        {{ 0x00089111e4596bb4, 0x000456e4901bbed5, 0x00009ba190d798a1, 0x0003e197d6419c71, 0x00002d587b9dd586 }, { 0x00051bb0af6d78db, 0x000bf337b60554e1, 0x0007d8c6d2f4ac07, 0x000c32235ea5f888, 0x00000285ce6252e2 }},
        {{ 0x000dd50e6646b5c5, 0x000ca7443b30ca81, 0x000abbbe371cf9fa, 0x0005b4bade10b5bb, 0x00002c1185fd706e }, { 0x0002c679e115a447, 0x0005654cff2b821f, 0x000c54619d7f138a, 0x000808fe33882555, 0x0000813626061721 }},
        {{ 0x0006e2ff3e3cf7e4, 0x000a4156d2a57687, 0x0001a53b7eef03f6, 0x0005810437f227bd, 0x0000890e9085f693 }, { 0x000ab8ff4c5d4c80, 0x0000d6fa3177ec75, 0x0000c87000a8d96c, 0x0006d958ebf731d5, 0x0000d478202f55ec }},
        {{ 0x0000a38f5769375d, 0x000035635edc56d2, 0x00095df07e21caaa, 0x0004f10cb5a69d34, 0x000046b4e39da99a }, { 0x0002061ea52a1372, 0x0007590fac3a15b2, 0x000e58496fc6d1e0, 0x0002758866ed7ea2, 0x00009cae713eaa92 }},
        {{ 0x0009654db53619a3, 0x0003ce6f546783f8, 0x0009d0a51668ee67, 0x000c4e665d38bc6d, 0x000084a7bd921bf8 }, { 0x0005f504d34681c8, 0x000f79a26e1adee2, 0x0004113d9000faba, 0x000febbfe291f376, 0x000014210635841d }},
        {{ 0x000e987967ce4bc6, 0x000c85ec3a0ee43c, 0x000999001f97950e, 0x0007a775670aaebc, 0x0000679e9d41df8f }, { 0x0004eac8b0e95f7e, 0x000ed195e26310aa, 0x0003933fc3f81098, 0x000d5b1e49e43548, 0x00002169b080d0d8 }},
        {{ 0x000fb265b41cf750, 0x0002512c1ad8cf84, 0x000e4a2e50e3a4c2, 0x000f10df0193b4e9, 0x0000b160e21161e4 }, { 0x0001099ad8ca10c5, 0x000b8cde51e1ae29, 0x000652cf15bd680d, 0x0004f84988e3a8ca, 0x00007fc6eb96d788 }},
        {{ 0x0001c9c40bf7f829, 0x000d01c93aa96b8d, 0x000aae006a072e1f, 0x0003a5e230fc8716, 0x00002dcd3ed4529d }, { 0x000363fc5ef58287, 0x000778642b7c094a, 0x000c71dfc3c2c438, 0x0003c765d1873439, 0x00005e33d9307f67 }},
        {{ 0x000b6c8350158412, 0x000200df862eaacc, 0x00025759c48cda62, 0x000c2c22814419c6, 0x0000de581dff2c38 }, { 0x000c9b41c4227517, 0x0002345d30f917c7, 0x00091bde6e29ed24, 0x000dae963d4c4b04, 0x0000c95bedff5a4d }},
        {{ 0x000bfb7557c62b30, 0x000cc2f2f8ba2c73, 0x000f406c34d0f5d5, 0x0004d39720966eb2, 0x0000f79e276fbe94 }, { 0x00021e887875bdea, 0x00065235a2b96204, 0x00072d90be459d5e, 0x000593923bff3ed2, 0x0000becee3d55908 }},
        {{ 0x00064e33579d3be5, 0x000737a25a416322, 0x00047d0d393b726c, 0x000afd3dbc081412, 0x0000f3a2065b9dbc }, { 0x000961bb956822d9, 0x00083cdc87c51e26, 0x000ae6bca67f4701, 0x00015a9ce16b3015, 0x000099b9f2e2caa2 }},
        {{ 0x000154c9356b909d, 0x000be8deff7a3a0d, 0x0007e08aa9f0cff1, 0x000cfad1fbec3373, 0x0000ce02e948472d }, { 0x00011828d5dd509d, 0x000b36f1cc7a906c, 0x0001ab3de5b9cee2, 0x000ed9c4473bb539, 0x000009806170e848 }},
        {{ 0x00085a892ec2931e, 0x0002de13e4b12ff9, 0x0008952ddd4d7a3d, 0x000ee02cfc3aad9f, 0x000060ef6671c961 }, { 0x000785426e98218f, 0x000231b3bb85fa06, 0x000026771ffb6393, 0x000c11deaf95375e, 0x00008e2bebfcfde6 }},
        {{ 0x00099fe2c7f70e55, 0x000c2c8d8ceb147d, 0x000ddd6a728c6e8c, 0x0005d71349b649ee, 0x0000f688ee12edb5 }, { 0x0003baeb490b7ac9, 0x0001e628d3abba2e, 0x0006c3dccc50256c, 0x0001b7d4d18fee6e, 0x00005a845d13c4bb }},
        {{ 0x000157c30c88e767, 0x0007447f23206d55, 0x000fcbce3e45a30e, 0x000a8919a2f6d341, 0x0000644f481a46b6 }, { 0x000508455987e93f, 0x00086c52d4fb936f, 0x000bf1494782ed2e, 0x0009b3b64ef22f7a, 0x0000e271957d8d37 }},
    },
    {
        /* digit=33 (1,2,..,64)*(2^{231})*G */
        {{ 0x000595cd1fa2a781, 0x000d8d8df7304d44, 0x000bb98b416f08ee, 0x000983c60b71bcf1, 0x0000fe06b3f76c34 }, { 0x0009429622589d67, 0x000cb9a4835859dc, 0x000cb478d834436f, 0x000f5234e4a0f0d2, 0x000076555e5f3c86 }},
        {{ 0x0004c9b20d431706, 0x0002e62f23f2d925, 0x000d0bef8e6b4e0a, 0x00055d8206cab71b, 0x0000c95a2a5e55d4 }, { 0x00031469c615f1a7, 0x00031aa9f2ac02f8, 0x00069d5a83ea26e1, 0x00016bd3403f8e61, 0x00007f5ad3cb507d }},
        {{ 0x0006930a553f291e, 0x000c1248f9813540, 0x000c2ef5980445f8, 0x000777b2c587e23e, 0x00005d0484062c98 }, { 0x000c4f3993ac460c, 0x00001ebbe5bbe21a, 0x0004784247eb5cec, 0x000c04560a7ffa5b, 0x00003ce6bf87455c }},
        {{ 0x0008b75007f2d3e8, 0x000b9583ae9c1bde, 0x0007fac5923887d3, 0x000d750b4e0af6d9, 0x0000d1ef5fdad135 }, { 0x0004069ea597b54b, 0x0003bafe02358f2b, 0x0006368b73835819, 0x000671cfc31b8b85, 0x0000fde8d8c56c72 }},
        {{ 0x0002f45c13822eb4, 0x00022e27fa03cb3d, 0x000a3ce92174b004, 0x0009d485ddb938e5, 0x000021b3fc2ab9aa }, { 0x000cc71d2d719fc7, 0x00059b57883e6142, 0x000891b541f8e786, 0x000c81ae35fd27e6, 0x0000dc5876ef7ba3 }},
        {{ 0x00028ad2a84e6c41, 0x0004fcde36d07576, 0x00017717c0d9fc24, 0x000484cdff722d7b, 0x0000ebe7611ea3b6 }, { 0x000c336b09195735, 0x0008ab520226040e, 0x000064d14bbbb3e8, 0x000ac8c886c34ecb, 0x0000b85de43f45af }},
        {{ 0x0001830b48c43912, 0x00050a21e206ee56, 0x00002ac69aea8304, 0x000bda0c3f1e2748, 0x000032ea619c2425 }, { 0x0004c11642e0594c, 0x0001c46e78e815f7, 0x000ed740f1e78774, 0x000d57765fe40139, 0x0000edbe53f588b4 }},
        {{ 0x000dcefcddd2868c, 0x000b79e558df0194, 0x000a5ea22e49ccca, 0x00091ce24230da4b, 0x00002dd640a90582 }, { 0x00072d824d931811, 0x000feb2a47d4c5ab, 0x000efbe07e5114c0, 0x000d6c17355ac9ab, 0x0000fcd91a520be1 }},
        {{ 0x000b1757ee1625e4, 0x00036744513e2cae, 0x00014d0cb33eb84c, 0x00094c0500b41861, 0x0000123028257abf }, { 0x000558a2a835c536, 0x00008283dfac6d55, 0x000df349954d38f7, 0x00030de99b2431e2, 0x0000130b37f17b1d }},
        {{ 0x000b8b3bb657e0d1, 0x0002823cbb13d1b0, 0x0002507060487a33, 0x0003073a998799f4, 0x0000a4673e731318 }, { 0x00071b9484805f36, 0x000b92e3ecebb211, 0x000b1f5665228e4a, 0x0000fe71f17cb6c4, 0x0000fab132e7caf2 }},
        {{ 0x0008f096fe69263f, 0x0006f2d7ef0329c3, 0x000258d3a92be87e, 0x000ae1cbc0850922, 0x00004f94dbc53746 }, { 0x000b4c0dfa96c3eb, 0x000a074a1dec0270, 0x0007447bcec03082, 0x000dc7853e8cf10e, 0x00006175addc465d }},
        {{ 0x000e8ef838c9fbc1, 0x000c452de7e5c194, 0x000973312d33ad9f, 0x000d8b33dfab2860, 0x000073e06257b5e1 }, { 0x000ec0507e35022e, 0x000df264cb1bf3d1, 0x0009551e8ee3b962, 0x000b4deed4c1f9d0, 0x00009cc539a7598a }},
        {{ 0x00038fd905d568c2, 0x0009ac7cde450794, 0x000a0ddaf0def815, 0x000076f5cde173c9, 0x000000e3536d8eec }, { 0x00053add2c142e87, 0x0003e06c7d9630b4, 0x000baf4270493d17, 0x0006bf9e86457a87, 0x0000eaa209f092f9 }},
        {{ 0x00094abebcf27683, 0x0008e09a607419d5, 0x000262210bc3d71d, 0x0004daef3faa71e6, 0x000036cd41505615 }, { 0x000a29a36702adec, 0x000ba91eb78399f1, 0x00052519283d6b50, 0x000ce18f048aafb3, 0x000081d651ae4804 }},
        {{ 0x000568d2e1c64bfe, 0x00042e51c21f2bf8, 0x000c1f9ca80c3bee, 0x0005f092ef99a513, 0x0000b0e4e1a943f6 }, { 0x0007a6dafc4f015a, 0x0003aebd3f69d9f8, 0x000280091fb54144, 0x000e1d6f4181a973, 0x0000bdac310c7a67 }},
        {{ 0x000a55b52ea9cdb8, 0x000fab5ef7d92893, 0x0007df294c2baacf, 0x00036dbb62480d4a, 0x00005fd14d5bdd1a }, { 0x000d5e9d7d31477a, 0x0000715309eadb09, 0x0009f58728a80fa4, 0x00022c35adea7de5, 0x0000ddf8bf0e739b }},
        {{ 0x0004d8163beea5c0, 0x000fef5206d3ffce, 0x000298a8759903f1, 0x000a735ba23d5e03, 0x0000c5541d12d6e3 }, { 0x000219da8243f413, 0x000d6efd3082e8e5, 0x000baf7918ddca19, 0x0009adb0a4fe1246, 0x00009d116a39ce07 }},
        {{ 0x0003fcac89cfa8ee, 0x000b7c1f7ea25aaa, 0x000f57b3d0d5ac4d, 0x0003e9871d165e6b, 0x0000a30effd4d7ba }, { 0x000e263feae7d9d1, 0x0009ed4f0baa8abc, 0x0001149b82faf7d1, 0x000fe87495a4adac, 0x00006b868aef4658 }},
        {{ 0x0005547c4f5be3f1, 0x00012d307032c722, 0x000e53b4ec796f84, 0x0000587f07f40d5c, 0x00006bdd6dfb6e81 }, { 0x0007df465d70f0e6, 0x0009ffd65fb2a9eb, 0x000e66d9d13f40f5, 0x0005f5e42ce69336, 0x0000c86a6ebbc9a1 }},
        {{ 0x00058de498e2b259, 0x000aba3315980cd1, 0x0002e4ae075ed625, 0x000a16c04693bc49, 0x00003b029d23553f }, { 0x000836ce7e2b454a, 0x000abc7cfda83685, 0x000320a048ffe7ec, 0x0007c9a6ed2d7bbb, 0x000066b26f8ebcc8 }},
        {{ 0x0002b52f65c725d1, 0x0007e1994f834cbc, 0x00037fe9a81cbbe2, 0x000bfb143d35653d, 0x000092a9b3930992 }, { 0x000f186cdd7031a0, 0x000821b18f23d962, 0x000c1039e5d77384, 0x000a47402cb05aba, 0x0000c20b19c6ae17 }},
        {{ 0x0005624623a128a9, 0x00043e59a29a5c5b, 0x0006860a3ddf3f1a, 0x000b07d84b19b3c0, 0x000095b53ccaa378 }, { 0x0005a9d0868addaf, 0x000bf653c04fa024, 0x000c4900692381db, 0x000c3b43f83a7176, 0x0000f9656734975f }},
        {{ 0x000b179393f74d6f, 0x000bd1d3dc090bc7, 0x000f2f422dd3e906, 0x0009a8d4d82c996e, 0x00000d602d115417 }, { 0x0007c5dff255af16, 0x0008825f8c5dbfdc, 0x000d576c9b126092, 0x000dee6a84d9d7a1, 0x00005c2588a5a8d3 }},
        {{ 0x000ea886daec0451, 0x000078d04c82bdd5, 0x0004998917222f76, 0x000e744806800553, 0x00005c6be61fdaec }, { 0x000de96ef613e33c, 0x000ce01252a0b4d7, 0x000db80131abd46f, 0x0000659631bffb4f, 0x00007cc0d1f5bbc5 }},
        {{ 0x000fe64809d55519, 0x000140cd77d970c2, 0x0003a358b1c17566, 0x000cc06e903cfb93, 0x0000d055c34cf7b5 }, { 0x0003d31b05a74dde, 0x0007824c2ac98ccf, 0x000f2666232e242f, 0x0001827212a1372b, 0x0000ee7c4caf1682 }},
        {{ 0x00061891a0a38e82, 0x00067df5ece6e7ce, 0x000e299469b4ff0c, 0x00013b1a50603ac8, 0x0000588d5e4bda3a }, { 0x0003b117efd74152, 0x00051d299bf1603d, 0x00091dd30d6b1e14, 0x00058ec021f43afc, 0x000028e06b2d515a }},
        {{ 0x000103fa05977e5a, 0x000801365d1d131c, 0x00070276d6394917, 0x000ae7edd86f1636, 0x0000eb35b19898ae }, { 0x0001ae47f2c3b604, 0x0003f44f4b478f01, 0x0008cf69819a940d, 0x00069bd2ece91edd, 0x0000bdb21f33bd5f }},
        {{ 0x000d53ae910e4572, 0x0005d8d20f8a2425, 0x000d94da6c47cb3e, 0x0007f8cb37aff5b1, 0x00007219eed53f71 }, { 0x000e86515094eea4, 0x000d505ec0418032, 0x0006e5b9f737bc0c, 0x00073e98f85e3c97, 0x00007c74d04d9d17 }},
        {{ 0x000a04ca6726822e, 0x0007ca46a67a6b02, 0x0005885821d9ef2a, 0x00044421ef4cdee2, 0x00003f466bb9f3be }, { 0x0004a874138b9a64, 0x000ec18ef0543fbf, 0x0004dd08640a5f75, 0x0007b6f33f709a4f, 0x0000bdf2825ed29d }},
        {{ 0x0007b80581df3f89, 0x000a6a74d82f4c2b, 0x000fb04cea31d778, 0x00083b95e90725a5, 0x0000bfe40a6fc4ce }, { 0x00087f6f736e6376, 0x000631cdd1fc74cd, 0x000b0f04375170e3, 0x000a155badc1c62c, 0x0000d6efb5c07fab }},
        {{ 0x0007be9ed2271299, 0x00036c1317f4a765, 0x000b3944ccfbdba2, 0x000d491568d2703e, 0x0000e1677466f47e }, { 0x0000d9e943522c82, 0x000c371ef0b17acc, 0x000e183e31f4b70a, 0x00066db0e74b2655, 0x0000795a4aa76537 }},
        {{ 0x0008d2fe5724a2d6, 0x0008a411e84e0e5b, 0x0002f5d04e3bf4c5, 0x000d0d132a5db84d, 0x00001765a592c24b }, { 0x0008b4a422ebc11c, 0x0004336f3eb82fab, 0x000c454ae73559b6, 0x000b3a5108cb20cc, 0x0000bc49662e3c97 }},
        {{ 0x000ed06e285bdac8, 0x000fbea6fe8208b7, 0x0003a99e0ad61a08, 0x000fe721abd74b9e, 0x0000d5e8ed41f37b }, { 0x0003186b23df1bb9, 0x000a1586be8b800c, 0x0001152e70049303, 0x000ccd5d3428299d, 0x000034b1cedc477c }},
        {{ 0x00017aae8c429f29, 0x000039e8977d99b6, 0x0001227a835b6392, 0x0003488d4be94433, 0x0000ff63573b35d6 }, { 0x0001efae75205711, 0x0002ec4282aec3b3, 0x000fc5c949ae1fbe, 0x000ade84cc6063d2, 0x0000bb575870e312 }},
        {{ 0x000184a526e63b6f, 0x000fd6d40a9b97c5, 0x0002fde1bd2fd49f, 0x0004943e314702c6, 0x00006773701eca6b }, { 0x00059cd28024561a, 0x0003937003c32ea4, 0x000cdabca247db9a, 0x000b7bd1a5aa374d, 0x00006c792f7fd426 }},
        {{ 0x0003e1589a804b9d, 0x000e461459c2b92c, 0x0007b20d365b2696, 0x00064c6102f020fa, 0x0000ca6d9d3092b0 }, { 0x0008466ee83f88e8, 0x0006bfeaa3f451a6, 0x00035a933dc49cd6, 0x0009efd8b3a3c7f2, 0x000087060571aa2b }},
        {{ 0x00013994bea33721, 0x0002cd41fa266138, 0x000b3c8908a4f4fb, 0x000dbb557a2dc6da, 0x0000d41b64ebec33 }, { 0x0000b65bedddf81b, 0x0009c22fd4f75534, 0x000bb2ed0f327ed3, 0x0005d82f55135ff8, 0x00001de1ffe7c18b }},
        {{ 0x000331c0941a20d5, 0x0001da43734e5205, 0x0002cd6a1c26bf6b, 0x0007386355e3abbe, 0x00003e3eb679bfa2 }, { 0x0008f2c0da9416e2, 0x0001c2976b60c80e, 0x000a91db561dbd79, 0x00014b5842285593, 0x000090564385f22a }},
        {{ 0x0005c558090f5634, 0x0003d74c1df065c4, 0x000b19a020f15227, 0x000e520a8afeeb5a, 0x0000ebe08fef2016 }, { 0x000ab226cebd5622, 0x00094c0ecc8c1778, 0x0004ced0c5db8594, 0x000b1cd8e799ea9b, 0x0000fb29019d75f8 }},
        {{ 0x000a741ef1df7c17, 0x0002ae57ea973dc8, 0x000f9614703c94db, 0x0008185f15c00ca6, 0x000063d1e4dbdd66 }, { 0x0003b6c6786317cb, 0x000512f196b4cec3, 0x000395c7df9fe4ad, 0x00099bf2b70d08e4, 0x00008cc26ea5e3b2 }},
        {{ 0x000ef2c5fb93c237, 0x000adf807ac97946, 0x00004b14057e32a0, 0x000171de51eccf05, 0x0000be44bff72854 }, { 0x00006e0b10d8d552, 0x00011d2679784532, 0x0005fcf73fe7d9eb, 0x000e85837a0cf379, 0x0000f1351989d9a5 }},
        {{ 0x000c6469812142a9, 0x000055fd5ecfefc4, 0x0008765b8f7b26db, 0x000ef9c9607084fd, 0x0000b7551edf99c8 }, { 0x000ae28483d757ee, 0x000bd4ed65abfc39, 0x000e78719a43c760, 0x000116212e01cb91, 0x00006fcb78045826 }},
        {{ 0x00091c285c2a4c9a, 0x000b29ea66d80a18, 0x0003ecd9d0b57b14, 0x0006db5c1680a3e4, 0x00001cad3822f1a4 }, { 0x0006f8b13781c29d, 0x000dfeb799ece3c0, 0x00093ae7ccf04903, 0x00021942329b910a, 0x0000f5e1879fd271 }},
        {{ 0x00043be0aaf3f89b, 0x0005ffd385470908, 0x000f01a760b26589, 0x000ea802719ab8b9, 0x0000ed6be57be9d3 }, { 0x00085f39d0ed1528, 0x000c543a6db836ca, 0x000359452235069c, 0x000ac7b772a1330e, 0x00000393cfe1f4a8 }},
        {{ 0x000ad599c449623a, 0x000546d4ae528342, 0x0009494e42ebf43f, 0x000d510ef5a403b3, 0x0000d5d9e1fe46a8 }, { 0x000827cf81a805c4, 0x0006bb9014b3bf81, 0x0004de4580857334, 0x0002e3d027309268, 0x00005f977d08a51c }},
        {{ 0x000b5212970e3e2d, 0x000d8f6f10ff9276, 0x0000b8218d2fb9f7, 0x0008395ccd387145, 0x00005c10024eb9bc }, { 0x0003ea9e97f9fabf, 0x00005f96e3cb4a55, 0x000123a0215f0c0c, 0x000937b45e47d3f4, 0x0000f7e86aadc6c4 }},
        {{ 0x000a9b83894c7656, 0x000ba6e39cbb688c, 0x0002289ce9a8fb03, 0x000435e21532401d, 0x0000c8103121db65 }, { 0x0002767c4b7d6814, 0x000174e2093913eb, 0x0001b74604c733da, 0x0007f9ec7533d2f9, 0x0000110a91af736e }},
        {{ 0x000581cf9346f15c, 0x000a387aedbae9f1, 0x000921063ac18234, 0x0005bb2717daf0eb, 0x000088c3f46a6255 }, { 0x0003d318ac75f3ac, 0x000fc873ac1f4527, 0x00098797d751c8f8, 0x00007e1c3b417fce, 0x0000aca7b2139a79 }},
        {{ 0x000f8a35b721d031, 0x0000f0f728693e5f, 0x000a340ef236c630, 0x0001a6810e87d814, 0x00001fb62f488436 }, { 0x00014052c2a4d35e, 0x000f5d77f200d34a, 0x00076e24ea1e5f64, 0x000cd0579470f1e6, 0x0000eb853f10fab4 }},
        {{ 0x00078fe4024f3dda, 0x00020b526bc27979, 0x00060261836f1c35, 0x000006ed113670d1, 0x000079b7c4f64fcf }, { 0x000768d7e586bd52, 0x000a992d41c5dd45, 0x00077049a5bfcf2f, 0x0008264065ff8e24, 0x00002216a5fc8c7a }},
        {{ 0x000c3672af71c96a, 0x00080a360017a5ff, 0x0003a02e5bec495b, 0x0007474076b18e6e, 0x00009937b8bda59e }, { 0x000c948f1455702e, 0x0007f77d06283a1a, 0x000038410e854096, 0x000d0b9c2542fc65, 0x00006954911b34d4 }},
        {{ 0x000fc87df6be7f7a, 0x0001c82b04465c31, 0x000a44a03f1eaecd, 0x000b2137ee3d062f, 0x00008c43ce25b5a0 }, { 0x000f663391d373da, 0x0006ca80591f6528, 0x0006c92fee18905d, 0x000bd5a8ca861b8f, 0x00006b9e0397c8fa }},
        {{ 0x000aab4e6a3238d0, 0x000a8acb591ffa38, 0x0004c2349d649d6e, 0x00015d6b3027ca9c, 0x0000be661b873b12 }, { 0x00068aae83dc529a, 0x000564fb474d6b85, 0x0003add089d147e5, 0x000f552aa6597ffd, 0x00008040d2d5caed }},
        {{ 0x00043d8baa45a8fd, 0x0007164c73b23834, 0x00078d3ffbb7c8ed, 0x000f760ab49eb344, 0x0000e0df7be97747 }, { 0x000bf53de2fa0299, 0x000caa5ea7610d6e, 0x0004e2276b961c38, 0x000b80832e47e418, 0x000046bc71e68286 }},
        {{ 0x000de1666fe499b7, 0x00045b4d3c8c1cab, 0x00029f234c99a8a2, 0x0004e1c950d914ef, 0x0000577ca7b4a559 }, { 0x000e9faf6f83c4d3, 0x000f76ed449ab198, 0x000259fa7703581c, 0x00006a7538902b32, 0x00006ea68c6f0f57 }},
        {{ 0x000da8564dcf371f, 0x00016da9a85788bc, 0x000def5dd0869c6e, 0x0003cbd64d21f03a, 0x0000cb0b2e54fa19 }, { 0x0004ee3dffcc2d2b, 0x0004f5cde92c68ca, 0x0000ce43fec47260, 0x00053d47c78e035a, 0x0000a966314f762c }},
        {{ 0x000e2ab3decb0797, 0x0009e07af4e18f85, 0x0004a8483f498d7a, 0x000959a86967ec9b, 0x00007b0ac9fcfa8a }, { 0x000c0c1a5b8ce6b8, 0x000b64d11c226dff, 0x00028b6615b91ec6, 0x000a19f19f9783f4, 0x00001a2b243e738d }},
        {{ 0x0002f2a62f45d86b, 0x0008149571d92ac3, 0x000e139fab5045e2, 0x0007639af569e85f, 0x0000dc961974e55b }, { 0x0006e5ba0c5b997b, 0x000533a06d89b85d, 0x00095911bffc3d29, 0x0005e1aa2b90741a, 0x0000a4a9e49fefa6 }},
        {{ 0x000b842ac7fd440d, 0x000775112e4c9366, 0x000d5187b6b3804a, 0x000e21ec9ae419f7, 0x0000853743dec424 }, { 0x0006ae3fc4509621, 0x000d3e9440d68277, 0x000e78d3bb4fe338, 0x000b99a925f69f4d, 0x00001ccb54d831d2 }},
        {{ 0x000f9a722461f223, 0x000d3c2e9740b335, 0x000ce9c8647cec97, 0x000deea15677bc85, 0x0000f16ec53b5b06 }, { 0x000eb6094dff8468, 0x00051164ad859617, 0x000767a1a1fc7112, 0x000c1b57e5f8c47f, 0x0000305f263682f7 }},
        {{ 0x0008ca06c000315d, 0x000c0fd341905477, 0x00037793ca900aec, 0x000ade7e68a394f4, 0x0000d89206088d97 }, { 0x0001ba76c1885187, 0x000ad626ffeec228, 0x0009b5136239e6b9, 0x000b437479d91bce, 0x0000a4b31a467b18 }},
        {{ 0x000b0c03e9bb6ab5, 0x0006fcb28c682c67, 0x00080b06537181b0, 0x0007bb99827f2522, 0x0000e7f5e678902d }, { 0x000287223f5705ed, 0x000e33015b548e76, 0x000fd62c48b8a6ea, 0x0002b1f40a8be76d, 0x00002b41597b0af8 }},
        {{ 0x000fcf45b93bfe6a, 0x000f6558b399320d, 0x0005e04f008d2b8f, 0x00027b05e61fe3fa, 0x00000320b1aab1c7 }, { 0x000c421d977c1dfe, 0x0009c512f684b432, 0x00049adfeccf138d, 0x00091a78ff796c65, 0x0000756d6c50e779 }},
        {{ 0x000dc9d1ec6f170f, 0x000bea3af2ebc2fd, 0x000ced852c3855dd, 0x000b601b1a0af843, 0x0000efe50594d52d }, { 0x000740f316de5b77, 0x00051bb62771deb1, 0x000a7a84b76a9d83, 0x000c81aae829277a, 0x0000e1420c11bfa4 }},
    },
    {
        /* digit=34 (1,2,..,64)*(2^{238})*G */
        {{ 0x000fcf57c64d0fe1, 0x000089aeeaf8c495, 0x000d0e557f623c19, 0x0001b346cee7aa73, 0x0000e3399090fc9a }, { 0x0006eb38aad81cf7, 0x000a739b6057604e, 0x0000db9b5314c754, 0x000f3472d7d343c0, 0x00004c1dfbe3a0a5 }},
        {{ 0x000848949c160154, 0x0000e4f8be762b49, 0x00006d459f779d3d, 0x000d6b1fa2a9ee4d, 0x000073144722cd51 }, { 0x000631594e5d8939, 0x0009ee5d23a84178, 0x000d6db14456f57f, 0x000650e1f892f3b1, 0x0000d7a0b3f34fe0 }},
        {{ 0x000a613a9e60b8e8, 0x000423cde918dcce, 0x00033c9c94ad800f, 0x00012332d2e7b599, 0x0000a4f91cafaad2 }, { 0x00000da7133e2037, 0x000286cf1fdc9f70, 0x000887923f6f84cb, 0x000fc857fb6eae8b, 0x0000f963508ecd1a }},
        {{ 0x000585113ba8dd62, 0x00080ea1f095615b, 0x00065f617af767a6, 0x000aab6be0a28ad7, 0x0000085f54531694 }, { 0x00033ed1deec48b1, 0x000ae0b30df8aa1c, 0x000b877911cd914a, 0x000e64fcbaddda07, 0x00004abbcd21ba68 }},
        {{ 0x000610ffa32530f1, 0x000285d80e877662, 0x000877ce4b1cfd91, 0x000b658349e99e6d, 0x0000e408841fab40 }, { 0x00090cb5663d4ea5, 0x000a2d4751c8159a, 0x0005062dddfd0fce, 0x000f8d733697b498, 0x000058975ab9e049 }},
        {{ 0x00091be071c8396f, 0x000c94836f024cb1, 0x00071ce580f69081, 0x0004498f185be1f7, 0x0000741b76e70b14 }, { 0x00048a876e5adf25, 0x0007aa5cca5e55cb, 0x000c4345c73e6437, 0x0009de0db3864ba3, 0x0000cb37b2d41132 }},
        {{ 0x00053fc7d8535a0e, 0x000b856580e40a2f, 0x000b7d40a0e49c2f, 0x0001f82c9f194ed8, 0x000020d26cc66d51 }, { 0x0003b7d63457b0ca, 0x000e49b78593e7d5, 0x0002ee238fa36714, 0x000f3099b2cac4ee, 0x000023acbadeefac }},
        {{ 0x00032b33d4f558d6, 0x00037483e07113c9, 0x0008ea602fd9386b, 0x0000f762b6e4a5d3, 0x000040b7b04b7684 }, { 0x000f178169d618d6, 0x00020510dbbf08dd, 0x000885aab8e91f1c, 0x00043069b211d07c, 0x0000c105f022a618 }},
        {{ 0x000b741973430cb0, 0x00096350a712229b, 0x00074db32619911c, 0x000070c8393cbe44, 0x0000fe88492c7c4c }, { 0x000da7715e131810, 0x000f9a256dbe6e72, 0x00062cb5c88e8a5a, 0x000078a31e578ec5, 0x0000121ed1b3bd72 }},
        {{ 0x00084867a70ea8a3, 0x0008d1753dfabf0f, 0x0007c09b2ab277d6, 0x000b4f658728d207, 0x00008772d5dc2672 }, { 0x000c9017adeb92a2, 0x0000f3d70e35b1c7, 0x00079d20267edcd3, 0x0008c8f210241446, 0x00002a5362cc2cc0 }},
        {{ 0x000461d767714a05, 0x00068e90a482873d, 0x0000e31c7d6d3a7f, 0x0003383e74265d6b, 0x00004a05abbc1234 }, { 0x000626833fe129d4, 0x000cac564a7e489c, 0x000cdab872b2a050, 0x00033754ecc5675c, 0x000089b5043845b7 }},
        {{ 0x000568d0917c4acb, 0x000c55a83cdd60ed, 0x0004cbae30f32f9c, 0x000d8992a50602fb, 0x0000910f7a813c33 }, { 0x000ec2057b354ced, 0x00035c70259217fd, 0x0008e56f3c6925cc, 0x000490b6e93831cd, 0x0000a336551d0bd7 }},
        {{ 0x000563cce247e851, 0x0005499f98a759a3, 0x0002c25e6f3b7c66, 0x000dca58755a0a7a, 0x00005ae8843b044c }, { 0x000eea4056f758e4, 0x0005117e7b1fbe12, 0x00092088f88989d2, 0x0007764db8a7dc97, 0x00000ee98d72925b }},
        {{ 0x000ee376f827fa0f, 0x00059a7ba772b34e, 0x0000706aae49b0d1, 0x000f205dbe16ed34, 0x0000b5d4498eade7 }, { 0x0009a59c3b7533ce, 0x000dc1783674c027, 0x0006de9d65ef7473, 0x00070cb0f65832f9, 0x0000512581da9ee6 }},
        {{ 0x000fa352b73388b3, 0x0009a2ebd19bb161, 0x00009382dacf6953, 0x000135847c6c7cfd, 0x0000dfe8b3751963 }, { 0x00059df31f52f920, 0x00035b8973db6d32, 0x0000c347c49fe544, 0x000c2371f7183685, 0x0000f1f4cc727793 }},
        {{ 0x0008835dedbc740c, 0x0007d25913cc16df, 0x000ce91f116c9a1b, 0x000808dae18bc5b6, 0x000071a394c1d139 }, { 0x000ce433b9ac0821, 0x000ead865a40ab05, 0x0008bc6703c50add, 0x0000029696559643, 0x00009a0bfc436be6 }},
        {{ 0x0001e29ee4d40158, 0x000984f6ad651437, 0x000764feac1214e7, 0x0009468c71151182, 0x0000fa9f57c32bae }, { 0x000b4d4bd6ce4114, 0x000e80963717b46b, 0x000e251d88cbd747, 0x00081172202be02f, 0x00001fa60920ef36 }},
        {{ 0x0001cc10cacd0147, 0x000ad9620efd6628, 0x000043c91c4987a3, 0x000963dc292ba6d2, 0x00000e78ee2053ff }, { 0x0005c6097c119aa3, 0x000cdad4b4d7f6ce, 0x00059edbd3562f1a, 0x000ad44975c9f1d9, 0x0000702d0c39051f }},
        {{ 0x000ea0e79fe0f90d, 0x000a2a6328e5e200, 0x000a45e06db05fe3, 0x000c6bc78f7b5244, 0x0000518bfd167866 }, { 0x000e3a0690854dfd, 0x00099eb646c7ecfb, 0x00028633e40d1fc1, 0x0005dd269eb81149, 0x000077f2043ce75a }},
        {{ 0x00066dd6d2079f06, 0x0004c3d42e06189e, 0x000067cd28417e6e, 0x000130050bf07794, 0x0000de2c929cf8ba }, { 0x000333b1e86df344, 0x000b967ff99abde3, 0x00009477132566ad, 0x000536ed7c990355, 0x0000973cee61a30e }},
        {{ 0x000982aad98c4636, 0x000db3c6c491c14e, 0x0007975a94ae8585, 0x00092fced0597668, 0x0000469878650d9d }, { 0x00092d6943d57f62, 0x000f1478be0dcf0e, 0x000d7c9ceaae9a12, 0x000bd457f700080a, 0x0000c7c4650a126e }},
        {{ 0x00038b135b3697ef, 0x0009bf885dfe9aef, 0x0004599f736f9c08, 0x000480120d825d9a, 0x00009a6f60743203 }, { 0x000847fb2ab7025d, 0x0002de062a7150b5, 0x000db3ec1370558f, 0x000b9932ef31fd8d, 0x0000640e5b498396 }},
        {{ 0x000ce5e866272fbc, 0x0008bdcdccbac8b2, 0x0007da0e9d6243e6, 0x0000930dfb642aa7, 0x00006481890aff49 }, { 0x0009424317123bb7, 0x0008088c98c44004, 0x000da8b80f62d7e6, 0x0008318d78104a8c, 0x0000c585f14062f4 }},
        {{ 0x0000428dffccb096, 0x000cfe784d6365d8, 0x000f0f3311b8f429, 0x000378330bcb7443, 0x0000aa82351ab1ae }, { 0x00058c1fb6bdb9c6, 0x000e9efa4faf8433, 0x000dca1d65a933e3, 0x0008101ceced8538, 0x0000d4a8d9e6e600 }},
        {{ 0x0008aba53a05bf33, 0x0008de5fd349961a, 0x0000db761dc2a9ea, 0x000e35a382b2cfa5, 0x0000b7879468dded }, { 0x000eebef3fda587c, 0x000b11d6e96eb8db, 0x00000cc3a6e277a8, 0x0000c796c65d8dea, 0x0000e172dfe107d8 }},
        {{ 0x0005a760658ad3df, 0x0008e98ce7852541, 0x0002a5bd40fc7160, 0x000a56bddd50e8d7, 0x00005c3561e661be }, { 0x000e509af3870a6a, 0x0003822961f23d6a, 0x0007c59c25bcd2e9, 0x0007893e4378e443, 0x000027a005b340d3 }},
        {{ 0x0005626678970f60, 0x0008e46577f44a13, 0x000c80af7872b352, 0x000e85884e09b748, 0x0000e421f514b45e }, { 0x000ceafe50229ea3, 0x0002218bf8d21e6b, 0x0000f51aa91d1556, 0x000a1d0c022d9653, 0x0000b69daf2a9172 }},
        {{ 0x000e6c984782812a, 0x000b4f7be42a5821, 0x00004697d39f6d6b, 0x00067b1a9d3fc608, 0x0000472dc3a58087 }, { 0x0006fec314288ecc, 0x0007d25d30c31f70, 0x000aa52b5e0e12c8, 0x000798de7b7eb72b, 0x00005698fa1698c6 }},
        {{ 0x000a05261d35c7d3, 0x000e910f8c22049d, 0x0006ad59994bee9c, 0x0006e5f6b9bfbcce, 0x000017336b3b0d0d }, { 0x000c744e8ee14501, 0x0005adceba7efc8d, 0x00094967709fe15a, 0x00030f027471237c, 0x0000b4abf8266348 }},
        {{ 0x0007b411642b925d, 0x0009016a56edb730, 0x0007244c6d0c0d15, 0x000ee570064357e3, 0x000045c5461cf724 }, { 0x0008434e8e56e691, 0x0008d3274c898834, 0x0000e03f995f5bf2, 0x0001d53823c16282, 0x0000bfbc0e792c1e }},
        {{ 0x00017b66ac0eda90, 0x000624f2c7ae0b94, 0x0000d7be4d9e6b1a, 0x0007eceb1c46fbf8, 0x000073445e243c47 }, { 0x000d06252a7f7cde, 0x00054b2c5ceec209, 0x000f449ba5b7dfa6, 0x00085a8e6d69661f, 0x0000bd415537281b }},
        {{ 0x0009650f1aafc77c, 0x00018b07e50122b3, 0x000b118b6d8a8280, 0x000784d7bdd744f8, 0x0000b7a7039bee26 }, { 0x0003cde18959c8fa, 0x0004df71fa38477c, 0x0009110e30c5b652, 0x000a5bb8a3d815f1, 0x00000fab64c70c58 }},
        {{ 0x000af52d3611da39, 0x000ba7bf9539a488, 0x0005e913946f430f, 0x0007dd3bbb0100e0, 0x00008c0902a3dbd8 }, { 0x0008a53a687694aa, 0x000f347dbbf698b0, 0x0009fb36e20b6928, 0x000d5b3f82961a1e, 0x0000e28e2ca89070 }},
        {{ 0x0009d84d2e9b0fce, 0x0002d481bee45d41, 0x0006229033078bdf, 0x0001f5669b3cee34, 0x000090c773d3e729 }, { 0x000fb1fb1bd23754, 0x0006fc51c4d349f6, 0x0003a68b5ea950a0, 0x000747aae6596584, 0x0000f1319ee53ae6 }},
        {{ 0x000f0bece2f566fc, 0x000010607bf8fe31, 0x0008155b86689659, 0x00090f4d346259c5, 0x0000d4aef04c0740 }, { 0x000849f911d26148, 0x0006cb3f3c72914e, 0x0004546b3a782574, 0x0000a254629e8ea9, 0x0000ef344da514b1 }},
        {{ 0x00007d28f8169af0, 0x0003fbbf4737f217, 0x000af18cccb4682b, 0x0005ebdbed8dba5a, 0x00005fcad39eb18f }, { 0x0008f0c981a47c9b, 0x000ab0bcaa999eab, 0x000c63a21b4ad294, 0x00058242dba7b174, 0x0000fa27d77ce474 }},
        {{ 0x0009619367bed35c, 0x0008e9a324b47379, 0x00029946a8624378, 0x0005f80edfba9e42, 0x00002193d51edbe3 }, { 0x000898978136c517, 0x000dc5754b7032b4, 0x0009dbaa3dc9b204, 0x000cbaf5af936b3b, 0x00005c1090706f4b }},
        {{ 0x000ed617a5a061dc, 0x000f95a13d4f95eb, 0x0004eec56d5b0d63, 0x00012183d737cff8, 0x00001b08a819d5ea }, { 0x000791e248259760, 0x000f4017dfd8ab26, 0x000dd2968e62408c, 0x000108545ae7b89a, 0x000093d95add0afe }},
        {{ 0x000d5c3a7fff1f96, 0x000a0ad324fa34d3, 0x00070f6273a886e1, 0x000bd9c99ea11144, 0x0000b3c3fd9df0bf }, { 0x000a251c3731c8ea, 0x000f1678d0412d2e, 0x000212b74f637f2d, 0x0005fbf82d92625e, 0x0000b4da522a951c }},
        {{ 0x000cf06fd6232db8, 0x000d132790691bf5, 0x0008a9c6c6d87a37, 0x000a8836eed61058, 0x0000653769772e9d }, { 0x000cf4a1aa26fac0, 0x0007facd9588e411, 0x00020e9e83d4703b, 0x000800ace1f9bf76, 0x000079b3d81e1a3f }},
        {{ 0x000fc099ea249902, 0x0008849dc1818d01, 0x000983a180c61c4b, 0x00005f4390643ff3, 0x00003e5c40cc22bf }, { 0x00058d9a82959817, 0x000a2c3eccd28fd0, 0x0002795eba28fb98, 0x0001c5eb6d743cbd, 0x0000a1953590ed50 }},
        {{ 0x000603700554b317, 0x0000b9291262b72f, 0x0008c50b6efe112d, 0x0006d514296f924f, 0x000024bbaa00c3a9 }, { 0x00040606f5dab0ef, 0x0007a9e18d7dd96b, 0x000217a51d9f847f, 0x000dafa49381e878, 0x0000e4b811835102 }},
        {{ 0x00020c00eed74984, 0x0003439fc6751a47, 0x0004899c000050c9, 0x000d0a4751619621, 0x000037a8d658db74 }, { 0x0007d16ea9dbaf18, 0x00086b0d820007cf, 0x000fabc9e97be6ba, 0x000705184880ee65, 0x0000435446efea6b }},
        {{ 0x0005512b4e835c1f, 0x00056b2c0519a238, 0x000dca182794a57a, 0x000cd2b8a9ebf94f, 0x000043f096ab191e }, { 0x0003ebbb9869ab72, 0x00019a899b16a0de, 0x000bd6898d0eaeea, 0x000c0741595c2430, 0x0000e438ccd17837 }},
        {{ 0x0003ec66f6fddbd1, 0x0006bfe18549fdb7, 0x000edbe1509ba54b, 0x000cfd46cf6e200b, 0x000098de96d82185 }, { 0x000c5616c5a42a40, 0x0006dd7d1b9a305c, 0x0008eb5804d7e8ed, 0x000a1c94aa959c5b, 0x00007e60385e5cc7 }},
        {{ 0x000a06b329f8877b, 0x0005f01e6a33f7ce, 0x0006b347bd83ae53, 0x00039e07cc6217a0, 0x00007de49a6fee11 }, { 0x00012c4acbdef85a, 0x000b7cea5f5e7b9c, 0x000fb7501b28384f, 0x0001e515466dfc30, 0x00003b6c73cc9d54 }},
        {{ 0x0005ad49e51ce957, 0x000361a1e407dd9b, 0x000bdc025fb2c747, 0x0008e60a5e8a3587, 0x0000375165792d8e }, { 0x0003f99fc369de26, 0x0008cb27006085ab, 0x000fd3c5cefed71b, 0x0005240a6a88ff33, 0x0000a54d6a89ede4 }},
        {{ 0x00017969cd6413a4, 0x0009b5eb2e636456, 0x0007234ed864a3a1, 0x0000131cfbe57e54, 0x000097da13fc490e }, { 0x0008c4f4f07e5b7f, 0x000e9cd8a7017a0a, 0x000a15d689f8531a, 0x000ba987677b4d19, 0x0000622e9398a31d }},
        {{ 0x000b8c3c1ca6d7d7, 0x0009acc67b3de77b, 0x000da7c33ce88288, 0x0009929c9ae3c0b8, 0x00005e532fdba5e3 }, { 0x00018487aab79e34, 0x000b4eac51b35836, 0x000f33a15e97b5d3, 0x00056d5f37591f5a, 0x0000464d53dbd05c }},
        {{ 0x0007623bd0306c82, 0x00063f79d61718a7, 0x000bfe3096bc0a06, 0x000cfb3cf00413bc, 0x0000bd36a5580365 }, { 0x000b0231a0568154, 0x000d33bdcdd2a50c, 0x000a3aff52ce886d, 0x0004595294a741f9, 0x00006cbf37c0b0b8 }},
        {{ 0x000a0c030ad04bbc, 0x000940d084bae24a, 0x00068a312388eb95, 0x000933c112f0ae8d, 0x000059d33f6ee7ac }, { 0x00089e38dd68ef7a, 0x0009361a0f442dea, 0x00067f1300f605ff, 0x000e94228f92e181, 0x00000cdb381fd9a1 }},
        {{ 0x000ff5b8f078049a, 0x0001e65d2fdca23e, 0x0007dc993fa509ea, 0x0008bbc145a15ee0, 0x00008afe9e821989 }, { 0x0006f2789726e911, 0x000b9bbcb1c8a0ef, 0x000cc1d5c062c9d7, 0x0006103c02e1ee9c, 0x00007b357e712163 }},
        {{ 0x00078ec0a2491df2, 0x000fd1c89ee6256c, 0x000da3a67aa3a41a, 0x000cb11b3f44ee1f, 0x00008112a4f22527 }, { 0x00023aae4f91b961, 0x000d224b7dccbc61, 0x000304010dd5da46, 0x000347705d96dd1f, 0x000040b329358dcb }},
        {{ 0x000ccb9b14c45c47, 0x000b5f5a26262cd2, 0x000cd39550c052c7, 0x0008e5caf1f7f4d5, 0x0000e4892d07ecee }, { 0x0002fe771f54a03a, 0x000e2ad52324aa65, 0x00061f496834cafb, 0x0002935826320052, 0x0000a358d4c12366 }},
        {{ 0x000d971c910c2659, 0x000483db1001684e, 0x000db3d982e44846, 0x000b52d72b56b4ee, 0x00008b44f48218c9 }, { 0x0008b44cb030e094, 0x000f291e3a07b296, 0x000c7c024a091d9c, 0x000e755ae7570d9e, 0x0000f4a48be7b74a }},
        {{ 0x000e62960e80d5a9, 0x0008bb3bf44da801, 0x00093c509857d0c7, 0x000a2f700283834f, 0x000004d0d39a88e1 }, { 0x0006394239b24d7d, 0x000357be6901c550, 0x00081ec479e5e93a, 0x0005f457d80e7853, 0x00009467d044177d }},
        {{ 0x0006f9d9edf5287c, 0x000ce40376c1944a, 0x0003bce03b234092, 0x000ec0e3b7cb6269, 0x0000dbfe013373c1 }, { 0x0006ce2871c6ea89, 0x000a3a2880dd2123, 0x00036f4311374ef1, 0x00024fe85903c381, 0x000070bac414fba5 }},
        {{ 0x000d3ed11bf23e7f, 0x0003128e10d9c12f, 0x000105f062b7d445, 0x000649c587573baa, 0x00002d7ca95fda75 }, { 0x000beec455ef108f, 0x0001b9a0cb2748e2, 0x000831811ac094cc, 0x00064aa418239f01, 0x00008c7d897f4683 }},
        {{ 0x000939e11117c855, 0x00001063f8b84cec, 0x0002c3fee75e7039, 0x000c0fedd7641708, 0x00003953dd909641 }, { 0x000585638f431edf, 0x000e3b35a914f2f1, 0x0003d1c62c0d57fb, 0x000fc50b039e35ed, 0x000031fe78392334 }},
        {{ 0x000e8d6d63ec1c25, 0x0003f6e9e2fce99d, 0x00003ff3181bb02f, 0x00027902a68a2108, 0x0000a6146405e04a }, { 0x0003b6c03c53dedd, 0x000386342146a0a4, 0x00021af375e61c53, 0x00034aa878ba826f, 0x00007332a2758132 }},
        {{ 0x00093c8fa198c421, 0x0008ebbb54dde39a, 0x000a32329dbdca62, 0x0002d37e1744e5e8, 0x000030383a3af937 }, { 0x00096449c0bb9f0d, 0x000ec56de058c5cc, 0x0001ca0b54b965e3, 0x000441167216235d, 0x0000a79f1b841b17 }},
        {{ 0x00022e45572f2733, 0x000f574170693bd8, 0x000d22301dde4f72, 0x0001b4044a28e14f, 0x0000500450e45c27 }, { 0x00086c3726a97a95, 0x000d9e2e6a285cd9, 0x0001117303c981ff, 0x000cb23445866f21, 0x00006f51ec40b74e }},
        {{ 0x000ad1ce128a1832, 0x000da2af7172277b, 0x000e51fae431e94f, 0x0002b5aa757b94f0, 0x00000f8d45d654a3 }, { 0x000ebc4f922ef269, 0x000849dd8f71caef, 0x000597638b146921, 0x0007aad37f43272a, 0x000009ef351a878e }},
        {{ 0x0009c02205f112dc, 0x0009e952f41deff1, 0x000acc7b67b111a0, 0x0002d1d510e44a59, 0x0000fa3e8511d623 }, { 0x00051ff946f13b54, 0x000dc69c4b7d692e, 0x000317f509655586, 0x000e6b4462e6392c, 0x0000a96c730c5b29 }},
    },
    {
        /* digit=35 (1,2,..,64)*(2^{245})*G */
        {{ 0x0000a3933301eb25, 0x000a651566c5c43e, 0x000854f45d136c9a, 0x000a9b44acff9c91, 0x0000afb49c7ad947 }, { 0x000a2e067e61d8ad, 0x0004dc10dc1eb2bf, 0x000135c5137c224f, 0x000bda57488cfd8c, 0x0000c44a7f8d059f }},
        {{ 0x0006c13cc1c1b2ba, 0x000707f3e5f9f11f, 0x0009e009feccf526, 0x0004546bd9afe153, 0x0000da180beb161c }, { 0x000f8417a1d44bc5, 0x000008e325fc3ee3, 0x00006399ac1041ab, 0x000580ee77109540, 0x0000ceab562965db }},
        {{ 0x000fab17514db1c6, 0x0008172813b230d7, 0x000e97892cd69e71, 0x000cdff6a634d0f5, 0x0000c7df5b396089 }, { 0x000b9815a106666f, 0x000a1a74f7c4f830, 0x000416725238afc5, 0x000ab9aa0a7d2edc, 0x000091170e9acf8b }},
        {{ 0x000db6d961011f15, 0x000439b319540c33, 0x000964ccfd972723, 0x0008f8acb18490f5, 0x00000165db5b23d9 }, { 0x0003f6e09555115e, 0x00005bb146110697, 0x000e9391de70734b, 0x0006302d2b19436a, 0x000046716ed749df }},
        {{ 0x0007e83eda22ea79, 0x0001a8eb6357f373, 0x0008f3f06b88b995, 0x000bf00fcc5d00f7, 0x000073b37b16df09 }, { 0x000b956bc10c800a, 0x000d8731560bf3d1, 0x0009017a42a05ea3, 0x000140abd1086e45, 0x000062690f21558e }},
        {{ 0x000610ad741347e3, 0x000d891c31b9c38e, 0x000a62deaa41683d, 0x000f3c86f0c5aa90, 0x0000f90c15e6f59b }, { 0x000921d5778fdb23, 0x0004ddf02ba8e15d, 0x0005b38621fa615d, 0x00050e06c0337edb, 0x0000f89b6644d026 }},
        {{ 0x000357c9e3ee8927, 0x0003d392c422f7ac, 0x000229dcbeb4ed3f, 0x000d2d5d1fb63536, 0x0000f4ae39a384d5 }, { 0x000c225a84c23dc0, 0x000f5405b2e5b8ce, 0x0000f4e05f1fef35, 0x000d1f1997e94693, 0x0000aeede3f666b8 }},
        {{ 0x000b8a4cbe978aa8, 0x000fbc8ee3c76cb3, 0x0003979fd2b05b7b, 0x000d941563cf1162, 0x0000e542d606a5d0 }, { 0x0005e56df6f115ed, 0x0003ca59ce6bb278, 0x00017eef4378ebbf, 0x0000d8c8dc4afaa5, 0x0000a21b92c92f88 }},
        {{ 0x0000ceb3fa8e1f73, 0x000163c2bf296984, 0x000496ffc2087462, 0x0007ae28f91fc19a, 0x000071c2f69ba91f }, { 0x0004cf10c29819fa, 0x0006660bc9ba369d, 0x000684a977ff8395, 0x000895207927e103, 0x0000429c81a856ee }},
        {{ 0x0009c0d0a7824c13, 0x0004034859dd614c, 0x000a4f0ff846b7f2, 0x000813ed59c475b5, 0x0000cd1b934f7785 }, { 0x000230a6e8f478a9, 0x000dd91c234296f5, 0x00044257d8cd2650, 0x00071f3f22a0a83a, 0x0000ddf899163e18 }},
        {{ 0x0000030a795ce50d, 0x000b20f72e3d5c1e, 0x00056f1e39259abd, 0x000fdf6343771744, 0x0000727bbec1ac3f }, { 0x000c8d087f282848, 0x00056bf1202e9ca6, 0x000c0959dd124705, 0x000b27cf8e79dde6, 0x0000ccb0bbb230c9 }},
        {{ 0x00025cd048f7a136, 0x000913dbce913af4, 0x000b032d3782b56e, 0x0004ed68a909e587, 0x0000f4c845403bdd }, { 0x0001af92d3eba46f, 0x0004abe90e9d8bf9, 0x000aac93883ac9f8, 0x000767dff991616d, 0x00000f2d9eeda667 }},
        {{ 0x000653fe47f4d959, 0x0006992e93fda293, 0x0007b3299374a645, 0x000d388e676bb6c1, 0x00002b338f3417ba }, { 0x000d70a99798c8cb, 0x00040280f680e752, 0x000bc14618d4da4f, 0x000a4773675eefd1, 0x000070102614081d }},
        {{ 0x00005cf3d8688b24, 0x000be0720445c36d, 0x0001892a7d9c11b3, 0x000144fe971d1ac8, 0x000058a7f36a1f2c }, { 0x000e16f5c25e1654, 0x0008e0b7b1c7129b, 0x000f98876c302942, 0x0003df82dffadcc1, 0x000015505c986cfb }},
        {{ 0x000e029db51b3c16, 0x0004fb8aad581350, 0x0008f57808095893, 0x00033c0a0622c211, 0x0000848670b38a49 }, { 0x000627cfbfce8544, 0x000803794fba0377, 0x0006ebdfc7e6bd5a, 0x000ab8f8eaefbd68, 0x0000239802ac6bee }},
        {{ 0x00035dc5396eb247, 0x000c8ef4e4ca4631, 0x00072209072a4380, 0x000c5c200bcef728, 0x0000b72cb6549160 }, { 0x000b0c5738b12183, 0x000822e27bf1bc68, 0x000f7cd34933b26d, 0x000119eda10a224e, 0x0000c80f888434bc }},
        {{ 0x0002cf43d8bba995, 0x000ba2a346060743, 0x00086e27c0a14c45, 0x000f74e0680fe7fe, 0x0000fac5f4d27a25 }, { 0x000dcd1a75979538, 0x000715f5f02ffe6f, 0x0003829db97527f4, 0x00033ad5e3d9eb9a, 0x0000209af43545e2 }},
        {{ 0x000e5c0304618223, 0x000f48bb95560983, 0x000f41c035820ec0, 0x000e596124090a1d, 0x0000cc98be20fb39 }, { 0x000626289bc31852, 0x000ea4b70dceecab, 0x000461f93bf28884, 0x0001ae25cbb2d3c1, 0x0000160674019bab }},
        {{ 0x000865a96da8e141, 0x00085598baebdcf4, 0x00092d795877cd89, 0x0007735d81a28ae0, 0x00006e178e1b6c8e }, { 0x00097f5dfc26037f, 0x0009ccd237936b24, 0x000b16c21697ffd9, 0x00006a247332f4f4, 0x0000ffad165e6c11 }},
        {{ 0x000ec6787acea84e, 0x000febd32be5e0fc, 0x000648d99afacf1e, 0x0003024b17475bcb, 0x0000b743664d4e82 }, { 0x000fe814369d2cde, 0x0008214d8578c70c, 0x0007a8d35754fdd3, 0x000ce5a06c9ad99a, 0x0000e934ddca5774 }},
        {{ 0x0008474a9996f7eb, 0x0002a822f4fb344d, 0x000f6ff85c520bf6, 0x000348c49b882344, 0x0000f420ddd2a3ce }, { 0x0007a033ec204ecb, 0x0004a4d5d1c00d06, 0x0008a69e46c0c3ed, 0x000d712f87fd581d, 0x000024e3f35d2d04 }},
        {{ 0x00023c422eac602c, 0x000d2c19ef2d2e4a, 0x000e3364a85064df, 0x00011200a1bf6767, 0x00001c6223e12f57 }, { 0x00013599f87f98e9, 0x000822d540f83afd, 0x000d84b4acaf5dc6, 0x000912908c2c2c27, 0x00007299ce914de7 }},
        {{ 0x000f627fd7a63a1a, 0x000315cd7b32c6ee, 0x000ffb93983d2957, 0x0005001bb61a5a63, 0x00007c449b79b37a }, { 0x00080cf93461f5c1, 0x0006efcc4c86bc81, 0x000aee0840c9e22d, 0x00095d433e6f9231, 0x0000b966edba2633 }},
        {{ 0x000e0efb3b8b1b5d, 0x00038959884aaf7a, 0x000259a44d6afe4d, 0x000f91f87b1959be, 0x0000337331701bb0 }, { 0x000b01a0216bb368, 0x0000c5eca8c325e7, 0x000671fd9f4f814d, 0x000e76c1d3c91169, 0x000010645e8443c5 }},
        {{ 0x0003a375e672dfca, 0x000e89363a1483f9, 0x0000e68119b07752, 0x000518de589eabe7, 0x0000537b9e85ecc1 }, { 0x000e0eb3eb3def9e, 0x000f5436a2afe69a, 0x0000335814a3a97e, 0x000978293c4f8910, 0x00004bb46fe72211 }},
        {{ 0x0006fcd2c98f37de, 0x0009b0d9d79046e6, 0x000d05f136ad2051, 0x000cf40ca4d650f2, 0x0000c7cbce68d0b3 }, { 0x0006ddebdb5a0c55, 0x00009b5696605f18, 0x00071e6ee7b1b45a, 0x000ccfbcfe77e5d0, 0x000090641c6e2ee9 }},
        {{ 0x000de285025363f9, 0x000513dfddf74157, 0x0008c18c08581775, 0x000318675f7fa130, 0x0000499a04911b96 }, { 0x00079deed354d5f0, 0x00080789e2aeb6b4, 0x0004aedc64ef1a47, 0x000c7ffc3ba2b9ff, 0x0000ff43ce38948a }},
        {{ 0x000eaefbd10b4a81, 0x000379b3a8631e8d, 0x000d9b723b75c7a3, 0x000c1de118de855d, 0x0000b4dea0f1e6a4 }, { 0x00072cd4b1febb5e, 0x0003c8dd150892d4, 0x0005da5ea4d30538, 0x0007c011a37b8468, 0x00003fedb7f726b8 }},
        {{ 0x000910e7019eb79c, 0x0005a51e9f05de9b, 0x000873d8a4a77728, 0x00080664cd404506, 0x0000f06a0d5cacfc }, { 0x000d896e2f49d140, 0x00016eab3aa8f0df, 0x000681650677644b, 0x0007b0620bab2a50, 0x0000225a11e3b301 }},
        {{ 0x000d55e62f7b147f, 0x0002f53308fbd93d, 0x000d256d89ab7031, 0x00090048546cd5a2, 0x00008f07fbe7613f }, { 0x000f410b11223a4e, 0x0001ca02a81ede7f, 0x0003f9e643967f06, 0x0000701866860dd1, 0x000037f2d5f3b15e }},
        {{ 0x00004aa9d3156f13, 0x000b93849ad467b6, 0x000baa288c1e44c4, 0x000e49060efb48b6, 0x00009eca1fbdd19b }, { 0x0003345e17323e5d, 0x0006786754e48e96, 0x0004d412c0772c5f, 0x000fcdab6693cb45, 0x00001eca39d6b65a }},
        {{ 0x0005b4488b3fb1e8, 0x000d26b23a5d8961, 0x00056a2fedc3595e, 0x00081de771fe19e3, 0x00005b981b48385a }, { 0x0002c4f79da9b17b, 0x00017541a1f22bff, 0x000fb8bc6c4a7592, 0x00033ea8e920a8a6, 0x000037f6be35ab5e }},
        {{ 0x0003272fd83d30cb, 0x000f8d04aa38d1b5, 0x000c62b22135dd8a, 0x0003013c03b0db35, 0x00000b015e47a001 }, { 0x0008a2e909f55c72, 0x000d74f07a8e1078, 0x0006e1dea8f06a53, 0x000ee63444a89e13, 0x0000850dcd40ec46 }},
        {{ 0x0002e8a6205f5be2, 0x0007be3b44a3f84f, 0x000324de206fb879, 0x000c74190acb2f34, 0x0000169e7bbd47b0 }, { 0x000baae31cbf3f3b, 0x0006fb69c2d76521, 0x0003d1eae29fe835, 0x00042b610fb5307e, 0x0000e643f0cba58a }},
        {{ 0x000959c3f21417f3, 0x000255bcbea62e7d, 0x000e119c6c6709d7, 0x00009921ce8ac9ea, 0x00004c4448bc0b85 }, { 0x0008f6c667720f2c, 0x000692520e40ae55, 0x0002aeab843a32e2, 0x0008386bc320789e, 0x0000f23eb6cec001 }},
        {{ 0x00005e16bb852160, 0x00052bbc55d2d8b9, 0x00076498b39b48fc, 0x00066161441d58bb, 0x0000a86b13ffc50a }, { 0x00053f7146de7703, 0x0004e40f661acacd, 0x000ff5e2ff690eca, 0x000c929ba3b8784e, 0x00006044fa1265ab }},
        {{ 0x000d45e2d4b575cf, 0x000bfe5e5681e833, 0x0005b966bb760ac6, 0x000f54b386212610, 0x0000920d33916add }, { 0x0000ca34a27332f1, 0x000d33976e8ae983, 0x000b8eea7b6bf672, 0x000057b89b53b2ee, 0x00008733d0036fff }},
        {{ 0x0008a92a0eca49dd, 0x000fb6c783417303, 0x00087c7cb839b4d4, 0x00005e2388ef5beb, 0x0000914d653ceeed }, { 0x00093b5505c32a66, 0x000543f22433ccf6, 0x00063963b0a3b74d, 0x000ee34fcde8362d, 0x000054d95941e49d }},
        {{ 0x000cbb9da03036b6, 0x000b5c0cd1797a18, 0x000e67fc118d1fef, 0x000c2923eea17b47, 0x0000697a3f1abc0d }, { 0x000986f3fa1f44ea, 0x000139c4d8424cff, 0x0000b195410c7287, 0x0003d1ca3db7048f, 0x0000c7c46d5a24e3 }},
        {{ 0x0006210d9e58a61a, 0x00029b5224c08dca, 0x0003db93fef2cd7b, 0x00002ae4f2bbb09b, 0x00004673f3e36b54 }, { 0x000acede4893a3e0, 0x000167a09cb54d69, 0x000fb53a3b5bd9ea, 0x0006f485791eef6d, 0x0000d389cb15387b }},
        {{ 0x000410cbabdf6c15, 0x000067190b771c33, 0x000486535b12a4e1, 0x0004bedff563b71e, 0x0000ecda5577d7de }, { 0x0003d7d5dee03f61, 0x000cc2630da84dd0, 0x0009ea44ecbb751b, 0x000151692d0711ae, 0x000014aba617814f }},
        {{ 0x0004912ebf56ac82, 0x000ada8ced806b84, 0x000424ede5b8b06d, 0x0005059bf222e613, 0x000077ee32509e55 }, { 0x00024cc23645bf6a, 0x000cc742496c27c5, 0x000bcdaf647a66d4, 0x0001f552c8ea3176, 0x000068aa54cc75ce }},
        {{ 0x0005468b2dae9bab, 0x00036d89f6f1b18b, 0x00048f86582e286b, 0x000f286fc2552f00, 0x00007485d84694f9 }, { 0x000ee3525470e487, 0x000b7883ff90f2ba, 0x000e802701cf5d83, 0x000725c2877947bd, 0x00000092a2e7377c }},
        {{ 0x000c1f75e1fc6e4d, 0x000e97d3567962f6, 0x000f15694a198229, 0x00082e3a1379ed61, 0x000016040a7a66fe }, { 0x0005b5c318e65050, 0x00016423b4a79fce, 0x000c6f39817a3245, 0x000c19ca1f20bc83, 0x0000e78407cf44aa }},
        {{ 0x000450a8760aef29, 0x000895d13221bc57, 0x0001f6ee42812871, 0x0004b6ca1707baad, 0x0000156dc61f6269 }, { 0x0002697b59a57f39, 0x0008453e5d0338fe, 0x000e491ec0628d37, 0x000e4b3d01e77f7a, 0x0000a053efd0726a }},
        {{ 0x000d3b14f7467390, 0x00015bdda85d5347, 0x000f825cf9f2e0fe, 0x0008ff5e753e78c1, 0x000059772a4a8d49 }, { 0x0005e906b119a241, 0x00033b0cb8f2bd25, 0x00062c96a2e1fd6c, 0x00050eaab2d50b48, 0x0000699177c6e9dd }},
        {{ 0x000dc63a9d42470f, 0x00013fc8fe280d19, 0x0005151ada0e3238, 0x000c459b0cd8e0de, 0x00004eff38b04e9e }, { 0x000f4187bc92bdf7, 0x000c53b4be1495fb, 0x0007d4178a70e616, 0x0004bd4dc25eb3db, 0x0000eb13072f9e75 }},
        {{ 0x00078c2ec1e3ed4f, 0x00051691416a6a5e, 0x00085c63595aa0e1, 0x000163b5d8f860c7, 0x0000283d923fc854 }, { 0x0005beef7ad3f5b3, 0x0007b815ad735813, 0x000023a1511c7557, 0x0006b79bab4cc9e0, 0x0000199d8c4e721f }},
        {{ 0x0000a1b1af1e78d3, 0x000f4cfdbcf41b9d, 0x0009de07e610f9cc, 0x0008a23d85703662, 0x000010d9697c4fc5 }, { 0x00023c2688aafc37, 0x0007c7503c41c4b1, 0x000af155a9b9f44b, 0x0003f540aa023829, 0x00006df810563aa5 }},
        {{ 0x000f8aff74187332, 0x0008ecc36ba06eee, 0x00044e47cc9819db, 0x000c0b71fface311, 0x000023dc4d264ada }, { 0x00009a815983bd68, 0x00045042d05841bb, 0x000e84cfa0889ade, 0x0001ffa7e42f0e3d, 0x00006a7649f271c7 }},
        {{ 0x000cee582b9889e1, 0x00045a10fee104a6, 0x0007ca159e31ee59, 0x000326733ad7788f, 0x000065aa11dd04af }, { 0x00044cc9c627cd7f, 0x0002267e8f55a7b8, 0x000655974f477505, 0x0003eaf9defbee3c, 0x00000dd76f26bfac }},
        {{ 0x00065e1d58e34a25, 0x00069cd66926e04d, 0x000eaab2a4614ef5, 0x000205a070001e38, 0x00007f9aa8b3ba19 }, { 0x00091abd3f792d82, 0x000bf41be76ca2da, 0x000c3f7835fb599f, 0x00043b7d27895cf5, 0x000000657fe7cdbd }},
        {{ 0x000e6ea8cc26f72a, 0x0001e8387eb36bff, 0x000192dce6e80e03, 0x000c29f5fc5f0710, 0x000060b97966ad86 }, { 0x000db3af99846589, 0x000c765edfa9284e, 0x00013dd8ddcbf735, 0x0008321fd6c510bd, 0x0000a06e6ac55647 }},
        {{ 0x000b32a66e2632f7, 0x000809d6295d4269, 0x0006a471070d02cc, 0x0005b2f040c4f6ac, 0x00004f774520e665 }, { 0x000688122ab3d6ee, 0x0004566f0476ce2b, 0x0002ef7ab5896bbd, 0x000783a097554d95, 0x000014bb84887b9e }},
        {{ 0x0009e6c0473150eb, 0x000218f508efb637, 0x0007f0d62f80b77c, 0x000f0c04f9eb5b5c, 0x000069accc4a1882 }, { 0x000dea6143dd9627, 0x000522ce37a1f698, 0x000a7569ca1c970e, 0x0002be4d3cbd252c, 0x00004fc80b2b91d6 }},
        {{ 0x000e42f66dd9a70c, 0x0000846de6736294, 0x000e21f936b36c3b, 0x000215cc757f7aa1, 0x0000e42c4db111bf }, { 0x000f78959874b51d, 0x00067dc910778965, 0x000f6350f2c2eb73, 0x000e53487a0d690c, 0x00008713f1619ac6 }},
        {{ 0x0009401711af5ba4, 0x000394752b16dba7, 0x0005bbe4aa9f7237, 0x000dc9293deefc0e, 0x0000a97c7a345d34 }, { 0x000c122b07c3fd19, 0x00093e9f21a15cb6, 0x000383dc6fe3e1fc, 0x00079e86c73d2616, 0x00000e249332ac22 }},
        {{ 0x000e79dab8354f95, 0x00047e715f096905, 0x0001cf8235118233, 0x000890ac6508b9b1, 0x0000c766eb659330 }, { 0x000d4ca47748ce93, 0x0008ef06c775aa2e, 0x000b4a7b482a1c21, 0x000611206df09d8c, 0x00003cf5f801f0a6 }},
        {{ 0x00087fc1cec8e5a8, 0x000363610bbb8b54, 0x0002bdc1ebd45d49, 0x0009d74cc0737ce8, 0x000003ccfa0ebc20 }, { 0x000a1b8ea3c1eb6f, 0x0000bb3b8bb36591, 0x000e6b42b27cef52, 0x000ff9344dea5df3, 0x000047617165cfc3 }},
        {{ 0x000fbbcec8aedc8a, 0x000e71967b0867ce, 0x0007ad7a76578ca6, 0x0005466e38d5e0d0, 0x00007711fd87562a }, { 0x000079e8e108ea31, 0x000cbab0263ad100, 0x000dfc57a6a990e5, 0x00006d517d8199e6, 0x0000296d52bf5ee1 }},
        {{ 0x0003b3afef9ade77, 0x000219f8a3422839, 0x000c147c62f6768b, 0x0008c19018097799, 0x0000ebf4142ddbef }, { 0x000ac70895b5ba6b, 0x000cf2a2db3d5ba7, 0x00020ef8474730ce, 0x0007efb73263dc46, 0x00004bc34642342a }},
        {{ 0x00025b0834bbbd7c, 0x000524d5c2f342bb, 0x000a02d27279ffdf, 0x000a440d0bf80907, 0x000081a156479865 }, { 0x000ccd9a92ebac1a, 0x00074a56c3a1ddad, 0x000a29fb8884da05, 0x0007030fefe4ec7e, 0x0000acc66637fdc1 }},
        {{ 0x000105a92d7a6a29, 0x000b4dadb58da1f8, 0x000f37c53dc1205c, 0x000f2214322bbc12, 0x0000f4c05d3cc006 }, { 0x00023213222f51a0, 0x0003bf37a859d51f, 0x00053e0f0648de72, 0x000bec7c9e3ec7ce, 0x0000fa1715d802dd }},
        {{ 0x000bdec2116e9b29, 0x000389c76497ee80, 0x0006bb3874fd1cc1, 0x000343fe15d2b0ac, 0x0000a3fadcb3a4e1 }, { 0x00037248e9d64745, 0x00087efa63b10110, 0x0007dd6bd1db932c, 0x000c6c78bf9e3fa9, 0x00009e31d0655466 }},
    },
    {
        /* digit=36 (1,2,..,64)*(2^{252})*G */
        {{ 0x000be5ed0e405bae, 0x0007fc91ec34f9e7, 0x0004b79b18f54024, 0x000b106f3d8772d3, 0x000037037c975e12 }, { 0x000aec44147d71ff, 0x0003d1931e82b100, 0x0007bf1327384e2c, 0x0002fb55ec63d285, 0x0000df2ba6d3b215 }},
        {{ 0x0007a23f356e3059, 0x00041e71e29a3efb, 0x000f94f0f0f98de8, 0x0007a880ecbe906e, 0x00002d869e92df60 }, { 0x000cf0bbf87a34ec, 0x00089c2efd2119d3, 0x000ebfbd0dd06fa3, 0x0006c851303198c6, 0x000030a29d4bfdd3 }},
        {{ 0x00089caef387625f, 0x000fcb72b7247593, 0x00017a38174726e5, 0x0001b102c945353e, 0x0000fcd0db4d1457 }, { 0x0003429bba3484a4, 0x0001753db65ef147, 0x000e6a574289160a, 0x0006d9c5f2dc2cb7, 0x00005d42b1ac334f }},
        {{ 0x00012c9e1ee0d0a1, 0x0003490b01e6e274, 0x000da05414bdd548, 0x000de03a9047e2cc, 0x0000c371569c9623 }, { 0x00078851bc8c9a7d, 0x0000d36794075521, 0x0006dbaa6726fc38, 0x000ce611949c5013, 0x000053af2d9b1059 }},
        {{ 0x00035aa24062df0d, 0x0002a52fb85fa4ca, 0x000d94e1eaa94f02, 0x0001840aed61257c, 0x0000fb93c2113070 }, { 0x0000d05aabec972c, 0x000f75d4421fc835, 0x00007292ec8f71dc, 0x000b37558f6df8ee, 0x0000d9d09ec67e4f }},
        {{ 0x0000235a102cb4ec, 0x000fb35a64785f45, 0x000b0f0672f75fa1, 0x0002e6467bc56637, 0x00008030444d7012 }, { 0x000881065be741fe, 0x000b8d8f2c4aa658, 0x0000fb14fbdf31d8, 0x000b607bac347583, 0x00006adf01034a09 }},
        {{ 0x000f12502f6f25d9, 0x0000c22cc1b5e838, 0x0000f6f93bcc544b, 0x0008c7ddfde2d4ad, 0x0000c68d49d6c0f8 }, { 0x0003511dcdcef6f9, 0x000a7dc7783920a0, 0x000dbd6b689c8148, 0x0000f52a6e80014a, 0x0000730b2f927704 }},
        {{ 0x000211e48a709134, 0x000a06d95a7b1a29, 0x0005aa515d70a8ec, 0x000baface9c4e7ad, 0x00001420ee199ed7 }, { 0x0005f92e47355a0e, 0x000716ec78ef1f42, 0x000fe05e173edf3f, 0x000e4dfe82b2c090, 0x00005f26894a26e6 }},
        {{ 0x0000e5daec35c312, 0x000444631890a36e, 0x000f77190f6f4a99, 0x000791e30703ef5f, 0x0000248d7cab9079 }, { 0x000a1bd7b6437663, 0x00041c3c9c9f1554, 0x00066b1d61acbaae, 0x000aa5c297e4bf17, 0x0000f13072c92c6b }},
        {{ 0x000372def7fe9e8e, 0x000089946191d3d6, 0x000d1815135fa99d, 0x00071e788281fc8d, 0x0000e04cb9db1b55 }, { 0x0003bcefb0c7e3f0, 0x000cb480d8972cae, 0x000672598a8f1310, 0x0004d1cf43f2dd4b, 0x00005a02d3e37cbc }},
        {{ 0x0005e8048f16243a, 0x0002f1ed785e73d3, 0x0005f8bdffd849c1, 0x000a2174aa5db5ab, 0x0000f3e31e72b2c9 }, { 0x000343d185ef14b5, 0x0005c8f1adaa0f81, 0x0005718a48d45076, 0x0005191b828ec3b4, 0x000095c49bafe83d }},
        {{ 0x00037b515fb7aa40, 0x000b844caaa358cb, 0x0003e5a8c6cb2e2d, 0x000dd1d9a1a1db7e, 0x000027da7bc2cde4 }, { 0x0005a481c0d7db4b, 0x00089097b165fbe5, 0x00098c626e327572, 0x000dc967efc322ed, 0x0000c100250a4181 }},
        {{ 0x00056b31e0992f90, 0x000c312c23fc9759, 0x0005706af79af358, 0x0005f4d391571580, 0x000032a43ce4ae5e }, { 0x000b37b634d74c5f, 0x00016d5ff7b33a68, 0x00098146b10a6fb6, 0x000e2c698c8d5dc3, 0x0000a98da6d07fa6 }},
        {{ 0x000f40ac3c4dd2af, 0x000cfd68b11888c9, 0x000de0fd79a08293, 0x000b4f34a5e0e8c3, 0x00005ee377d3a06b }, { 0x000ef1ec28436dca, 0x000149534e0f63a7, 0x0001ec713cd110f6, 0x000f727b7a128bf9, 0x00005d0d98ab5496 }},
        {{ 0x0000f1a43a4a1ad5, 0x0004f0fa36a27378, 0x0005e27077875e3c, 0x000e0939735dc136, 0x0000d7436395198a }, { 0x0000d7c67d880277, 0x000b04bf8cb0ba05, 0x00062fa0ff6e7415, 0x000ff59efb0b5376, 0x00001a3237f9a695 }},
        {{ 0x0008910bd3c74498, 0x000d8af3aeac968f, 0x0008cc252692ed9e, 0x00067be64e4f7fee, 0x000092836060ef1c }, { 0x000890896b28139e, 0x000470cc9c0726e3, 0x0009a0eab753f427, 0x00046bc1b1530956, 0x000038600b6014d5 }},
        {{ 0x00053794e8af97d5, 0x0007c5ca5618c937, 0x000b02a1ade8a949, 0x0005f92a8e665432, 0x0000f8ed81c1d707 }, { 0x000c21fb8d99ba56, 0x000957ce8c0dbccf, 0x00025d6478156ff5, 0x0008c850031e0261, 0x0000b4fb313fc9e3 }},
        {{ 0x000b3bcc8886aff4, 0x0000d16b0843e0bb, 0x0007ae7478a94254, 0x00027e43053b1b3a, 0x000070a88a7167ff }, { 0x00047e1384f31100, 0x0001100c93dce831, 0x00018d1416a58415, 0x000039ce87e79da6, 0x000008370d177e00 }},
        {{ 0x00034714e4870f2c, 0x00057d3af077f30f, 0x00031311a055c13d, 0x0006bc053ea78f84, 0x00006124e88ddd7b }, { 0x000ba0c58a21a931, 0x0002c15c2376c840, 0x00093179e1f91646, 0x000dbf9908315b65, 0x00005e507e2ffacd }},
        {{ 0x000b7587e7d8f6a9, 0x000e47a529a6d482, 0x000d977c0a1b5071, 0x0003784f99c7f250, 0x0000a3bfb164138f }, { 0x00043e21ae8a2808, 0x000fa33e0182a650, 0x0007dfa2f0dba6d3, 0x000d3018fb82ecf8, 0x000055f7520eb309 }},
        {{ 0x000e6559e9dabe39, 0x00010d666c834ea0, 0x000838dbe89dd09d, 0x000dbf1473be43e1, 0x000026c068976f46 }, { 0x000e63ef6977674b, 0x000b38e0c6615b48, 0x000cc7e1ae0786b6, 0x00076b88614abc8f, 0x0000e564e456248e }},
        {{ 0x000faca8d0cebb8d, 0x00074b453b31e91a, 0x0003948a0c0742a8, 0x000bf70660cba7ad, 0x000094b060ddcf41 }, { 0x0008fb9a616bc52b, 0x000311d9ee9761ca, 0x000b47c4be2e0a39, 0x0003f554c3497a16, 0x00001a01ac4f5b97 }},
        {{ 0x00024d301b78fa64, 0x000cefe392511143, 0x000c987a034b5852, 0x000ace55c36bf25b, 0x0000a71003604b62 }, { 0x000b211a0e5ee431, 0x000310cd670407e5, 0x000188521ff34062, 0x000def3e00ea8d4f, 0x0000dbd759ec2451 }},
        {{ 0x000e220933648ca0, 0x000dd205ffe7b376, 0x000aa925c9a6a480, 0x000c325193b7f3ef, 0x0000519935e92c50 }, { 0x0006fa48a2afcd4b, 0x00015dfb486d440d, 0x000568e6ebc32df4, 0x0001776517fcfec4, 0x0000575cad80abc7 }},
        {{ 0x000f59e604baf773, 0x000dc265872cd88f, 0x000b2831aa9d0b38, 0x000f9491c806c1d3, 0x00005ff0f174210f }, { 0x000212f245cd494b, 0x000c4327a0b85dd7, 0x000a21d9cfd1d70b, 0x0008cd713dec5720, 0x00009a66bf0ffb9c }},
        {{ 0x000b75466ceb202d, 0x0001411b5cae787e, 0x00073287eab73556, 0x0005581fb13cb7f5, 0x0000e12d755e6d26 }, { 0x0000fbf8f7c29358, 0x000173bdb6cff322, 0x000a4100e1a2b244, 0x00091aa8b7440f37, 0x000020bd4b0b9af2 }},
        {{ 0x000913b93cb994fa, 0x00098e1df83097f7, 0x000bdc374ec3de91, 0x00039352f28738f6, 0x0000d11cf177dae9 }, { 0x0000deb384609912, 0x00033931fe896366, 0x000f96a98121a3f9, 0x0007b93a17b98443, 0x0000d01181097cf0 }},
        {{ 0x000ab007761cf94e, 0x00006c601a982e6c, 0x000069e1c5b02fb1, 0x000c41d051dd35e1, 0x000099e98577e95f }, { 0x0009c7a367e425f4, 0x000a71f6565e15ae, 0x0008a2c80cbe203f, 0x0007aea37bf30f52, 0x00007bf67fb4b3c4 }},
        {{ 0x000f733e7e5b9a30, 0x0006ca552f03ad8b, 0x000842daf7427fea, 0x000299789c7e8dbe, 0x0000e03dc9ad2e74 }, { 0x000abfec927f95e5, 0x00016f5b8444823a, 0x0007c1385ba9089a, 0x000291140b7224e8, 0x00005f27d833f8e6 }},
        {{ 0x000cff1ae2810e9b, 0x0007a10079ea1bd1, 0x000cb35e424422ff, 0x000d7f24f1ad9add, 0x0000223f4eaec892 }, { 0x0002b37f69fa04b0, 0x0000749d4cee1172, 0x000b1d83599fe6e3, 0x000489c388e9f275, 0x000066a58613b395 }},
        {{ 0x000f59bf5ce72e10, 0x000e1fdb40146044, 0x000939e7911b9cb2, 0x0004b1efea8ce991, 0x0000b81a0bac185a }, { 0x0002588530df2495, 0x0009d4f923a277d8, 0x000ce2b6aed5cd2a, 0x000152751d9bc9b7, 0x0000c023c697a0ab }},
        {{ 0x000cd15a3cc6fb9b, 0x000a19f82e4c6346, 0x0003cd730abb95f1, 0x000222f258efb831, 0x000068413078deb3 }, { 0x0001e7c1ed462bd3, 0x000d5fdbfcd8fb51, 0x000f5c6d0b354d1b, 0x000acc02f31db2ee, 0x00009cf8f2c231b2 }},
        {{ 0x000c1b23430b5424, 0x000e10cdc9a151e9, 0x00079f9585161afe, 0x000026de79edbfc0, 0x0000dd01a58284f2 }, { 0x000676c6277d628a, 0x000928b7f8474b9c, 0x00063aff51f1e745, 0x0007dad5a90657c0, 0x0000a3eff6f709d8 }},
        {{ 0x0006cc32b494aa16, 0x0001c47b7d6dee80, 0x00077299605ebcb8, 0x000011fed0948d92, 0x0000bd274de4ad55 }, { 0x0004428c5ee2c46b, 0x0006b64061ef338c, 0x0001fdc05d8e8c6b, 0x000d73e9b1606fa0, 0x0000c773ba6a62af }},
        {{ 0x000205fb46da5307, 0x00021aef91b0b2ab, 0x00033e3801f8f3cd, 0x000ace1edce870a2, 0x0000468f39384030 }, { 0x0007205ff5fc9697, 0x00068e744cefa48d, 0x0007ca7816e1c2c7, 0x000209163b6fc37a, 0x000044e445df560c }},
        {{ 0x000b1f606a304a7a, 0x00093328887de41e, 0x000b9b7f44ca4fe8, 0x0000c70175de0df0, 0x0000e1591f40e90c }, { 0x000901a9494fafda, 0x0006996e30ea15cb, 0x0001084797dc0097, 0x000bbd98f575816a, 0x00004054cb1612ae }},
        {{ 0x0009f2630bb6056d, 0x000a208d8fc35489, 0x00003d4999fd2fa9, 0x000f8f85a5010b5d, 0x000008b55032749d }, { 0x00083d306cbd6562, 0x00019afb4e3b76fc, 0x000bf00347df1b02, 0x0009b30ce14e51dd, 0x00007b2794a6a001 }},
        {{ 0x0007c824b04ea420, 0x000471577d4715c5, 0x000913a80d8575a1, 0x000c5c986d30019f, 0x0000d81758c67e87 }, { 0x000c66382097faea, 0x000692fd8f92a230, 0x00048f5357b258f0, 0x000eb2f9a4a557fd, 0x00000e310f808729 }},
        {{ 0x000f50250aaa7e2c, 0x000928ff83a92661, 0x0006e1b1983af0c4, 0x0003f70ebc7741c9, 0x00004f20a513065d }, { 0x000e8025b4418b73, 0x0007bc81477b6547, 0x000d976fbfe6b5d7, 0x000bb3597c65b900, 0x0000304bd4ae9453 }},
        {{ 0x000683125ec1f5d5, 0x00090ef12045cf9f, 0x000ea649d65eec4a, 0x000d990256fcb2ca, 0x00006feb137cf030 }, { 0x0005f11e11f4e767, 0x0003deae312410e0, 0x000766e7ca1eab36, 0x00013d80767e2867, 0x0000b79acfa040a4 }},
        {{ 0x0004f1f61b1ed88a, 0x00070ef9d3472c26, 0x0007628b1d2a9b6f, 0x0003d25ee3adff69, 0x00007d01488de9be }, { 0x000bb88d5112808b, 0x00094ab9bfb9093f, 0x000ea55ab0ade7e8, 0x0004be60d3df8ae8, 0x0000c9e427871646 }},
        {{ 0x000b283e83436d45, 0x000d64ce88c50d10, 0x00079d4eddcc4a93, 0x000ba49e47532e80, 0x0000ad9d0c90c59e }, { 0x0000afcff7b09b52, 0x000a09149e692749, 0x000787b8e224e80b, 0x0000d3e8eb96e966, 0x00009458dc2fcb01 }},
        {{ 0x0001382d0c39d19c, 0x0000783210b08568, 0x00025db81b7d5f44, 0x000933909a5a8060, 0x00007b38a809ed42 }, { 0x0004fa1871dda89f, 0x0001b5381927e27e, 0x000165fae9106b66, 0x00004c3ce46c63b3, 0x0000a770fae667f6 }},
        {{ 0x00070bd00e57f777, 0x00059bcd7466b255, 0x000a84178c56f6d2, 0x000449f8884ddbe1, 0x0000555f11ecdfbe }, { 0x000c7d249df1def7, 0x000bf29128bfa0f9, 0x000ccfa016cb0872, 0x0008881b192fce1b, 0x00005893d5343ec4 }},
        {{ 0x000ac38b3da9a9f7, 0x00031bf0bbc6229b, 0x000e25158959fa61, 0x000d9e8051c83c05, 0x0000da4e85bd1746 }, { 0x000896ef10770631, 0x00086e3ec0dfcfd2, 0x000e627395068017, 0x0000c7966f2ecda8, 0x000082a16d1f7601 }},
        {{ 0x0004a0885be18687, 0x000916a0fe9a745f, 0x000c5529e8801da0, 0x00096254908249fc, 0x0000b2610ba18a62 }, { 0x0009895a50a60550, 0x00067317d7ead588, 0x00058f0f76b3c827, 0x000274a89f96cf7d, 0x00000ccc62724ca9 }},
        {{ 0x00023b9b36df10cd, 0x000ebe5573cecfa3, 0x000e9f2368affd6b, 0x000f7772066497ea, 0x0000e7189610325f }, { 0x0009369a44ac6ad3, 0x000660a0115a3c12, 0x000b042f2fca2382, 0x000e839789414c40, 0x000036e7483f7882 }},
        {{ 0x000ee3e9bd875a0b, 0x00079f5903fa2711, 0x00029af6a861120e, 0x000561354e6da0fd, 0x00000c0f6913abd6 }, { 0x000948148819fd8a, 0x0008e1ce27a94979, 0x0003f4d9497c8870, 0x0002f21ca36d254e, 0x00009bc3a89fe40b }},
        {{ 0x0007862d5db75ad2, 0x0002c6940d9cae17, 0x000f9a1ae1ab94f3, 0x00036e44b0586957, 0x00009909fd1ff97d }, { 0x00039531fedb222b, 0x0005a6e503491266, 0x00010e2ff490e472, 0x0009425a62be30bd, 0x0000fb87c6381af0 }},
        {{ 0x0007bbabb687d4ad, 0x000bd11b29353593, 0x000677d066b77a80, 0x0008b697fda38aa9, 0x00008397f4e80b81 }, { 0x000dd9c0966b3966, 0x00013e02c9477784, 0x00052c405d304bb0, 0x00066490cd69a517, 0x000086e14cf5ff32 }},
        {{ 0x00059cf5ae217581, 0x000af4d64f16ea85, 0x000628d490757f4e, 0x000f26f3e8b1cac1, 0x0000aed72ff0985a }, { 0x000c6e2c5d3b9f54, 0x00023bb0e2af19ae, 0x0000b4b46034cba9, 0x000cf35201f717ec, 0x000060bfeddfa2f6 }},
        {{ 0x00077eaf4cee5651, 0x000997d3fca4752b, 0x00086a9346dd5414, 0x0009165b5f12c094, 0x0000c95b968e8b8d }, { 0x000b6d88c7322484, 0x000a381fda78fd37, 0x000faae161f6e327, 0x000d9a7c4848a83a, 0x0000f137df7a291d }},
        {{ 0x000cc1bade2b6b43, 0x0007647ce417d99b, 0x0002cecfb1e654f4, 0x000bbf0234b93dcc, 0x00004be18fe3d4f5 }, { 0x0000e6a5a4e609ba, 0x00076114bc5ee855, 0x000470a83c3f2818, 0x000b337f1a561be7, 0x00002e0b52e72fc0 }},
        {{ 0x000ae9bcf25208f9, 0x000071fa00b500bb, 0x000ba2941103788d, 0x0003b71855c098cf, 0x0000543d6d693c3a }, { 0x000d4f3fa4da32f1, 0x00035838435a2c5b, 0x000b8e5e069e748f, 0x0009a03e38ceaed6, 0x0000c306dcaca2f2 }},
        {{ 0x0000ccfe8e6e9bc7, 0x000ea10a3db32927, 0x000a4469c7dc6903, 0x000ae1fa9e28a37d, 0x000071256bb0ed1f }, { 0x0008ff50ee6ae731, 0x000b3e03ba464b2d, 0x000dd3aa52a9b6ac, 0x00044e1ab0f39d60, 0x0000e50f6112c18d }},
        {{ 0x000bd9abbfb15903, 0x0000d58fa28a7e00, 0x0005001c40b5393d, 0x000c99366c726cf2, 0x0000598cccad0929 }, { 0x00032a8b5f2075fd, 0x000f2074dfc47b3a, 0x000269286dbe5403, 0x00032c0e2d9c65fc, 0x0000df0b4df59f16 }},
        {{ 0x000db8b22b2b7d9a, 0x000e26112e832330, 0x0002cf110cfd3ee9, 0x000fa75610fbb351, 0x0000c773f37679e1 }, { 0x000b47b17faba833, 0x0009efb89be01445, 0x00066f0f5fbb5ec7, 0x00061961c8c77d24, 0x0000cb969988de53 }},
        {{ 0x000d2b61710647ca, 0x000d8b016287b52d, 0x000ff3a22d3f5266, 0x0008dbb73a134351, 0x00001e8a651d5fbf }, { 0x000061da3d7648b8, 0x00081291da3b3b75, 0x0008b862d1d32fd2, 0x000cf1d835f946d7, 0x00009b66ecb267fb }},
        {{ 0x0005057a8f03a393, 0x000010dec1b49919, 0x0000a1ef0b1f2d70, 0x000769be0f12195e, 0x0000fd15ee98236f }, { 0x0007f25d916cad26, 0x000a3112e1301367, 0x0005bbd3af843715, 0x00053c82b97dfbb3, 0x000049bfae8df046 }},
        {{ 0x0001f37cbd3c62c7, 0x00088986d840cd64, 0x0000a440abcf9eb9, 0x00065da61471d62c, 0x00009f893242a192 }, { 0x000904261943916d, 0x000350f4b7099851, 0x000bd6a472422402, 0x000fcf7467a7bd33, 0x0000da56f786a44f }},
        {{ 0x00056fbb9354d802, 0x0008467cc36975fb, 0x0007e7ef6267c1ac, 0x00063a89402acf16, 0x000003023c61410c }, { 0x0001b7d9bedec678, 0x000950de9e405698, 0x000eb390c630ef9a, 0x0002b13977b99c65, 0x00009969974b5c5f }},
        {{ 0x0007c7289562f9a3, 0x00075c0c716cc0a1, 0x00006da1f3b47558, 0x0003bc7c6f4b5ff9, 0x00000ff02fe0d66b }, { 0x000283d331c15563, 0x000ed85da6be7de9, 0x000da07e001d37e7, 0x0009ee5e8ff71530, 0x00002b7baaf41117 }},
        {{ 0x000b9d6b1bdf4ea1, 0x0003c992a8498705, 0x000ae5b9f8196de5, 0x0004d3fe5a716964, 0x00004e830f707f38 }, { 0x000e1481a495c70b, 0x000ccd4a52f313bc, 0x000f9565c4bba5c4, 0x00077168fc9dd959, 0x0000dff7b5e2bd75 }},
        {{ 0x0004af860fe1d658, 0x0005c3a43228d831, 0x00003626b989c96b, 0x000ceba2924ae1c3, 0x0000c45b79310a64 }, { 0x0002ceb1de0d0667, 0x00088613f714aa18, 0x000d68a9c780c9b4, 0x000a36f94f51865a, 0x0000055e19d4f0d9 }},
    },
};
/* clang-format on */

#endif /* #if !defined(_DISABLE_ECP_256R1_HARDCODED_BP_TBL_) */

IPP_OWN_DEFN(const cpPrecompAP*, gfpec_precom_nistP256r1_radix52_fun, (void))
{
    static cpPrecompAP t = { /* w */ 7,
                             /* select function */ p256r1_select_ap_w7_ifma,
                             /* precomputed data */ (BNU_CHUNK_T*)ifma_ec_nistp256r1_bp_precomp };
    return &t;
}

#endif // (_IPP32E >= _IPP32E_K1)

#endif /* #ifndef IFMA_ECPRECOMP7_P256_H */
