/*
 * Copyright 2021-2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.events.git;

import org.apiguardian.api.API;
import org.opentest4j.reporting.events.api.Factory;

import static org.apiguardian.api.API.Status.EXPERIMENTAL;

/**
 * Factory for elements of the Git namespace.
 *
 * @since 0.2.0
 */
@API(status = EXPERIMENTAL, since = "0.2.0")
public class GitFactory {

	private GitFactory() {
	}

	/**
	 * Create a factory for {@link Repository} elements.
	 *
	 * @return Repository factory
	 */
	public static Factory<Repository> repository() {
		return Repository::new;
	}

	/**
	 * Create a factory for {@link Branch} elements.
	 *
	 * @param name the branch name
	 * @return Branch factory
	 */
	public static Factory<Branch> branch(String name) {
		return context -> new Branch(context, name);
	}

	/**
	 * Create a factory for {@link Commit} elements.
	 *
	 * @param hash the commit hash
	 * @return Commit factory
	 */
	public static Factory<Commit> commit(String hash) {
		return context -> new Commit(context, hash);
	}

	/**
	 * Create a factory for {@link Status} elements.
	 *
	 * @param output the status output of Git
	 * @return Status factory
	 */
	public static Factory<Status> status(String output) {
		return context -> new Status(context, output);
	}

}
