//===-- Implementation header for expf16 ------------------------*- C++ -*-===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_LIBC_SRC___SUPPORT_MATH_EXPF16_H
#define LLVM_LIBC_SRC___SUPPORT_MATH_EXPF16_H

#include "include/llvm-libc-macros/float16-macros.h"

#ifdef LIBC_TYPES_HAS_FLOAT16

#include "hdr/errno_macros.h"
#include "hdr/fenv_macros.h"
#include "src/__support/FPUtil/FEnvImpl.h"
#include "src/__support/FPUtil/FPBits.h"
#include "src/__support/FPUtil/PolyEval.h"
#include "src/__support/FPUtil/cast.h"
#include "src/__support/FPUtil/except_value_utils.h"
#include "src/__support/FPUtil/rounding_mode.h"
#include "src/__support/common.h"
#include "src/__support/macros/config.h"
#include "src/__support/macros/optimization.h"

#include "expf16_utils.h"

namespace LIBC_NAMESPACE_DECL {

namespace math {

static constexpr float16 expf16(float16 x) {
#ifndef LIBC_MATH_HAS_SKIP_ACCURATE_PASS
  constexpr fputil::ExceptValues<float16, 2> EXPF16_EXCEPTS_LO = {{
      // (input, RZ output, RU offset, RD offset, RN offset)
      // x = 0x1.de4p-8, expf16(x) = 0x1.01cp+0 (RZ)
      {0x1f79U, 0x3c07U, 1U, 0U, 0U},
      // x = 0x1.73cp-6, expf16(x) = 0x1.05cp+0 (RZ)
      {0x25cfU, 0x3c17U, 1U, 0U, 0U},
  }};

  constexpr fputil::ExceptValues<float16, 3> EXPF16_EXCEPTS_HI = {{
      // (input, RZ output, RU offset, RD offset, RN offset)
      // x = 0x1.c34p+0, expf16(x) = 0x1.74cp+2 (RZ)
      {0x3f0dU, 0x45d3U, 1U, 0U, 1U},
      // x = -0x1.488p-5, expf16(x) = 0x1.ebcp-1 (RZ)
      {0xa922U, 0x3bafU, 1U, 0U, 0U},
      // x = -0x1.55p-5, expf16(x) = 0x1.ebp-1 (RZ)
      {0xa954U, 0x3bacU, 1U, 0U, 0U},
  }};
#endif // !LIBC_MATH_HAS_SKIP_ACCURATE_PASS

  using FPBits = fputil::FPBits<float16>;
  FPBits x_bits(x);

  uint16_t x_u = x_bits.uintval();
  uint16_t x_abs = x_u & 0x7fffU;

  // When 0 < |x| <= 2^(-5), or |x| >= 12, or x is NaN.
  if (LIBC_UNLIKELY(x_abs <= 0x2800U || x_abs >= 0x4a00U)) {
    // exp(NaN) = NaN
    if (x_bits.is_nan()) {
      if (x_bits.is_signaling_nan()) {
        fputil::raise_except_if_required(FE_INVALID);
        return FPBits::quiet_nan().get_val();
      }

      return x;
    }

    // When x >= 12.
    if (x_bits.is_pos() && x_abs >= 0x4a00U) {
      // exp(+inf) = +inf
      if (x_bits.is_inf())
        return FPBits::inf().get_val();

      switch (fputil::quick_get_round()) {
      case FE_TONEAREST:
      case FE_UPWARD:
        fputil::set_errno_if_required(ERANGE);
        fputil::raise_except_if_required(FE_OVERFLOW);
        return FPBits::inf().get_val();
      default:
        return FPBits::max_normal().get_val();
      }
    }

    // When x <= -18.
    if (x_u >= 0xcc80U) {
      // exp(-inf) = +0
      if (x_bits.is_inf())
        return FPBits::zero().get_val();

      fputil::set_errno_if_required(ERANGE);
      fputil::raise_except_if_required(FE_UNDERFLOW | FE_INEXACT);

      switch (fputil::quick_get_round()) {
      case FE_UPWARD:
        return FPBits::min_subnormal().get_val();
      default:
        return FPBits::zero().get_val();
      }
    }

    // When 0 < |x| <= 2^(-5).
    if (x_abs <= 0x2800U && !x_bits.is_zero()) {
#ifndef LIBC_MATH_HAS_SKIP_ACCURATE_PASS
      if (auto r = EXPF16_EXCEPTS_LO.lookup(x_u); LIBC_UNLIKELY(r.has_value()))
        return r.value();
#endif // !LIBC_MATH_HAS_SKIP_ACCURATE_PASS

      float xf = x;
      // Degree-3 minimax polynomial generated by Sollya with the following
      // commands:
      //   > display = hexadecimal;
      //   > P = fpminimax(expm1(x)/x, 2, [|SG...|], [-2^-5, 2^-5]);
      //   > 1 + x * P;
      return fputil::cast<float16>(
          fputil::polyeval(xf, 0x1p+0f, 0x1p+0f, 0x1.0004p-1f, 0x1.555778p-3f));
    }
  }

#ifndef LIBC_MATH_HAS_SKIP_ACCURATE_PASS
  if (auto r = EXPF16_EXCEPTS_HI.lookup(x_u); LIBC_UNLIKELY(r.has_value()))
    return r.value();
#endif // !LIBC_MATH_HAS_SKIP_ACCURATE_PASS

  // exp(x) = exp(hi + mid) * exp(lo)
  auto [exp_hi_mid, exp_lo] = exp_range_reduction(x);
  return fputil::cast<float16>(exp_hi_mid * exp_lo);
}

} // namespace math

} // namespace LIBC_NAMESPACE_DECL

#endif // LIBC_TYPES_HAS_FLOAT16

#endif // LLVM_LIBC_SRC___SUPPORT_MATH_EXPF16_H
