## MIME types

`resources/mime-types.json` and `resources/generated-mime-types.json` contain MIME type tests. The tests are encoded as a JSON array. String values in the array serve as documentation. All other values are objects with the following fields:

* `input`: The string to be parsed.
* `output`: Null if parsing resulted in failure and the MIME type record serialized as string otherwise.
* `navigable`: True if the MIME type can be used for a document to be loaded in a browsing context (i.e., does not result in a download) and omitted otherwise.
* `encoding`: The encoding that can be extracted from the MIME type or null if no encoding can be extracted, and omitted otherwise.

Note: the object description implies that there tests without `navigable` or `encoding` set.

A wrapper for these JSON MIME type tests needs to take care that not all `input` values can be tested in all entrypoints. Some entrypoints only accept bytes and some have further restrictions. A function such as the one below can be used to differentiate:

```js
function isByteCompatible(str) {
  // see https://fetch.spec.whatwg.org/#concept-header-value-normalize
  if(/^[\u0009\u0020\u000A\u000D]+|[\u0009\u0020\u000A\u000D]+$/.test(str)) {
    return "header-value-incompatible";
  }

  for(let i = 0; i < str.length; i++) {
    const charCode = str.charCodeAt(i);
    // See https://fetch.spec.whatwg.org/#concept-header-value
    if(charCode > 0xFF) {
      return "incompatible";
    } else if(charCode === 0x00 || charCode === 0x0A || charCode === 0x0D) {
      return "header-value-incompatible";
    }
  }
  return "compatible";
}
```

`resources/generated-mime-types.json` is generated by running `resources/generated-mime-types.py`. Modify the latter to correct the former.

These tests are used by resources in this directory to test various aspects of MIME types.

## MIME type groups

`resources/mime-groups.json` contains MIME type group-membership tests. The tests are encoded as a JSON array. String values in the array serve as documentation. All other values are objects with the following fields:

* `input`: The MIME type to test.
* `groups`: An array of zero or more groups to which the MIME type belongs.

In order to pass the tests an implementation must treat each MIME type as belonging to the exact set of groups listed, with no additions or omissions.

Note: As MIME type groups are used only while determining the computed MIME type of a resource and are not exposed in any API, no browser-based test harness is available.
