##  Copyright (C) 2006,2007,2008,2009,2010  Carlo de Falco, Massimiliano Culpo
##
##  This file is part of:
##         FPL - Fem PLotting package for octave
##
##  FPL is free software; you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation; either version 3 of the License, or
##  (at your option) any later version.
##
##  FPL is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with FPL; If not, see <http://www.gnu.org/licenses/>.
##
##  author: Carlo de Falco     <cdf _AT_ users.sourceforge.net>
##  author: Massimiliano Culpo <culpo _AT_ users.sourceforge.net>

## -*- texinfo -*-
## @deftypefn {Function File} {} fpl_dx_write_series (@var{basename}, @
## @var{mesh}, @var{u}, @var{sp}, @var{attr_name}, @var{attr_rank},  @
## @var{attr_shape})
##
## Output data series in ASCII Open-DX format.
##
## @var{basename} is a string containing the base-name of the dx file where the
## data will be saved.
##
## @var{mesh} is a PDE-tool like mesh, like the ones generated by the
## "msh" package.
##
## @var{u} is the series to be saved. It should represent scalar, vector
## or tensor of doubles. @var{sp} is the vector of the sampled points
## (e.g. time points in the case of a time series).
##
## @var{attr_name} is a descriptive name for the series @var{u}, while
## @var{attr_rank} is the rank of the field items (0 for scalar, 1 for
## vector, etc.) and @var{attr_shape} is the number of components of the
## field items (assumed 1 for scalar).
##
## @seealso{fpl_dx_write_field}
##
## @end deftypefn

function fpl_dx_write_series(basename,mesh,u,tp,attr_name,attr_rank,attr_shape)

  ## FIXME: add append capabilities as in fpl_dx_write_field??

  if nargin!=7
    error("fpl_dx_write_series: wrong number of input");
  endif

  if !ischar(basename)
    error("fpl_dx_write_series: basename should be a valid string");
  elseif !( isstruct(mesh) )
    error("fpl_dx_write_series: mesh should be a valid structure");
  elseif !ismatrix(u)
    error("fpl_dx_write_series: u should be a valid matrix");
  elseif !ischar(attr_name)
    error("fpl_dx_write_series: attr_name should be a valid string");
  elseif !isscalar(attr_rank)
    error("fpl_dx_write_series: attr_rank should be a valid scalar");
  elseif !isscalar(attr_shape)
    error("fpl_dx_write_series: attr_shape should be a valid scalar");
    ##elseif !isscalar(endfile)
    ##error("fpl_dx_write_series: endfile should be a valid scalar");
  endif

  filename = [basename ".dx"];

  npoints = length(tp);

  p   = mesh.p';
  dim = columns(p); # 2D or 3D
  
  if dim == 2
    t = mesh.t(1:3,:)';
  elseif dim == 3
    t = mesh.t(1:4,:)';
  else
    error("fpl_dx_write_series: neither 2D triangle nor 3D tetrahedral mesh");    
  endif
  
  nnodes = rows(p);
  nelems = rows(t);
  ndatas = rows(u);

  if ndatas == nnodes
    dep = "positions";
  elseif ndatas == nelems
    dep = "connections";
  else
    error("fpl_dx_write_series: neither position nor connection data type")
  endif

  ## Write field items to file
  idx = (1:attr_shape);
  for ii = 1:npoints
    field = u(:,idx);
    fname = sprintf("%s.%d",attr_name,ii);
    fpl_dx_write_field(basename,mesh,field,fname,attr_rank,attr_shape,0);
    idx  += attr_shape;
  endfor
  
  ##if endfile
  fid = fopen(filename,"a");
  fprintf (fid, "object \"%s_series\" class series\n",attr_name);
  for ii = 1:npoints
    fname = sprintf("%s.%d",attr_name,ii);
    fprintf (fid,"member %d position %g value \"%s\"\n",ii-1,tp(ii),fname);
  endfor
  fprintf (fid, "\nend\n");
  fclose(fid);
  ##endif

endfunction