/*
 *  Copyright (c) 2008,2009,2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Parser_p.h"

#include "GTLCore/Debug.h"

#include "Compiler_p.h"
#include "Lexer_p.h"

#include <GTLCore/AST/Expression.h>
#include <GTLCore/AST/Statement.h>
#include <GTLCore/AST/Tree.h>
#include <GTLCore/CompilerBase_p.h>
#include <GTLCore/Macros_p.h>
#include <GTLCore/Type.h>
#include <GTLCore/TypesManager.h>
#include <GTLCore/Value.h>
#include <GTLCore/VariableNG_p.h>
#include <GTLCore/VariablesManager_p.h>
#include <GTLCore/Function.h>
#include <GTLCore/Parameter.h>
#include <GTLCore/ConvertCenter_p.h>
#include <GTLCore/AST/BinaryExpression.h>
#include <GTLCore/AST/FunctionDeclaration.h>

namespace AST = GTLCore::AST;
using namespace GTLFragment;

struct Parser::Private {
  GTLCore::String kernelName;
  GTLCore::AST::Tree* tree;
  Compiler* compiler;
  std::list<GTLCore::VariableNG*> dependents;
  std::list<GTLCore::AST::Statement*> dependentsExpressions;
  bool parsingStarted;
  bool dependentFunctionParsed;
};

Parser::Parser( Compiler* _compiler , Lexer* _lexer) : ParserBase(_compiler, _lexer), d(new Private)
{
  d->compiler = _compiler;
  d->tree = new GTLCore::AST::Tree();
  d->parsingStarted = false;
  d->dependentFunctionParsed = false;
}

Parser::~Parser()
{
  delete d;
}

GTLCore::AST::Tree* Parser::parse()
{
  GTL_DEBUG("Parsing with ParserNG");
  GTL_ASSERT( d->tree );
  GTL_ASSERT( not d->parsingStarted );

  d->parsingStarted = true;
  variablesManager()->startContext();
  
  getNextToken();
  // Start by ingoring metadata if any
  if( currentToken().type == GTLCore::Token::INFERIOR )
  {
    int count = 0;
    do
    {
      switch( currentToken().type )
      {
        case GTLCore::Token::INFERIOR:
          ++count;
          break;
        case GTLCore::Token::SUPPERIOR:
          --count;
          break;
        default:
          break;
      }
      getNextToken();
    } while( count > 0 and currentToken().type != GTLCore::Token::END_OF_FILE  );
    if( isOfType( currentToken(), GTLCore::Token::SEMI ) )
    {
      getNextToken();
    }
  }
  
  // Parse "import"
  while( currentToken().type == GTLCore::Token::IMPORT )
  {
    getNextToken();
    if( isOfType( currentToken(), GTLCore::Token::STRING_CONSTANT ) )
    {
      d->compiler->importModule( currentToken().string );
    }
    checkNextTokenIsSemi();
    getNextToken();
  }
  
  GTL_DEBUG("Parse body");
  // Parse "kernel" or "library"
  if( (d->compiler->libraryType() == Library::SHIVA_KERNEL and isOfType( currentToken(), GTLCore::Token::KERNEL ) )
      or (d->compiler->libraryType() == Library::RIJN_SKETCH and isOfType( currentToken(), GTLCore::Token::SKETCH ) )
      or isOfType( currentToken(), GTLCore::Token::LIBRARY ) )
  {
    getNextToken();
    if( isOfType( currentToken(), GTLCore::Token::IDENTIFIER ) )
    {
      d->kernelName = currentToken().string;
      GTL_ASSERT( d->kernelName != "" );
      GTL_DEBUG( d->compiler->whichStdLibrary() << " " << d->kernelName );
      if( d->compiler->whichStdLibrary() == NotAStdLibrary || d->compiler->whichStdLibrary() == RijnWrappersLibrary  || d->compiler->whichStdLibrary() == ShivaWrappersLibrary )
      {
        setNameSpace( d->kernelName );
        GTL_ASSERT( d->kernelName == nameSpace() );
      }
      GTL_DEBUG("Set parameters as constants");
      // Set parameters as constants
      for( std::map< GTLCore::String, GTLCore::Value >::const_iterator it = d->compiler->parameters().begin();
          it != d->compiler->parameters().end(); ++it )
      {
        GTLCore::ScopedName scopedName( nameSpace(), it->first );
        const GTLCore::Type* declarationType = it->second.type();
        if (declarationType == GTLCore::Type::Color)
        {
          declarationType = typesManager()->getStructure("color");
          GTL_ASSERT(declarationType);
        }
        AST::GlobalConstantDeclaration* gcd = new AST::GlobalConstantDeclaration(
            scopedName, declarationType,
            GTLCore::AST::Expression::fromValue( it->second, declarationType ), true, false );
        variablesManager()->declareConstant( scopedName, gcd->variable() );
        tree()->append( gcd );
      }
      getNextToken();
      if( isOfType( currentToken(), GTLCore::Token::STARTBRACE ) )
      {
        getNextToken();
        parseKernelBody();
        isOfType( currentToken(), GTLCore::Token::ENDBRACE );
        getNextToken();
      }
    }
  }
  if(not d->dependentFunctionParsed and not d->dependentsExpressions.empty())
  {
    // Create a dependent function
    AST::FunctionDeclaration* fd = new AST::FunctionDeclaration( GTLCore::ScopedName(nameSpace(), "evaluateDependents"), GTLCore::Type::Void, std::vector<AST::FunctionParameter*>() );
    fd->setStatement(new AST::StatementsList(d->dependentsExpressions));
    d->tree->append(fd);
  }
  
  return d->tree;
}

void Parser::parseKernelBody()
{
  while(true)
  {
    switch(currentToken().type)
    {
      case GTLCore::Token::END_OF_FILE:
      case GTLCore::Token::ENDBRACE:
        return;
      case GTLCore::Token::CONST:
        parseConstantDeclaration();
        break;
      case GTLCore::Token::STRUCT:
        parseStructDefinition();
        break;
      case GTLCore::Token::DEPENDENT:
        parseDependentDeclaration();
        break;
      default:
      {
        if( currentToken().isFunctionType() )
        {
          parseFunction();
        } else {
          GTL_DEBUG("unexpected " << GTLCore::Token::typeToString( currentToken().type ) );
          reportUnexpected( currentToken() );
          getNextToken();
        }
      }
    }
  }
}

GTLCore::AST::Tree* Parser::tree()
{
  return d->tree;
}

AST::Statement* Parser::parseStatement()
{
  if( isType(currentToken()) )
  {
    return parseVariableDeclaration();
  }
  switch( currentToken().type )
  {
    case GTLCore::Token::CONST:
      return parseVariableDeclaration();
    case GTLCore::Token::STRUCT:
    {
      parseStructDefinition();
      return new AST::DummyStatement();
    }
    case GTLCore::Token::STARTBRACKET:
    case GTLCore::Token::IDENTIFIER:
      return parseExpressionStatement();
    case GTLCore::Token::IF:
      return parseIfStatement();
    case GTLCore::Token::WHILE:
      return parseWhileStatement();
    case GTLCore::Token::FOR:
      return parseForStatement();
    case GTLCore::Token::RETURN:
      return parseReturnStatement();
    case GTLCore::Token::STARTBRACE:
    {
      variablesManager()->startContext();
      AST::Statement* statement = appendCurrentContextGarbageCollecting(parseStatementList());
      variablesManager()->endContext();
      return statement;
    }
    case GTLCore::Token::PRINT:
    {
      return parsePrintStatement();
    }
    default:
      if( currentToken().isUnaryOperator() or currentToken().isConstant() )
      {
        return parseExpressionStatement();
      }
      GTL_DEBUG("unexpected");
      reportUnexpected( currentToken() );
      getNextToken();
      return 0;
  }
}

const GTLCore::Type* Parser::parseType()
{
  switch( currentToken().type )
  {
    case GTLCore::Token::BOOL2:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Boolean, 2 );
    case GTLCore::Token::BOOL3:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Boolean, 3 );
    case GTLCore::Token::BOOL4:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Boolean, 4 );
    case GTLCore::Token::BOOLN:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Boolean, d->compiler->channelsNb() );
    case GTLCore::Token::INT2:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Integer32, 2 );
    case GTLCore::Token::INT3:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Integer32, 3 );
    case GTLCore::Token::INT4:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Integer32, 4 );
    case GTLCore::Token::INTN:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Integer32, d->compiler->channelsNb() );
    case GTLCore::Token::FLOAT2:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Float32, 2 );
    case GTLCore::Token::FLOAT3:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Float32, 3 );
    case GTLCore::Token::FLOAT4:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Float32, 4 );
    case GTLCore::Token::FLOATN:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Float32, d->compiler->channelsNb() );
    default:
      return GTLCore::ParserBase::parseType();
  }
}

void Parser::parseDependentDeclaration()
{
  if(d->dependentFunctionParsed)
  {
    reportError("Dependents must be declared before the dependent function", currentToken());
  }
  getNextToken();
  const GTLCore::Type* type = parseType();
  if( not type ) return;
  if( type->dataType() == GTLCore::Type::STRUCTURE 
      and ( type->structName().startWith("pixel")
            or type->structName().startWith("image") ) )
  {
    reportError( type->structName() + " can't be a dependent", currentToken());
  }
  while(true)
  {
    if( isOfType( currentToken(), GTLCore::Token::IDENTIFIER ) )
    {
      GTLCore::String name = currentToken().string;
      getNextToken();
      GTL_DEBUG("Dependent: " << name);
      std::list<int> memberArraySize = expressionsListToIntegersList( parseArraySize(true) );
      type = d->compiler->typesManager()->getArray( type, memberArraySize.size() );
      GTLCore::ScopedName scopedName( nameSpace(), name );
      if( tree()->containsGlobalConstant( scopedName ) )
      {
        reportError("Constant '" + scopedName.toString() + "' has already been declared", currentToken());
      } else {
        AST::GlobalConstantDeclaration* gcd = new AST::GlobalConstantDeclaration( scopedName, type , memberArraySize, true, false );
        variablesManager()->declareConstant( scopedName, gcd->variable() );
        d->dependents.push_back( gcd->variable() );
        tree()->append( gcd );
        if(currentToken().type == GTLCore::Token::EQUAL) {
          getNextToken();
          AST::Expression* expression = parseExpression(false, gcd->type());
          expression = d->compiler->convertCenter()->createConvertExpression(expression, gcd->type());
          gcd->variable()->setConstant(false);
          d->dependentsExpressions.push_back(new GTLCore::AST::AssignementBinaryExpression( new GTLCore::AST::VariableAccessorExpression(gcd->variable()), expression ));
          gcd->variable()->setConstant(true);
        }
      }
    } else {
      GTL_DEBUG("Unexpected");
      reportUnexpected( currentToken() );
      reachNextSemi();
    }
    if( currentToken().type == GTLCore::Token::COMA ) {
      getNextToken();
    } else {
      break;
    }
  }
  isOfType( currentToken(), GTLCore::Token::SEMI );
  getNextToken();
}

AST::StatementsList* Parser::startParsingFunction( const GTLCore::String& _name )
{
  if( _name == "evaluateDependents" )
  {
    d->dependentFunctionParsed = true;
    foreach( GTLCore::VariableNG* var, d->dependents )
    {
      var->setConstant( false );
    }
    if(d->dependentsExpressions.empty())
    {
      return 0;
    } else {
      return new AST::StatementsList(d->dependentsExpressions);
    }
  }
  return 0;
}

void Parser::endParsingFunction( const GTLCore::String& _name )
{
  if( _name == "evaluateDependents" )
  {
    foreach( GTLCore::VariableNG* var, d->dependents )
    {
      var->setConstant( true );
    }
  }
}

bool isRegion(const GTLCore::Type* t)
{
  return t->isStructure() and t->structName() == "region";
}

bool Parser::validateFunctionParameters(const GTLCore::Function* function)
{
  if(function->name().name() == "evaluatePixel") {
    if(function->parameters().empty()) return false;
    std::size_t last_idx = function->parameters().size() - 1;
    for(std::size_t i = 0; i < last_idx; ++i)
    {
      const GTLCore::Type* t = function->parameters()[i].type();
      if(not t->isStructure() and function->parameters()[i].isOutput() and not t->structName().startWith("image"))
      {
        return false;
      }
    }
    const GTLCore::Type* t = function->parameters()[last_idx].type();
    return t->isStructure() and function->parameters()[last_idx].isOutput() and t->structName().startWith("pixel");
  } else if(function->name().name() == "changed" or function->name().name() == "needed") {
    if(function->parameters().size() != 3) return false;
    const GTLCore::Type* t0 = function->parameters()[0].type();
    const GTLCore::Type* t1 = function->parameters()[1].type();
    const GTLCore::Type* t2 = function->parameters()[2].type();
    return isRegion(function->returnType()) and isRegion(t0) and t1 == GTLCore::Type::Integer32 and t2->dataType() == GTLCore::Type::ARRAY and isRegion(t2->embeddedType());
  } else if(function->name().name() == "generated") {
    return function->parameters().empty() and isRegion(function->returnType());
  } 
  return true;
}
