#-*- coding:utf-8 -*-

"""
This file is part of OpenSesame.

OpenSesame is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

OpenSesame is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with OpenSesame.  If not, see <http://www.gnu.org/licenses/>.
"""

from libopensesame import item, exceptions, generic_response, debug
import openexp.canvas
import openexp.exceptions

class sketchpad(item.item, generic_response.generic_response):
	
	"""The sketchpad item"""
	
	def __init__(self, name, experiment, string=None):

		"""
		Constructor

		Arguments:
		name		--	name of the item
		experiment	--	experiment

		Keyword arguments:
		string		--	definition string (default=None)
		"""

		self.duration = 'keypress'
		self.start_response_interval = 'no'
		self.item_type = 'sketchpad'
		self.elements = []
		if not hasattr(self, 'description'):
			self.description = 'Displays stimuli'
		item.item.__init__(self, name, experiment, string)
		
	def from_string(self, string):

		"""
		Reads a sketchpad from string.

		Arguments:
		string		--	the unicode string containing the sketchpad definition
		"""
		
		for line in string.split(u'\n'):
			if not self.parse_variable(line):
				l = self.split(line)
				if len(l) > 0:
					if l[0] == u'draw':
						if len(l) == 1:
							raise exceptions.script_error( \
								"Incomplete draw command '%s'" % line)
						element_name = l[1]
						if not hasattr(elements, element_name):
							raise exceptions.script_error( \
								"Unknown sketchpad element '%s'" % element_name)
						element_class = getattr(elements, element_name)
						element = element_class(self, l)
						self.elements.append(element)
					else:
						raise exceptions.script_error("Unknown command '%s'" \
							% line)
							
	def prepare(self):
		
		"""Prepares the sketchpad canvas."""
		
		self.canvas = canvas(self.experiment)
		for element in self.elements:
			element.render()
	
	def run(self):
		
		"""Shows the sketchpad canvas and optionally sleep or wait for input."""

		self.set_item_onset(self.canvas.show())
		self.set_sri(self._reset)
		self.process_response()
		return True
		
	def to_string(self):

		"""
		Encode sketchpad item as string

		Returns:
		A definition string
		"""

		s = item.item.to_string(self, self.item_type)
		for element in self.elements:
			s += u'\t%s\n' % element.to_string()
		return s

	def var_info(self):

		"""
		Return a list of dictionaries with variable descriptions

		Returns:
		A list of (name, description) tuples
		"""

		l = item.item.var_info(self)
		if self.get("duration", _eval=False) in ["keypress", "mouseclick"]:
			l += generic_response.generic_response.var_info(self)
		return l