[manpage_begin tepam n 0.4.0]
[copyright {2009/2010, Andreas Drollinger}]
[moddesc {Tcl's Enhanced Procedure and Argument Manager}]
[titledesc {An introduction into TEPAM, Tcl's Enhanced Procedure and Argument Manager}]
[category {Procedures, arguments, parameters, options}]

[description]
[para]

This document is an informal introduction into TEPAM, the Tcl's Enhanced Procedure and Argument Manager. Detailed information to the TEPAM package is provided in the [emph tepam::procedure] and [emph tepam::argument_dialogbox] reference manuals.

[section "OVERVIEW"]

This package provides a new Tcl procedure declaration syntax that simplifies the implementation of procedure subcommands and the handling of the different types of procedure arguments like flags or switches, options, unnamed arguments, optional and mandatory options and arguments, default values, etc. Procedure declarations can be enriched with detailed information about the procedure and its arguments. This information is used for the following purposes:
[para]
First of all, a preamble is added in front of the body of a procedure that is declared with TEPAM. This preamble calls an argument manager that that uses the provided information to check the validity of the argument types and values before the procedure body is executed. Then, the information is used to generate help and usage texts if requested, or to generate clear error message in case an argument validation fails. The information also allows generating automatically graphical forms that allows an interactive definition of all arguments, in case a procedure is called interactively. And finally, the additional information helps self-commenting in a clean way the declaration of a procedure and of all its arguments.
[para]
The graphical form generator that creates the necessary argument specification forms for the interactive procedure calls is also available for other purposes than for procedure argument specifications. It allows creating code efficiently complex parameter entry forms that are usable independently from TEPAM's new procedure definition method.

[para]

Here is a short overview about all major TEPAM features:

[list_begin itemized]

[item]
New self-documenting procedure declaration syntax: The additional information to declare properly a procedure has not to be provided with additional statements, but can be added in a natural syntax directly into the procedure header.

[item]
Easy way to specify subcommands: A subcommand is declared like a procedure, simply with a procedure name composed by a base name followed by a subcommand name. Sub-subcommands are created identically using simply procedure names composed by 3 words.

[item]
Flexible usage of flags (switches), options (named arguments) and unnamed arguments. Option names are optionally automatically completed.

[item]
Support for default values, mandatory/optional options and arguments, choice lists, value ranges, multiple usable options/arguments.

[item]
Choice of a [emph {named arguments first, unnamed arguments later}] procedure calling style (typical for Tcl commands) or of an [emph {unnamed arguments first, named arguments later}] procedure calling style (typical for Tk commands).

[item]
In case the [emph {named arguments first, unnamed arguments later}] style (Tcl) is selected:  Clear separation between options and arguments via the "--" flag. The unnamed arguments can optionally be accessed as options (named arguments).

[item]
Automatic type and value check before the procedure body is executed, taking into account validation ranges, choice lists and custom validation commands. Generation of clear error message if necessary.

[item]
Many predefined types exist (integer, boolean, double, color, file, font, ...). Other application specific types can easily be added.

[item]
Automatic help and usage text generation if a procedure is called with the [arg -help] flag.

[item]
Automatic generation of an interactive argument definition form, in case a procedure is called with the [arg -interactive] flag.

[item]
Procedure calls can be logged which is useful to get for interactively called procedures the command call lines.

[item]
Powerful and code efficient generation of complex parameter definition forms.

[list_end]

[section "PROCEDURE DECLARATION"]

TEPAM's procedure declaration syntax is simple and self-explaining. Instead of declaring a procedure with the Tcl key word [cmd proc], a procedure is declared with the TEPAM command [cmd procedure] which takes as [cmd proc] also 3 arguments: The procedure name, the procedure header and the procedure body.
[para]
The following example declares the subcommand [cmd message] of the procedure [cmd display]. This command has several named and unnamed arguments:

[example_begin][cmd tepam::procedure] {display message} {
   -return            -
   -short_description "Displays a simple message box"
   -description       "This procedure allows displaying a configurable message box.
   -args {
      {-mtype -default Warning -choices {Info Warning Error} -description "Message type"}
      {-font -type font -default {Arial 10 italic} -description "Message text font"}
      {-level -type integer -optional -range {1 10} -description "Message level"}
      {-fg -type color -default black -description "Message color"}
      {-bg -type color -optional -description "Background color"}
      {-no_border -type none -description "Use a splash window style (no border)"}
      {-log_file -type file -optional -description "Optional message log file"}
      {text -type string -multiple -description "Multiple text lines to display"}
   }
} {
[emph {   puts "display message:"
   foreach var {mtype font level fg bg no_border log_file text} {
      if {[info exists $var]} {
         puts  "  $var=[set $var]"
      }
   }
}]}[example_end]

A call of procedure that has been declared in this way will first invoke the TEPAM argument manager, before the procedure body is executed. The argument manager parses the provided arguments, validates them, completes them eventually with some default values, and makes them finally available to the procedure body as local variables. In case an argument is missing or has a wrong type, the argument manager generates an error message that explains the reason for the error. 
[para]
As the example above shows, the TEPAM command [cmd procedure] accepts subcommand definitions as procedure name and allows defining much more information than just the argument list inside the procedure header. The procedure body on the other hand is identical between a command declared with [cmd proc] and a command declared with [cmd procedure].
[para]
The procedure header allows defining in addition to the arguments some procedure attributes, like a description, information concerning the return value, etc. This information is basically used for the automatic generation of comprehensive help and usage texts.
[para]
A list of argument definition statements assigned to the [arg -args] argument is defining the procedure arguments. Each argument definition statement starts with the argument name, optionally followed by some argument attributes.
[para]
Three types of arguments can be defined: Unnamed arguments, named arguments and flags. The distinction between the named and unnamed arguments is made by the first argument name character which is simply "-" for named arguments. A flag is defined as named argument that has the type  [emph none].
[para]
Named and unnamed arguments are mandatory, unless they are declared with the [arg -optional] flag and unless they have a default value specified with the [arg -default] option. Named arguments and the last unnamed argument can have the attribute [arg -multiple], which means that they can be defined multiple times. The expected argument data type is specified with the [arg -type] option. TEPAM defines a large set of standard data types which can easily be completed with application specific data types.
[para]
The argument declaration order has only an importance for unnamed arguments that are by default parsed after the named arguments (Tcl style). A variable allows changing this behavior in a way that unnamed arguments are parsed first, before the named arguments (Tk style).

[section "PROCEDURE HELP"]

The declared procedure can simply be called with the [arg -help] option to get the information about the usage of the procedure and its arguments:

[example_begin][cmd {display message}] -help
[emph {  ->
NAME
      display message - Displays a simple message box
SYNOPSYS
      display message
            [-mtype <mtype>] :
               Message type, default: "Warning", choices: {Info Warning Error}
            [-font <font>] :
               Message text font, type: font, default: Arial 10 italic
            [-level <level>] :
               Message level, type: integer, range: 1..10
            [-fg <fg>] :
               Message color, type: color, default: black
            [-bg <bg>] :
               Background color, type: color
            [-no_border ] :
               Use a splash window style (no border)
            [-log_file <log_file>] :
               Optional message log file, type: file
            <text> :
               Multiple text lines to display, type: string
DESCRIPTION
      This procedure allows displaying a configurable message box.}][example_end]

[section "PROCEDURE CALL"]

The specified procedure can be called in many ways. The following listing shows some valid procedure calls:

[example_begin][cmd {display message}] "The document hasn't yet been saved!"
[emph {-> display message:
     mtype=Warning
     font=Arial 10 italic
     fg=black
     no_border=0
     text={The document hasn't yet been saved!}}]
 
[cmd {display message}] -fg red -bg black "Please save first the document"
[emph {-> display message:
     mtype=Warning
     font=Arial 10 italic
     fg=red
     bg=black
     no_border=0
     text={Please save first the document}}]
 
[cmd {display message}] -mtype Error -no_border "Why is here no border?"
[emph {-> display message:
     mtype=Error
     font=Arial 10 italic
     fg=black
     no_border=1
     text={Why is here no border?}}]
 
[cmd {display message}] -font {Courier 12} -level 10 \
   "Is there enough space?" "Reduce otherwise the font size!"
[emph {-> display message:
     mtype=Warning
     font=Courier 12
     level=10
     fg=black
     no_border=0
     text={Is there enough space?} {Reduce otherwise the font size!}}][example_end]
 
The next lines show how wrong arguments are recognized. The [arg text] argument that is mandatory is missing in the first procedure call:

[example_begin][cmd {display message}] -font {Courier 12}
[emph {  -> display message: Required argument is missing: text}][example_end]

Only known arguments are accepted:

[example_begin][cmd {display message}] -category warning Hello
[emph {  -> display message: Argument '-category' not known}][example_end]

Argument types are automatically checked and an error message is generated in case the argument value has not the expected type:

[example_begin][cmd {display message}] -fg MyColor "Hello"
[emph {  -> display message: Argument 'fg' requires type 'color'. \ 
            Provided value: 'MyColor'}][example_end]

Selection choices have to be respected ...

[example_begin][cmd {display message}] -mtype Fatal Hello
[emph {  -> display message: Argument (mtype) has to be one of the \ 
            following elements: Info, Warning, Error}][example_end]

... as well as valid value ranges:

[example_begin][cmd {display message}] -level 12 Hello
[emph {  -> display message: Argument (level) has to be between 1 and 10}][example_end]

[section "INTERACTIVE PROCEDURE CALLS"]

The most intuitive way to call the procedure is using an form that allows specifying all arguments interactively. This form will automatically be generated if the declared procedure is called with the [arg -interactive] flag.

[example_begin][cmd {display message}] -interactive[example_end]

The generated form contains for each argument a data entry widget that is adapted to the argument type. Check buttons are used to specify flags, radio boxes for tiny choice lists, disjoint list boxes for larger choice lists and files, directories, fonts and colors can be selected with dedicated browsers. 
[para]
After acknowledging the specified argument data via an OK button, the entered data are first validated, before the provided arguments are transformed into local variables and the procedure body is executed. In case the entered data are invalid, a message appears and the user can correct them until they are valid.
[para]
The procedure calls can optionally be logged in a variable. This is for example useful to get the command call lines of interactively called procedures.

[section "FLEXIBLE ARGUMENT DIALOG BOX"]

The form generator that creates in the previous example the argument dialog box for the interactive procedure call is also available for other purposes than for the definition of procedure arguments. TEPAM's provided argument dialog box allows crating complex parameter definition forms in a very efficient way.
[para]
The following example tries to illustrate the simplicity to create complex data entry forms. It creates an input mask that allows specifying a file to copy, a destination folder as well as a checkbox that allows specifying if an eventual existing file can be overwritten. Comfortable browsers can be used to select files and directories. And finally, the form offers also the possibility to accept and decline the selection. Here is the code snippet that is doing all this:

[example_begin][cmd tepam::argument_dialogbox] \
   [cmd -existingfile] {-label "Source file" -variable SourceFile} \
   [cmd -existingdirectory] {-label "Destination folder" -variable DestDir} \
   [cmd -checkbutton] {-label "Overwrite existing file" -variable Overwrite}[example_end]

The [cmd argument_dialogbox] returns [const ok] if the entered data are validated. It will return [const cancel] if the data entry has been canceled. After the validation of the entered data, the [cmd argument_dialogbox] defines all the specified variables with the entered data inside the calling context.
[para]
An [cmd argument_dialogbox] requires a pair of arguments for each variable that it has to handle. The first argument defines the entry widget type used to select the variable's value and the second one is a lists of attributes related to the variable and the entry widget.
[para]
Many entry widget types are available: Beside the simple generic entries, there are different kinds of list and combo boxes available, browsers for existing and new files and directories, check and radio boxes and buttons, as well as color and font pickers. If necessary, additional entry widgets can be defined.
[para]
The attribute list contains pairs of attribute names and attribute data. The primary attribute is [arg -variable] used to specify the variable in the calling context into which the entered data has to be stored. Another often used attribute is [arg -label] that allows adding a label to the data entry widget. Other attributes are available that allow specifying default values, the expected data types, valid data ranges, etc.
[para]
The next example of a more complex argument dialog box provides a good overview about the different available entry widget types and parameter attributes. The example contains also some formatting instructions like [arg -frame] and [arg -sep] which allows organizing the different entry widgets in frames and sections:
[example_begin]set ChoiceList {"Choice 1" "Choice 2" "Choice 3" "Choice 4" "Choice 5" "Choice 6"}
 
set Result [lb][cmd tepam::argument_dialogbox] \
   [cmd -title] "System configuration" \
   [cmd -context] test_1 \
   [cmd -frame] {-label "Entries"} \
      [cmd -entry] {-label Entry1 -variable Entry1} \
      [cmd -entry] {-label Entry2 -variable Entry2 -default "my default"} \
   [cmd -frame] {-label "Listbox & combobox"} \
      [cmd -listbox] {-label "Listbox, single selection" -variable Listbox1 \
                -choices {1 2 3 4 5 6 7 8} -default 1 -height 3} \
      [cmd -listbox] {-label "Listbox, multiple selection" -variable Listbox2 
                -choicevariable ChoiceList -default {"Choice 2" "Choice 3"} 
                -multiple_selection 1 -height 3} \
      [cmd -disjointlistbox] {-label "Disjoined listbox" -variable DisJntListbox 
                        -choicevariable ChoiceList \
                        -default {"Choice 3" "Choice 5"} -height 3} \
      [cmd -combobox] {-label "Combobox" -variable Combobox \
                 -choices {1 2 3 4 5 6 7 8} -default 3} \
   [cmd -frame] {-label "Checkbox, radiobox and checkbutton"} \
      [cmd -checkbox] {-label Checkbox -variable Checkbox 
                 -choices {bold italic underline} -choicelabels {Bold Italic Underline} \
                 -default italic} \
      [cmd -radiobox] {-label Radiobox -variable Radiobox 
                 -choices {bold italic underline} -choicelabels {Bold Italic Underline} \
                 -default underline} \
      [cmd -checkbutton] {-label CheckButton -variable Checkbutton -default 1} \
   [cmd -frame] {-label "Files & directories"} \
      [cmd -existingfile] {-label "Input file" -variable InputFile} \
      [cmd -file] {-label "Output file" -variable OutputFile} \
      [cmd -sep] {} \
      [cmd -existingdirectory] {-label "Input directory" -variable InputDirectory} \
      [cmd -directory] {-label "Output irectory" -variable OutputDirectory} \
   [cmd -frame] {-label "Colors and fonts"} \
      [cmd -color] {-label "Background color" -variable Color -default red} \
      [cmd -sep] {} \
      [cmd -font] {-label "Font" -variable Font -default {Courier 12 italic}}[rb][example_end]

The [cmd argument_dialogbox] defines all the specified variables with the entered data and returns [const ok] if the data have been validated via the Ok button. If the data entry is cancelled by activating the Cancel button, the [cmd argument_dialogbox] returns [const cancel].

[example_begin]if {$Result=="cancel"} {
   puts "Canceled"
} else { # $Result=="ok"
   puts "Arguments: "
   foreach Var {
      Entry1 Entry2
      Listbox1 Listbox2 DisJntListbox 
      Combobox Checkbox Radiobox Checkbutton
      InputFile OutputFile InputDirectory OutputDirectory
      Color Font
   } {
      puts "  $Var: '[lb]set $Var[rb]'"
   }
}
[emph {-> Arguments:
   Entry1: 'Hello, this is a trial'
   Entry2: 'my default'
   Listbox1: '1'
   Listbox2: '{Choice 2} {Choice 3}'
   DisJntListbox: '{Choice 3} {Choice 5}'
   Combobox: '3'
   Checkbox: 'italic'
   Radiobox: 'underline'
   Checkbutton: '1'
   InputFile: 'c:\tepam\in.txt'
   OutputFile: 'c:\tepam\out.txt'
   InputDirectory: 'c:\tepam\input'
   OutputDirectory: 'c:\tepam\output'
   Color: 'red'
   Font: 'Courier 12 italic'}][example_end]

[see_also tepam::argument_dialogbox(n) tepam::procedure(n)]

[keywords procedure arguments {argument validation} {argument integrity} subcommand {entry mask} {parameter entry form}]
[manpage_end]
